package drr.regulation.fca.ukemir.refit.trade.reports;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.reports.ReportFunction;
import drr.regulation.common.ReportableDelivery;
import drr.regulation.common.TransactionReportInstruction;
import drr.regulation.common.functions.GetReportableDelivery;
import drr.regulation.fca.ukemir.refit.trade.DeliveryReport;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;


@ImplementedBy(DeliveryRule.DeliveryRuleDefault.class)
public abstract class DeliveryRule implements ReportFunction<TransactionReportInstruction, List<? extends DeliveryReport>> {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected CurrencyOfThePriceTimeIntervalQuantityRule currencyOfThePriceTimeIntervalQuantityRule;
	@Inject protected DaysOfTheWeekRule daysOfTheWeekRule;
	@Inject protected DeliveryCapacityRule deliveryCapacityRule;
	@Inject protected DeliveryEndDateRule deliveryEndDateRule;
	@Inject protected DeliveryIntervalEndTimeRule deliveryIntervalEndTimeRule;
	@Inject protected DeliveryIntervalStartTimeRule deliveryIntervalStartTimeRule;
	@Inject protected DeliveryStartDateRule deliveryStartDateRule;
	@Inject protected DurationRule durationRule;
	@Inject protected GetReportableDelivery getReportableDelivery;
	@Inject protected PriceTimeIntervalQuantityRule priceTimeIntervalQuantityRule;
	@Inject protected QuantityUnitRule quantityUnitRule;

	/**
	* @param input 
	* @return output 
	*/
	@Override
	public List<? extends DeliveryReport> evaluate(TransactionReportInstruction input) {
		List<DeliveryReport.DeliveryReportBuilder> outputBuilder = doEvaluate(input);
		
		final List<? extends DeliveryReport> output;
		if (outputBuilder == null) {
			output = null;
		} else {
			output = outputBuilder.stream().map(DeliveryReport::build).collect(Collectors.toList());
			objectValidator.validate(DeliveryReport.class, output);
		}
		
		return output;
	}

	protected abstract List<DeliveryReport.DeliveryReportBuilder> doEvaluate(TransactionReportInstruction input);

	public static class DeliveryRuleDefault extends DeliveryRule {
		@Override
		protected List<DeliveryReport.DeliveryReportBuilder> doEvaluate(TransactionReportInstruction input) {
			List<DeliveryReport.DeliveryReportBuilder> output = new ArrayList<>();
			return assignOutput(output, input);
		}
		
		protected List<DeliveryReport.DeliveryReportBuilder> assignOutput(List<DeliveryReport.DeliveryReportBuilder> output, TransactionReportInstruction input) {
			final MapperC<ReportableDelivery> thenArg = MapperS.of(input)
				.mapSingleToList(item -> MapperC.<ReportableDelivery>of(getReportableDelivery.evaluate(item.get())));
			output = toBuilder(thenArg
				.mapItem(item -> MapperS.of(DeliveryReport.builder()
					.setDeliveryStartTime(deliveryIntervalStartTimeRule.evaluate(item.get()))
					.setDeliveryEndTime(deliveryIntervalEndTimeRule.evaluate(item.get()))
					.setDeliveryStartDate(deliveryStartDateRule.evaluate(item.get()))
					.setDeliveryEndDate(deliveryEndDateRule.evaluate(item.get()))
					.setDuration(durationRule.evaluate(item.get()))
					.setDaysOfTheWeek(daysOfTheWeekRule.evaluate(item.get()))
					.setDeliveryCapacity(deliveryCapacityRule.evaluate(item.get()))
					.setQuantityUnit(quantityUnitRule.evaluate(item.get()))
					.setPriceTimeIntervalQuantity(priceTimeIntervalQuantityRule.evaluate(item.get()))
					.setCurrencyOfThePriceTimeIntervalQuantity(currencyOfThePriceTimeIntervalQuantityRule.evaluate(item.get()))
					.build())).getMulti());
			
			return Optional.ofNullable(output)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
