package drr.regulation.asic.rewrite.trade.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.asic.rewrite.trade.ASICTransactionReport;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkNumber;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkString;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

public class ASICTransactionReportTypeFormatValidator implements Validator<ASICTransactionReport> {

	private List<ComparisonResult> getComparisonResults(ASICTransactionReport o) {
		return Lists.<ComparisonResult>newArrayList(
				checkString("uniqueTransactionIdentifier", o.getUniqueTransactionIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("utiProprietaryID", o.getUtiProprietaryID(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("secondaryTransactionIdentifier", o.getSecondaryTransactionIdentifier(), 1, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("assetClass", o.getAssetClass(), 1, of(4), empty()), 
				checkString("reportingEntity", o.getReportingEntity(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty1", o.getCounterparty1(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty2", o.getCounterparty2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("counterparty2Name", o.getCounterparty2Name(), 1, of(150), empty()), 
				checkString("beneficiary1", o.getBeneficiary1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("broker", o.getBroker(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("cleared", o.getCleared(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("centralCounterparty", o.getCentralCounterparty(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("clearingMember", o.getClearingMember(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("platformIdentifier", o.getPlatformIdentifier(), 1, of(4), empty()), 
				checkNumber("notionalLeg1", o.getNotionalLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalLeg2", o.getNotionalLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("totalNotionalQuantityLeg1", o.getTotalNotionalQuantityLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("totalNotionalQuantityLeg2", o.getTotalNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalQuantityLeg1", o.getNotionalQuantityLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalQuantityLeg2", o.getNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("callAmount", o.getCallAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("putAmount", o.getPutAmount(), of(25), of(5), empty(), empty()), 
				checkString("quantityUnitOfMeasureLeg1", o.getQuantityUnitOfMeasureLeg1(), 1, of(4), empty()), 
				checkString("quantityUnitOfMeasureLeg2", o.getQuantityUnitOfMeasureLeg2(), 1, of(4), empty()), 
				checkString("priceUnitOfMeasure", o.getPriceUnitOfMeasure(), 1, of(4), empty()), 
				checkNumber("fixedRateLeg1", o.getFixedRateLeg1(), of(11), of(10), empty(), empty()), 
				checkNumber("fixedRateLeg2", o.getFixedRateLeg2(), of(11), of(10), empty(), empty()), 
				checkString("spreadNotationLeg1", o.getSpreadNotationLeg1(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("spreadNotationLeg2", o.getSpreadNotationLeg2(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkNumber("optionPremiumAmount", o.getOptionPremiumAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("exchangeRate", o.getExchangeRate(), of(18), of(13), empty(), empty()), 
				checkString("exchangeRateBasis", o.getExchangeRateBasis(), 7, of(7), of(Pattern.compile("[A-Z]{3,3}/[A-Z]{3,3}"))), 
				checkString("paymentFrequencyPeriodLeg1", o.getPaymentFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("paymentFrequencyPeriodLeg2", o.getPaymentFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkNumber("paymentFrequencyPeriodMultiplierLeg1", o.getPaymentFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("paymentFrequencyPeriodMultiplierLeg2", o.getPaymentFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkString("underlyingIdentification", o.getUnderlyingIdentification(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("underlyingIdentificationType", o.getUnderlyingIdentificationType(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("underlyingIdOther", o.getUnderlyingIdOther(), 1, of(210), of(Pattern.compile("[a-zA-Z0-9]{1,210}"))), 
				checkString("indicatorOfTheFloatingRateLeg2", o.getIndicatorOfTheFloatingRateLeg2(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("floatingRateReferencePeriodOfLeg2TimePeriod", o.getFloatingRateReferencePeriodOfLeg2TimePeriod(), 1, of(4), empty()), 
				checkNumber("floatingRateReferencePeriodOfLeg2Multiplier", o.getFloatingRateReferencePeriodOfLeg2Multiplier(), of(3), of(0), empty(), empty()), 
				checkNumber("cdSIndexAttachmentPoint", o.getCdSIndexAttachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("cdSIndexDetachmentPoint", o.getCdSIndexDetachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("indexFactor", o.getIndexFactor(), of(11), of(10), empty(), empty()), 
				checkString("packageIdentifier", o.getPackageIdentifier(), 1, of(35), of(Pattern.compile("[a-zA-Z0-9]{1,35}"))), 
				checkString("priorUTI", o.getPriorUTI(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("priorUtiProprietary", o.getPriorUtiProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("eventIdentifier", o.getEventIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("newDerivativeTradeRepository", o.getNewDerivativeTradeRepository(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkNumber("valuationAmount", o.getValuationAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("delta", o.getDelta(), of(25), of(5), empty(), empty()), 
				checkString("collateralPortfolioCodeInitialMargin", o.getCollateralPortfolioCodeInitialMargin(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("collateralPortfolioCodeVariationMargin", o.getCollateralPortfolioCodeVariationMargin(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("executionAgentOfCounterparty2DTCC", o.getExecutionAgentOfCounterparty2DTCC(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}")))
			);
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, ASICTransactionReport o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("ASICTransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "ASICTransactionReport", path, "", res.getError());
				}
				return success("ASICTransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "ASICTransactionReport", path, "");
			})
			.collect(toList());
	}

}
