package drr.regulation.cftc.rewrite.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.cftc.rewrite.CFTCPart45TransactionReport;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkNumber;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkString;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

public class CFTCPart45TransactionReportTypeFormatValidator implements Validator<CFTCPart45TransactionReport> {

	private List<ComparisonResult> getComparisonResults(CFTCPart45TransactionReport o) {
		return Lists.<ComparisonResult>newArrayList(
				checkString("cleared", o.getCleared(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("centralCounterparty", o.getCentralCounterparty(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("clearingAccountOrigin", o.getClearingAccountOrigin(), 1, of(4), empty()), 
				checkString("clearingMember", o.getClearingMember(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("originalSwapUTI", o.getOriginalSwapUTI(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("originalSwapSDRIdentifier", o.getOriginalSwapSDRIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty1", o.getCounterparty1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("counterparty2", o.getCounterparty2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("counterparty2IdentifierSource", o.getCounterparty2IdentifierSource(), 1, of(4), empty()), 
				checkString("buyerIdentifier", o.getBuyerIdentifier(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("sellerIdentifier", o.getSellerIdentifier(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("payerIdentifierLeg1", o.getPayerIdentifierLeg1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("payerIdentifierLeg2", o.getPayerIdentifierLeg2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("receiverIdentifierLeg1", o.getReceiverIdentifierLeg1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("receiverIdentifierLeg2", o.getReceiverIdentifierLeg2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("submitterIdentifier", o.getSubmitterIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("actionType", o.getActionType(), 1, of(4), empty()), 
				checkString("eventType", o.getEventType(), 1, of(4), empty()), 
				checkString("eventIdentifier", o.getEventIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkNumber("notionalAmountLeg1", o.getNotionalAmountLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalAmountLeg2", o.getNotionalAmountLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("callAmount", o.getCallAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("putAmount", o.getPutAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalQuantityLeg1", o.getNotionalQuantityLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalQuantityLeg2", o.getNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkString("quantityFrequencyLeg1", o.getQuantityFrequencyLeg1(), 1, of(4), empty()), 
				checkString("quantityFrequencyLeg2", o.getQuantityFrequencyLeg2(), 1, of(4), empty()), 
				checkNumber("quantityFrequencyMultiplierLeg1", o.getQuantityFrequencyMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("quantityFrequencyMultiplierLeg2", o.getQuantityFrequencyMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkString("quantityUnitOfMeasureLeg1", o.getQuantityUnitOfMeasureLeg1(), 1, of(4), empty()), 
				checkString("quantityUnitOfMeasureLeg2", o.getQuantityUnitOfMeasureLeg2(), 1, of(4), empty()), 
				checkNumber("totalNotionalQuantityLeg1", o.getTotalNotionalQuantityLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("totalNotionalQuantityLeg2", o.getTotalNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkString("packageTransactionPriceNotation", o.getPackageTransactionPriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("packageTransactionSpreadNotation", o.getPackageTransactionSpreadNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("floatingRateResetFrequencyPeriodLeg1", o.getFloatingRateResetFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("floatingRateResetFrequencyPeriodLeg2", o.getFloatingRateResetFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierLeg1", o.getFloatingRateResetFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierLeg2", o.getFloatingRateResetFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkString("fixedRatePaymentFrequencyPeriodLeg1", o.getFixedRatePaymentFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("fixedRatePaymentFrequencyPeriodLeg2", o.getFixedRatePaymentFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkString("floatingRatePaymentFrequencyPeriodLeg1", o.getFloatingRatePaymentFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("floatingRatePaymentFrequencyPeriodLeg2", o.getFloatingRatePaymentFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierLeg1", o.getFixedRatePaymentFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierLeg2", o.getFixedRatePaymentFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierLeg1", o.getFloatingRatePaymentFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierLeg2", o.getFloatingRatePaymentFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("exchangeRateLeg1", o.getExchangeRateLeg1(), of(18), of(13), empty(), empty()), 
				checkString("exchangeRateBasis", o.getExchangeRateBasis(), 7, of(7), of(Pattern.compile("[A-Z]{3,3}/[A-Z]{3,3}"))), 
				checkNumber("fixedRateLeg1", o.getFixedRateLeg1(), of(11), of(10), empty(), empty()), 
				checkNumber("fixedRateLeg2", o.getFixedRateLeg2(), of(11), of(10), empty(), empty()), 
				checkNumber("price", o.getPrice(), of(11), of(10), empty(), empty()), 
				checkString("priceNotation", o.getPriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("priceUnitOfMeasure", o.getPriceUnitOfMeasure(), 1, of(4), empty()), 
				checkString("spreadNotationLeg1", o.getSpreadNotationLeg1(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("spreadNotationLeg2", o.getSpreadNotationLeg2(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("strikePriceNotation", o.getStrikePriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkNumber("optionPremiumAmount", o.getOptionPremiumAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("cDSIndexAttachmentPoint", o.getCDSIndexAttachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("cDSIndexDetachmentPoint", o.getCDSIndexDetachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("indexFactor", o.getIndexFactor(), of(11), of(10), empty(), empty()), 
				checkString("embeddedOptionType", o.getEmbeddedOptionType(), 1, of(4), empty()), 
				checkString("settlementLocationLeg1", o.getSettlementLocationLeg1(), 1, of(2), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("settlementLocationLeg2", o.getSettlementLocationLeg2(), 1, of(2), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("allocationIndicator", o.getAllocationIndicator(), 1, of(4), empty()), 
				checkString("platformIdentifier", o.getPlatformIdentifier(), 1, of(4), empty()), 
				checkString("priorUSI", o.getPriorUSI(), 1, of(42), of(Pattern.compile("[a-zA-Z0-9]{1,42}"))), 
				checkString("priorUTI", o.getPriorUTI(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("uniqueSwapIdentifier", o.getUniqueSwapIdentifier(), 1, of(42), of(Pattern.compile("[a-zA-Z0-9]{1,42}"))), 
				checkString("uniqueTransactionIdentifier", o.getUniqueTransactionIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("jurisdiction", o.getJurisdiction(), 1, of(4), empty()), 
				checkString("newSDRIdentifier", o.getNewSDRIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("initialMarginCollateralPortfolioCode", o.getInitialMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("variationMarginCollateralPortfolioCode", o.getVariationMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}")))
			);
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, CFTCPart45TransactionReport o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("CFTCPart45TransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "CFTCPart45TransactionReport", path, "", res.getError());
				}
				return success("CFTCPart45TransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "CFTCPart45TransactionReport", path, "");
			})
			.collect(toList());
	}

}
