package drr.regulation.common;

import com.rosetta.model.lib.annotations.RosettaEnum;
import com.rosetta.model.lib.annotations.RosettaEnumValue;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * Whether the financial instrument is admitted to trading, or traded on a trading venue or for which a request for admission was made
 * @version 5.52.0
 */
@RosettaEnum("AdmittedToTradingVenue")
public enum AdmittedToTradingVenue {

	/**
	 * The financial instrument is admitted to trading, or traded on a trading venue or a request for admission was made
	 */
	@RosettaEnumValue(value = "Admitted") 
	ADMITTED("Admitted", null),
	
	/**
	 * The financial instrument is not admitted to trading or traded on a trading venue or no request for admission has been made
	 */
	@RosettaEnumValue(value = "NotAdmitted") 
	NOT_ADMITTED("NotAdmitted", null),
	
	/**
	 * If the reporting party cannot determine whether the instrument is listed or not, as per jurisdictional requirements
	 */
	@RosettaEnumValue(value = "Unknown") 
	UNKNOWN("Unknown", null)
;
	private static Map<String, AdmittedToTradingVenue> values;
	static {
        Map<String, AdmittedToTradingVenue> map = new ConcurrentHashMap<>();
		for (AdmittedToTradingVenue instance : AdmittedToTradingVenue.values()) {
			map.put(instance.toDisplayString(), instance);
		}
		values = Collections.unmodifiableMap(map);
    }

	private final String rosettaName;
	private final String displayName;

	AdmittedToTradingVenue(String rosettaName, String displayName) {
		this.rosettaName = rosettaName;
		this.displayName = displayName;
	}

	public static AdmittedToTradingVenue fromDisplayName(String name) {
		AdmittedToTradingVenue value = values.get(name);
		if (value == null) {
			throw new IllegalArgumentException("No enum constant with display name \"" + name + "\".");
		}
		return value;
	}

	@Override
	public String toString() {
		return toDisplayString();
	}

	public String toDisplayString() {
		return displayName != null ?  displayName : rosettaName;
	}
}
