# pylint: disable=line-too-long, invalid-name, missing-function-docstring
# pylint: disable=bad-indentation, trailing-whitespace, superfluous-parens
# pylint: disable=wrong-import-position, unused-import, unused-wildcard-import
# pylint: disable=wildcard-import, wrong-import-order, missing-class-docstring
# pylint: disable=missing-module-docstring
from __future__ import annotations
from typing import List, Optional
import datetime
import inspect
from decimal import Decimal
from pydantic import Field
from rosetta.runtime.utils import (
    BaseDataClass, rosetta_condition, rosetta_resolve_attr, rosetta_resolve_deep_attr
)
from rosetta.runtime.utils import *

__all__ = ['Payout']


class Payout(BaseDataClass):
    """
    A class to represent the set of future cashflow methodologies in the form of specific payout class(es) that can be associated for the purpose of specifying a financial product. For example, two interest rate payouts can be combined to specify an interest rate swap, or one interest rate payout can be combined with a credit default payout to specify a credit default swap.
    """
    interestRatePayout: List[cdm.product.asset.InterestRatePayout.InterestRatePayout] = Field([], description="All of the terms necessary to define and calculate a cash flow based on a fixed, a floating or an inflation index rate. The interest rate payout can be applied to interest rate swaps and FRA (which both have two associated interest rate payouts), credit default swaps (to represent the fee leg when subject to periodic payments) and equity swaps (to represent the funding leg).")
    """
    All of the terms necessary to define and calculate a cash flow based on a fixed, a floating or an inflation index rate. The interest rate payout can be applied to interest rate swaps and FRA (which both have two associated interest rate payouts), credit default swaps (to represent the fee leg when subject to periodic payments) and equity swaps (to represent the funding leg).
    """
    creditDefaultPayout: Optional[cdm.product.asset.CreditDefaultPayout.CreditDefaultPayout] = Field(None, description="The credit default payout, which provides the details necessary for determining when a credit payout will be triggered as well as the parameters for calculating the payout and the settlement terms.")
    """
    The credit default payout, which provides the details necessary for determining when a credit payout will be triggered as well as the parameters for calculating the payout and the settlement terms.
    """
    optionPayout: List[cdm.product.template.OptionPayout.OptionPayout] = Field([], description="The option payout.")
    """
    The option payout.
    """
    commodityPayout: List[cdm.product.asset.CommodityPayout.CommodityPayout] = Field([], description="Defines the payout for the floating leg of a Commodity Swap.")
    """
    Defines the payout for the floating leg of a Commodity Swap.
    """
    forwardPayout: List[cdm.product.template.ForwardPayout.ForwardPayout] = Field([], description="Represents a forward settling payout. The 'Underlier' attribute captures the underlying payout, which is settled according to the 'SettlementTerms' attribute. Both FX Spot and FX Forward should use this component.")
    """
    Represents a forward settling payout. The 'Underlier' attribute captures the underlying payout, which is settled according to the 'SettlementTerms' attribute. Both FX Spot and FX Forward should use this component.
    """
    fixedPricePayout: List[cdm.product.template.FixedPricePayout.FixedPricePayout] = Field([], description="Defines a payout in which one or more payouts are defined as a fixed price.")
    """
    Defines a payout in which one or more payouts are defined as a fixed price.
    """
    securityPayout: List[cdm.product.template.SecurityPayout.SecurityPayout] = Field([], description="The security payout when the product involves some form of securities, such as collateral in a securities financing transaction")
    """
    The security payout when the product involves some form of securities, such as collateral in a securities financing transaction
    """
    cashflow: List[cdm.product.common.settlement.Cashflow.Cashflow] = Field([], description="A cashflow between the parties to the trade. For interest rate and equity products, this corresponds to the FpML additionalPayment element. For credit default swaps, this corresponds to the FpML initialPayment element and the singlePayment element of the fee leg. For option products, it represents the FpML premium element.")
    """
    A cashflow between the parties to the trade. For interest rate and equity products, this corresponds to the FpML additionalPayment element. For credit default swaps, this corresponds to the FpML initialPayment element and the singlePayment element of the fee leg. For option products, it represents the FpML premium element.
    """
    performancePayout: List[cdm.product.template.PerformancePayout.PerformancePayout] = Field([], description="The performance payout, which encompasses the equity price returns, dividend returns, volatility return, variance return and correlation provisions.")
    """
    The performance payout, which encompasses the equity price returns, dividend returns, volatility return, variance return and correlation provisions.
    """
    assetPayout: List[cdm.product.template.AssetPayout.AssetPayout] = Field([], description="The security payout when the product involves some form of securities, such as collateral in a securities financing transaction")
    """
    The security payout when the product involves some form of securities, such as collateral in a securities financing transaction
    """
    
    @rosetta_condition
    def condition_0_ReturnType_Total_Requires_Dividends(self):
        """
        A total return implies both a price and a dividend return
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "performancePayout"), "returnTerms"), "dividendReturnTerms"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn(all_elements(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "performancePayout"), "returnTerms"), "priceReturnTerms"), "returnType"), "=", rosetta_resolve_attr(ReturnTypeEnum, "TOTAL")), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_1_LastRegularPaymentDate(self):
        """
        FpML specifies that lastRegularPaymentDate must only be included if there is a final stub. As part of the CDM, this data rule has been adjusted to specify that it only applies to interest rate swaps, as the credit derivatives products can have a specified lastRegularPaymentDate while the stub is typically not applicable to those.
        """
        item = self
        def _then_fn0():
            return all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2)
        
        def _else_fn0():
            return True
        
        return if_cond_fn((rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"), "lastRegularPaymentDate")) and rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "stubPeriod"), "finalStub"))), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_2_PayRelativeTo(self):
        """
        FpML specifies a required payRelativeTo element as part of the PaymentDates. As standardized CDS don't have such payRelativeTo provision, the cardinality has been relaxed as part of the CDM. This data rule specifies that if the product has two interest rate streams, this provision must exist.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"), "payRelativeTo"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn((all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2) and rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"))), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_3_PaymentDatesAdjustments(self):
        """
        FpML specifies a required paymentDatesAdjustments element as part of the PaymentDates. As standardized CDS don't have such paymentDatesAdjustments provision, the cardinality has been relaxed as part of the CDM. This data rule specifies that if the product has two interest rate streams, this provision must exist.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"), "paymentDatesAdjustments"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn((all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2) and rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"))), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_4_PaymentFrequency(self):
        """
        FpML specifies a required paymentFrequency element as part of the PaymentDates. As standardized CDS may not have such paymentFrequency provision, the cardinality has been relaxed as part of the CDM. This data rule specifies that if the product has two interest rate streams, this provision must exist.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"), "paymentFrequency"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn((all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2) and rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "paymentDates"))), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_5_Quantity(self):
        """
        When there is an OptionPayout the quantity can be expressed as part of the payoutQuantity, or as part of the underlier in the case of a Swaption.  For all other payouts that extend PayoutBase the payoutQuantity is a mandatory attribute.
        """
        item = self
        def _then_fn0():
            return ((rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "optionPayout"), "priceQuantity")) or all_elements(rosetta_count(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "optionPayout"), "underlier"), "contractualProduct"), "economicTerms"), "payout"), "interestRatePayout")), "=", 2)) or rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "optionPayout"), "underlier"), "foreignExchange")))
        
        def _else_fn0():
            return True
        
        return if_cond_fn(rosetta_attr_exists(rosetta_resolve_attr(self, "optionPayout")), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_6_DayCountFraction(self):
        """
        FpML specifies a required dayCountFraction element as part of the swapStream/calculationPeriodAmount/calculation. As standardized CDS don't have such specified day count fraction, the cardinality has been relaxed as part of the CDM. This data rule specifies that if the product has two interest rate streams, this provision must exist.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "dayCountFraction"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn(all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_7_PaymentDates(self):
        """
        FpML specifies a required paymentDates element as part of the swapStream. As standardized CDS may not have specified payment dates, the cardinality has been relaxed as part of the CDM. This data rule specifies that if the product has two interest rate streams, this provision must exist.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "dayCountFraction"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn(all_elements(rosetta_count(rosetta_resolve_attr(self, "interestRatePayout")), "=", 2), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_8_MarketPrice(self):
        """
        FpML specifies that marketFixedRate and marketPrice only have meaning in a credit index trade
        """
        item = self
        def _then_fn0():
            return ((not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "creditDefaultPayout"), "transactedPrice"), "marketFixedRate"))) and (not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "creditDefaultPayout"), "transactedPrice"), "marketPrice"))))
        
        def _else_fn0():
            return True
        
        return if_cond_fn((not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "creditDefaultPayout"), "generalTerms"), "indexReferenceInformation"))), _then_fn0, _else_fn0)
    
    @rosetta_condition
    def condition_9_NotionalResetOnPerformancePayout(self):
        """
        Notional reset only applies to return swaps, and therefore can only exist on an performancePayout.
        """
        item = self
        return ((((not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "interestRatePayout"), "priceQuantity"), "reset"))) and (not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "creditDefaultPayout"), "priceQuantity"), "reset")))) and (not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "optionPayout"), "priceQuantity"), "reset")))) and (not rosetta_attr_exists(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "cashflow"), "priceQuantity"), "reset"))))
    
    @rosetta_condition
    def condition_10_NotionalResetInterestRatePayoutExists(self):
        """
        As the performancePayout->payoutQuantity->reset attribute applies to return swaps, the interestRatePayout needs to be present alongside it.
        """
        item = self
        def _then_fn0():
            return rosetta_attr_exists(rosetta_resolve_attr(self, "interestRatePayout"))
        
        def _else_fn0():
            return True
        
        return if_cond_fn(contains(rosetta_resolve_attr(rosetta_resolve_attr(rosetta_resolve_attr(self, "performancePayout"), "priceQuantity"), "reset"), True), _then_fn0, _else_fn0)

import cdm 
import cdm.product.asset.InterestRatePayout
import cdm.product.asset.CreditDefaultPayout
import cdm.product.template.OptionPayout
import cdm.product.asset.CommodityPayout
import cdm.product.template.ForwardPayout
import cdm.product.template.FixedPricePayout
import cdm.product.template.SecurityPayout
import cdm.product.common.settlement.Cashflow
import cdm.product.template.PerformancePayout
import cdm.product.template.AssetPayout
from cdm.product.asset.ReturnTypeEnum import ReturnTypeEnum
