package drr.regulation.common;

import cdm.event.common.PositionIdentifier;
import cdm.observable.asset.PriceSchedule;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import com.rosetta.util.ListEquals;
import drr.enrichment.common.EnrichmentData;
import drr.regulation.common.meta.ReportableInformationMeta;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Specifies the relevant regulators or other supervisory bodies the transaction is relevant for, and regime specific information required relating to the transaction and the parties.
 * @version 5.52.1
 */
@RosettaDataType(value="ReportableInformation", builder=ReportableInformation.ReportableInformationBuilderImpl.class, version="5.52.1")
@RuneDataType(value="ReportableInformation", model="drr", builder=ReportableInformation.ReportableInformationBuilderImpl.class, version="5.52.1")
public interface ReportableInformation extends RosettaModelObject {

	ReportableInformationMeta metaData = new ReportableInformationMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Specifies the method for confirmation of the transacion.
	 */
	ConfirmationMethodEnum getConfirmationMethod();
	/**
	 * Specifies the type of venue where a trade could be executed.
	 */
	ExecutionVenueTypeEnum getExecutionVenueType();
	/**
	 * Specifies whether the contract was entered into as an intragroup transaction, defined in Article 3 of Regulation (EU) No 648/2012.
	 */
	Boolean getIntragroup();
	/**
	 * Specifies whether the sender of this trade considers it to be a large notional trade or block trade for reporting purposes, and thus eligible for delayed public reporting. Normally this will only be applicable for off-facility trades.
	 */
	Boolean getLargeSizeTrade();
	/**
	 * Specifies party specific information required for reporting of the transaction
	 */
	List<? extends PartyInformation> getPartyInformation();
	/**
	 * Indicator of whether the swap was executed anonymously on a SEF or DCM. When set to true, counterparty details will be masked on reporting.
	 */
	Boolean getSefOrDcmAnonymousIndicator();
	/**
	 * Date and time a transaction was originally executed, resulting in the generation of a new UTI. This data element remains unchanged throughout the life of the UTI.
	 */
	ZonedDateTime getOriginalExecutionTimestamp();
	EnrichmentData getEnrichment();
	/**
	 * Indicator whether the derivative is based on crypto-assets.
	 */
	Boolean getCryptoBased();
	/**
	 * Last Available Spot Price of the underlying asset. This price is only needed for a regulatory purpose, that being the reporting of the notional amount for commodity basis products
	 */
	PriceSchedule getUnderlyingAssetLastAvailableSpotPrice();
	List<? extends TransactionInformation> getTransactionInformation();
	/**
	 * Specifies the regulatory lifecycle event action for the ReportableEvent
	 */
	ReportableActionEnum getReportableAction();
	/**
	 * The UTI of the position in which a derivative transaction is included.
	 */
	PositionIdentifier getSubsequentPositionUTI();
	/**
	 * The prior position UTI (eg in case of a corporate action of type StockIdentifierChange).
	 */
	PositionIdentifier getCounterpartyPositionPriorUTI();
	/**
	 * Information pertaining to Custom Basket
	 */
	CustomBasket getCustomBasket();
	/**
	 * Date as per the contract, by which all transfer of cash or assets should take place and the counterparties should no longer have any outstanding obligations to each other under that contract.
	 */
	Date getFinalContractualSettlementDate();

	/*********************** Build Methods  ***********************/
	ReportableInformation build();
	
	ReportableInformation.ReportableInformationBuilder toBuilder();
	
	static ReportableInformation.ReportableInformationBuilder builder() {
		return new ReportableInformation.ReportableInformationBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportableInformation> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportableInformation> getType() {
		return ReportableInformation.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
		processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
		processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
		processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
		processRosetta(path.newSubPath("partyInformation"), processor, PartyInformation.class, getPartyInformation());
		processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
		processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
		processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.class, getEnrichment());
		processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
		processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.class, getUnderlyingAssetLastAvailableSpotPrice());
		processRosetta(path.newSubPath("transactionInformation"), processor, TransactionInformation.class, getTransactionInformation());
		processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
		processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.class, getSubsequentPositionUTI());
		processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.class, getCounterpartyPositionPriorUTI());
		processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.class, getCustomBasket());
		processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportableInformationBuilder extends ReportableInformation, RosettaModelObjectBuilder {
		PartyInformation.PartyInformationBuilder getOrCreatePartyInformation(int index);
		@Override
		List<? extends PartyInformation.PartyInformationBuilder> getPartyInformation();
		EnrichmentData.EnrichmentDataBuilder getOrCreateEnrichment();
		@Override
		EnrichmentData.EnrichmentDataBuilder getEnrichment();
		PriceSchedule.PriceScheduleBuilder getOrCreateUnderlyingAssetLastAvailableSpotPrice();
		@Override
		PriceSchedule.PriceScheduleBuilder getUnderlyingAssetLastAvailableSpotPrice();
		TransactionInformation.TransactionInformationBuilder getOrCreateTransactionInformation(int index);
		@Override
		List<? extends TransactionInformation.TransactionInformationBuilder> getTransactionInformation();
		PositionIdentifier.PositionIdentifierBuilder getOrCreateSubsequentPositionUTI();
		@Override
		PositionIdentifier.PositionIdentifierBuilder getSubsequentPositionUTI();
		PositionIdentifier.PositionIdentifierBuilder getOrCreateCounterpartyPositionPriorUTI();
		@Override
		PositionIdentifier.PositionIdentifierBuilder getCounterpartyPositionPriorUTI();
		CustomBasket.CustomBasketBuilder getOrCreateCustomBasket();
		@Override
		CustomBasket.CustomBasketBuilder getCustomBasket();
		ReportableInformation.ReportableInformationBuilder setConfirmationMethod(ConfirmationMethodEnum confirmationMethod);
		ReportableInformation.ReportableInformationBuilder setExecutionVenueType(ExecutionVenueTypeEnum executionVenueType);
		ReportableInformation.ReportableInformationBuilder setIntragroup(Boolean intragroup);
		ReportableInformation.ReportableInformationBuilder setLargeSizeTrade(Boolean largeSizeTrade);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation partyInformation);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation partyInformation, int idx);
		ReportableInformation.ReportableInformationBuilder addPartyInformation(List<? extends PartyInformation> partyInformation);
		ReportableInformation.ReportableInformationBuilder setPartyInformation(List<? extends PartyInformation> partyInformation);
		ReportableInformation.ReportableInformationBuilder setSefOrDcmAnonymousIndicator(Boolean sefOrDcmAnonymousIndicator);
		ReportableInformation.ReportableInformationBuilder setOriginalExecutionTimestamp(ZonedDateTime originalExecutionTimestamp);
		ReportableInformation.ReportableInformationBuilder setEnrichment(EnrichmentData enrichment);
		ReportableInformation.ReportableInformationBuilder setCryptoBased(Boolean cryptoBased);
		ReportableInformation.ReportableInformationBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule underlyingAssetLastAvailableSpotPrice);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation transactionInformation);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation transactionInformation, int idx);
		ReportableInformation.ReportableInformationBuilder addTransactionInformation(List<? extends TransactionInformation> transactionInformation);
		ReportableInformation.ReportableInformationBuilder setTransactionInformation(List<? extends TransactionInformation> transactionInformation);
		ReportableInformation.ReportableInformationBuilder setReportableAction(ReportableActionEnum reportableAction);
		ReportableInformation.ReportableInformationBuilder setSubsequentPositionUTI(PositionIdentifier subsequentPositionUTI);
		ReportableInformation.ReportableInformationBuilder setCounterpartyPositionPriorUTI(PositionIdentifier counterpartyPositionPriorUTI);
		ReportableInformation.ReportableInformationBuilder setCustomBasket(CustomBasket customBasket);
		ReportableInformation.ReportableInformationBuilder setFinalContractualSettlementDate(Date finalContractualSettlementDate);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
			processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
			processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
			processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
			processRosetta(path.newSubPath("partyInformation"), processor, PartyInformation.PartyInformationBuilder.class, getPartyInformation());
			processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
			processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
			processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.EnrichmentDataBuilder.class, getEnrichment());
			processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
			processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.PriceScheduleBuilder.class, getUnderlyingAssetLastAvailableSpotPrice());
			processRosetta(path.newSubPath("transactionInformation"), processor, TransactionInformation.TransactionInformationBuilder.class, getTransactionInformation());
			processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
			processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getSubsequentPositionUTI());
			processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getCounterpartyPositionPriorUTI());
			processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.CustomBasketBuilder.class, getCustomBasket());
			processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
		}
		

		ReportableInformation.ReportableInformationBuilder prune();
	}

	/*********************** Immutable Implementation of ReportableInformation  ***********************/
	class ReportableInformationImpl implements ReportableInformation {
		private final ConfirmationMethodEnum confirmationMethod;
		private final ExecutionVenueTypeEnum executionVenueType;
		private final Boolean intragroup;
		private final Boolean largeSizeTrade;
		private final List<? extends PartyInformation> partyInformation;
		private final Boolean sefOrDcmAnonymousIndicator;
		private final ZonedDateTime originalExecutionTimestamp;
		private final EnrichmentData enrichment;
		private final Boolean cryptoBased;
		private final PriceSchedule underlyingAssetLastAvailableSpotPrice;
		private final List<? extends TransactionInformation> transactionInformation;
		private final ReportableActionEnum reportableAction;
		private final PositionIdentifier subsequentPositionUTI;
		private final PositionIdentifier counterpartyPositionPriorUTI;
		private final CustomBasket customBasket;
		private final Date finalContractualSettlementDate;
		
		protected ReportableInformationImpl(ReportableInformation.ReportableInformationBuilder builder) {
			this.confirmationMethod = builder.getConfirmationMethod();
			this.executionVenueType = builder.getExecutionVenueType();
			this.intragroup = builder.getIntragroup();
			this.largeSizeTrade = builder.getLargeSizeTrade();
			this.partyInformation = ofNullable(builder.getPartyInformation()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.sefOrDcmAnonymousIndicator = builder.getSefOrDcmAnonymousIndicator();
			this.originalExecutionTimestamp = builder.getOriginalExecutionTimestamp();
			this.enrichment = ofNullable(builder.getEnrichment()).map(f->f.build()).orElse(null);
			this.cryptoBased = builder.getCryptoBased();
			this.underlyingAssetLastAvailableSpotPrice = ofNullable(builder.getUnderlyingAssetLastAvailableSpotPrice()).map(f->f.build()).orElse(null);
			this.transactionInformation = ofNullable(builder.getTransactionInformation()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.reportableAction = builder.getReportableAction();
			this.subsequentPositionUTI = ofNullable(builder.getSubsequentPositionUTI()).map(f->f.build()).orElse(null);
			this.counterpartyPositionPriorUTI = ofNullable(builder.getCounterpartyPositionPriorUTI()).map(f->f.build()).orElse(null);
			this.customBasket = ofNullable(builder.getCustomBasket()).map(f->f.build()).orElse(null);
			this.finalContractualSettlementDate = builder.getFinalContractualSettlementDate();
		}
		
		@Override
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		public ConfirmationMethodEnum getConfirmationMethod() {
			return confirmationMethod;
		}
		
		@Override
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		public ExecutionVenueTypeEnum getExecutionVenueType() {
			return executionVenueType;
		}
		
		@Override
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		public Boolean getIntragroup() {
			return intragroup;
		}
		
		@Override
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		public Boolean getLargeSizeTrade() {
			return largeSizeTrade;
		}
		
		@Override
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		public List<? extends PartyInformation> getPartyInformation() {
			return partyInformation;
		}
		
		@Override
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		public Boolean getSefOrDcmAnonymousIndicator() {
			return sefOrDcmAnonymousIndicator;
		}
		
		@Override
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		public ZonedDateTime getOriginalExecutionTimestamp() {
			return originalExecutionTimestamp;
		}
		
		@Override
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		public EnrichmentData getEnrichment() {
			return enrichment;
		}
		
		@Override
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		public Boolean getCryptoBased() {
			return cryptoBased;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		public PriceSchedule getUnderlyingAssetLastAvailableSpotPrice() {
			return underlyingAssetLastAvailableSpotPrice;
		}
		
		@Override
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		public List<? extends TransactionInformation> getTransactionInformation() {
			return transactionInformation;
		}
		
		@Override
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		public ReportableActionEnum getReportableAction() {
			return reportableAction;
		}
		
		@Override
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		public PositionIdentifier getSubsequentPositionUTI() {
			return subsequentPositionUTI;
		}
		
		@Override
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		public PositionIdentifier getCounterpartyPositionPriorUTI() {
			return counterpartyPositionPriorUTI;
		}
		
		@Override
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		public CustomBasket getCustomBasket() {
			return customBasket;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public Date getFinalContractualSettlementDate() {
			return finalContractualSettlementDate;
		}
		
		@Override
		public ReportableInformation build() {
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder toBuilder() {
			ReportableInformation.ReportableInformationBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportableInformation.ReportableInformationBuilder builder) {
			ofNullable(getConfirmationMethod()).ifPresent(builder::setConfirmationMethod);
			ofNullable(getExecutionVenueType()).ifPresent(builder::setExecutionVenueType);
			ofNullable(getIntragroup()).ifPresent(builder::setIntragroup);
			ofNullable(getLargeSizeTrade()).ifPresent(builder::setLargeSizeTrade);
			ofNullable(getPartyInformation()).ifPresent(builder::setPartyInformation);
			ofNullable(getSefOrDcmAnonymousIndicator()).ifPresent(builder::setSefOrDcmAnonymousIndicator);
			ofNullable(getOriginalExecutionTimestamp()).ifPresent(builder::setOriginalExecutionTimestamp);
			ofNullable(getEnrichment()).ifPresent(builder::setEnrichment);
			ofNullable(getCryptoBased()).ifPresent(builder::setCryptoBased);
			ofNullable(getUnderlyingAssetLastAvailableSpotPrice()).ifPresent(builder::setUnderlyingAssetLastAvailableSpotPrice);
			ofNullable(getTransactionInformation()).ifPresent(builder::setTransactionInformation);
			ofNullable(getReportableAction()).ifPresent(builder::setReportableAction);
			ofNullable(getSubsequentPositionUTI()).ifPresent(builder::setSubsequentPositionUTI);
			ofNullable(getCounterpartyPositionPriorUTI()).ifPresent(builder::setCounterpartyPositionPriorUTI);
			ofNullable(getCustomBasket()).ifPresent(builder::setCustomBasket);
			ofNullable(getFinalContractualSettlementDate()).ifPresent(builder::setFinalContractualSettlementDate);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableInformation _that = getType().cast(o);
		
			if (!Objects.equals(confirmationMethod, _that.getConfirmationMethod())) return false;
			if (!Objects.equals(executionVenueType, _that.getExecutionVenueType())) return false;
			if (!Objects.equals(intragroup, _that.getIntragroup())) return false;
			if (!Objects.equals(largeSizeTrade, _that.getLargeSizeTrade())) return false;
			if (!ListEquals.listEquals(partyInformation, _that.getPartyInformation())) return false;
			if (!Objects.equals(sefOrDcmAnonymousIndicator, _that.getSefOrDcmAnonymousIndicator())) return false;
			if (!Objects.equals(originalExecutionTimestamp, _that.getOriginalExecutionTimestamp())) return false;
			if (!Objects.equals(enrichment, _that.getEnrichment())) return false;
			if (!Objects.equals(cryptoBased, _that.getCryptoBased())) return false;
			if (!Objects.equals(underlyingAssetLastAvailableSpotPrice, _that.getUnderlyingAssetLastAvailableSpotPrice())) return false;
			if (!ListEquals.listEquals(transactionInformation, _that.getTransactionInformation())) return false;
			if (!Objects.equals(reportableAction, _that.getReportableAction())) return false;
			if (!Objects.equals(subsequentPositionUTI, _that.getSubsequentPositionUTI())) return false;
			if (!Objects.equals(counterpartyPositionPriorUTI, _that.getCounterpartyPositionPriorUTI())) return false;
			if (!Objects.equals(customBasket, _that.getCustomBasket())) return false;
			if (!Objects.equals(finalContractualSettlementDate, _that.getFinalContractualSettlementDate())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (confirmationMethod != null ? confirmationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (executionVenueType != null ? executionVenueType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (intragroup != null ? intragroup.hashCode() : 0);
			_result = 31 * _result + (largeSizeTrade != null ? largeSizeTrade.hashCode() : 0);
			_result = 31 * _result + (partyInformation != null ? partyInformation.hashCode() : 0);
			_result = 31 * _result + (sefOrDcmAnonymousIndicator != null ? sefOrDcmAnonymousIndicator.hashCode() : 0);
			_result = 31 * _result + (originalExecutionTimestamp != null ? originalExecutionTimestamp.hashCode() : 0);
			_result = 31 * _result + (enrichment != null ? enrichment.hashCode() : 0);
			_result = 31 * _result + (cryptoBased != null ? cryptoBased.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetLastAvailableSpotPrice != null ? underlyingAssetLastAvailableSpotPrice.hashCode() : 0);
			_result = 31 * _result + (transactionInformation != null ? transactionInformation.hashCode() : 0);
			_result = 31 * _result + (reportableAction != null ? reportableAction.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (subsequentPositionUTI != null ? subsequentPositionUTI.hashCode() : 0);
			_result = 31 * _result + (counterpartyPositionPriorUTI != null ? counterpartyPositionPriorUTI.hashCode() : 0);
			_result = 31 * _result + (customBasket != null ? customBasket.hashCode() : 0);
			_result = 31 * _result + (finalContractualSettlementDate != null ? finalContractualSettlementDate.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformation {" +
				"confirmationMethod=" + this.confirmationMethod + ", " +
				"executionVenueType=" + this.executionVenueType + ", " +
				"intragroup=" + this.intragroup + ", " +
				"largeSizeTrade=" + this.largeSizeTrade + ", " +
				"partyInformation=" + this.partyInformation + ", " +
				"sefOrDcmAnonymousIndicator=" + this.sefOrDcmAnonymousIndicator + ", " +
				"originalExecutionTimestamp=" + this.originalExecutionTimestamp + ", " +
				"enrichment=" + this.enrichment + ", " +
				"cryptoBased=" + this.cryptoBased + ", " +
				"underlyingAssetLastAvailableSpotPrice=" + this.underlyingAssetLastAvailableSpotPrice + ", " +
				"transactionInformation=" + this.transactionInformation + ", " +
				"reportableAction=" + this.reportableAction + ", " +
				"subsequentPositionUTI=" + this.subsequentPositionUTI + ", " +
				"counterpartyPositionPriorUTI=" + this.counterpartyPositionPriorUTI + ", " +
				"customBasket=" + this.customBasket + ", " +
				"finalContractualSettlementDate=" + this.finalContractualSettlementDate +
			'}';
		}
	}

	/*********************** Builder Implementation of ReportableInformation  ***********************/
	class ReportableInformationBuilderImpl implements ReportableInformation.ReportableInformationBuilder {
	
		protected ConfirmationMethodEnum confirmationMethod;
		protected ExecutionVenueTypeEnum executionVenueType;
		protected Boolean intragroup;
		protected Boolean largeSizeTrade;
		protected List<PartyInformation.PartyInformationBuilder> partyInformation = new ArrayList<>();
		protected Boolean sefOrDcmAnonymousIndicator;
		protected ZonedDateTime originalExecutionTimestamp;
		protected EnrichmentData.EnrichmentDataBuilder enrichment;
		protected Boolean cryptoBased;
		protected PriceSchedule.PriceScheduleBuilder underlyingAssetLastAvailableSpotPrice;
		protected List<TransactionInformation.TransactionInformationBuilder> transactionInformation = new ArrayList<>();
		protected ReportableActionEnum reportableAction;
		protected PositionIdentifier.PositionIdentifierBuilder subsequentPositionUTI;
		protected PositionIdentifier.PositionIdentifierBuilder counterpartyPositionPriorUTI;
		protected CustomBasket.CustomBasketBuilder customBasket;
		protected Date finalContractualSettlementDate;
		
		@Override
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		public ConfirmationMethodEnum getConfirmationMethod() {
			return confirmationMethod;
		}
		
		@Override
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		public ExecutionVenueTypeEnum getExecutionVenueType() {
			return executionVenueType;
		}
		
		@Override
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		public Boolean getIntragroup() {
			return intragroup;
		}
		
		@Override
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		public Boolean getLargeSizeTrade() {
			return largeSizeTrade;
		}
		
		@Override
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		public List<? extends PartyInformation.PartyInformationBuilder> getPartyInformation() {
			return partyInformation;
		}
		
		@Override
		public PartyInformation.PartyInformationBuilder getOrCreatePartyInformation(int index) {
			if (partyInformation==null) {
				this.partyInformation = new ArrayList<>();
			}
			return getIndex(partyInformation, index, () -> {
						PartyInformation.PartyInformationBuilder newPartyInformation = PartyInformation.builder();
						return newPartyInformation;
					});
		}
		
		@Override
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		public Boolean getSefOrDcmAnonymousIndicator() {
			return sefOrDcmAnonymousIndicator;
		}
		
		@Override
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		public ZonedDateTime getOriginalExecutionTimestamp() {
			return originalExecutionTimestamp;
		}
		
		@Override
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		public EnrichmentData.EnrichmentDataBuilder getEnrichment() {
			return enrichment;
		}
		
		@Override
		public EnrichmentData.EnrichmentDataBuilder getOrCreateEnrichment() {
			EnrichmentData.EnrichmentDataBuilder result;
			if (enrichment!=null) {
				result = enrichment;
			}
			else {
				result = enrichment = EnrichmentData.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		public Boolean getCryptoBased() {
			return cryptoBased;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		public PriceSchedule.PriceScheduleBuilder getUnderlyingAssetLastAvailableSpotPrice() {
			return underlyingAssetLastAvailableSpotPrice;
		}
		
		@Override
		public PriceSchedule.PriceScheduleBuilder getOrCreateUnderlyingAssetLastAvailableSpotPrice() {
			PriceSchedule.PriceScheduleBuilder result;
			if (underlyingAssetLastAvailableSpotPrice!=null) {
				result = underlyingAssetLastAvailableSpotPrice;
			}
			else {
				result = underlyingAssetLastAvailableSpotPrice = PriceSchedule.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		public List<? extends TransactionInformation.TransactionInformationBuilder> getTransactionInformation() {
			return transactionInformation;
		}
		
		@Override
		public TransactionInformation.TransactionInformationBuilder getOrCreateTransactionInformation(int index) {
			if (transactionInformation==null) {
				this.transactionInformation = new ArrayList<>();
			}
			return getIndex(transactionInformation, index, () -> {
						TransactionInformation.TransactionInformationBuilder newTransactionInformation = TransactionInformation.builder();
						return newTransactionInformation;
					});
		}
		
		@Override
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		public ReportableActionEnum getReportableAction() {
			return reportableAction;
		}
		
		@Override
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		public PositionIdentifier.PositionIdentifierBuilder getSubsequentPositionUTI() {
			return subsequentPositionUTI;
		}
		
		@Override
		public PositionIdentifier.PositionIdentifierBuilder getOrCreateSubsequentPositionUTI() {
			PositionIdentifier.PositionIdentifierBuilder result;
			if (subsequentPositionUTI!=null) {
				result = subsequentPositionUTI;
			}
			else {
				result = subsequentPositionUTI = PositionIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		public PositionIdentifier.PositionIdentifierBuilder getCounterpartyPositionPriorUTI() {
			return counterpartyPositionPriorUTI;
		}
		
		@Override
		public PositionIdentifier.PositionIdentifierBuilder getOrCreateCounterpartyPositionPriorUTI() {
			PositionIdentifier.PositionIdentifierBuilder result;
			if (counterpartyPositionPriorUTI!=null) {
				result = counterpartyPositionPriorUTI;
			}
			else {
				result = counterpartyPositionPriorUTI = PositionIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		public CustomBasket.CustomBasketBuilder getCustomBasket() {
			return customBasket;
		}
		
		@Override
		public CustomBasket.CustomBasketBuilder getOrCreateCustomBasket() {
			CustomBasket.CustomBasketBuilder result;
			if (customBasket!=null) {
				result = customBasket;
			}
			else {
				result = customBasket = CustomBasket.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public Date getFinalContractualSettlementDate() {
			return finalContractualSettlementDate;
		}
		
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		@Override
		public ReportableInformation.ReportableInformationBuilder setConfirmationMethod(ConfirmationMethodEnum _confirmationMethod) {
			this.confirmationMethod = _confirmationMethod == null ? null : _confirmationMethod;
			return this;
		}
		
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		@Override
		public ReportableInformation.ReportableInformationBuilder setExecutionVenueType(ExecutionVenueTypeEnum _executionVenueType) {
			this.executionVenueType = _executionVenueType == null ? null : _executionVenueType;
			return this;
		}
		
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		@Override
		public ReportableInformation.ReportableInformationBuilder setIntragroup(Boolean _intragroup) {
			this.intragroup = _intragroup == null ? null : _intragroup;
			return this;
		}
		
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		@Override
		public ReportableInformation.ReportableInformationBuilder setLargeSizeTrade(Boolean _largeSizeTrade) {
			this.largeSizeTrade = _largeSizeTrade == null ? null : _largeSizeTrade;
			return this;
		}
		
		@RosettaAttribute(value="partyInformation", isRequired=true)
		@RuneAttribute(value="partyInformation", isRequired=true)
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation _partyInformation) {
			if (_partyInformation != null) {
				this.partyInformation.add(_partyInformation.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(PartyInformation _partyInformation, int idx) {
			getIndex(this.partyInformation, idx, () -> _partyInformation.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addPartyInformation(List<? extends PartyInformation> partyInformations) {
			if (partyInformations != null) {
				for (final PartyInformation toAdd : partyInformations) {
					this.partyInformation.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("partyInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder setPartyInformation(List<? extends PartyInformation> partyInformations) {
			if (partyInformations == null) {
				this.partyInformation = new ArrayList<>();
			} else {
				this.partyInformation = partyInformations.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		@Override
		public ReportableInformation.ReportableInformationBuilder setSefOrDcmAnonymousIndicator(Boolean _sefOrDcmAnonymousIndicator) {
			this.sefOrDcmAnonymousIndicator = _sefOrDcmAnonymousIndicator == null ? null : _sefOrDcmAnonymousIndicator;
			return this;
		}
		
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		@Override
		public ReportableInformation.ReportableInformationBuilder setOriginalExecutionTimestamp(ZonedDateTime _originalExecutionTimestamp) {
			this.originalExecutionTimestamp = _originalExecutionTimestamp == null ? null : _originalExecutionTimestamp;
			return this;
		}
		
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		@Override
		public ReportableInformation.ReportableInformationBuilder setEnrichment(EnrichmentData _enrichment) {
			this.enrichment = _enrichment == null ? null : _enrichment.toBuilder();
			return this;
		}
		
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCryptoBased(Boolean _cryptoBased) {
			this.cryptoBased = _cryptoBased == null ? null : _cryptoBased;
			return this;
		}
		
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		@Override
		public ReportableInformation.ReportableInformationBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule _underlyingAssetLastAvailableSpotPrice) {
			this.underlyingAssetLastAvailableSpotPrice = _underlyingAssetLastAvailableSpotPrice == null ? null : _underlyingAssetLastAvailableSpotPrice.toBuilder();
			return this;
		}
		
		@RosettaAttribute("transactionInformation")
		@RuneAttribute("transactionInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation _transactionInformation) {
			if (_transactionInformation != null) {
				this.transactionInformation.add(_transactionInformation.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(TransactionInformation _transactionInformation, int idx) {
			getIndex(this.transactionInformation, idx, () -> _transactionInformation.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder addTransactionInformation(List<? extends TransactionInformation> transactionInformations) {
			if (transactionInformations != null) {
				for (final TransactionInformation toAdd : transactionInformations) {
					this.transactionInformation.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("transactionInformation")
		@Override
		public ReportableInformation.ReportableInformationBuilder setTransactionInformation(List<? extends TransactionInformation> transactionInformations) {
			if (transactionInformations == null) {
				this.transactionInformation = new ArrayList<>();
			} else {
				this.transactionInformation = transactionInformations.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		@Override
		public ReportableInformation.ReportableInformationBuilder setReportableAction(ReportableActionEnum _reportableAction) {
			this.reportableAction = _reportableAction == null ? null : _reportableAction;
			return this;
		}
		
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		@Override
		public ReportableInformation.ReportableInformationBuilder setSubsequentPositionUTI(PositionIdentifier _subsequentPositionUTI) {
			this.subsequentPositionUTI = _subsequentPositionUTI == null ? null : _subsequentPositionUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCounterpartyPositionPriorUTI(PositionIdentifier _counterpartyPositionPriorUTI) {
			this.counterpartyPositionPriorUTI = _counterpartyPositionPriorUTI == null ? null : _counterpartyPositionPriorUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		@Override
		public ReportableInformation.ReportableInformationBuilder setCustomBasket(CustomBasket _customBasket) {
			this.customBasket = _customBasket == null ? null : _customBasket.toBuilder();
			return this;
		}
		
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		@Override
		public ReportableInformation.ReportableInformationBuilder setFinalContractualSettlementDate(Date _finalContractualSettlementDate) {
			this.finalContractualSettlementDate = _finalContractualSettlementDate == null ? null : _finalContractualSettlementDate;
			return this;
		}
		
		@Override
		public ReportableInformation build() {
			return new ReportableInformation.ReportableInformationImpl(this);
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformation.ReportableInformationBuilder prune() {
			partyInformation = partyInformation.stream().filter(b->b!=null).<PartyInformation.PartyInformationBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (enrichment!=null && !enrichment.prune().hasData()) enrichment = null;
			if (underlyingAssetLastAvailableSpotPrice!=null && !underlyingAssetLastAvailableSpotPrice.prune().hasData()) underlyingAssetLastAvailableSpotPrice = null;
			transactionInformation = transactionInformation.stream().filter(b->b!=null).<TransactionInformation.TransactionInformationBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (subsequentPositionUTI!=null && !subsequentPositionUTI.prune().hasData()) subsequentPositionUTI = null;
			if (counterpartyPositionPriorUTI!=null && !counterpartyPositionPriorUTI.prune().hasData()) counterpartyPositionPriorUTI = null;
			if (customBasket!=null && !customBasket.prune().hasData()) customBasket = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getConfirmationMethod()!=null) return true;
			if (getExecutionVenueType()!=null) return true;
			if (getIntragroup()!=null) return true;
			if (getLargeSizeTrade()!=null) return true;
			if (getPartyInformation()!=null && getPartyInformation().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getSefOrDcmAnonymousIndicator()!=null) return true;
			if (getOriginalExecutionTimestamp()!=null) return true;
			if (getEnrichment()!=null && getEnrichment().hasData()) return true;
			if (getCryptoBased()!=null) return true;
			if (getUnderlyingAssetLastAvailableSpotPrice()!=null && getUnderlyingAssetLastAvailableSpotPrice().hasData()) return true;
			if (getTransactionInformation()!=null && getTransactionInformation().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getReportableAction()!=null) return true;
			if (getSubsequentPositionUTI()!=null && getSubsequentPositionUTI().hasData()) return true;
			if (getCounterpartyPositionPriorUTI()!=null && getCounterpartyPositionPriorUTI().hasData()) return true;
			if (getCustomBasket()!=null && getCustomBasket().hasData()) return true;
			if (getFinalContractualSettlementDate()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformation.ReportableInformationBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			ReportableInformation.ReportableInformationBuilder o = (ReportableInformation.ReportableInformationBuilder) other;
			
			merger.mergeRosetta(getPartyInformation(), o.getPartyInformation(), this::getOrCreatePartyInformation);
			merger.mergeRosetta(getEnrichment(), o.getEnrichment(), this::setEnrichment);
			merger.mergeRosetta(getUnderlyingAssetLastAvailableSpotPrice(), o.getUnderlyingAssetLastAvailableSpotPrice(), this::setUnderlyingAssetLastAvailableSpotPrice);
			merger.mergeRosetta(getTransactionInformation(), o.getTransactionInformation(), this::getOrCreateTransactionInformation);
			merger.mergeRosetta(getSubsequentPositionUTI(), o.getSubsequentPositionUTI(), this::setSubsequentPositionUTI);
			merger.mergeRosetta(getCounterpartyPositionPriorUTI(), o.getCounterpartyPositionPriorUTI(), this::setCounterpartyPositionPriorUTI);
			merger.mergeRosetta(getCustomBasket(), o.getCustomBasket(), this::setCustomBasket);
			
			merger.mergeBasic(getConfirmationMethod(), o.getConfirmationMethod(), this::setConfirmationMethod);
			merger.mergeBasic(getExecutionVenueType(), o.getExecutionVenueType(), this::setExecutionVenueType);
			merger.mergeBasic(getIntragroup(), o.getIntragroup(), this::setIntragroup);
			merger.mergeBasic(getLargeSizeTrade(), o.getLargeSizeTrade(), this::setLargeSizeTrade);
			merger.mergeBasic(getSefOrDcmAnonymousIndicator(), o.getSefOrDcmAnonymousIndicator(), this::setSefOrDcmAnonymousIndicator);
			merger.mergeBasic(getOriginalExecutionTimestamp(), o.getOriginalExecutionTimestamp(), this::setOriginalExecutionTimestamp);
			merger.mergeBasic(getCryptoBased(), o.getCryptoBased(), this::setCryptoBased);
			merger.mergeBasic(getReportableAction(), o.getReportableAction(), this::setReportableAction);
			merger.mergeBasic(getFinalContractualSettlementDate(), o.getFinalContractualSettlementDate(), this::setFinalContractualSettlementDate);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableInformation _that = getType().cast(o);
		
			if (!Objects.equals(confirmationMethod, _that.getConfirmationMethod())) return false;
			if (!Objects.equals(executionVenueType, _that.getExecutionVenueType())) return false;
			if (!Objects.equals(intragroup, _that.getIntragroup())) return false;
			if (!Objects.equals(largeSizeTrade, _that.getLargeSizeTrade())) return false;
			if (!ListEquals.listEquals(partyInformation, _that.getPartyInformation())) return false;
			if (!Objects.equals(sefOrDcmAnonymousIndicator, _that.getSefOrDcmAnonymousIndicator())) return false;
			if (!Objects.equals(originalExecutionTimestamp, _that.getOriginalExecutionTimestamp())) return false;
			if (!Objects.equals(enrichment, _that.getEnrichment())) return false;
			if (!Objects.equals(cryptoBased, _that.getCryptoBased())) return false;
			if (!Objects.equals(underlyingAssetLastAvailableSpotPrice, _that.getUnderlyingAssetLastAvailableSpotPrice())) return false;
			if (!ListEquals.listEquals(transactionInformation, _that.getTransactionInformation())) return false;
			if (!Objects.equals(reportableAction, _that.getReportableAction())) return false;
			if (!Objects.equals(subsequentPositionUTI, _that.getSubsequentPositionUTI())) return false;
			if (!Objects.equals(counterpartyPositionPriorUTI, _that.getCounterpartyPositionPriorUTI())) return false;
			if (!Objects.equals(customBasket, _that.getCustomBasket())) return false;
			if (!Objects.equals(finalContractualSettlementDate, _that.getFinalContractualSettlementDate())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (confirmationMethod != null ? confirmationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (executionVenueType != null ? executionVenueType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (intragroup != null ? intragroup.hashCode() : 0);
			_result = 31 * _result + (largeSizeTrade != null ? largeSizeTrade.hashCode() : 0);
			_result = 31 * _result + (partyInformation != null ? partyInformation.hashCode() : 0);
			_result = 31 * _result + (sefOrDcmAnonymousIndicator != null ? sefOrDcmAnonymousIndicator.hashCode() : 0);
			_result = 31 * _result + (originalExecutionTimestamp != null ? originalExecutionTimestamp.hashCode() : 0);
			_result = 31 * _result + (enrichment != null ? enrichment.hashCode() : 0);
			_result = 31 * _result + (cryptoBased != null ? cryptoBased.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetLastAvailableSpotPrice != null ? underlyingAssetLastAvailableSpotPrice.hashCode() : 0);
			_result = 31 * _result + (transactionInformation != null ? transactionInformation.hashCode() : 0);
			_result = 31 * _result + (reportableAction != null ? reportableAction.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (subsequentPositionUTI != null ? subsequentPositionUTI.hashCode() : 0);
			_result = 31 * _result + (counterpartyPositionPriorUTI != null ? counterpartyPositionPriorUTI.hashCode() : 0);
			_result = 31 * _result + (customBasket != null ? customBasket.hashCode() : 0);
			_result = 31 * _result + (finalContractualSettlementDate != null ? finalContractualSettlementDate.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformationBuilder {" +
				"confirmationMethod=" + this.confirmationMethod + ", " +
				"executionVenueType=" + this.executionVenueType + ", " +
				"intragroup=" + this.intragroup + ", " +
				"largeSizeTrade=" + this.largeSizeTrade + ", " +
				"partyInformation=" + this.partyInformation + ", " +
				"sefOrDcmAnonymousIndicator=" + this.sefOrDcmAnonymousIndicator + ", " +
				"originalExecutionTimestamp=" + this.originalExecutionTimestamp + ", " +
				"enrichment=" + this.enrichment + ", " +
				"cryptoBased=" + this.cryptoBased + ", " +
				"underlyingAssetLastAvailableSpotPrice=" + this.underlyingAssetLastAvailableSpotPrice + ", " +
				"transactionInformation=" + this.transactionInformation + ", " +
				"reportableAction=" + this.reportableAction + ", " +
				"subsequentPositionUTI=" + this.subsequentPositionUTI + ", " +
				"counterpartyPositionPriorUTI=" + this.counterpartyPositionPriorUTI + ", " +
				"customBasket=" + this.customBasket + ", " +
				"finalContractualSettlementDate=" + this.finalContractualSettlementDate +
			'}';
		}
	}
}
