package drr.regulation.common.functions;

import cdm.base.staticdata.asset.common.Commodity;
import cdm.base.staticdata.asset.common.Index;
import cdm.base.staticdata.asset.common.Loan;
import cdm.base.staticdata.asset.common.ProductIdentifier;
import cdm.base.staticdata.asset.common.Security;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaCommodity;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier;
import cdm.event.common.Trade;
import cdm.product.asset.BasketReferenceInformation;
import cdm.product.asset.CommodityPayout;
import cdm.product.asset.CreditDefaultPayout;
import cdm.product.asset.GeneralTerms;
import cdm.product.asset.ReferenceObligation;
import cdm.product.asset.ReferencePair;
import cdm.product.asset.ReferencePool;
import cdm.product.asset.ReferencePoolItem;
import cdm.product.template.Basket;
import cdm.product.template.Payout;
import cdm.product.template.PerformancePayout;
import cdm.product.template.Product;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperListOfLists;
import com.rosetta.model.lib.mapper.MapperS;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GetBasketConstituentsProductIdentifier.GetBasketConstituentsProductIdentifierDefault.class)
public abstract class GetBasketConstituentsProductIdentifier implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected EconomicTermsForProduct economicTermsForProduct;
	@Inject protected GetProductIdentifierFilteringISIN getProductIdentifierFilteringISIN;
	@Inject protected IsSingleCommodityPayoutProduct isSingleCommodityPayoutProduct;
	@Inject protected ProductForTrade productForTrade;
	@Inject protected UnderlierForProduct underlierForProduct;

	/**
	* @param trade 
	* @return productIdentifiers Single product identifier per basket constituent.
	*/
	public List<? extends ProductIdentifier> evaluate(Trade trade) {
		List<ProductIdentifier.ProductIdentifierBuilder> productIdentifiersBuilder = doEvaluate(trade);
		
		final List<? extends ProductIdentifier> productIdentifiers;
		if (productIdentifiersBuilder == null) {
			productIdentifiers = null;
		} else {
			productIdentifiers = productIdentifiersBuilder.stream().map(ProductIdentifier::build).collect(Collectors.toList());
			objectValidator.validate(ProductIdentifier.class, productIdentifiers);
		}
		
		return productIdentifiers;
	}

	protected abstract List<ProductIdentifier.ProductIdentifierBuilder> doEvaluate(Trade trade);

	protected abstract MapperS<? extends Product> product(Trade trade);

	protected abstract MapperC<? extends Product> basketConstituents(Trade trade);

	protected abstract MapperC<? extends ReferenceObligation> referenceObligations(Trade trade);

	public static class GetBasketConstituentsProductIdentifierDefault extends GetBasketConstituentsProductIdentifier {
		@Override
		protected List<ProductIdentifier.ProductIdentifierBuilder> doEvaluate(Trade trade) {
			List<ProductIdentifier.ProductIdentifierBuilder> productIdentifiers = new ArrayList<>();
			return assignOutput(productIdentifiers, trade);
		}
		
		protected List<ProductIdentifier.ProductIdentifierBuilder> assignOutput(List<ProductIdentifier.ProductIdentifierBuilder> productIdentifiers, Trade trade) {
			final MapperListOfLists<ReferenceWithMetaProductIdentifier> thenArg0 = basketConstituents(trade)
				.mapItemToList(item -> {
					if (exists(item.<Security>map("getSecurity", _product -> _product.getSecurity())).getOrDefault(false)) {
						return item.<Security>map("getSecurity", _product -> _product.getSecurity()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", security -> security.getProductIdentifier());
					}
					if (exists(item.<Loan>map("getLoan", _product -> _product.getLoan())).getOrDefault(false)) {
						return item.<Loan>map("getLoan", _product -> _product.getLoan()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", loan -> loan.getProductIdentifier());
					}
					if (exists(item.<ReferenceWithMetaCommodity>map("getCommodity", _product -> _product.getCommodity())).getOrDefault(false)) {
						return item.<ReferenceWithMetaCommodity>map("getCommodity", _product -> _product.getCommodity()).<Commodity>map("Type coercion", referenceWithMetaCommodity -> referenceWithMetaCommodity == null ? null : referenceWithMetaCommodity.getValue()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", commodity -> commodity.getProductIdentifier());
					}
					if (exists(item.<Index>map("getIndex", _product -> _product.getIndex())).getOrDefault(false)) {
						return item.<Index>map("getIndex", _product -> _product.getIndex()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", index -> index.getProductIdentifier());
					}
					return MapperC.<ReferenceWithMetaProductIdentifier>ofNull();
				});
			productIdentifiers.addAll(toBuilder(thenArg0
				.mapListToItem(item -> MapperS.of(getProductIdentifierFilteringISIN.evaluate(item.<ProductIdentifier>map("Type coercion", referenceWithMetaProductIdentifier -> referenceWithMetaProductIdentifier.getValue()).getMulti()))).getMulti()));
			
			final MapperListOfLists<ReferenceWithMetaProductIdentifier> thenArg1 = referenceObligations(trade)
				.mapItemToList(item -> {
					if (exists(item.<Security>map("getSecurity", referenceObligation -> referenceObligation.getSecurity())).getOrDefault(false)) {
						return item.<Security>map("getSecurity", referenceObligation -> referenceObligation.getSecurity()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", security -> security.getProductIdentifier());
					}
					if (exists(item.<Loan>map("getLoan", referenceObligation -> referenceObligation.getLoan())).getOrDefault(false)) {
						return item.<Loan>map("getLoan", referenceObligation -> referenceObligation.getLoan()).<ReferenceWithMetaProductIdentifier>mapC("getProductIdentifier", loan -> loan.getProductIdentifier());
					}
					return MapperC.<ReferenceWithMetaProductIdentifier>ofNull();
				});
			productIdentifiers.addAll(toBuilder(thenArg1
				.mapListToItem(item -> MapperS.of(getProductIdentifierFilteringISIN.evaluate(item.<ProductIdentifier>map("Type coercion", referenceWithMetaProductIdentifier -> referenceWithMetaProductIdentifier.getValue()).getMulti()))).getMulti()));
			
			return Optional.ofNullable(productIdentifiers)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
		
		@Override
		protected MapperS<? extends Product> product(Trade trade) {
			return MapperS.of(productForTrade.evaluate(trade));
		}
		
		@Override
		protected MapperC<? extends Product> basketConstituents(Trade trade) {
			if (exists(MapperS.of(underlierForProduct.evaluate(product(trade).get())).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).getOrDefault(false)) {
				return MapperS.of(underlierForProduct.evaluate(product(trade).get())).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<PerformancePayout>mapC("getPerformancePayout", payout -> payout.getPerformancePayout()).<Product>map("getUnderlier", performancePayout -> performancePayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<PerformancePayout>mapC("getPerformancePayout", payout -> payout.getPerformancePayout()).<Product>map("getUnderlier", performancePayout -> performancePayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CommodityPayout>mapC("getCommodityPayout", payout -> payout.getCommodityPayout()).<Product>map("getUnderlier", commodityPayout -> commodityPayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).and(areEqual(MapperS.of(isSingleCommodityPayoutProduct.evaluate(product(trade).get())), MapperS.of(false), CardinalityOperator.All)).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CommodityPayout>mapC("getCommodityPayout", payout -> payout.getCommodityPayout()).<Product>map("getUnderlier", commodityPayout -> commodityPayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			return MapperC.<Product>ofNull();
		}
		
		@Override
		protected MapperC<? extends ReferenceObligation> referenceObligations(Trade trade) {
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<BasketReferenceInformation>map("getBasketReferenceInformation", generalTerms -> generalTerms.getBasketReferenceInformation()).<ReferencePool>map("getReferencePool", basketReferenceInformation -> basketReferenceInformation.getReferencePool()).<ReferencePoolItem>mapC("getReferencePoolItem", referencePool -> referencePool.getReferencePoolItem()).<ReferencePair>map("getReferencePair", referencePoolItem -> referencePoolItem.getReferencePair()).<ReferenceObligation>map("getReferenceObligation", referencePair -> referencePair.getReferenceObligation())).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<BasketReferenceInformation>map("getBasketReferenceInformation", generalTerms -> generalTerms.getBasketReferenceInformation()).<ReferencePool>map("getReferencePool", basketReferenceInformation -> basketReferenceInformation.getReferencePool()).<ReferencePoolItem>mapC("getReferencePoolItem", referencePool -> referencePool.getReferencePoolItem()).<ReferencePair>map("getReferencePair", referencePoolItem -> referencePoolItem.getReferencePair()).<ReferenceObligation>map("getReferenceObligation", referencePair -> referencePair.getReferenceObligation());
			}
			return MapperC.<ReferenceObligation>ofNull();
		}
	}
}
