package drr.regulation.jfsa.rewrite.trade.validation.datarule;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.jfsa.rewrite.trade.StrikePriceScheduleReport;
import drr.standards.iso.PriceNotationEnum;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.52.1
 */
@RosettaDataRule("StrikePriceScheduleReportMonetary")
@ImplementedBy(StrikePriceScheduleReportMonetary.Default.class)
public interface StrikePriceScheduleReportMonetary extends Validator<StrikePriceScheduleReport> {
	
	String NAME = "StrikePriceScheduleReportMonetary";
	String DEFINITION = "if strikePriceNotationInEffectOnAssociatedEffectiveDate = PriceNotationEnum -> Monetary then strikePriceInEffectOnAssociatedEffectiveDateMonetary exists";
	
	class Default implements StrikePriceScheduleReportMonetary {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, StrikePriceScheduleReport strikePriceScheduleReport) {
			ComparisonResult result = executeDataRule(strikePriceScheduleReport);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "StrikePriceScheduleReport", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationResult.ValidationType.DATA_RULE, "StrikePriceScheduleReport", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(StrikePriceScheduleReport strikePriceScheduleReport) {
			try {
				if (areEqual(MapperS.of(strikePriceScheduleReport).<PriceNotationEnum>map("getStrikePriceNotationInEffectOnAssociatedEffectiveDate", _strikePriceScheduleReport -> _strikePriceScheduleReport.getStrikePriceNotationInEffectOnAssociatedEffectiveDate()), MapperS.of(PriceNotationEnum.MONETARY), CardinalityOperator.All).getOrDefault(false)) {
					return exists(MapperS.of(strikePriceScheduleReport).<BigDecimal>map("getStrikePriceInEffectOnAssociatedEffectiveDateMonetary", _strikePriceScheduleReport -> _strikePriceScheduleReport.getStrikePriceInEffectOnAssociatedEffectiveDateMonetary()));
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements StrikePriceScheduleReportMonetary {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, StrikePriceScheduleReport strikePriceScheduleReport) {
			return Collections.emptyList();
		}
	}
}
