package drr.regulation.jfsa.rewrite.trade.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.jfsa.rewrite.trade.JFSATransactionReport;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkNumber;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkString;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

public class JFSATransactionReportTypeFormatValidator implements Validator<JFSATransactionReport> {

	private List<ComparisonResult> getComparisonResults(JFSATransactionReport o) {
		return Lists.<ComparisonResult>newArrayList(
				checkString("entityResponsibleForReporting", o.getEntityResponsibleForReporting(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty1", o.getCounterparty1(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty2", o.getCounterparty2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("direction1", o.getDirection1(), 1, of(4), empty()), 
				checkString("direction2Leg1", o.getDirection2Leg1(), 1, of(4), empty()), 
				checkString("direction2Leg2", o.getDirection2Leg2(), 1, of(4), empty()), 
				checkString("submitterIdentifier", o.getSubmitterIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("newSDRIdentifier", o.getNewSDRIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("originalSwapSDRIdentifier", o.getOriginalSwapSDRIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("cleared", o.getCleared(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("centralCounterparty", o.getCentralCounterparty(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("clearingMember", o.getClearingMember(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("platformIdentifier", o.getPlatformIdentifier(), 1, of(4), empty()), 
				checkString("confirmed", o.getConfirmed(), 1, of(4), empty()), 
				checkString("clearingAccountOrigin", o.getClearingAccountOrigin(), 1, of(4), empty()), 
				checkString("originalSwapUTI", o.getOriginalSwapUTI(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18}[0-9]{2}[A-Z0-9]{0,32}"))), 
				checkString("originalSwapUTIProprietary", o.getOriginalSwapUTIProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("uti", o.getUti(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18}[0-9]{2}[A-Z0-9]{0,32}"))), 
				checkString("utiProprietary", o.getUtiProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("priorUti", o.getPriorUti(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18}[0-9]{2}[A-Z0-9]{0,32}"))), 
				checkString("priorUtiProprietary", o.getPriorUtiProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("fixedRatePaymentFrequencyPeriodLeg1", o.getFixedRatePaymentFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("fixedRatePaymentFrequencyPeriodLeg2", o.getFixedRatePaymentFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkString("floatingRatePaymentFrequencyPeriodLeg1", o.getFloatingRatePaymentFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("floatingRatePaymentFrequencyPeriodLeg2", o.getFloatingRatePaymentFrequencyPeriodLeg2(), 1, of(4), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierLeg1", o.getFixedRatePaymentFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierLeg2", o.getFixedRatePaymentFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierLeg1", o.getFloatingRatePaymentFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierLeg2", o.getFloatingRatePaymentFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkString("floatingRateResetFrequencyPeriodLeg1", o.getFloatingRateResetFrequencyPeriodLeg1(), 1, of(4), empty()), 
				checkString("floatingRateResetFrequencyPeriodOfLeg2", o.getFloatingRateResetFrequencyPeriodOfLeg2(), 1, of(4), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierLeg1", o.getFloatingRateResetFrequencyPeriodMultiplierLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierLeg2", o.getFloatingRateResetFrequencyPeriodMultiplierLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("valuationAmount", o.getValuationAmount(), of(25), of(5), empty(), empty()), 
				checkString("initialMarginCollateralPortfolioCode", o.getInitialMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("variationMarginCollateralPortfolioCode", o.getVariationMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("priceNotation", o.getPriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("priceUnitOfMeasure", o.getPriceUnitOfMeasure(), 1, of(4), empty()), 
				checkNumber("fixedRateLeg1", o.getFixedRateLeg1(), of(11), of(10), empty(), empty()), 
				checkNumber("fixedRateLeg2", o.getFixedRateLeg2(), of(11), of(10), empty(), empty()), 
				checkString("spreadNotationOfLeg1", o.getSpreadNotationOfLeg1(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("spreadNotationOfLeg2", o.getSpreadNotationOfLeg2(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("strikePriceNotation", o.getStrikePriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkNumber("optionPremiumAmount", o.getOptionPremiumAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("exchangeRate", o.getExchangeRate(), of(18), of(13), empty(), empty()), 
				checkString("exchangeRateBasis", o.getExchangeRateBasis(), 7, of(7), of(Pattern.compile("[A-Z]{3,3}/[A-Z]{3,3}"))), 
				checkNumber("notionalLeg1", o.getNotionalLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalLeg2", o.getNotionalLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("callAmount", o.getCallAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("putAmount", o.getPutAmount(), of(25), of(5), empty(), empty()), 
				checkString("quantityUnitOfMeasureLeg1", o.getQuantityUnitOfMeasureLeg1(), 1, of(4), empty()), 
				checkString("quantityUnitOfMeasureLeg2", o.getQuantityUnitOfMeasureLeg2(), 1, of(4), empty()), 
				checkNumber("totalNotionalQuantityLeg1", o.getTotalNotionalQuantityLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("totalNotionalQuantityLeg2", o.getTotalNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkString("actionType", o.getActionType(), 1, of(4), empty()), 
				checkString("eventType", o.getEventType(), 1, of(4), empty()), 
				checkString("eventIdentifier", o.getEventIdentifier(), 1, of(52), empty()), 
				checkNumber("indexFactor", o.getIndexFactor(), of(11), of(10), empty(), empty()), 
				checkString("embeddedOptionType", o.getEmbeddedOptionType(), 1, of(4), empty()), 
				checkString("deliveryType", o.getDeliveryType(), 1, of(4), empty()), 
				checkString("assetClass", o.getAssetClass(), 1, of(4), empty()), 
				checkString("underlyingIdentificationType", o.getUnderlyingIdentificationType(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("underlyingIdentification", o.getUnderlyingIdentification(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("underlyingIndexIndicator", o.getUnderlyingIndexIndicator(), 1, of(4), empty()), 
				checkString("nameOfTheUnderlyingIndex", o.getNameOfTheUnderlyingIndex(), 1, of(50), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_\\&]{1,50}"))), 
				checkString("indicatorOfTheFloatingRateOfLeg1", o.getIndicatorOfTheFloatingRateOfLeg1(), 1, of(4), empty()), 
				checkString("indicatorOfTheFloatingRateOfLeg2", o.getIndicatorOfTheFloatingRateOfLeg2(), 1, of(4), empty()), 
				checkString("nameOfTheFloatingRateOfLeg1", o.getNameOfTheFloatingRateOfLeg1(), 1, of(50), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_\\&]{1,50}"))), 
				checkString("nameOfTheFloatingRateOfLeg2", o.getNameOfTheFloatingRateOfLeg2(), 1, of(50), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_\\&]{1,50}"))), 
				checkString("floatingRateReferencePeriodOfLeg1TimePeriod", o.getFloatingRateReferencePeriodOfLeg1TimePeriod(), 1, of(4), empty()), 
				checkString("floatingRateReferencePeriodOfLeg2TimePeriod", o.getFloatingRateReferencePeriodOfLeg2TimePeriod(), 1, of(4), empty()), 
				checkNumber("floatingRateReferencePeriodOfLeg1Multiplier", o.getFloatingRateReferencePeriodOfLeg1Multiplier(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRateReferencePeriodOfLeg2Multiplier", o.getFloatingRateReferencePeriodOfLeg2Multiplier(), of(3), of(0), empty(), empty()), 
				checkString("seniority", o.getSeniority(), 1, of(4), empty()), 
				checkNumber("series", o.getSeries(), of(5), of(0), empty(), empty()), 
				checkNumber("version", o.getVersion(), of(5), of(0), empty(), empty()), 
				checkNumber("cdSIndexAttachmentPoint", o.getCdSIndexAttachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("cdSIndexDetachmentPoint", o.getCdSIndexDetachmentPoint(), of(11), of(10), empty(), empty()), 
				checkString("customBasketCode", o.getCustomBasketCode(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("contractType", o.getContractType(), 1, of(4), empty()), 
				checkString("optionStyle", o.getOptionStyle(), 1, of(4), empty()), 
				checkString("optionType", o.getOptionType(), 1, of(4), empty()), 
				checkString("packageIdentifier", o.getPackageIdentifier(), 1, of(35), of(Pattern.compile("[a-zA-Z0-9]{1,35}"))), 
				checkString("packageTransactionPriceNotation", o.getPackageTransactionPriceNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkString("packageTransactionSpreadNotation", o.getPackageTransactionSpreadNotation(), 1, of(1), of(Pattern.compile("[1-4]"))), 
				checkNumber("delta", o.getDelta(), of(25), of(5), empty(), empty())
			);
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, JFSATransactionReport o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("JFSATransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "JFSATransactionReport", path, "", res.getError());
				}
				return success("JFSATransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "JFSATransactionReport", path, "");
			})
			.collect(toList());
	}

}
