package drr.regulation.jfsa.rewrite.trade.reports;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.records.Date;
import com.rosetta.model.lib.reports.ReportFunction;
import drr.regulation.common.PriceReportablePeriod;
import drr.regulation.common.TransactionReportInstruction;
import drr.regulation.common.functions.GetReportablePricePeriod;
import drr.regulation.common.functions.IsAllowableAction;
import drr.regulation.jfsa.rewrite.trade.PriceScheduleReport;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;


@ImplementedBy(PriceScheduleRule.PriceScheduleRuleDefault.class)
public abstract class PriceScheduleRule implements ReportFunction<TransactionReportInstruction, List<? extends PriceScheduleReport>> {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected ContractPriceScheduleEffectiveDateRule contractPriceScheduleEffectiveDateRule;
	@Inject protected ContractPriceScheduleEndDateRule contractPriceScheduleEndDateRule;
	@Inject protected GetReportablePricePeriod getReportablePricePeriod;
	@Inject protected IsAllowableAction isAllowableAction;
	@Inject protected PriceScheduleAmountFormatDecimalRule priceScheduleAmountFormatDecimalRule;
	@Inject protected PriceScheduleAmountFormatMonetaryRule priceScheduleAmountFormatMonetaryRule;

	/**
	* @param input 
	* @return output 
	*/
	@Override
	public List<? extends PriceScheduleReport> evaluate(TransactionReportInstruction input) {
		List<PriceScheduleReport.PriceScheduleReportBuilder> outputBuilder = doEvaluate(input);
		
		final List<? extends PriceScheduleReport> output;
		if (outputBuilder == null) {
			output = null;
		} else {
			output = outputBuilder.stream().map(PriceScheduleReport::build).collect(Collectors.toList());
			objectValidator.validate(PriceScheduleReport.class, output);
		}
		
		return output;
	}

	protected abstract List<PriceScheduleReport.PriceScheduleReportBuilder> doEvaluate(TransactionReportInstruction input);

	public static class PriceScheduleRuleDefault extends PriceScheduleRule {
		@Override
		protected List<PriceScheduleReport.PriceScheduleReportBuilder> doEvaluate(TransactionReportInstruction input) {
			List<PriceScheduleReport.PriceScheduleReportBuilder> output = new ArrayList<>();
			return assignOutput(output, input);
		}
		
		protected List<PriceScheduleReport.PriceScheduleReportBuilder> assignOutput(List<PriceScheduleReport.PriceScheduleReportBuilder> output, TransactionReportInstruction input) {
			final MapperS<TransactionReportInstruction> thenArg0 = MapperS.of(input)
				.filterSingleNullSafe(item -> isAllowableAction.evaluate(item.get()));
			final MapperC<PriceReportablePeriod> thenArg1 = thenArg0
				.mapSingleToList(item -> MapperC.<PriceReportablePeriod>of(getReportablePricePeriod.evaluate(item.get())));
			final MapperC<PriceScheduleReport> thenArg2 = thenArg1
				.mapItem(item -> MapperS.of(PriceScheduleReport.builder()
					.setEffectiveDate(contractPriceScheduleEffectiveDateRule.evaluate(item.get()))
					.setEndDate(contractPriceScheduleEndDateRule.evaluate(item.get()))
					.setPriceMonetary(priceScheduleAmountFormatMonetaryRule.evaluate(item.get()))
					.setPriceDecimal(priceScheduleAmountFormatDecimalRule.evaluate(item.get()))
					.build()));
			output = toBuilder(thenArg2
				.sort(item -> item.<Date>map("getEffectiveDate", priceScheduleReport -> priceScheduleReport.getEffectiveDate())).getMulti());
			
			return Optional.ofNullable(output)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
