package drr.regulation.jfsa.rewrite.trade.reports;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.records.Date;
import com.rosetta.model.lib.reports.ReportFunction;
import drr.regulation.common.PriceReportablePeriod;
import drr.regulation.common.TransactionReportInstruction;
import drr.regulation.common.functions.GetReportableStrikePricePeriod;
import drr.regulation.common.functions.IsAllowableAction;
import drr.regulation.jfsa.rewrite.trade.StrikePriceScheduleReport;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;


@ImplementedBy(StrikePriceScheduleRule.StrikePriceScheduleRuleDefault.class)
public abstract class StrikePriceScheduleRule implements ReportFunction<TransactionReportInstruction, List<? extends StrikePriceScheduleReport>> {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected EffectiveDateStrikePriceRule effectiveDateStrikePriceRule;
	@Inject protected EndDateStrikePriceRule endDateStrikePriceRule;
	@Inject protected GetReportableStrikePricePeriod getReportableStrikePricePeriod;
	@Inject protected IsAllowableAction isAllowableAction;
	@Inject protected StrikePriceInEffectOnAssociatedEffectiveDateDecimalRule strikePriceInEffectOnAssociatedEffectiveDateDecimalRule;
	@Inject protected StrikePriceInEffectOnAssociatedEffectiveDateMonetaryRule strikePriceInEffectOnAssociatedEffectiveDateMonetaryRule;

	/**
	* @param input 
	* @return output 
	*/
	@Override
	public List<? extends StrikePriceScheduleReport> evaluate(TransactionReportInstruction input) {
		List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> outputBuilder = doEvaluate(input);
		
		final List<? extends StrikePriceScheduleReport> output;
		if (outputBuilder == null) {
			output = null;
		} else {
			output = outputBuilder.stream().map(StrikePriceScheduleReport::build).collect(Collectors.toList());
			objectValidator.validate(StrikePriceScheduleReport.class, output);
		}
		
		return output;
	}

	protected abstract List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> doEvaluate(TransactionReportInstruction input);

	public static class StrikePriceScheduleRuleDefault extends StrikePriceScheduleRule {
		@Override
		protected List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> doEvaluate(TransactionReportInstruction input) {
			List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> output = new ArrayList<>();
			return assignOutput(output, input);
		}
		
		protected List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> assignOutput(List<StrikePriceScheduleReport.StrikePriceScheduleReportBuilder> output, TransactionReportInstruction input) {
			final MapperS<TransactionReportInstruction> thenArg0 = MapperS.of(input)
				.filterSingleNullSafe(item -> isAllowableAction.evaluate(item.get()));
			final MapperC<PriceReportablePeriod> thenArg1 = thenArg0
				.mapSingleToList(item -> MapperC.<PriceReportablePeriod>of(getReportableStrikePricePeriod.evaluate(item.get())));
			final MapperC<StrikePriceScheduleReport> thenArg2 = thenArg1
				.mapItem(item -> MapperS.of(StrikePriceScheduleReport.builder()
					.setEffectiveDateStrikePrice(effectiveDateStrikePriceRule.evaluate(item.get()))
					.setEndDateStrikePrice(endDateStrikePriceRule.evaluate(item.get()))
					.setStrikePriceInEffectOnAssociatedEffectiveDateMonetary(strikePriceInEffectOnAssociatedEffectiveDateMonetaryRule.evaluate(item.get()))
					.setStrikePriceInEffectOnAssociatedEffectiveDateDecimal(strikePriceInEffectOnAssociatedEffectiveDateDecimalRule.evaluate(item.get()))
					.setStrikePriceNotationInEffectOnAssociatedEffectiveDate(null)
					.build()));
			output = toBuilder(thenArg2
				.sort(item -> item.<Date>map("getEffectiveDateStrikePrice", strikePriceScheduleReport -> strikePriceScheduleReport.getEffectiveDateStrikePrice())).getMulti());
			
			return Optional.ofNullable(output)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
