package drr.regulation.common;

import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.metafields.FieldWithMetaString;
import drr.regulation.common.meta.ReportingRegimeMeta;
import drr.regulation.common.metafields.FieldWithMetaRegimeNameEnum;
import drr.regulation.common.metafields.FieldWithMetaSupervisoryBodyEnum;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * Specifies the regulatory regime requiring reporting of the transaction.
 * @version 5.57.0
 */
@RosettaDataType(value="ReportingRegime", builder=ReportingRegime.ReportingRegimeBuilderImpl.class, version="5.57.0")
@RuneDataType(value="ReportingRegime", model="drr", builder=ReportingRegime.ReportingRegimeBuilderImpl.class, version="5.57.0")
public interface ReportingRegime extends RosettaModelObject {

	ReportingRegimeMeta metaData = new ReportingRegimeMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Identifies the reporting regime under which this data is reported. For example, Dodd-Frank, MiFID, HongKongOTCDRepository, ODRF
	 */
	FieldWithMetaRegimeNameEnum getRegimeName();
	/**
	 * Identifies the specific regulator or other supervisory body for which this data is produced. For example, CFTC, SEC, UKFSA, ODRF, SFC, ESMA.
	 */
	FieldWithMetaSupervisoryBodyEnum getSupervisoryBody();
	/**
	 * Indicator of whether the transaction is mandatorily clearable.
	 */
	MandatorilyClearableEnum getMandatorilyClearable();
	/**
	 * Identifies the type of clearing exception or exemption that the counterparty has elected.
	 */
	ClearingException getClearingException();
	/**
	 * Indication of whether the transaction satisfies the definition of &#39;post-priced swap&#39;.
	 */
	Boolean getPostPricedIndicator();
	/**
	 * Party information required for reporting a transaction to the ASIC.
	 */
	ASICPartyInformation getAsicPartyInformation();
	/**
	 * Party information required for reporting a transaction to the CFTC.
	 */
	CFTCPartyInformation getCftcPartyInformation();
	/**
	 * Party information required for reporting a transaction under ESMA.
	 */
	ESMAPartyInformation getEsmaPartyInformation();
	/**
	 * Party information required for reporting a transaction under FCA.
	 */
	FCAPartyInformation getFcaPartyInformation();
	/**
	 * Party information required for reporting a transaction under SEC.
	 */
	SECPartyInformation getSecPartyInformation();
	/**
	 * Specifies the reporting role of the party with respect to the regime.
	 */
	ReportingRoleEnum getReportingRole();
	/**
	 * Specifies the Trading Capacity of the specified person with respect to the regime.
	 */
	ReportingRoleEnum getTradingCapacity();
	/**
	 * Unique technical identification of the original data for which the status is provided for the BDR.
	 */
	FieldWithMetaString getTechnicalRecordId();

	/*********************** Build Methods  ***********************/
	ReportingRegime build();
	
	ReportingRegime.ReportingRegimeBuilder toBuilder();
	
	static ReportingRegime.ReportingRegimeBuilder builder() {
		return new ReportingRegime.ReportingRegimeBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportingRegime> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportingRegime> getType() {
		return ReportingRegime.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("regimeName"), processor, FieldWithMetaRegimeNameEnum.class, getRegimeName());
		processRosetta(path.newSubPath("supervisoryBody"), processor, FieldWithMetaSupervisoryBodyEnum.class, getSupervisoryBody());
		processor.processBasic(path.newSubPath("mandatorilyClearable"), MandatorilyClearableEnum.class, getMandatorilyClearable(), this);
		processRosetta(path.newSubPath("clearingException"), processor, ClearingException.class, getClearingException());
		processor.processBasic(path.newSubPath("postPricedIndicator"), Boolean.class, getPostPricedIndicator(), this);
		processRosetta(path.newSubPath("asicPartyInformation"), processor, ASICPartyInformation.class, getAsicPartyInformation());
		processRosetta(path.newSubPath("cftcPartyInformation"), processor, CFTCPartyInformation.class, getCftcPartyInformation());
		processRosetta(path.newSubPath("esmaPartyInformation"), processor, ESMAPartyInformation.class, getEsmaPartyInformation());
		processRosetta(path.newSubPath("fcaPartyInformation"), processor, FCAPartyInformation.class, getFcaPartyInformation());
		processRosetta(path.newSubPath("secPartyInformation"), processor, SECPartyInformation.class, getSecPartyInformation());
		processor.processBasic(path.newSubPath("reportingRole"), ReportingRoleEnum.class, getReportingRole(), this);
		processor.processBasic(path.newSubPath("tradingCapacity"), ReportingRoleEnum.class, getTradingCapacity(), this);
		processRosetta(path.newSubPath("technicalRecordId"), processor, FieldWithMetaString.class, getTechnicalRecordId());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportingRegimeBuilder extends ReportingRegime, RosettaModelObjectBuilder {
		FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getOrCreateRegimeName();
		@Override
		FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getRegimeName();
		FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getOrCreateSupervisoryBody();
		@Override
		FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getSupervisoryBody();
		ClearingException.ClearingExceptionBuilder getOrCreateClearingException();
		@Override
		ClearingException.ClearingExceptionBuilder getClearingException();
		ASICPartyInformation.ASICPartyInformationBuilder getOrCreateAsicPartyInformation();
		@Override
		ASICPartyInformation.ASICPartyInformationBuilder getAsicPartyInformation();
		CFTCPartyInformation.CFTCPartyInformationBuilder getOrCreateCftcPartyInformation();
		@Override
		CFTCPartyInformation.CFTCPartyInformationBuilder getCftcPartyInformation();
		ESMAPartyInformation.ESMAPartyInformationBuilder getOrCreateEsmaPartyInformation();
		@Override
		ESMAPartyInformation.ESMAPartyInformationBuilder getEsmaPartyInformation();
		FCAPartyInformation.FCAPartyInformationBuilder getOrCreateFcaPartyInformation();
		@Override
		FCAPartyInformation.FCAPartyInformationBuilder getFcaPartyInformation();
		SECPartyInformation.SECPartyInformationBuilder getOrCreateSecPartyInformation();
		@Override
		SECPartyInformation.SECPartyInformationBuilder getSecPartyInformation();
		FieldWithMetaString.FieldWithMetaStringBuilder getOrCreateTechnicalRecordId();
		@Override
		FieldWithMetaString.FieldWithMetaStringBuilder getTechnicalRecordId();
		ReportingRegime.ReportingRegimeBuilder setRegimeName(FieldWithMetaRegimeNameEnum regimeName);
		ReportingRegime.ReportingRegimeBuilder setRegimeNameValue(RegimeNameEnum regimeName);
		ReportingRegime.ReportingRegimeBuilder setSupervisoryBody(FieldWithMetaSupervisoryBodyEnum supervisoryBody);
		ReportingRegime.ReportingRegimeBuilder setSupervisoryBodyValue(SupervisoryBodyEnum supervisoryBody);
		ReportingRegime.ReportingRegimeBuilder setMandatorilyClearable(MandatorilyClearableEnum mandatorilyClearable);
		ReportingRegime.ReportingRegimeBuilder setClearingException(ClearingException clearingException);
		ReportingRegime.ReportingRegimeBuilder setPostPricedIndicator(Boolean postPricedIndicator);
		ReportingRegime.ReportingRegimeBuilder setAsicPartyInformation(ASICPartyInformation asicPartyInformation);
		ReportingRegime.ReportingRegimeBuilder setCftcPartyInformation(CFTCPartyInformation cftcPartyInformation);
		ReportingRegime.ReportingRegimeBuilder setEsmaPartyInformation(ESMAPartyInformation esmaPartyInformation);
		ReportingRegime.ReportingRegimeBuilder setFcaPartyInformation(FCAPartyInformation fcaPartyInformation);
		ReportingRegime.ReportingRegimeBuilder setSecPartyInformation(SECPartyInformation secPartyInformation);
		ReportingRegime.ReportingRegimeBuilder setReportingRole(ReportingRoleEnum reportingRole);
		ReportingRegime.ReportingRegimeBuilder setTradingCapacity(ReportingRoleEnum tradingCapacity);
		ReportingRegime.ReportingRegimeBuilder setTechnicalRecordId(FieldWithMetaString technicalRecordId);
		ReportingRegime.ReportingRegimeBuilder setTechnicalRecordIdValue(String technicalRecordId);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("regimeName"), processor, FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder.class, getRegimeName());
			processRosetta(path.newSubPath("supervisoryBody"), processor, FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder.class, getSupervisoryBody());
			processor.processBasic(path.newSubPath("mandatorilyClearable"), MandatorilyClearableEnum.class, getMandatorilyClearable(), this);
			processRosetta(path.newSubPath("clearingException"), processor, ClearingException.ClearingExceptionBuilder.class, getClearingException());
			processor.processBasic(path.newSubPath("postPricedIndicator"), Boolean.class, getPostPricedIndicator(), this);
			processRosetta(path.newSubPath("asicPartyInformation"), processor, ASICPartyInformation.ASICPartyInformationBuilder.class, getAsicPartyInformation());
			processRosetta(path.newSubPath("cftcPartyInformation"), processor, CFTCPartyInformation.CFTCPartyInformationBuilder.class, getCftcPartyInformation());
			processRosetta(path.newSubPath("esmaPartyInformation"), processor, ESMAPartyInformation.ESMAPartyInformationBuilder.class, getEsmaPartyInformation());
			processRosetta(path.newSubPath("fcaPartyInformation"), processor, FCAPartyInformation.FCAPartyInformationBuilder.class, getFcaPartyInformation());
			processRosetta(path.newSubPath("secPartyInformation"), processor, SECPartyInformation.SECPartyInformationBuilder.class, getSecPartyInformation());
			processor.processBasic(path.newSubPath("reportingRole"), ReportingRoleEnum.class, getReportingRole(), this);
			processor.processBasic(path.newSubPath("tradingCapacity"), ReportingRoleEnum.class, getTradingCapacity(), this);
			processRosetta(path.newSubPath("technicalRecordId"), processor, FieldWithMetaString.FieldWithMetaStringBuilder.class, getTechnicalRecordId());
		}
		

		ReportingRegime.ReportingRegimeBuilder prune();
	}

	/*********************** Immutable Implementation of ReportingRegime  ***********************/
	class ReportingRegimeImpl implements ReportingRegime {
		private final FieldWithMetaRegimeNameEnum regimeName;
		private final FieldWithMetaSupervisoryBodyEnum supervisoryBody;
		private final MandatorilyClearableEnum mandatorilyClearable;
		private final ClearingException clearingException;
		private final Boolean postPricedIndicator;
		private final ASICPartyInformation asicPartyInformation;
		private final CFTCPartyInformation cftcPartyInformation;
		private final ESMAPartyInformation esmaPartyInformation;
		private final FCAPartyInformation fcaPartyInformation;
		private final SECPartyInformation secPartyInformation;
		private final ReportingRoleEnum reportingRole;
		private final ReportingRoleEnum tradingCapacity;
		private final FieldWithMetaString technicalRecordId;
		
		protected ReportingRegimeImpl(ReportingRegime.ReportingRegimeBuilder builder) {
			this.regimeName = ofNullable(builder.getRegimeName()).map(f->f.build()).orElse(null);
			this.supervisoryBody = ofNullable(builder.getSupervisoryBody()).map(f->f.build()).orElse(null);
			this.mandatorilyClearable = builder.getMandatorilyClearable();
			this.clearingException = ofNullable(builder.getClearingException()).map(f->f.build()).orElse(null);
			this.postPricedIndicator = builder.getPostPricedIndicator();
			this.asicPartyInformation = ofNullable(builder.getAsicPartyInformation()).map(f->f.build()).orElse(null);
			this.cftcPartyInformation = ofNullable(builder.getCftcPartyInformation()).map(f->f.build()).orElse(null);
			this.esmaPartyInformation = ofNullable(builder.getEsmaPartyInformation()).map(f->f.build()).orElse(null);
			this.fcaPartyInformation = ofNullable(builder.getFcaPartyInformation()).map(f->f.build()).orElse(null);
			this.secPartyInformation = ofNullable(builder.getSecPartyInformation()).map(f->f.build()).orElse(null);
			this.reportingRole = builder.getReportingRole();
			this.tradingCapacity = builder.getTradingCapacity();
			this.technicalRecordId = ofNullable(builder.getTechnicalRecordId()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		public FieldWithMetaRegimeNameEnum getRegimeName() {
			return regimeName;
		}
		
		@Override
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		public FieldWithMetaSupervisoryBodyEnum getSupervisoryBody() {
			return supervisoryBody;
		}
		
		@Override
		@RosettaAttribute("mandatorilyClearable")
		@RuneAttribute("mandatorilyClearable")
		public MandatorilyClearableEnum getMandatorilyClearable() {
			return mandatorilyClearable;
		}
		
		@Override
		@RosettaAttribute("clearingException")
		@RuneAttribute("clearingException")
		public ClearingException getClearingException() {
			return clearingException;
		}
		
		@Override
		@RosettaAttribute("postPricedIndicator")
		@RuneAttribute("postPricedIndicator")
		public Boolean getPostPricedIndicator() {
			return postPricedIndicator;
		}
		
		@Override
		@RosettaAttribute("asicPartyInformation")
		@RuneAttribute("asicPartyInformation")
		public ASICPartyInformation getAsicPartyInformation() {
			return asicPartyInformation;
		}
		
		@Override
		@RosettaAttribute("cftcPartyInformation")
		@RuneAttribute("cftcPartyInformation")
		public CFTCPartyInformation getCftcPartyInformation() {
			return cftcPartyInformation;
		}
		
		@Override
		@RosettaAttribute("esmaPartyInformation")
		@RuneAttribute("esmaPartyInformation")
		public ESMAPartyInformation getEsmaPartyInformation() {
			return esmaPartyInformation;
		}
		
		@Override
		@RosettaAttribute("fcaPartyInformation")
		@RuneAttribute("fcaPartyInformation")
		public FCAPartyInformation getFcaPartyInformation() {
			return fcaPartyInformation;
		}
		
		@Override
		@RosettaAttribute("secPartyInformation")
		@RuneAttribute("secPartyInformation")
		public SECPartyInformation getSecPartyInformation() {
			return secPartyInformation;
		}
		
		@Override
		@RosettaAttribute("reportingRole")
		@RuneAttribute("reportingRole")
		public ReportingRoleEnum getReportingRole() {
			return reportingRole;
		}
		
		@Override
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		public ReportingRoleEnum getTradingCapacity() {
			return tradingCapacity;
		}
		
		@Override
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		public FieldWithMetaString getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		public ReportingRegime build() {
			return this;
		}
		
		@Override
		public ReportingRegime.ReportingRegimeBuilder toBuilder() {
			ReportingRegime.ReportingRegimeBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportingRegime.ReportingRegimeBuilder builder) {
			ofNullable(getRegimeName()).ifPresent(builder::setRegimeName);
			ofNullable(getSupervisoryBody()).ifPresent(builder::setSupervisoryBody);
			ofNullable(getMandatorilyClearable()).ifPresent(builder::setMandatorilyClearable);
			ofNullable(getClearingException()).ifPresent(builder::setClearingException);
			ofNullable(getPostPricedIndicator()).ifPresent(builder::setPostPricedIndicator);
			ofNullable(getAsicPartyInformation()).ifPresent(builder::setAsicPartyInformation);
			ofNullable(getCftcPartyInformation()).ifPresent(builder::setCftcPartyInformation);
			ofNullable(getEsmaPartyInformation()).ifPresent(builder::setEsmaPartyInformation);
			ofNullable(getFcaPartyInformation()).ifPresent(builder::setFcaPartyInformation);
			ofNullable(getSecPartyInformation()).ifPresent(builder::setSecPartyInformation);
			ofNullable(getReportingRole()).ifPresent(builder::setReportingRole);
			ofNullable(getTradingCapacity()).ifPresent(builder::setTradingCapacity);
			ofNullable(getTechnicalRecordId()).ifPresent(builder::setTechnicalRecordId);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportingRegime _that = getType().cast(o);
		
			if (!Objects.equals(regimeName, _that.getRegimeName())) return false;
			if (!Objects.equals(supervisoryBody, _that.getSupervisoryBody())) return false;
			if (!Objects.equals(mandatorilyClearable, _that.getMandatorilyClearable())) return false;
			if (!Objects.equals(clearingException, _that.getClearingException())) return false;
			if (!Objects.equals(postPricedIndicator, _that.getPostPricedIndicator())) return false;
			if (!Objects.equals(asicPartyInformation, _that.getAsicPartyInformation())) return false;
			if (!Objects.equals(cftcPartyInformation, _that.getCftcPartyInformation())) return false;
			if (!Objects.equals(esmaPartyInformation, _that.getEsmaPartyInformation())) return false;
			if (!Objects.equals(fcaPartyInformation, _that.getFcaPartyInformation())) return false;
			if (!Objects.equals(secPartyInformation, _that.getSecPartyInformation())) return false;
			if (!Objects.equals(reportingRole, _that.getReportingRole())) return false;
			if (!Objects.equals(tradingCapacity, _that.getTradingCapacity())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (regimeName != null ? regimeName.hashCode() : 0);
			_result = 31 * _result + (supervisoryBody != null ? supervisoryBody.hashCode() : 0);
			_result = 31 * _result + (mandatorilyClearable != null ? mandatorilyClearable.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (clearingException != null ? clearingException.hashCode() : 0);
			_result = 31 * _result + (postPricedIndicator != null ? postPricedIndicator.hashCode() : 0);
			_result = 31 * _result + (asicPartyInformation != null ? asicPartyInformation.hashCode() : 0);
			_result = 31 * _result + (cftcPartyInformation != null ? cftcPartyInformation.hashCode() : 0);
			_result = 31 * _result + (esmaPartyInformation != null ? esmaPartyInformation.hashCode() : 0);
			_result = 31 * _result + (fcaPartyInformation != null ? fcaPartyInformation.hashCode() : 0);
			_result = 31 * _result + (secPartyInformation != null ? secPartyInformation.hashCode() : 0);
			_result = 31 * _result + (reportingRole != null ? reportingRole.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradingCapacity != null ? tradingCapacity.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportingRegime {" +
				"regimeName=" + this.regimeName + ", " +
				"supervisoryBody=" + this.supervisoryBody + ", " +
				"mandatorilyClearable=" + this.mandatorilyClearable + ", " +
				"clearingException=" + this.clearingException + ", " +
				"postPricedIndicator=" + this.postPricedIndicator + ", " +
				"asicPartyInformation=" + this.asicPartyInformation + ", " +
				"cftcPartyInformation=" + this.cftcPartyInformation + ", " +
				"esmaPartyInformation=" + this.esmaPartyInformation + ", " +
				"fcaPartyInformation=" + this.fcaPartyInformation + ", " +
				"secPartyInformation=" + this.secPartyInformation + ", " +
				"reportingRole=" + this.reportingRole + ", " +
				"tradingCapacity=" + this.tradingCapacity + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}';
		}
	}

	/*********************** Builder Implementation of ReportingRegime  ***********************/
	class ReportingRegimeBuilderImpl implements ReportingRegime.ReportingRegimeBuilder {
	
		protected FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder regimeName;
		protected FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder supervisoryBody;
		protected MandatorilyClearableEnum mandatorilyClearable;
		protected ClearingException.ClearingExceptionBuilder clearingException;
		protected Boolean postPricedIndicator;
		protected ASICPartyInformation.ASICPartyInformationBuilder asicPartyInformation;
		protected CFTCPartyInformation.CFTCPartyInformationBuilder cftcPartyInformation;
		protected ESMAPartyInformation.ESMAPartyInformationBuilder esmaPartyInformation;
		protected FCAPartyInformation.FCAPartyInformationBuilder fcaPartyInformation;
		protected SECPartyInformation.SECPartyInformationBuilder secPartyInformation;
		protected ReportingRoleEnum reportingRole;
		protected ReportingRoleEnum tradingCapacity;
		protected FieldWithMetaString.FieldWithMetaStringBuilder technicalRecordId;
		
		@Override
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		public FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getRegimeName() {
			return regimeName;
		}
		
		@Override
		public FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder getOrCreateRegimeName() {
			FieldWithMetaRegimeNameEnum.FieldWithMetaRegimeNameEnumBuilder result;
			if (regimeName!=null) {
				result = regimeName;
			}
			else {
				result = regimeName = FieldWithMetaRegimeNameEnum.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		public FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getSupervisoryBody() {
			return supervisoryBody;
		}
		
		@Override
		public FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder getOrCreateSupervisoryBody() {
			FieldWithMetaSupervisoryBodyEnum.FieldWithMetaSupervisoryBodyEnumBuilder result;
			if (supervisoryBody!=null) {
				result = supervisoryBody;
			}
			else {
				result = supervisoryBody = FieldWithMetaSupervisoryBodyEnum.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("mandatorilyClearable")
		@RuneAttribute("mandatorilyClearable")
		public MandatorilyClearableEnum getMandatorilyClearable() {
			return mandatorilyClearable;
		}
		
		@Override
		@RosettaAttribute("clearingException")
		@RuneAttribute("clearingException")
		public ClearingException.ClearingExceptionBuilder getClearingException() {
			return clearingException;
		}
		
		@Override
		public ClearingException.ClearingExceptionBuilder getOrCreateClearingException() {
			ClearingException.ClearingExceptionBuilder result;
			if (clearingException!=null) {
				result = clearingException;
			}
			else {
				result = clearingException = ClearingException.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("postPricedIndicator")
		@RuneAttribute("postPricedIndicator")
		public Boolean getPostPricedIndicator() {
			return postPricedIndicator;
		}
		
		@Override
		@RosettaAttribute("asicPartyInformation")
		@RuneAttribute("asicPartyInformation")
		public ASICPartyInformation.ASICPartyInformationBuilder getAsicPartyInformation() {
			return asicPartyInformation;
		}
		
		@Override
		public ASICPartyInformation.ASICPartyInformationBuilder getOrCreateAsicPartyInformation() {
			ASICPartyInformation.ASICPartyInformationBuilder result;
			if (asicPartyInformation!=null) {
				result = asicPartyInformation;
			}
			else {
				result = asicPartyInformation = ASICPartyInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("cftcPartyInformation")
		@RuneAttribute("cftcPartyInformation")
		public CFTCPartyInformation.CFTCPartyInformationBuilder getCftcPartyInformation() {
			return cftcPartyInformation;
		}
		
		@Override
		public CFTCPartyInformation.CFTCPartyInformationBuilder getOrCreateCftcPartyInformation() {
			CFTCPartyInformation.CFTCPartyInformationBuilder result;
			if (cftcPartyInformation!=null) {
				result = cftcPartyInformation;
			}
			else {
				result = cftcPartyInformation = CFTCPartyInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("esmaPartyInformation")
		@RuneAttribute("esmaPartyInformation")
		public ESMAPartyInformation.ESMAPartyInformationBuilder getEsmaPartyInformation() {
			return esmaPartyInformation;
		}
		
		@Override
		public ESMAPartyInformation.ESMAPartyInformationBuilder getOrCreateEsmaPartyInformation() {
			ESMAPartyInformation.ESMAPartyInformationBuilder result;
			if (esmaPartyInformation!=null) {
				result = esmaPartyInformation;
			}
			else {
				result = esmaPartyInformation = ESMAPartyInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("fcaPartyInformation")
		@RuneAttribute("fcaPartyInformation")
		public FCAPartyInformation.FCAPartyInformationBuilder getFcaPartyInformation() {
			return fcaPartyInformation;
		}
		
		@Override
		public FCAPartyInformation.FCAPartyInformationBuilder getOrCreateFcaPartyInformation() {
			FCAPartyInformation.FCAPartyInformationBuilder result;
			if (fcaPartyInformation!=null) {
				result = fcaPartyInformation;
			}
			else {
				result = fcaPartyInformation = FCAPartyInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("secPartyInformation")
		@RuneAttribute("secPartyInformation")
		public SECPartyInformation.SECPartyInformationBuilder getSecPartyInformation() {
			return secPartyInformation;
		}
		
		@Override
		public SECPartyInformation.SECPartyInformationBuilder getOrCreateSecPartyInformation() {
			SECPartyInformation.SECPartyInformationBuilder result;
			if (secPartyInformation!=null) {
				result = secPartyInformation;
			}
			else {
				result = secPartyInformation = SECPartyInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("reportingRole")
		@RuneAttribute("reportingRole")
		public ReportingRoleEnum getReportingRole() {
			return reportingRole;
		}
		
		@Override
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		public ReportingRoleEnum getTradingCapacity() {
			return tradingCapacity;
		}
		
		@Override
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		public FieldWithMetaString.FieldWithMetaStringBuilder getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		public FieldWithMetaString.FieldWithMetaStringBuilder getOrCreateTechnicalRecordId() {
			FieldWithMetaString.FieldWithMetaStringBuilder result;
			if (technicalRecordId!=null) {
				result = technicalRecordId;
			}
			else {
				result = technicalRecordId = FieldWithMetaString.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute(value="regimeName", isRequired=true)
		@RuneAttribute(value="regimeName", isRequired=true)
		@Override
		public ReportingRegime.ReportingRegimeBuilder setRegimeName(FieldWithMetaRegimeNameEnum _regimeName) {
			this.regimeName = _regimeName == null ? null : _regimeName.toBuilder();
			return this;
		}
		
		@Override
		public ReportingRegime.ReportingRegimeBuilder setRegimeNameValue(RegimeNameEnum _regimeName) {
			this.getOrCreateRegimeName().setValue(_regimeName);
			return this;
		}
		
		@RosettaAttribute("supervisoryBody")
		@RuneAttribute("supervisoryBody")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setSupervisoryBody(FieldWithMetaSupervisoryBodyEnum _supervisoryBody) {
			this.supervisoryBody = _supervisoryBody == null ? null : _supervisoryBody.toBuilder();
			return this;
		}
		
		@Override
		public ReportingRegime.ReportingRegimeBuilder setSupervisoryBodyValue(SupervisoryBodyEnum _supervisoryBody) {
			this.getOrCreateSupervisoryBody().setValue(_supervisoryBody);
			return this;
		}
		
		@RosettaAttribute("mandatorilyClearable")
		@RuneAttribute("mandatorilyClearable")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setMandatorilyClearable(MandatorilyClearableEnum _mandatorilyClearable) {
			this.mandatorilyClearable = _mandatorilyClearable == null ? null : _mandatorilyClearable;
			return this;
		}
		
		@RosettaAttribute("clearingException")
		@RuneAttribute("clearingException")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setClearingException(ClearingException _clearingException) {
			this.clearingException = _clearingException == null ? null : _clearingException.toBuilder();
			return this;
		}
		
		@RosettaAttribute("postPricedIndicator")
		@RuneAttribute("postPricedIndicator")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setPostPricedIndicator(Boolean _postPricedIndicator) {
			this.postPricedIndicator = _postPricedIndicator == null ? null : _postPricedIndicator;
			return this;
		}
		
		@RosettaAttribute("asicPartyInformation")
		@RuneAttribute("asicPartyInformation")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setAsicPartyInformation(ASICPartyInformation _asicPartyInformation) {
			this.asicPartyInformation = _asicPartyInformation == null ? null : _asicPartyInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("cftcPartyInformation")
		@RuneAttribute("cftcPartyInformation")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setCftcPartyInformation(CFTCPartyInformation _cftcPartyInformation) {
			this.cftcPartyInformation = _cftcPartyInformation == null ? null : _cftcPartyInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("esmaPartyInformation")
		@RuneAttribute("esmaPartyInformation")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setEsmaPartyInformation(ESMAPartyInformation _esmaPartyInformation) {
			this.esmaPartyInformation = _esmaPartyInformation == null ? null : _esmaPartyInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("fcaPartyInformation")
		@RuneAttribute("fcaPartyInformation")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setFcaPartyInformation(FCAPartyInformation _fcaPartyInformation) {
			this.fcaPartyInformation = _fcaPartyInformation == null ? null : _fcaPartyInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("secPartyInformation")
		@RuneAttribute("secPartyInformation")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setSecPartyInformation(SECPartyInformation _secPartyInformation) {
			this.secPartyInformation = _secPartyInformation == null ? null : _secPartyInformation.toBuilder();
			return this;
		}
		
		@RosettaAttribute("reportingRole")
		@RuneAttribute("reportingRole")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setReportingRole(ReportingRoleEnum _reportingRole) {
			this.reportingRole = _reportingRole == null ? null : _reportingRole;
			return this;
		}
		
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setTradingCapacity(ReportingRoleEnum _tradingCapacity) {
			this.tradingCapacity = _tradingCapacity == null ? null : _tradingCapacity;
			return this;
		}
		
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		@Override
		public ReportingRegime.ReportingRegimeBuilder setTechnicalRecordId(FieldWithMetaString _technicalRecordId) {
			this.technicalRecordId = _technicalRecordId == null ? null : _technicalRecordId.toBuilder();
			return this;
		}
		
		@Override
		public ReportingRegime.ReportingRegimeBuilder setTechnicalRecordIdValue(String _technicalRecordId) {
			this.getOrCreateTechnicalRecordId().setValue(_technicalRecordId);
			return this;
		}
		
		@Override
		public ReportingRegime build() {
			return new ReportingRegime.ReportingRegimeImpl(this);
		}
		
		@Override
		public ReportingRegime.ReportingRegimeBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportingRegime.ReportingRegimeBuilder prune() {
			if (regimeName!=null && !regimeName.prune().hasData()) regimeName = null;
			if (supervisoryBody!=null && !supervisoryBody.prune().hasData()) supervisoryBody = null;
			if (clearingException!=null && !clearingException.prune().hasData()) clearingException = null;
			if (asicPartyInformation!=null && !asicPartyInformation.prune().hasData()) asicPartyInformation = null;
			if (cftcPartyInformation!=null && !cftcPartyInformation.prune().hasData()) cftcPartyInformation = null;
			if (esmaPartyInformation!=null && !esmaPartyInformation.prune().hasData()) esmaPartyInformation = null;
			if (fcaPartyInformation!=null && !fcaPartyInformation.prune().hasData()) fcaPartyInformation = null;
			if (secPartyInformation!=null && !secPartyInformation.prune().hasData()) secPartyInformation = null;
			if (technicalRecordId!=null && !technicalRecordId.prune().hasData()) technicalRecordId = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getRegimeName()!=null) return true;
			if (getSupervisoryBody()!=null) return true;
			if (getMandatorilyClearable()!=null) return true;
			if (getClearingException()!=null && getClearingException().hasData()) return true;
			if (getPostPricedIndicator()!=null) return true;
			if (getAsicPartyInformation()!=null && getAsicPartyInformation().hasData()) return true;
			if (getCftcPartyInformation()!=null && getCftcPartyInformation().hasData()) return true;
			if (getEsmaPartyInformation()!=null && getEsmaPartyInformation().hasData()) return true;
			if (getFcaPartyInformation()!=null && getFcaPartyInformation().hasData()) return true;
			if (getSecPartyInformation()!=null && getSecPartyInformation().hasData()) return true;
			if (getReportingRole()!=null) return true;
			if (getTradingCapacity()!=null) return true;
			if (getTechnicalRecordId()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportingRegime.ReportingRegimeBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			ReportingRegime.ReportingRegimeBuilder o = (ReportingRegime.ReportingRegimeBuilder) other;
			
			merger.mergeRosetta(getRegimeName(), o.getRegimeName(), this::setRegimeName);
			merger.mergeRosetta(getSupervisoryBody(), o.getSupervisoryBody(), this::setSupervisoryBody);
			merger.mergeRosetta(getClearingException(), o.getClearingException(), this::setClearingException);
			merger.mergeRosetta(getAsicPartyInformation(), o.getAsicPartyInformation(), this::setAsicPartyInformation);
			merger.mergeRosetta(getCftcPartyInformation(), o.getCftcPartyInformation(), this::setCftcPartyInformation);
			merger.mergeRosetta(getEsmaPartyInformation(), o.getEsmaPartyInformation(), this::setEsmaPartyInformation);
			merger.mergeRosetta(getFcaPartyInformation(), o.getFcaPartyInformation(), this::setFcaPartyInformation);
			merger.mergeRosetta(getSecPartyInformation(), o.getSecPartyInformation(), this::setSecPartyInformation);
			merger.mergeRosetta(getTechnicalRecordId(), o.getTechnicalRecordId(), this::setTechnicalRecordId);
			
			merger.mergeBasic(getMandatorilyClearable(), o.getMandatorilyClearable(), this::setMandatorilyClearable);
			merger.mergeBasic(getPostPricedIndicator(), o.getPostPricedIndicator(), this::setPostPricedIndicator);
			merger.mergeBasic(getReportingRole(), o.getReportingRole(), this::setReportingRole);
			merger.mergeBasic(getTradingCapacity(), o.getTradingCapacity(), this::setTradingCapacity);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportingRegime _that = getType().cast(o);
		
			if (!Objects.equals(regimeName, _that.getRegimeName())) return false;
			if (!Objects.equals(supervisoryBody, _that.getSupervisoryBody())) return false;
			if (!Objects.equals(mandatorilyClearable, _that.getMandatorilyClearable())) return false;
			if (!Objects.equals(clearingException, _that.getClearingException())) return false;
			if (!Objects.equals(postPricedIndicator, _that.getPostPricedIndicator())) return false;
			if (!Objects.equals(asicPartyInformation, _that.getAsicPartyInformation())) return false;
			if (!Objects.equals(cftcPartyInformation, _that.getCftcPartyInformation())) return false;
			if (!Objects.equals(esmaPartyInformation, _that.getEsmaPartyInformation())) return false;
			if (!Objects.equals(fcaPartyInformation, _that.getFcaPartyInformation())) return false;
			if (!Objects.equals(secPartyInformation, _that.getSecPartyInformation())) return false;
			if (!Objects.equals(reportingRole, _that.getReportingRole())) return false;
			if (!Objects.equals(tradingCapacity, _that.getTradingCapacity())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (regimeName != null ? regimeName.hashCode() : 0);
			_result = 31 * _result + (supervisoryBody != null ? supervisoryBody.hashCode() : 0);
			_result = 31 * _result + (mandatorilyClearable != null ? mandatorilyClearable.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (clearingException != null ? clearingException.hashCode() : 0);
			_result = 31 * _result + (postPricedIndicator != null ? postPricedIndicator.hashCode() : 0);
			_result = 31 * _result + (asicPartyInformation != null ? asicPartyInformation.hashCode() : 0);
			_result = 31 * _result + (cftcPartyInformation != null ? cftcPartyInformation.hashCode() : 0);
			_result = 31 * _result + (esmaPartyInformation != null ? esmaPartyInformation.hashCode() : 0);
			_result = 31 * _result + (fcaPartyInformation != null ? fcaPartyInformation.hashCode() : 0);
			_result = 31 * _result + (secPartyInformation != null ? secPartyInformation.hashCode() : 0);
			_result = 31 * _result + (reportingRole != null ? reportingRole.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradingCapacity != null ? tradingCapacity.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportingRegimeBuilder {" +
				"regimeName=" + this.regimeName + ", " +
				"supervisoryBody=" + this.supervisoryBody + ", " +
				"mandatorilyClearable=" + this.mandatorilyClearable + ", " +
				"clearingException=" + this.clearingException + ", " +
				"postPricedIndicator=" + this.postPricedIndicator + ", " +
				"asicPartyInformation=" + this.asicPartyInformation + ", " +
				"cftcPartyInformation=" + this.cftcPartyInformation + ", " +
				"esmaPartyInformation=" + this.esmaPartyInformation + ", " +
				"fcaPartyInformation=" + this.fcaPartyInformation + ", " +
				"secPartyInformation=" + this.secPartyInformation + ", " +
				"reportingRole=" + this.reportingRole + ", " +
				"tradingCapacity=" + this.tradingCapacity + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}';
		}
	}
}
