package drr.regulation.esma.emir.refit.trade.validation.datarule;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.esma.emir.refit.trade.PackageTransactionSpread;
import drr.standards.iso.PriceNotationEnum;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperatorsNullSafe.*;

/**
 * @version 5.57.0
 */
@RosettaDataRule("PackageTransactionSpreadPercentage")
@ImplementedBy(PackageTransactionSpreadPercentage.Default.class)
public interface PackageTransactionSpreadPercentage extends Validator<PackageTransactionSpread> {
	
	String NAME = "PackageTransactionSpreadPercentage";
	String DEFINITION = "if packageTransactionSpreadNotation = PriceNotationEnum -> Percentage then packageTransactionSpreadPercentage exists";
	
	class Default implements PackageTransactionSpreadPercentage {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, PackageTransactionSpread packageTransactionSpread) {
			ComparisonResult result = executeDataRule(packageTransactionSpread);
			if (result.getOrDefault(true)) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "PackageTransactionSpread", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationResult.ValidationType.DATA_RULE, "PackageTransactionSpread", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(PackageTransactionSpread packageTransactionSpread) {
			try {
				if (areEqual(MapperS.of(packageTransactionSpread).<PriceNotationEnum>map("getPackageTransactionSpreadNotation", _packageTransactionSpread -> _packageTransactionSpread.getPackageTransactionSpreadNotation()), MapperS.of(PriceNotationEnum.PERCENTAGE), CardinalityOperator.All).getOrDefault(false)) {
					return exists(MapperS.of(packageTransactionSpread).<BigDecimal>map("getPackageTransactionSpreadPercentage", _packageTransactionSpread -> _packageTransactionSpread.getPackageTransactionSpreadPercentage()));
				}
				return ComparisonResult.ofEmpty();
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements PackageTransactionSpreadPercentage {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, PackageTransactionSpread packageTransactionSpread) {
			return Collections.emptyList();
		}
	}
}
