package iso20022.auth030.esma.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import iso20022.auth030.esma.AssetClassCommodityFreight4Choice;
import iso20022.auth030.esma.FreightCommodityContainerShip2;
import iso20022.auth030.esma.FreightCommodityDry3;
import iso20022.auth030.esma.FreightCommodityOther2;
import iso20022.auth030.esma.FreightCommodityWet3;
import java.util.List;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkCardinality;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

public class AssetClassCommodityFreight4ChoiceValidator implements Validator<AssetClassCommodityFreight4Choice> {

	private List<ComparisonResult> getComparisonResults(AssetClassCommodityFreight4Choice o) {
		return Lists.<ComparisonResult>newArrayList(
				checkCardinality("dry", (FreightCommodityDry3) o.getDry() != null ? 1 : 0, 0, 1), 
				checkCardinality("wet", (FreightCommodityWet3) o.getWet() != null ? 1 : 0, 0, 1), 
				checkCardinality("cntnrShip", (FreightCommodityContainerShip2) o.getCntnrShip() != null ? 1 : 0, 0, 1), 
				checkCardinality("othr", (FreightCommodityOther2) o.getOthr() != null ? 1 : 0, 0, 1)
			);
	}

	@Override
	public ValidationResult<AssetClassCommodityFreight4Choice> validate(RosettaPath path, AssetClassCommodityFreight4Choice o) {
		String error = getComparisonResults(o)
			.stream()
			.filter(res -> !res.get())
			.map(res -> res.getError())
			.collect(joining("; "));

		if (!isNullOrEmpty(error)) {
			return failure("AssetClassCommodityFreight4Choice", ValidationType.CARDINALITY, "AssetClassCommodityFreight4Choice", path, "", error);
		}
		return success("AssetClassCommodityFreight4Choice", ValidationType.CARDINALITY, "AssetClassCommodityFreight4Choice", path, "");
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, AssetClassCommodityFreight4Choice o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("AssetClassCommodityFreight4Choice", ValidationType.CARDINALITY, "AssetClassCommodityFreight4Choice", path, "", res.getError());
				}
				return success("AssetClassCommodityFreight4Choice", ValidationType.CARDINALITY, "AssetClassCommodityFreight4Choice", path, "");
			})
			.collect(toList());
	}

}
