package drr.regulation.mas.rewrite.trade.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.mas.rewrite.trade.MASTransactionReport;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperatorsNullSafe.checkNumber;
import static com.rosetta.model.lib.expression.ExpressionOperatorsNullSafe.checkString;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

public class MASTransactionReportTypeFormatValidator implements Validator<MASTransactionReport> {

	private List<ComparisonResult> getComparisonResults(MASTransactionReport o) {
		return Lists.<ComparisonResult>newArrayList(
				checkString("counterparty1", o.getCounterparty1(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty2", o.getCounterparty2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("reportingSpecifiedPerson", o.getReportingSpecifiedPerson(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("uniqueTransactionIdentifier", o.getUniqueTransactionIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18}[0-9]{2}[A-Z0-9]{0,32}"))), 
				checkString("uniqueTransactionIdentifierProprietary", o.getUniqueTransactionIdentifierProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("priorUTI", o.getPriorUTI(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18}[0-9]{2}[A-Z0-9]{0,32}"))), 
				checkString("priorUTIProprietary", o.getPriorUTIProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("assetClass", o.getAssetClass(), 1, of(4), empty()), 
				checkString("underlying", o.getUnderlying(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("underlyingIdentificationType", o.getUnderlyingIdentificationType(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("underlyingIdOther", o.getUnderlyingIdOther(), 1, of(210), of(Pattern.compile("[a-zA-Z0-9]{1,210}"))), 
				checkString("nameOfTheUnderlyingIndex", o.getNameOfTheUnderlyingIndex(), 1, of(50), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_\\&]{1,50}"))), 
				checkNumber("fixedRateOfLeg1", o.getFixedRateOfLeg1(), of(11), of(10), empty(), empty()), 
				checkString("identifierOfFloatingRateOfLeg1", o.getIdentifierOfFloatingRateOfLeg1(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("nameOfTheFloatingRateOfLeg1", o.getNameOfTheFloatingRateOfLeg1(), 1, of(350), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_]{1,350}"))), 
				checkNumber("fixedRateOfLeg2", o.getFixedRateOfLeg2(), of(11), of(10), empty(), empty()), 
				checkString("identifierOfFloatingRateOfLeg2", o.getIdentifierOfFloatingRateOfLeg2(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}[A-Z0-9]{9,9}[0-9]{1,1}"))), 
				checkString("nameOfTheFloatingRateOfLeg2", o.getNameOfTheFloatingRateOfLeg2(), 1, of(350), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_]{1,350}"))), 
				checkString("optionType", o.getOptionType(), 1, of(4), empty()), 
				checkString("optionStyle", o.getOptionStyle(), 1, of(4), empty()), 
				checkString("swapLinkID", o.getSwapLinkID(), 1, of(35), of(Pattern.compile("[a-zA-Z0-9]{1,35}"))), 
				checkString("packageIdentifier", o.getPackageIdentifier(), 1, of(35), of(Pattern.compile("[a-zA-Z0-9]{1,35}"))), 
				checkString("cleared", o.getCleared(), 1, of(1), of(Pattern.compile("[a-zA-Z]"))), 
				checkString("centralCounterparty", o.getCentralCounterparty(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("clearingMember", o.getClearingMember(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("platformIdentifier", o.getPlatformIdentifier(), 1, of(4), empty()), 
				checkString("bookingLocation", o.getBookingLocation(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}"))), 
				checkString("traderLocation", o.getTraderLocation(), 0, empty(), of(Pattern.compile("[A-Z]{2,2}"))), 
				checkNumber("valuationAmount", o.getValuationAmount(), of(25), of(5), empty(), empty()), 
				checkString("initialMarginCollateralPortfolioCode", o.getInitialMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("variationMarginCollateralPortfolioCode", o.getVariationMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkNumber("notionalAmountOfTheContractOrLeg1", o.getNotionalAmountOfTheContractOrLeg1(), of(25), of(5), empty(), empty()), 
				checkNumber("notionalAmountOfLeg2", o.getNotionalAmountOfLeg2(), of(25), of(5), empty(), empty()), 
				checkNumber("totalNotionalQuantityOfTheContractOrLeg1", o.getTotalNotionalQuantityOfTheContractOrLeg1(), of(25), of(5), empty(), empty()), 
				checkString("quantityUnitOfMeasureForTheContractOrLeg1", o.getQuantityUnitOfMeasureForTheContractOrLeg1(), 1, of(4), empty()), 
				checkNumber("notionalQuantityOfTheContractOrLeg1", o.getNotionalQuantityOfTheContractOrLeg1(), of(25), of(5), empty(), empty()), 
				checkString("quantityFrequencyOfTheContractOrLeg1", o.getQuantityFrequencyOfTheContractOrLeg1(), 1, of(4), empty()), 
				checkNumber("quantityFrequencyMultiplierOfTheContractOrLeg1", o.getQuantityFrequencyMultiplierOfTheContractOrLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("totalNotionalQuantityLeg2", o.getTotalNotionalQuantityLeg2(), of(25), of(5), empty(), empty()), 
				checkString("quantityUnitOfMeasureLeg2", o.getQuantityUnitOfMeasureLeg2(), 1, of(4), empty()), 
				checkNumber("notionalQuantityOfLeg2", o.getNotionalQuantityOfLeg2(), of(25), of(5), empty(), empty()), 
				checkString("quantityFrequencyOfLeg2", o.getQuantityFrequencyOfLeg2(), 1, of(4), empty()), 
				checkNumber("quantityFrequencyMultiplierOfLeg2", o.getQuantityFrequencyMultiplierOfLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("callAmount", o.getCallAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("putAmount", o.getPutAmount(), of(25), of(5), empty(), empty()), 
				checkString("priceUnitOfMeasure", o.getPriceUnitOfMeasure(), 1, of(4), empty()), 
				checkNumber("optionPremiumAmount", o.getOptionPremiumAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("exchangeRate", o.getExchangeRate(), of(18), of(13), empty(), empty()), 
				checkString("exchangeRateBasis", o.getExchangeRateBasis(), 7, of(7), of(Pattern.compile("[A-Z]{3,3}/[A-Z]{3,3}"))), 
				checkString("floatingRateResetFrequencyPeriodOfLeg1", o.getFloatingRateResetFrequencyPeriodOfLeg1(), 1, of(4), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierOfLeg1", o.getFloatingRateResetFrequencyPeriodMultiplierOfLeg1(), of(3), of(0), empty(), empty()), 
				checkString("floatingRateResetFrequencyPeriodOfLeg2", o.getFloatingRateResetFrequencyPeriodOfLeg2(), 1, of(4), empty()), 
				checkNumber("floatingRateResetFrequencyPeriodMultiplierOfLeg2", o.getFloatingRateResetFrequencyPeriodMultiplierOfLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierOfLeg1", o.getFixedRatePaymentFrequencyPeriodMultiplierOfLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("fixedRatePaymentFrequencyPeriodMultiplierOfLeg2", o.getFixedRatePaymentFrequencyPeriodMultiplierOfLeg2(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierOfLeg1", o.getFloatingRatePaymentFrequencyPeriodMultiplierOfLeg1(), of(3), of(0), empty(), empty()), 
				checkNumber("floatingRatePaymentFrequencyPeriodMultiplierOfLeg2", o.getFloatingRatePaymentFrequencyPeriodMultiplierOfLeg2(), of(3), of(0), empty(), empty()), 
				checkString("counterparty2Name", o.getCounterparty2Name(), 1, of(150), empty()), 
				checkString("executionAgentOfCounterparty2DTCC", o.getExecutionAgentOfCounterparty2DTCC(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}")))
			);
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, MASTransactionReport o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("MASTransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "MASTransactionReport", path, "", res.getError());
				}
				return success("MASTransactionReport", ValidationResult.ValidationType.TYPE_FORMAT, "MASTransactionReport", path, "");
			})
			.collect(toList());
	}

}
