/*
 * Decompiled with CFR 0.152.
 */
package com.opengamma.strata.basics.schedule;

import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.Messages;
import java.io.Serializable;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.LocalDate;
import java.time.Period;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAmount;
import java.time.temporal.TemporalUnit;
import java.util.List;
import java.util.Locale;
import org.joda.convert.FromString;
import org.joda.convert.ToString;

public final class Frequency
implements TemporalAmount,
Serializable {
    private static final long serialVersionUID = 1L;
    private static final int MAX_YEARS = 1000;
    private static final int MAX_MONTHS = 12000;
    private static final int TERM_YEARS = 10000;
    public static final Frequency P1D = Frequency.ofDays(1);
    public static final Frequency P1W = new Frequency(Period.ofWeeks(1), "P1W");
    public static final Frequency P2W = new Frequency(Period.ofWeeks(2), "P2W");
    public static final Frequency P4W = new Frequency(Period.ofWeeks(4), "P4W");
    public static final Frequency P13W = new Frequency(Period.ofWeeks(13), "P13W");
    public static final Frequency P26W = new Frequency(Period.ofWeeks(26), "P26W");
    public static final Frequency P52W = new Frequency(Period.ofWeeks(52), "P52W");
    public static final Frequency P1M = new Frequency(Period.ofMonths(1));
    public static final Frequency P2M = new Frequency(Period.ofMonths(2));
    public static final Frequency P3M = new Frequency(Period.ofMonths(3));
    public static final Frequency P4M = new Frequency(Period.ofMonths(4));
    public static final Frequency P6M = new Frequency(Period.ofMonths(6));
    public static final Frequency P12M = new Frequency(Period.ofMonths(12));
    public static final Frequency TERM = new Frequency(Period.ofYears(10000), "Term");
    private final Period period;
    private final String name;
    private final transient int eventsPerYear;
    private final transient double eventsPerYearEstimate;

    public static Frequency of(Period period) {
        ArgChecker.notNull((Object)period, (String)"period");
        int days = period.getDays();
        long months = period.toTotalMonths();
        if (months == 0L && days != 0) {
            return Frequency.ofDays(days);
        }
        if (months > 12000L) {
            throw new IllegalArgumentException("Period must not exceed 1000 years");
        }
        return new Frequency(period);
    }

    public static Frequency ofDays(int days) {
        if (days % 7 == 0) {
            return Frequency.ofWeeks(days / 7);
        }
        return new Frequency(Period.ofDays(days));
    }

    public static Frequency ofWeeks(int weeks) {
        switch (weeks) {
            case 1: {
                return P1W;
            }
            case 2: {
                return P2W;
            }
            case 4: {
                return P4W;
            }
            case 13: {
                return P13W;
            }
            case 26: {
                return P26W;
            }
            case 52: {
                return P52W;
            }
        }
        return new Frequency(Period.ofWeeks(weeks), "P" + weeks + "W");
    }

    public static Frequency ofMonths(int months) {
        switch (months) {
            case 1: {
                return P1M;
            }
            case 2: {
                return P2M;
            }
            case 3: {
                return P3M;
            }
            case 4: {
                return P4M;
            }
            case 6: {
                return P6M;
            }
            case 12: {
                return P12M;
            }
        }
        if (months > 12000) {
            throw new IllegalArgumentException(Frequency.maxMonthMsg());
        }
        return new Frequency(Period.ofMonths(months));
    }

    private static String maxMonthMsg() {
        DecimalFormat formatter = new DecimalFormat("#,###", new DecimalFormatSymbols(Locale.ENGLISH));
        return "Months must not exceed " + formatter.format(12000L);
    }

    public static Frequency ofYears(int years) {
        if (years > 1000) {
            throw new IllegalArgumentException(Frequency.maxYearMsg());
        }
        return new Frequency(Period.ofYears(years));
    }

    private static String maxYearMsg() {
        DecimalFormat formatter = new DecimalFormat("#,###", new DecimalFormatSymbols(Locale.ENGLISH));
        return "Years must not exceed " + formatter.format(1000L);
    }

    @FromString
    public static Frequency parse(String toParse) {
        ArgChecker.notNull((Object)toParse, (String)"toParse");
        if (toParse.equalsIgnoreCase("Term") || toParse.equalsIgnoreCase("0T") || toParse.equalsIgnoreCase("1T")) {
            return TERM;
        }
        String prefixed = toParse.startsWith("P") ? toParse : "P" + toParse;
        try {
            return Frequency.of(Period.parse(prefixed));
        }
        catch (DateTimeParseException ex) {
            throw new IllegalArgumentException(ex);
        }
    }

    private Frequency(Period period) {
        this(period, period.toString());
    }

    private Frequency(Period period, String name) {
        ArgChecker.notNull((Object)period, (String)"period");
        ArgChecker.isFalse((boolean)period.isZero(), (String)"Period must not be zero");
        ArgChecker.isFalse((boolean)period.isNegative(), (String)"Period must not be negative");
        this.period = period;
        this.name = name;
        long monthsLong = period.toTotalMonths();
        if (monthsLong > 12000L) {
            this.eventsPerYear = 0;
            this.eventsPerYearEstimate = 0.0;
        } else {
            int months = (int)monthsLong;
            int days = period.getDays();
            if (months > 0 && days == 0) {
                this.eventsPerYear = 12 % months == 0 ? 12 / months : -1;
                this.eventsPerYearEstimate = 12.0 / (double)months;
            } else if (days > 0 && months == 0) {
                this.eventsPerYear = 364 % days == 0 ? 364 / days : -1;
                this.eventsPerYearEstimate = 364.0 / (double)days;
            } else {
                this.eventsPerYear = -1;
                double estimatedSecs = (long)months * ChronoUnit.MONTHS.getDuration().getSeconds() + (long)days * ChronoUnit.DAYS.getDuration().getSeconds();
                this.eventsPerYearEstimate = (double)ChronoUnit.YEARS.getDuration().getSeconds() / estimatedSecs;
            }
        }
    }

    private Object readResolve() {
        if (this.equals(TERM)) {
            return TERM;
        }
        return Frequency.of(this.period);
    }

    public Period getPeriod() {
        return this.period;
    }

    public boolean isTerm() {
        return this == TERM;
    }

    public Frequency normalized() {
        Period norm = this.period.normalized();
        return norm != this.period ? Frequency.of(norm) : this;
    }

    public boolean isWeekBased() {
        return this.period.toTotalMonths() == 0L && this.period.getDays() % 7 == 0;
    }

    public boolean isMonthBased() {
        return this.period.toTotalMonths() > 0L && this.period.getDays() == 0 && !this.isTerm();
    }

    public boolean isAnnual() {
        return this.period.toTotalMonths() == 12L && this.period.getDays() == 0;
    }

    public int eventsPerYear() {
        if (this.eventsPerYear == -1) {
            throw new IllegalArgumentException("Unable to calculate events per year: " + this);
        }
        return this.eventsPerYear;
    }

    public double eventsPerYearEstimate() {
        return this.eventsPerYearEstimate;
    }

    public int exactDivide(Frequency other) {
        long accrualDays;
        long paymentDays;
        ArgChecker.notNull((Object)other, (String)"other");
        if (this.isMonthBased() && other.isMonthBased()) {
            long accrualMonths;
            long paymentMonths = this.getPeriod().toTotalMonths();
            if (paymentMonths % (accrualMonths = other.getPeriod().toTotalMonths()) == 0L) {
                return Math.toIntExact(paymentMonths / accrualMonths);
            }
        } else if (this.period.toTotalMonths() == 0L && other.period.toTotalMonths() == 0L && (paymentDays = (long)this.getPeriod().getDays()) % (accrualDays = (long)other.getPeriod().getDays()) == 0L) {
            return Math.toIntExact(paymentDays / accrualDays);
        }
        throw new IllegalArgumentException(Messages.format((String)"Frequency '{}' is not a multiple of '{}'", (Object[])new Object[]{this, other}));
    }

    @Override
    public long get(TemporalUnit unit) {
        return this.period.get(unit);
    }

    @Override
    public List<TemporalUnit> getUnits() {
        return this.period.getUnits();
    }

    @Override
    public Temporal addTo(Temporal temporal) {
        if (temporal instanceof LocalDate) {
            LocalDate date = (LocalDate)temporal;
            return date.plusMonths(this.period.toTotalMonths()).plusDays(this.period.getDays());
        }
        return this.period.addTo(temporal);
    }

    @Override
    public Temporal subtractFrom(Temporal temporal) {
        if (temporal instanceof LocalDate) {
            LocalDate date = (LocalDate)temporal;
            return date.minusMonths(this.period.toTotalMonths()).minusDays(this.period.getDays());
        }
        return this.period.subtractFrom(temporal);
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        Frequency other = (Frequency)obj;
        return this.period.equals(other.period);
    }

    public int hashCode() {
        return this.period.hashCode();
    }

    @ToString
    public String toString() {
        return this.name;
    }
}

