package cdm.base.staticdata.asset.common;

import cdm.base.staticdata.asset.common.Commodity;
import cdm.base.staticdata.asset.common.Commodity.CommodityBuilder;
import cdm.base.staticdata.asset.common.Commodity.CommodityBuilderImpl;
import cdm.base.staticdata.asset.common.Commodity.CommodityImpl;
import cdm.base.staticdata.asset.common.CommodityProductDefinition;
import cdm.base.staticdata.asset.common.CommodityProductDefinition.CommodityProductDefinitionBuilder;
import cdm.base.staticdata.asset.common.DeliveryDateParameters;
import cdm.base.staticdata.asset.common.DeliveryDateParameters.DeliveryDateParametersBuilder;
import cdm.base.staticdata.asset.common.Listing;
import cdm.base.staticdata.asset.common.Listing.ListingBuilder;
import cdm.base.staticdata.asset.common.Listing.ListingBuilderImpl;
import cdm.base.staticdata.asset.common.Listing.ListingImpl;
import cdm.base.staticdata.asset.common.ProductIdentifier;
import cdm.base.staticdata.asset.common.ProductTaxonomy;
import cdm.base.staticdata.asset.common.ProductTaxonomy.ProductTaxonomyBuilder;
import cdm.base.staticdata.asset.common.meta.CommodityMeta;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder;
import cdm.base.staticdata.party.LegalEntity;
import cdm.base.staticdata.party.LegalEntity.LegalEntityBuilder;
import cdm.observable.asset.QuotationSideEnum;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneScopedAttributeReference;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Identifies a specific commodity by referencing a product identifier or by a product definition.
 * @version 5.24.0
 */
@RosettaDataType(value="Commodity", builder=Commodity.CommodityBuilderImpl.class, version="5.24.0")
@RuneDataType(value="Commodity", model="cdm", builder=Commodity.CommodityBuilderImpl.class, version="5.24.0")
public interface Commodity extends Listing {

	CommodityMeta metaData = new CommodityMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Specifies the commodity underlier in the event that no ISDA Commodity Reference Benchmark exists.
	 */
	CommodityProductDefinition getCommodityProductDefinition();
	/**
	 * Describes the required quote type of the underlying price that will be observed. Example values include &#39;Bid, &#39;Ask&#39;, &#39;Settlement&#39; (for a futures contract) and &#39;WeightedAverage&#39; (for some published prices and indices).
	 */
	QuotationSideEnum getPriceQuoteType();
	/**
	 * Specifies the parameters for identifying the relevant contract date when the commodity reference price is a futures contract.
	 */
	DeliveryDateParameters getDeliveryDateReference();
	/**
	 * Provides additional information about the commodity underlier.
	 */
	String getDescription();

	/*********************** Build Methods  ***********************/
	Commodity build();
	
	Commodity.CommodityBuilder toBuilder();
	
	static Commodity.CommodityBuilder builder() {
		return new Commodity.CommodityBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends Commodity> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends Commodity> getType() {
		return Commodity.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.class, getProductTaxonomy());
		processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.class, getProductIdentifier());
		processRosetta(path.newSubPath("exchange"), processor, LegalEntity.class, getExchange());
		processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.class, getRelatedExchange());
		processRosetta(path.newSubPath("commodityProductDefinition"), processor, CommodityProductDefinition.class, getCommodityProductDefinition());
		processor.processBasic(path.newSubPath("priceQuoteType"), QuotationSideEnum.class, getPriceQuoteType(), this);
		processRosetta(path.newSubPath("deliveryDateReference"), processor, DeliveryDateParameters.class, getDeliveryDateReference());
		processor.processBasic(path.newSubPath("description"), String.class, getDescription(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface CommodityBuilder extends Commodity, Listing.ListingBuilder {
		CommodityProductDefinition.CommodityProductDefinitionBuilder getOrCreateCommodityProductDefinition();
		@Override
		CommodityProductDefinition.CommodityProductDefinitionBuilder getCommodityProductDefinition();
		DeliveryDateParameters.DeliveryDateParametersBuilder getOrCreateDeliveryDateReference();
		@Override
		DeliveryDateParameters.DeliveryDateParametersBuilder getDeliveryDateReference();
		@Override
		Commodity.CommodityBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy);
		@Override
		Commodity.CommodityBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy, int _idx);
		@Override
		Commodity.CommodityBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Commodity.CommodityBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Commodity.CommodityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier);
		@Override
		Commodity.CommodityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier, int _idx);
		@Override
		Commodity.CommodityBuilder addProductIdentifierValue(ProductIdentifier productIdentifier);
		@Override
		Commodity.CommodityBuilder addProductIdentifierValue(ProductIdentifier productIdentifier, int _idx);
		@Override
		Commodity.CommodityBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Commodity.CommodityBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Commodity.CommodityBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		@Override
		Commodity.CommodityBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		@Override
		Commodity.CommodityBuilder setExchange(LegalEntity exchange);
		@Override
		Commodity.CommodityBuilder addRelatedExchange(LegalEntity relatedExchange);
		@Override
		Commodity.CommodityBuilder addRelatedExchange(LegalEntity relatedExchange, int _idx);
		@Override
		Commodity.CommodityBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchange);
		@Override
		Commodity.CommodityBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchange);
		Commodity.CommodityBuilder setCommodityProductDefinition(CommodityProductDefinition commodityProductDefinition);
		Commodity.CommodityBuilder setPriceQuoteType(QuotationSideEnum priceQuoteType);
		Commodity.CommodityBuilder setDeliveryDateReference(DeliveryDateParameters deliveryDateReference);
		Commodity.CommodityBuilder setDescription(String description);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.ProductTaxonomyBuilder.class, getProductTaxonomy());
			processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder.class, getProductIdentifier());
			processRosetta(path.newSubPath("exchange"), processor, LegalEntity.LegalEntityBuilder.class, getExchange());
			processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.LegalEntityBuilder.class, getRelatedExchange());
			processRosetta(path.newSubPath("commodityProductDefinition"), processor, CommodityProductDefinition.CommodityProductDefinitionBuilder.class, getCommodityProductDefinition());
			processor.processBasic(path.newSubPath("priceQuoteType"), QuotationSideEnum.class, getPriceQuoteType(), this);
			processRosetta(path.newSubPath("deliveryDateReference"), processor, DeliveryDateParameters.DeliveryDateParametersBuilder.class, getDeliveryDateReference());
			processor.processBasic(path.newSubPath("description"), String.class, getDescription(), this);
		}
		

		Commodity.CommodityBuilder prune();
	}

	/*********************** Immutable Implementation of Commodity  ***********************/
	class CommodityImpl extends Listing.ListingImpl implements Commodity {
		private final CommodityProductDefinition commodityProductDefinition;
		private final QuotationSideEnum priceQuoteType;
		private final DeliveryDateParameters deliveryDateReference;
		private final String description;
		
		protected CommodityImpl(Commodity.CommodityBuilder builder) {
			super(builder);
			this.commodityProductDefinition = ofNullable(builder.getCommodityProductDefinition()).map(f->f.build()).orElse(null);
			this.priceQuoteType = builder.getPriceQuoteType();
			this.deliveryDateReference = ofNullable(builder.getDeliveryDateReference()).map(f->f.build()).orElse(null);
			this.description = builder.getDescription();
		}
		
		@Override
		@RosettaAttribute("commodityProductDefinition")
		@RuneAttribute("commodityProductDefinition")
		public CommodityProductDefinition getCommodityProductDefinition() {
			return commodityProductDefinition;
		}
		
		@Override
		@RosettaAttribute("priceQuoteType")
		@RuneAttribute("priceQuoteType")
		public QuotationSideEnum getPriceQuoteType() {
			return priceQuoteType;
		}
		
		@Override
		@RosettaAttribute("deliveryDateReference")
		@RuneAttribute("deliveryDateReference")
		public DeliveryDateParameters getDeliveryDateReference() {
			return deliveryDateReference;
		}
		
		@Override
		@RosettaAttribute("description")
		@RuneAttribute("description")
		public String getDescription() {
			return description;
		}
		
		@Override
		public Commodity build() {
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder toBuilder() {
			Commodity.CommodityBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(Commodity.CommodityBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getCommodityProductDefinition()).ifPresent(builder::setCommodityProductDefinition);
			ofNullable(getPriceQuoteType()).ifPresent(builder::setPriceQuoteType);
			ofNullable(getDeliveryDateReference()).ifPresent(builder::setDeliveryDateReference);
			ofNullable(getDescription()).ifPresent(builder::setDescription);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Commodity _that = getType().cast(o);
		
			if (!Objects.equals(commodityProductDefinition, _that.getCommodityProductDefinition())) return false;
			if (!Objects.equals(priceQuoteType, _that.getPriceQuoteType())) return false;
			if (!Objects.equals(deliveryDateReference, _that.getDeliveryDateReference())) return false;
			if (!Objects.equals(description, _that.getDescription())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (commodityProductDefinition != null ? commodityProductDefinition.hashCode() : 0);
			_result = 31 * _result + (priceQuoteType != null ? priceQuoteType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (deliveryDateReference != null ? deliveryDateReference.hashCode() : 0);
			_result = 31 * _result + (description != null ? description.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "Commodity {" +
				"commodityProductDefinition=" + this.commodityProductDefinition + ", " +
				"priceQuoteType=" + this.priceQuoteType + ", " +
				"deliveryDateReference=" + this.deliveryDateReference + ", " +
				"description=" + this.description +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of Commodity  ***********************/
	class CommodityBuilderImpl extends Listing.ListingBuilderImpl implements Commodity.CommodityBuilder {
	
		protected CommodityProductDefinition.CommodityProductDefinitionBuilder commodityProductDefinition;
		protected QuotationSideEnum priceQuoteType;
		protected DeliveryDateParameters.DeliveryDateParametersBuilder deliveryDateReference;
		protected String description;
		
		@Override
		@RosettaAttribute("commodityProductDefinition")
		@RuneAttribute("commodityProductDefinition")
		public CommodityProductDefinition.CommodityProductDefinitionBuilder getCommodityProductDefinition() {
			return commodityProductDefinition;
		}
		
		@Override
		public CommodityProductDefinition.CommodityProductDefinitionBuilder getOrCreateCommodityProductDefinition() {
			CommodityProductDefinition.CommodityProductDefinitionBuilder result;
			if (commodityProductDefinition!=null) {
				result = commodityProductDefinition;
			}
			else {
				result = commodityProductDefinition = CommodityProductDefinition.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("priceQuoteType")
		@RuneAttribute("priceQuoteType")
		public QuotationSideEnum getPriceQuoteType() {
			return priceQuoteType;
		}
		
		@Override
		@RosettaAttribute("deliveryDateReference")
		@RuneAttribute("deliveryDateReference")
		public DeliveryDateParameters.DeliveryDateParametersBuilder getDeliveryDateReference() {
			return deliveryDateReference;
		}
		
		@Override
		public DeliveryDateParameters.DeliveryDateParametersBuilder getOrCreateDeliveryDateReference() {
			DeliveryDateParameters.DeliveryDateParametersBuilder result;
			if (deliveryDateReference!=null) {
				result = deliveryDateReference;
			}
			else {
				result = deliveryDateReference = DeliveryDateParameters.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("description")
		@RuneAttribute("description")
		public String getDescription() {
			return description;
		}
		
		@Override
		@RosettaAttribute("productTaxonomy")
		@RuneAttribute("productTaxonomy")
		public Commodity.CommodityBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy) {
			if (_productTaxonomy != null) {
				this.productTaxonomy.add(_productTaxonomy.toBuilder());
			}
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy, int _idx) {
			getIndex(this.productTaxonomy, _idx, () -> _productTaxonomy.toBuilder());
			return this;
		}
		
		@Override 
		public Commodity.CommodityBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys != null) {
				for (final ProductTaxonomy toAdd : productTaxonomys) {
					this.productTaxonomy.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productTaxonomy")
		public Commodity.CommodityBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys == null) {
				this.productTaxonomy = new ArrayList<>();
			} else {
				this.productTaxonomy = productTaxonomys.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Commodity.CommodityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier) {
			if (_productIdentifier != null) {
				this.productIdentifier.add(_productIdentifier.toBuilder());
			}
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier, int _idx) {
			getIndex(this.productIdentifier, _idx, () -> _productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier) {
			this.getOrCreateProductIdentifier(-1).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier, int _idx) {
			this.getOrCreateProductIdentifier(_idx).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override 
		public Commodity.CommodityBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ReferenceWithMetaProductIdentifier toAdd : productIdentifiers) {
					this.productIdentifier.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Commodity.CommodityBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers == null) {
				this.productIdentifier = new ArrayList<>();
			} else {
				this.productIdentifier = productIdentifiers.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ProductIdentifier toAdd : productIdentifiers) {
					this.addProductIdentifierValue(toAdd);
				}
			}
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			this.productIdentifier.clear();
			if (productIdentifiers != null) {
				productIdentifiers.forEach(this::addProductIdentifierValue);
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("exchange")
		@RuneAttribute("exchange")
		public Commodity.CommodityBuilder setExchange(LegalEntity _exchange) {
			this.exchange = _exchange == null ? null : _exchange.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("relatedExchange")
		@RuneAttribute("relatedExchange")
		public Commodity.CommodityBuilder addRelatedExchange(LegalEntity _relatedExchange) {
			if (_relatedExchange != null) {
				this.relatedExchange.add(_relatedExchange.toBuilder());
			}
			return this;
		}
		
		@Override
		public Commodity.CommodityBuilder addRelatedExchange(LegalEntity _relatedExchange, int _idx) {
			getIndex(this.relatedExchange, _idx, () -> _relatedExchange.toBuilder());
			return this;
		}
		
		@Override 
		public Commodity.CommodityBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges != null) {
				for (final LegalEntity toAdd : relatedExchanges) {
					this.relatedExchange.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("relatedExchange")
		public Commodity.CommodityBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges == null) {
				this.relatedExchange = new ArrayList<>();
			} else {
				this.relatedExchange = relatedExchanges.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("commodityProductDefinition")
		@RuneAttribute("commodityProductDefinition")
		public Commodity.CommodityBuilder setCommodityProductDefinition(CommodityProductDefinition _commodityProductDefinition) {
			this.commodityProductDefinition = _commodityProductDefinition == null ? null : _commodityProductDefinition.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("priceQuoteType")
		@RuneAttribute("priceQuoteType")
		public Commodity.CommodityBuilder setPriceQuoteType(QuotationSideEnum _priceQuoteType) {
			this.priceQuoteType = _priceQuoteType == null ? null : _priceQuoteType;
			return this;
		}
		
		@Override
		@RosettaAttribute("deliveryDateReference")
		@RuneAttribute("deliveryDateReference")
		public Commodity.CommodityBuilder setDeliveryDateReference(DeliveryDateParameters _deliveryDateReference) {
			this.deliveryDateReference = _deliveryDateReference == null ? null : _deliveryDateReference.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("description")
		@RuneAttribute("description")
		public Commodity.CommodityBuilder setDescription(String _description) {
			this.description = _description == null ? null : _description;
			return this;
		}
		
		@Override
		public Commodity build() {
			return new Commodity.CommodityImpl(this);
		}
		
		@Override
		public Commodity.CommodityBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Commodity.CommodityBuilder prune() {
			super.prune();
			if (commodityProductDefinition!=null && !commodityProductDefinition.prune().hasData()) commodityProductDefinition = null;
			if (deliveryDateReference!=null && !deliveryDateReference.prune().hasData()) deliveryDateReference = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getCommodityProductDefinition()!=null && getCommodityProductDefinition().hasData()) return true;
			if (getPriceQuoteType()!=null) return true;
			if (getDeliveryDateReference()!=null && getDeliveryDateReference().hasData()) return true;
			if (getDescription()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Commodity.CommodityBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			
			Commodity.CommodityBuilder o = (Commodity.CommodityBuilder) other;
			
			merger.mergeRosetta(getCommodityProductDefinition(), o.getCommodityProductDefinition(), this::setCommodityProductDefinition);
			merger.mergeRosetta(getDeliveryDateReference(), o.getDeliveryDateReference(), this::setDeliveryDateReference);
			
			merger.mergeBasic(getPriceQuoteType(), o.getPriceQuoteType(), this::setPriceQuoteType);
			merger.mergeBasic(getDescription(), o.getDescription(), this::setDescription);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Commodity _that = getType().cast(o);
		
			if (!Objects.equals(commodityProductDefinition, _that.getCommodityProductDefinition())) return false;
			if (!Objects.equals(priceQuoteType, _that.getPriceQuoteType())) return false;
			if (!Objects.equals(deliveryDateReference, _that.getDeliveryDateReference())) return false;
			if (!Objects.equals(description, _that.getDescription())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (commodityProductDefinition != null ? commodityProductDefinition.hashCode() : 0);
			_result = 31 * _result + (priceQuoteType != null ? priceQuoteType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (deliveryDateReference != null ? deliveryDateReference.hashCode() : 0);
			_result = 31 * _result + (description != null ? description.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CommodityBuilder {" +
				"commodityProductDefinition=" + this.commodityProductDefinition + ", " +
				"priceQuoteType=" + this.priceQuoteType + ", " +
				"deliveryDateReference=" + this.deliveryDateReference + ", " +
				"description=" + this.description +
			'}' + " " + super.toString();
		}
	}
}
