package cdm.base.staticdata.asset.common;

import cdm.base.staticdata.asset.common.Listing;
import cdm.base.staticdata.asset.common.Listing.ListingBuilder;
import cdm.base.staticdata.asset.common.Listing.ListingBuilderImpl;
import cdm.base.staticdata.asset.common.Listing.ListingImpl;
import cdm.base.staticdata.asset.common.ProductBase;
import cdm.base.staticdata.asset.common.ProductBase.ProductBaseBuilder;
import cdm.base.staticdata.asset.common.ProductBase.ProductBaseBuilderImpl;
import cdm.base.staticdata.asset.common.ProductBase.ProductBaseImpl;
import cdm.base.staticdata.asset.common.ProductIdentifier;
import cdm.base.staticdata.asset.common.ProductTaxonomy;
import cdm.base.staticdata.asset.common.ProductTaxonomy.ProductTaxonomyBuilder;
import cdm.base.staticdata.asset.common.meta.ListingMeta;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder;
import cdm.base.staticdata.party.LegalEntity;
import cdm.base.staticdata.party.LegalEntity.LegalEntityBuilder;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneScopedAttributeReference;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.util.ListEquals;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Specifies the exchange where the asset is listed.
 * @version 5.24.0
 */
@RosettaDataType(value="Listing", builder=Listing.ListingBuilderImpl.class, version="5.24.0")
@RuneDataType(value="Listing", model="cdm", builder=Listing.ListingBuilderImpl.class, version="5.24.0")
public interface Listing extends ProductBase {

	ListingMeta metaData = new ListingMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Exchange refers to the principal financial market where the asset is listed. This is the main venue for the buying and selling of the asset, such as a stock exchange for equities or a futures exchange for commodities.
	 */
	LegalEntity getExchange();
	/**
	 * Related Exchange denotes any additional exchange or trading platform where the asset is listed. For instance, if the asset is a particular stock, the related exchange might be the exchange where the stock is listed.
	 */
	List<? extends LegalEntity> getRelatedExchange();

	/*********************** Build Methods  ***********************/
	Listing build();
	
	Listing.ListingBuilder toBuilder();
	
	static Listing.ListingBuilder builder() {
		return new Listing.ListingBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends Listing> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends Listing> getType() {
		return Listing.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.class, getProductTaxonomy());
		processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.class, getProductIdentifier());
		processRosetta(path.newSubPath("exchange"), processor, LegalEntity.class, getExchange());
		processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.class, getRelatedExchange());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ListingBuilder extends Listing, ProductBase.ProductBaseBuilder {
		LegalEntity.LegalEntityBuilder getOrCreateExchange();
		@Override
		LegalEntity.LegalEntityBuilder getExchange();
		LegalEntity.LegalEntityBuilder getOrCreateRelatedExchange(int _index);
		@Override
		List<? extends LegalEntity.LegalEntityBuilder> getRelatedExchange();
		@Override
		Listing.ListingBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy);
		@Override
		Listing.ListingBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy, int _idx);
		@Override
		Listing.ListingBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Listing.ListingBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Listing.ListingBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier);
		@Override
		Listing.ListingBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier, int _idx);
		@Override
		Listing.ListingBuilder addProductIdentifierValue(ProductIdentifier productIdentifier);
		@Override
		Listing.ListingBuilder addProductIdentifierValue(ProductIdentifier productIdentifier, int _idx);
		@Override
		Listing.ListingBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Listing.ListingBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Listing.ListingBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		@Override
		Listing.ListingBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		Listing.ListingBuilder setExchange(LegalEntity exchange);
		Listing.ListingBuilder addRelatedExchange(LegalEntity relatedExchange);
		Listing.ListingBuilder addRelatedExchange(LegalEntity relatedExchange, int _idx);
		Listing.ListingBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchange);
		Listing.ListingBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchange);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.ProductTaxonomyBuilder.class, getProductTaxonomy());
			processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder.class, getProductIdentifier());
			processRosetta(path.newSubPath("exchange"), processor, LegalEntity.LegalEntityBuilder.class, getExchange());
			processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.LegalEntityBuilder.class, getRelatedExchange());
		}
		

		Listing.ListingBuilder prune();
	}

	/*********************** Immutable Implementation of Listing  ***********************/
	class ListingImpl extends ProductBase.ProductBaseImpl implements Listing {
		private final LegalEntity exchange;
		private final List<? extends LegalEntity> relatedExchange;
		
		protected ListingImpl(Listing.ListingBuilder builder) {
			super(builder);
			this.exchange = ofNullable(builder.getExchange()).map(f->f.build()).orElse(null);
			this.relatedExchange = ofNullable(builder.getRelatedExchange()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
		}
		
		@Override
		@RosettaAttribute("exchange")
		@RuneAttribute("exchange")
		public LegalEntity getExchange() {
			return exchange;
		}
		
		@Override
		@RosettaAttribute("relatedExchange")
		@RuneAttribute("relatedExchange")
		public List<? extends LegalEntity> getRelatedExchange() {
			return relatedExchange;
		}
		
		@Override
		public Listing build() {
			return this;
		}
		
		@Override
		public Listing.ListingBuilder toBuilder() {
			Listing.ListingBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(Listing.ListingBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getExchange()).ifPresent(builder::setExchange);
			ofNullable(getRelatedExchange()).ifPresent(builder::setRelatedExchange);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Listing _that = getType().cast(o);
		
			if (!Objects.equals(exchange, _that.getExchange())) return false;
			if (!ListEquals.listEquals(relatedExchange, _that.getRelatedExchange())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (exchange != null ? exchange.hashCode() : 0);
			_result = 31 * _result + (relatedExchange != null ? relatedExchange.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "Listing {" +
				"exchange=" + this.exchange + ", " +
				"relatedExchange=" + this.relatedExchange +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of Listing  ***********************/
	class ListingBuilderImpl extends ProductBase.ProductBaseBuilderImpl implements Listing.ListingBuilder {
	
		protected LegalEntity.LegalEntityBuilder exchange;
		protected List<LegalEntity.LegalEntityBuilder> relatedExchange = new ArrayList<>();
		
		@Override
		@RosettaAttribute("exchange")
		@RuneAttribute("exchange")
		public LegalEntity.LegalEntityBuilder getExchange() {
			return exchange;
		}
		
		@Override
		public LegalEntity.LegalEntityBuilder getOrCreateExchange() {
			LegalEntity.LegalEntityBuilder result;
			if (exchange!=null) {
				result = exchange;
			}
			else {
				result = exchange = LegalEntity.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("relatedExchange")
		@RuneAttribute("relatedExchange")
		public List<? extends LegalEntity.LegalEntityBuilder> getRelatedExchange() {
			return relatedExchange;
		}
		
		@Override
		public LegalEntity.LegalEntityBuilder getOrCreateRelatedExchange(int _index) {
		
			if (relatedExchange==null) {
				this.relatedExchange = new ArrayList<>();
			}
			LegalEntity.LegalEntityBuilder result;
			return getIndex(relatedExchange, _index, () -> {
						LegalEntity.LegalEntityBuilder newRelatedExchange = LegalEntity.builder();
						return newRelatedExchange;
					});
		}
		
		@Override
		@RosettaAttribute("productTaxonomy")
		@RuneAttribute("productTaxonomy")
		public Listing.ListingBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy) {
			if (_productTaxonomy != null) {
				this.productTaxonomy.add(_productTaxonomy.toBuilder());
			}
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy, int _idx) {
			getIndex(this.productTaxonomy, _idx, () -> _productTaxonomy.toBuilder());
			return this;
		}
		
		@Override 
		public Listing.ListingBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys != null) {
				for (final ProductTaxonomy toAdd : productTaxonomys) {
					this.productTaxonomy.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productTaxonomy")
		public Listing.ListingBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys == null) {
				this.productTaxonomy = new ArrayList<>();
			} else {
				this.productTaxonomy = productTaxonomys.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Listing.ListingBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier) {
			if (_productIdentifier != null) {
				this.productIdentifier.add(_productIdentifier.toBuilder());
			}
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier, int _idx) {
			getIndex(this.productIdentifier, _idx, () -> _productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier) {
			this.getOrCreateProductIdentifier(-1).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier, int _idx) {
			this.getOrCreateProductIdentifier(_idx).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override 
		public Listing.ListingBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ReferenceWithMetaProductIdentifier toAdd : productIdentifiers) {
					this.productIdentifier.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Listing.ListingBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers == null) {
				this.productIdentifier = new ArrayList<>();
			} else {
				this.productIdentifier = productIdentifiers.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ProductIdentifier toAdd : productIdentifiers) {
					this.addProductIdentifierValue(toAdd);
				}
			}
			return this;
		}
		
		@Override
		public Listing.ListingBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			this.productIdentifier.clear();
			if (productIdentifiers != null) {
				productIdentifiers.forEach(this::addProductIdentifierValue);
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("exchange")
		@RuneAttribute("exchange")
		public Listing.ListingBuilder setExchange(LegalEntity _exchange) {
			this.exchange = _exchange == null ? null : _exchange.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("relatedExchange")
		@RuneAttribute("relatedExchange")
		public Listing.ListingBuilder addRelatedExchange(LegalEntity _relatedExchange) {
			if (_relatedExchange != null) {
				this.relatedExchange.add(_relatedExchange.toBuilder());
			}
			return this;
		}
		
		@Override
		public Listing.ListingBuilder addRelatedExchange(LegalEntity _relatedExchange, int _idx) {
			getIndex(this.relatedExchange, _idx, () -> _relatedExchange.toBuilder());
			return this;
		}
		
		@Override 
		public Listing.ListingBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges != null) {
				for (final LegalEntity toAdd : relatedExchanges) {
					this.relatedExchange.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("relatedExchange")
		public Listing.ListingBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges == null) {
				this.relatedExchange = new ArrayList<>();
			} else {
				this.relatedExchange = relatedExchanges.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public Listing build() {
			return new Listing.ListingImpl(this);
		}
		
		@Override
		public Listing.ListingBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Listing.ListingBuilder prune() {
			super.prune();
			if (exchange!=null && !exchange.prune().hasData()) exchange = null;
			relatedExchange = relatedExchange.stream().filter(b->b!=null).<LegalEntity.LegalEntityBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getExchange()!=null && getExchange().hasData()) return true;
			if (getRelatedExchange()!=null && getRelatedExchange().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Listing.ListingBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			
			Listing.ListingBuilder o = (Listing.ListingBuilder) other;
			
			merger.mergeRosetta(getExchange(), o.getExchange(), this::setExchange);
			merger.mergeRosetta(getRelatedExchange(), o.getRelatedExchange(), this::getOrCreateRelatedExchange);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Listing _that = getType().cast(o);
		
			if (!Objects.equals(exchange, _that.getExchange())) return false;
			if (!ListEquals.listEquals(relatedExchange, _that.getRelatedExchange())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (exchange != null ? exchange.hashCode() : 0);
			_result = 31 * _result + (relatedExchange != null ? relatedExchange.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ListingBuilder {" +
				"exchange=" + this.exchange + ", " +
				"relatedExchange=" + this.relatedExchange +
			'}' + " " + super.toString();
		}
	}
}
