package cdm.base.staticdata.asset.common;

import cdm.base.staticdata.asset.common.DebtType;
import cdm.base.staticdata.asset.common.DebtType.DebtTypeBuilder;
import cdm.base.staticdata.asset.common.EquityTypeEnum;
import cdm.base.staticdata.asset.common.FundProductTypeEnum;
import cdm.base.staticdata.asset.common.Listing;
import cdm.base.staticdata.asset.common.Listing.ListingBuilder;
import cdm.base.staticdata.asset.common.Listing.ListingBuilderImpl;
import cdm.base.staticdata.asset.common.Listing.ListingImpl;
import cdm.base.staticdata.asset.common.ProductIdentifier;
import cdm.base.staticdata.asset.common.ProductTaxonomy;
import cdm.base.staticdata.asset.common.ProductTaxonomy.ProductTaxonomyBuilder;
import cdm.base.staticdata.asset.common.Security;
import cdm.base.staticdata.asset.common.Security.SecurityBuilder;
import cdm.base.staticdata.asset.common.Security.SecurityBuilderImpl;
import cdm.base.staticdata.asset.common.Security.SecurityImpl;
import cdm.base.staticdata.asset.common.SecurityTypeEnum;
import cdm.base.staticdata.asset.common.meta.SecurityMeta;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder;
import cdm.base.staticdata.party.LegalEntity;
import cdm.base.staticdata.party.LegalEntity.LegalEntityBuilder;
import cdm.product.template.EconomicTerms;
import cdm.product.template.EconomicTerms.EconomicTermsBuilder;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneScopedAttributeReference;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Identifies a security by referencing a product identifier and by specifying the sector.
 * @version 5.24.0
 *
 * Body ICMA
 * Corpus MasterAgreement GMRA Global Master Repurchase Agreement GMRA 2011 "The Global Master Repurchase Agreement (GMRA) is a model legal agreement designed for parties transacting repos and is published by the International Capital Market Association (ICMA)." 
 * namingConvention "Purchased Security"
 *
 * Provision As defined in GMRA paragraph 2(oo) The Purchased Securities are the Securities sold or to be sold and any New Purchased Securities transferred by Seller to Buyer under paragraph 8 (Substitution).
 *
 */
@RosettaDataType(value="Security", builder=Security.SecurityBuilderImpl.class, version="5.24.0")
@RuneDataType(value="Security", model="cdm", builder=Security.SecurityBuilderImpl.class, version="5.24.0")
public interface Security extends Listing {

	SecurityMeta metaData = new SecurityMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Identifies the type of security using an enumerated list.
	 */
	SecurityTypeEnum getSecurityType();
	/**
	 * Identifies the type of debt and selected debt economics.
	 */
	DebtType getDebtType();
	/**
	 * Identifies the type of equity.
	 */
	EquityTypeEnum getEquityType();
	/**
	 * Identifies the type of fund.
	 */
	FundProductTypeEnum getFundType();
	/**
	 * The economic terms associated with a contractual product, i.e. the set of features that are price-forming.
	 */
	EconomicTerms getEconomicTerms();

	/*********************** Build Methods  ***********************/
	Security build();
	
	Security.SecurityBuilder toBuilder();
	
	static Security.SecurityBuilder builder() {
		return new Security.SecurityBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends Security> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends Security> getType() {
		return Security.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.class, getProductTaxonomy());
		processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.class, getProductIdentifier());
		processRosetta(path.newSubPath("exchange"), processor, LegalEntity.class, getExchange());
		processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.class, getRelatedExchange());
		processor.processBasic(path.newSubPath("securityType"), SecurityTypeEnum.class, getSecurityType(), this);
		processRosetta(path.newSubPath("debtType"), processor, DebtType.class, getDebtType());
		processor.processBasic(path.newSubPath("equityType"), EquityTypeEnum.class, getEquityType(), this);
		processor.processBasic(path.newSubPath("fundType"), FundProductTypeEnum.class, getFundType(), this);
		processRosetta(path.newSubPath("economicTerms"), processor, EconomicTerms.class, getEconomicTerms());
	}
	

	/*********************** Builder Interface  ***********************/
	interface SecurityBuilder extends Security, Listing.ListingBuilder {
		DebtType.DebtTypeBuilder getOrCreateDebtType();
		@Override
		DebtType.DebtTypeBuilder getDebtType();
		EconomicTerms.EconomicTermsBuilder getOrCreateEconomicTerms();
		@Override
		EconomicTerms.EconomicTermsBuilder getEconomicTerms();
		@Override
		Security.SecurityBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy);
		@Override
		Security.SecurityBuilder addProductTaxonomy(ProductTaxonomy productTaxonomy, int _idx);
		@Override
		Security.SecurityBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Security.SecurityBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomy);
		@Override
		Security.SecurityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier);
		@Override
		Security.SecurityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier, int _idx);
		@Override
		Security.SecurityBuilder addProductIdentifierValue(ProductIdentifier productIdentifier);
		@Override
		Security.SecurityBuilder addProductIdentifierValue(ProductIdentifier productIdentifier, int _idx);
		@Override
		Security.SecurityBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Security.SecurityBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		@Override
		Security.SecurityBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		@Override
		Security.SecurityBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		@Override
		Security.SecurityBuilder setExchange(LegalEntity exchange);
		@Override
		Security.SecurityBuilder addRelatedExchange(LegalEntity relatedExchange);
		@Override
		Security.SecurityBuilder addRelatedExchange(LegalEntity relatedExchange, int _idx);
		@Override
		Security.SecurityBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchange);
		@Override
		Security.SecurityBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchange);
		Security.SecurityBuilder setSecurityType(SecurityTypeEnum securityType);
		Security.SecurityBuilder setDebtType(DebtType debtType);
		Security.SecurityBuilder setEquityType(EquityTypeEnum equityType);
		Security.SecurityBuilder setFundType(FundProductTypeEnum fundType);
		Security.SecurityBuilder setEconomicTerms(EconomicTerms economicTerms);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("productTaxonomy"), processor, ProductTaxonomy.ProductTaxonomyBuilder.class, getProductTaxonomy());
			processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder.class, getProductIdentifier());
			processRosetta(path.newSubPath("exchange"), processor, LegalEntity.LegalEntityBuilder.class, getExchange());
			processRosetta(path.newSubPath("relatedExchange"), processor, LegalEntity.LegalEntityBuilder.class, getRelatedExchange());
			processor.processBasic(path.newSubPath("securityType"), SecurityTypeEnum.class, getSecurityType(), this);
			processRosetta(path.newSubPath("debtType"), processor, DebtType.DebtTypeBuilder.class, getDebtType());
			processor.processBasic(path.newSubPath("equityType"), EquityTypeEnum.class, getEquityType(), this);
			processor.processBasic(path.newSubPath("fundType"), FundProductTypeEnum.class, getFundType(), this);
			processRosetta(path.newSubPath("economicTerms"), processor, EconomicTerms.EconomicTermsBuilder.class, getEconomicTerms());
		}
		

		Security.SecurityBuilder prune();
	}

	/*********************** Immutable Implementation of Security  ***********************/
	class SecurityImpl extends Listing.ListingImpl implements Security {
		private final SecurityTypeEnum securityType;
		private final DebtType debtType;
		private final EquityTypeEnum equityType;
		private final FundProductTypeEnum fundType;
		private final EconomicTerms economicTerms;
		
		protected SecurityImpl(Security.SecurityBuilder builder) {
			super(builder);
			this.securityType = builder.getSecurityType();
			this.debtType = ofNullable(builder.getDebtType()).map(f->f.build()).orElse(null);
			this.equityType = builder.getEquityType();
			this.fundType = builder.getFundType();
			this.economicTerms = ofNullable(builder.getEconomicTerms()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute("securityType")
		@RuneAttribute("securityType")
		public SecurityTypeEnum getSecurityType() {
			return securityType;
		}
		
		@Override
		@RosettaAttribute("debtType")
		@RuneAttribute("debtType")
		public DebtType getDebtType() {
			return debtType;
		}
		
		@Override
		@RosettaAttribute("equityType")
		@RuneAttribute("equityType")
		public EquityTypeEnum getEquityType() {
			return equityType;
		}
		
		@Override
		@RosettaAttribute("fundType")
		@RuneAttribute("fundType")
		public FundProductTypeEnum getFundType() {
			return fundType;
		}
		
		@Override
		@RosettaAttribute("economicTerms")
		@RuneAttribute("economicTerms")
		public EconomicTerms getEconomicTerms() {
			return economicTerms;
		}
		
		@Override
		public Security build() {
			return this;
		}
		
		@Override
		public Security.SecurityBuilder toBuilder() {
			Security.SecurityBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(Security.SecurityBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getSecurityType()).ifPresent(builder::setSecurityType);
			ofNullable(getDebtType()).ifPresent(builder::setDebtType);
			ofNullable(getEquityType()).ifPresent(builder::setEquityType);
			ofNullable(getFundType()).ifPresent(builder::setFundType);
			ofNullable(getEconomicTerms()).ifPresent(builder::setEconomicTerms);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Security _that = getType().cast(o);
		
			if (!Objects.equals(securityType, _that.getSecurityType())) return false;
			if (!Objects.equals(debtType, _that.getDebtType())) return false;
			if (!Objects.equals(equityType, _that.getEquityType())) return false;
			if (!Objects.equals(fundType, _that.getFundType())) return false;
			if (!Objects.equals(economicTerms, _that.getEconomicTerms())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (securityType != null ? securityType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (debtType != null ? debtType.hashCode() : 0);
			_result = 31 * _result + (equityType != null ? equityType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (fundType != null ? fundType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (economicTerms != null ? economicTerms.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "Security {" +
				"securityType=" + this.securityType + ", " +
				"debtType=" + this.debtType + ", " +
				"equityType=" + this.equityType + ", " +
				"fundType=" + this.fundType + ", " +
				"economicTerms=" + this.economicTerms +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of Security  ***********************/
	class SecurityBuilderImpl extends Listing.ListingBuilderImpl implements Security.SecurityBuilder {
	
		protected SecurityTypeEnum securityType;
		protected DebtType.DebtTypeBuilder debtType;
		protected EquityTypeEnum equityType;
		protected FundProductTypeEnum fundType;
		protected EconomicTerms.EconomicTermsBuilder economicTerms;
		
		@Override
		@RosettaAttribute("securityType")
		@RuneAttribute("securityType")
		public SecurityTypeEnum getSecurityType() {
			return securityType;
		}
		
		@Override
		@RosettaAttribute("debtType")
		@RuneAttribute("debtType")
		public DebtType.DebtTypeBuilder getDebtType() {
			return debtType;
		}
		
		@Override
		public DebtType.DebtTypeBuilder getOrCreateDebtType() {
			DebtType.DebtTypeBuilder result;
			if (debtType!=null) {
				result = debtType;
			}
			else {
				result = debtType = DebtType.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("equityType")
		@RuneAttribute("equityType")
		public EquityTypeEnum getEquityType() {
			return equityType;
		}
		
		@Override
		@RosettaAttribute("fundType")
		@RuneAttribute("fundType")
		public FundProductTypeEnum getFundType() {
			return fundType;
		}
		
		@Override
		@RosettaAttribute("economicTerms")
		@RuneAttribute("economicTerms")
		public EconomicTerms.EconomicTermsBuilder getEconomicTerms() {
			return economicTerms;
		}
		
		@Override
		public EconomicTerms.EconomicTermsBuilder getOrCreateEconomicTerms() {
			EconomicTerms.EconomicTermsBuilder result;
			if (economicTerms!=null) {
				result = economicTerms;
			}
			else {
				result = economicTerms = EconomicTerms.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("productTaxonomy")
		@RuneAttribute("productTaxonomy")
		public Security.SecurityBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy) {
			if (_productTaxonomy != null) {
				this.productTaxonomy.add(_productTaxonomy.toBuilder());
			}
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addProductTaxonomy(ProductTaxonomy _productTaxonomy, int _idx) {
			getIndex(this.productTaxonomy, _idx, () -> _productTaxonomy.toBuilder());
			return this;
		}
		
		@Override 
		public Security.SecurityBuilder addProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys != null) {
				for (final ProductTaxonomy toAdd : productTaxonomys) {
					this.productTaxonomy.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productTaxonomy")
		public Security.SecurityBuilder setProductTaxonomy(List<? extends ProductTaxonomy> productTaxonomys) {
			if (productTaxonomys == null) {
				this.productTaxonomy = new ArrayList<>();
			} else {
				this.productTaxonomy = productTaxonomys.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Security.SecurityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier) {
			if (_productIdentifier != null) {
				this.productIdentifier.add(_productIdentifier.toBuilder());
			}
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier, int _idx) {
			getIndex(this.productIdentifier, _idx, () -> _productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier) {
			this.getOrCreateProductIdentifier(-1).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier, int _idx) {
			this.getOrCreateProductIdentifier(_idx).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override 
		public Security.SecurityBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ReferenceWithMetaProductIdentifier toAdd : productIdentifiers) {
					this.productIdentifier.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public Security.SecurityBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers == null) {
				this.productIdentifier = new ArrayList<>();
			} else {
				this.productIdentifier = productIdentifiers.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ProductIdentifier toAdd : productIdentifiers) {
					this.addProductIdentifierValue(toAdd);
				}
			}
			return this;
		}
		
		@Override
		public Security.SecurityBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			this.productIdentifier.clear();
			if (productIdentifiers != null) {
				productIdentifiers.forEach(this::addProductIdentifierValue);
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("exchange")
		@RuneAttribute("exchange")
		public Security.SecurityBuilder setExchange(LegalEntity _exchange) {
			this.exchange = _exchange == null ? null : _exchange.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("relatedExchange")
		@RuneAttribute("relatedExchange")
		public Security.SecurityBuilder addRelatedExchange(LegalEntity _relatedExchange) {
			if (_relatedExchange != null) {
				this.relatedExchange.add(_relatedExchange.toBuilder());
			}
			return this;
		}
		
		@Override
		public Security.SecurityBuilder addRelatedExchange(LegalEntity _relatedExchange, int _idx) {
			getIndex(this.relatedExchange, _idx, () -> _relatedExchange.toBuilder());
			return this;
		}
		
		@Override 
		public Security.SecurityBuilder addRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges != null) {
				for (final LegalEntity toAdd : relatedExchanges) {
					this.relatedExchange.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("relatedExchange")
		public Security.SecurityBuilder setRelatedExchange(List<? extends LegalEntity> relatedExchanges) {
			if (relatedExchanges == null) {
				this.relatedExchange = new ArrayList<>();
			} else {
				this.relatedExchange = relatedExchanges.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("securityType")
		@RuneAttribute("securityType")
		public Security.SecurityBuilder setSecurityType(SecurityTypeEnum _securityType) {
			this.securityType = _securityType == null ? null : _securityType;
			return this;
		}
		
		@Override
		@RosettaAttribute("debtType")
		@RuneAttribute("debtType")
		public Security.SecurityBuilder setDebtType(DebtType _debtType) {
			this.debtType = _debtType == null ? null : _debtType.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("equityType")
		@RuneAttribute("equityType")
		public Security.SecurityBuilder setEquityType(EquityTypeEnum _equityType) {
			this.equityType = _equityType == null ? null : _equityType;
			return this;
		}
		
		@Override
		@RosettaAttribute("fundType")
		@RuneAttribute("fundType")
		public Security.SecurityBuilder setFundType(FundProductTypeEnum _fundType) {
			this.fundType = _fundType == null ? null : _fundType;
			return this;
		}
		
		@Override
		@RosettaAttribute("economicTerms")
		@RuneAttribute("economicTerms")
		public Security.SecurityBuilder setEconomicTerms(EconomicTerms _economicTerms) {
			this.economicTerms = _economicTerms == null ? null : _economicTerms.toBuilder();
			return this;
		}
		
		@Override
		public Security build() {
			return new Security.SecurityImpl(this);
		}
		
		@Override
		public Security.SecurityBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Security.SecurityBuilder prune() {
			super.prune();
			if (debtType!=null && !debtType.prune().hasData()) debtType = null;
			if (economicTerms!=null && !economicTerms.prune().hasData()) economicTerms = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getSecurityType()!=null) return true;
			if (getDebtType()!=null && getDebtType().hasData()) return true;
			if (getEquityType()!=null) return true;
			if (getFundType()!=null) return true;
			if (getEconomicTerms()!=null && getEconomicTerms().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Security.SecurityBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			
			Security.SecurityBuilder o = (Security.SecurityBuilder) other;
			
			merger.mergeRosetta(getDebtType(), o.getDebtType(), this::setDebtType);
			merger.mergeRosetta(getEconomicTerms(), o.getEconomicTerms(), this::setEconomicTerms);
			
			merger.mergeBasic(getSecurityType(), o.getSecurityType(), this::setSecurityType);
			merger.mergeBasic(getEquityType(), o.getEquityType(), this::setEquityType);
			merger.mergeBasic(getFundType(), o.getFundType(), this::setFundType);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			Security _that = getType().cast(o);
		
			if (!Objects.equals(securityType, _that.getSecurityType())) return false;
			if (!Objects.equals(debtType, _that.getDebtType())) return false;
			if (!Objects.equals(equityType, _that.getEquityType())) return false;
			if (!Objects.equals(fundType, _that.getFundType())) return false;
			if (!Objects.equals(economicTerms, _that.getEconomicTerms())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (securityType != null ? securityType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (debtType != null ? debtType.hashCode() : 0);
			_result = 31 * _result + (equityType != null ? equityType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (fundType != null ? fundType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (economicTerms != null ? economicTerms.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "SecurityBuilder {" +
				"securityType=" + this.securityType + ", " +
				"debtType=" + this.debtType + ", " +
				"equityType=" + this.equityType + ", " +
				"fundType=" + this.fundType + ", " +
				"economicTerms=" + this.economicTerms +
			'}' + " " + super.toString();
		}
	}
}
