package cdm.observable.asset.validation.datarule;

import cdm.observable.asset.CrossRate;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("CrossRateCrossRate")
@ImplementedBy(CrossRateCrossRate.Default.class)
public interface CrossRateCrossRate extends Validator<CrossRate> {
	
	String NAME = "CrossRateCrossRate";
	String DEFINITION = "if forwardPoints exists then spotRate exists";
	
	class Default implements CrossRateCrossRate {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, CrossRate crossRate) {
			ComparisonResult result = executeDataRule(crossRate);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "CrossRate", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "CrossRate", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(CrossRate crossRate) {
			try {
				if (exists(MapperS.of(crossRate).<BigDecimal>map("getForwardPoints", _crossRate -> _crossRate.getForwardPoints())).getOrDefault(false)) {
					return exists(MapperS.of(crossRate).<BigDecimal>map("getSpotRate", _crossRate -> _crossRate.getSpotRate()));
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements CrossRateCrossRate {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, CrossRate crossRate) {
			return Collections.emptyList();
		}
	}
}
