package cdm.product.collateral;

import cdm.base.staticdata.party.CounterpartyRoleEnum;
import cdm.product.collateral.AssetCriteria;
import cdm.product.collateral.AssetCriteria.AssetCriteriaBuilder;
import cdm.product.collateral.CollateralCriteriaBase;
import cdm.product.collateral.CollateralCriteriaBase.CollateralCriteriaBaseBuilder;
import cdm.product.collateral.CollateralCriteriaBase.CollateralCriteriaBaseBuilderImpl;
import cdm.product.collateral.CollateralCriteriaBase.CollateralCriteriaBaseImpl;
import cdm.product.collateral.IssuerCriteria;
import cdm.product.collateral.IssuerCriteria.IssuerCriteriaBuilder;
import cdm.product.collateral.meta.CollateralCriteriaBaseMeta;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.util.ListEquals;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Represents a set of criteria used to specify and describe collateral.
 * @version 5.24.0
 */
@RosettaDataType(value="CollateralCriteriaBase", builder=CollateralCriteriaBase.CollateralCriteriaBaseBuilderImpl.class, version="5.24.0")
@RuneDataType(value="CollateralCriteriaBase", model="cdm", builder=CollateralCriteriaBase.CollateralCriteriaBaseBuilderImpl.class, version="5.24.0")
public interface CollateralCriteriaBase extends RosettaModelObject {

	CollateralCriteriaBaseMeta metaData = new CollateralCriteriaBaseMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Represents a filter based criteria related to the issuer.
	 */
	List<? extends IssuerCriteria> getIssuer();
	/**
	 * Represents a filter based on the criteria related to the asset.
	 */
	List<? extends AssetCriteria> getAsset();
	/**
	 * Specifies which of the two counterparties the criteria applies to (either one or both counterparties). This attribute is optional, in case the applicable party is already specified elsewhere within a party election.
	 */
	List<CounterpartyRoleEnum> getAppliesTo();

	/*********************** Build Methods  ***********************/
	CollateralCriteriaBase build();
	
	CollateralCriteriaBase.CollateralCriteriaBaseBuilder toBuilder();
	
	static CollateralCriteriaBase.CollateralCriteriaBaseBuilder builder() {
		return new CollateralCriteriaBase.CollateralCriteriaBaseBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends CollateralCriteriaBase> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends CollateralCriteriaBase> getType() {
		return CollateralCriteriaBase.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("issuer"), processor, IssuerCriteria.class, getIssuer());
		processRosetta(path.newSubPath("asset"), processor, AssetCriteria.class, getAsset());
		processor.processBasic(path.newSubPath("appliesTo"), CounterpartyRoleEnum.class, getAppliesTo(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface CollateralCriteriaBaseBuilder extends CollateralCriteriaBase, RosettaModelObjectBuilder {
		IssuerCriteria.IssuerCriteriaBuilder getOrCreateIssuer(int _index);
		@Override
		List<? extends IssuerCriteria.IssuerCriteriaBuilder> getIssuer();
		AssetCriteria.AssetCriteriaBuilder getOrCreateAsset(int _index);
		@Override
		List<? extends AssetCriteria.AssetCriteriaBuilder> getAsset();
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(IssuerCriteria issuer);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(IssuerCriteria issuer, int _idx);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(List<? extends IssuerCriteria> issuer);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder setIssuer(List<? extends IssuerCriteria> issuer);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(AssetCriteria asset);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(AssetCriteria asset, int _idx);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(List<? extends AssetCriteria> asset);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder setAsset(List<? extends AssetCriteria> asset);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(CounterpartyRoleEnum appliesTo);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(CounterpartyRoleEnum appliesTo, int _idx);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(List<CounterpartyRoleEnum> appliesTo);
		CollateralCriteriaBase.CollateralCriteriaBaseBuilder setAppliesTo(List<CounterpartyRoleEnum> appliesTo);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("issuer"), processor, IssuerCriteria.IssuerCriteriaBuilder.class, getIssuer());
			processRosetta(path.newSubPath("asset"), processor, AssetCriteria.AssetCriteriaBuilder.class, getAsset());
			processor.processBasic(path.newSubPath("appliesTo"), CounterpartyRoleEnum.class, getAppliesTo(), this);
		}
		

		CollateralCriteriaBase.CollateralCriteriaBaseBuilder prune();
	}

	/*********************** Immutable Implementation of CollateralCriteriaBase  ***********************/
	class CollateralCriteriaBaseImpl implements CollateralCriteriaBase {
		private final List<? extends IssuerCriteria> issuer;
		private final List<? extends AssetCriteria> asset;
		private final List<CounterpartyRoleEnum> appliesTo;
		
		protected CollateralCriteriaBaseImpl(CollateralCriteriaBase.CollateralCriteriaBaseBuilder builder) {
			this.issuer = ofNullable(builder.getIssuer()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.asset = ofNullable(builder.getAsset()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.appliesTo = ofNullable(builder.getAppliesTo()).filter(_l->!_l.isEmpty()).map(ImmutableList::copyOf).orElse(null);
		}
		
		@Override
		@RosettaAttribute("issuer")
		@RuneAttribute("issuer")
		public List<? extends IssuerCriteria> getIssuer() {
			return issuer;
		}
		
		@Override
		@RosettaAttribute("asset")
		@RuneAttribute("asset")
		public List<? extends AssetCriteria> getAsset() {
			return asset;
		}
		
		@Override
		@RosettaAttribute("appliesTo")
		@RuneAttribute("appliesTo")
		public List<CounterpartyRoleEnum> getAppliesTo() {
			return appliesTo;
		}
		
		@Override
		public CollateralCriteriaBase build() {
			return this;
		}
		
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder toBuilder() {
			CollateralCriteriaBase.CollateralCriteriaBaseBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(CollateralCriteriaBase.CollateralCriteriaBaseBuilder builder) {
			ofNullable(getIssuer()).ifPresent(builder::setIssuer);
			ofNullable(getAsset()).ifPresent(builder::setAsset);
			ofNullable(getAppliesTo()).ifPresent(builder::setAppliesTo);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CollateralCriteriaBase _that = getType().cast(o);
		
			if (!ListEquals.listEquals(issuer, _that.getIssuer())) return false;
			if (!ListEquals.listEquals(asset, _that.getAsset())) return false;
			if (!ListEquals.listEquals(appliesTo, _that.getAppliesTo())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (issuer != null ? issuer.hashCode() : 0);
			_result = 31 * _result + (asset != null ? asset.hashCode() : 0);
			_result = 31 * _result + (appliesTo != null ? appliesTo.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CollateralCriteriaBase {" +
				"issuer=" + this.issuer + ", " +
				"asset=" + this.asset + ", " +
				"appliesTo=" + this.appliesTo +
			'}';
		}
	}

	/*********************** Builder Implementation of CollateralCriteriaBase  ***********************/
	class CollateralCriteriaBaseBuilderImpl implements CollateralCriteriaBase.CollateralCriteriaBaseBuilder {
	
		protected List<IssuerCriteria.IssuerCriteriaBuilder> issuer = new ArrayList<>();
		protected List<AssetCriteria.AssetCriteriaBuilder> asset = new ArrayList<>();
		protected List<CounterpartyRoleEnum> appliesTo = new ArrayList<>();
		
		@Override
		@RosettaAttribute("issuer")
		@RuneAttribute("issuer")
		public List<? extends IssuerCriteria.IssuerCriteriaBuilder> getIssuer() {
			return issuer;
		}
		
		@Override
		public IssuerCriteria.IssuerCriteriaBuilder getOrCreateIssuer(int _index) {
		
			if (issuer==null) {
				this.issuer = new ArrayList<>();
			}
			IssuerCriteria.IssuerCriteriaBuilder result;
			return getIndex(issuer, _index, () -> {
						IssuerCriteria.IssuerCriteriaBuilder newIssuer = IssuerCriteria.builder();
						return newIssuer;
					});
		}
		
		@Override
		@RosettaAttribute("asset")
		@RuneAttribute("asset")
		public List<? extends AssetCriteria.AssetCriteriaBuilder> getAsset() {
			return asset;
		}
		
		@Override
		public AssetCriteria.AssetCriteriaBuilder getOrCreateAsset(int _index) {
		
			if (asset==null) {
				this.asset = new ArrayList<>();
			}
			AssetCriteria.AssetCriteriaBuilder result;
			return getIndex(asset, _index, () -> {
						AssetCriteria.AssetCriteriaBuilder newAsset = AssetCriteria.builder();
						return newAsset;
					});
		}
		
		@Override
		@RosettaAttribute("appliesTo")
		@RuneAttribute("appliesTo")
		public List<CounterpartyRoleEnum> getAppliesTo() {
			return appliesTo;
		}
		
		@Override
		@RosettaAttribute("issuer")
		@RuneAttribute("issuer")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(IssuerCriteria _issuer) {
			if (_issuer != null) {
				this.issuer.add(_issuer.toBuilder());
			}
			return this;
		}
		
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(IssuerCriteria _issuer, int _idx) {
			getIndex(this.issuer, _idx, () -> _issuer.toBuilder());
			return this;
		}
		
		@Override 
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addIssuer(List<? extends IssuerCriteria> issuers) {
			if (issuers != null) {
				for (final IssuerCriteria toAdd : issuers) {
					this.issuer.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("issuer")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder setIssuer(List<? extends IssuerCriteria> issuers) {
			if (issuers == null) {
				this.issuer = new ArrayList<>();
			} else {
				this.issuer = issuers.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("asset")
		@RuneAttribute("asset")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(AssetCriteria _asset) {
			if (_asset != null) {
				this.asset.add(_asset.toBuilder());
			}
			return this;
		}
		
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(AssetCriteria _asset, int _idx) {
			getIndex(this.asset, _idx, () -> _asset.toBuilder());
			return this;
		}
		
		@Override 
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAsset(List<? extends AssetCriteria> assets) {
			if (assets != null) {
				for (final AssetCriteria toAdd : assets) {
					this.asset.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("asset")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder setAsset(List<? extends AssetCriteria> assets) {
			if (assets == null) {
				this.asset = new ArrayList<>();
			} else {
				this.asset = assets.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("appliesTo")
		@RuneAttribute("appliesTo")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(CounterpartyRoleEnum _appliesTo) {
			if (_appliesTo != null) {
				this.appliesTo.add(_appliesTo);
			}
			return this;
		}
		
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(CounterpartyRoleEnum _appliesTo, int _idx) {
			getIndex(this.appliesTo, _idx, () -> _appliesTo);
			return this;
		}
		
		@Override 
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder addAppliesTo(List<CounterpartyRoleEnum> appliesTos) {
			if (appliesTos != null) {
				for (final CounterpartyRoleEnum toAdd : appliesTos) {
					this.appliesTo.add(toAdd);
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("appliesTo")
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder setAppliesTo(List<CounterpartyRoleEnum> appliesTos) {
			if (appliesTos == null) {
				this.appliesTo = new ArrayList<>();
			} else {
				this.appliesTo = appliesTos.stream()
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public CollateralCriteriaBase build() {
			return new CollateralCriteriaBase.CollateralCriteriaBaseImpl(this);
		}
		
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder prune() {
			issuer = issuer.stream().filter(b->b!=null).<IssuerCriteria.IssuerCriteriaBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			asset = asset.stream().filter(b->b!=null).<AssetCriteria.AssetCriteriaBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getIssuer()!=null && getIssuer().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getAsset()!=null && getAsset().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getAppliesTo()!=null && !getAppliesTo().isEmpty()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CollateralCriteriaBase.CollateralCriteriaBaseBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			CollateralCriteriaBase.CollateralCriteriaBaseBuilder o = (CollateralCriteriaBase.CollateralCriteriaBaseBuilder) other;
			
			merger.mergeRosetta(getIssuer(), o.getIssuer(), this::getOrCreateIssuer);
			merger.mergeRosetta(getAsset(), o.getAsset(), this::getOrCreateAsset);
			
			merger.mergeBasic(getAppliesTo(), o.getAppliesTo(), (Consumer<CounterpartyRoleEnum>) this::addAppliesTo);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CollateralCriteriaBase _that = getType().cast(o);
		
			if (!ListEquals.listEquals(issuer, _that.getIssuer())) return false;
			if (!ListEquals.listEquals(asset, _that.getAsset())) return false;
			if (!ListEquals.listEquals(appliesTo, _that.getAppliesTo())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (issuer != null ? issuer.hashCode() : 0);
			_result = 31 * _result + (asset != null ? asset.hashCode() : 0);
			_result = 31 * _result + (appliesTo != null ? appliesTo.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CollateralCriteriaBaseBuilder {" +
				"issuer=" + this.issuer + ", " +
				"asset=" + this.asset + ", " +
				"appliesTo=" + this.appliesTo +
			'}';
		}
	}
}
