package cdm.product.common.settlement.validation.datarule;

import cdm.base.math.ArithmeticOperationEnum;
import cdm.observable.asset.Observable;
import cdm.observable.asset.PriceSchedule;
import cdm.observable.asset.metafields.FieldWithMetaPriceSchedule;
import cdm.product.common.settlement.PriceQuantity;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("PriceQuantityObservableExists")
@ImplementedBy(PriceQuantityObservableExists.Default.class)
public interface PriceQuantityObservableExists extends Validator<PriceQuantity> {
	
	String NAME = "PriceQuantityObservableExists";
	String DEFINITION = "if price -> arithmeticOperator exists then observable exists";
	
	class Default implements PriceQuantityObservableExists {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, PriceQuantity priceQuantity) {
			ComparisonResult result = executeDataRule(priceQuantity);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "PriceQuantity", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "PriceQuantity", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(PriceQuantity priceQuantity) {
			try {
				if (exists(MapperS.of(priceQuantity).<FieldWithMetaPriceSchedule>mapC("getPrice", _priceQuantity -> _priceQuantity.getPrice()).<PriceSchedule>map("Type coercion", fieldWithMetaPriceSchedule -> fieldWithMetaPriceSchedule.getValue()).<ArithmeticOperationEnum>map("getArithmeticOperator", priceSchedule -> priceSchedule.getArithmeticOperator())).getOrDefault(false)) {
					return exists(MapperS.of(priceQuantity).<Observable>map("getObservable", _priceQuantity -> _priceQuantity.getObservable()));
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements PriceQuantityObservableExists {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, PriceQuantity priceQuantity) {
			return Collections.emptyList();
		}
	}
}
