package cdm.product.template;

import cdm.product.asset.CommodityPayout;
import cdm.product.asset.CommodityPayout.CommodityPayoutBuilder;
import cdm.product.asset.CreditDefaultPayout;
import cdm.product.asset.CreditDefaultPayout.CreditDefaultPayoutBuilder;
import cdm.product.asset.InterestRatePayout;
import cdm.product.asset.InterestRatePayout.InterestRatePayoutBuilder;
import cdm.product.common.settlement.Cashflow;
import cdm.product.common.settlement.Cashflow.CashflowBuilder;
import cdm.product.template.AssetPayout;
import cdm.product.template.AssetPayout.AssetPayoutBuilder;
import cdm.product.template.FixedPricePayout;
import cdm.product.template.FixedPricePayout.FixedPricePayoutBuilder;
import cdm.product.template.ForwardPayout;
import cdm.product.template.ForwardPayout.ForwardPayoutBuilder;
import cdm.product.template.OptionPayout;
import cdm.product.template.OptionPayout.OptionPayoutBuilder;
import cdm.product.template.Payout;
import cdm.product.template.Payout.PayoutBuilder;
import cdm.product.template.Payout.PayoutBuilderImpl;
import cdm.product.template.Payout.PayoutImpl;
import cdm.product.template.PerformancePayout;
import cdm.product.template.PerformancePayout.PerformancePayoutBuilder;
import cdm.product.template.SecurityPayout;
import cdm.product.template.SecurityPayout.SecurityPayoutBuilder;
import cdm.product.template.meta.PayoutMeta;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.GlobalKey;
import com.rosetta.model.lib.GlobalKey.GlobalKeyBuilder;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneMetaType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.metafields.MetaFields;
import com.rosetta.model.metafields.MetaFields.MetaFieldsBuilder;
import com.rosetta.util.ListEquals;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * A class to represent the set of future cashflow methodologies in the form of specific payout class(es) that can be associated for the purpose of specifying a financial product. For example, two interest rate payouts can be combined to specify an interest rate swap, or one interest rate payout can be combined with a credit default payout to specify a credit default swap.
 * @version 5.24.0
 */
@RosettaDataType(value="Payout", builder=Payout.PayoutBuilderImpl.class, version="5.24.0")
@RuneDataType(value="Payout", model="cdm", builder=Payout.PayoutBuilderImpl.class, version="5.24.0")
public interface Payout extends RosettaModelObject, GlobalKey {

	PayoutMeta metaData = new PayoutMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * All of the terms necessary to define and calculate a cash flow based on a fixed, a floating or an inflation index rate. The interest rate payout can be applied to interest rate swaps and FRA (which both have two associated interest rate payouts), credit default swaps (to represent the fee leg when subject to periodic payments) and equity swaps (to represent the funding leg).
	 */
	List<? extends InterestRatePayout> getInterestRatePayout();
	/**
	 * The credit default payout, which provides the details necessary for determining when a credit payout will be triggered as well as the parameters for calculating the payout and the settlement terms.
	 */
	CreditDefaultPayout getCreditDefaultPayout();
	/**
	 * The option payout.
	 */
	List<? extends OptionPayout> getOptionPayout();
	/**
	 * Defines the payout for the floating leg of a Commodity Swap.
	 */
	List<? extends CommodityPayout> getCommodityPayout();
	/**
	 * Represents a forward settling payout. The &#39;Underlier&#39; attribute captures the underlying payout, which is settled according to the &#39;SettlementTerms&#39; attribute. Both FX Spot and FX Forward should use this component.
	 */
	List<? extends ForwardPayout> getForwardPayout();
	/**
	 * Defines a payout in which one or more payouts are defined as a fixed price.
	 */
	List<? extends FixedPricePayout> getFixedPricePayout();
	/**
	 * The security payout when the product involves some form of securities, such as collateral in a securities financing transaction
	 */
	List<? extends SecurityPayout> getSecurityPayout();
	/**
	 * A cashflow between the parties to the trade. For interest rate and equity products, this corresponds to the FpML additionalPayment element. For credit default swaps, this corresponds to the FpML initialPayment element and the singlePayment element of the fee leg. For option products, it represents the FpML premium element.
	 */
	List<? extends Cashflow> getCashflow();
	/**
	 * The performance payout, which encompasses the equity price returns, dividend returns, volatility return, variance return and correlation provisions.
	 */
	List<? extends PerformancePayout> getPerformancePayout();
	/**
	 * The security payout when the product involves some form of securities, such as collateral in a securities financing transaction
	 */
	List<? extends AssetPayout> getAssetPayout();
	MetaFields getMeta();

	/*********************** Build Methods  ***********************/
	Payout build();
	
	Payout.PayoutBuilder toBuilder();
	
	static Payout.PayoutBuilder builder() {
		return new Payout.PayoutBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends Payout> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends Payout> getType() {
		return Payout.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("interestRatePayout"), processor, InterestRatePayout.class, getInterestRatePayout());
		processRosetta(path.newSubPath("creditDefaultPayout"), processor, CreditDefaultPayout.class, getCreditDefaultPayout());
		processRosetta(path.newSubPath("optionPayout"), processor, OptionPayout.class, getOptionPayout());
		processRosetta(path.newSubPath("commodityPayout"), processor, CommodityPayout.class, getCommodityPayout());
		processRosetta(path.newSubPath("forwardPayout"), processor, ForwardPayout.class, getForwardPayout());
		processRosetta(path.newSubPath("fixedPricePayout"), processor, FixedPricePayout.class, getFixedPricePayout());
		processRosetta(path.newSubPath("securityPayout"), processor, SecurityPayout.class, getSecurityPayout());
		processRosetta(path.newSubPath("cashflow"), processor, Cashflow.class, getCashflow());
		processRosetta(path.newSubPath("performancePayout"), processor, PerformancePayout.class, getPerformancePayout());
		processRosetta(path.newSubPath("assetPayout"), processor, AssetPayout.class, getAssetPayout());
		processRosetta(path.newSubPath("meta"), processor, MetaFields.class, getMeta());
	}
	

	/*********************** Builder Interface  ***********************/
	interface PayoutBuilder extends Payout, RosettaModelObjectBuilder, GlobalKey.GlobalKeyBuilder {
		InterestRatePayout.InterestRatePayoutBuilder getOrCreateInterestRatePayout(int _index);
		@Override
		List<? extends InterestRatePayout.InterestRatePayoutBuilder> getInterestRatePayout();
		CreditDefaultPayout.CreditDefaultPayoutBuilder getOrCreateCreditDefaultPayout();
		@Override
		CreditDefaultPayout.CreditDefaultPayoutBuilder getCreditDefaultPayout();
		OptionPayout.OptionPayoutBuilder getOrCreateOptionPayout(int _index);
		@Override
		List<? extends OptionPayout.OptionPayoutBuilder> getOptionPayout();
		CommodityPayout.CommodityPayoutBuilder getOrCreateCommodityPayout(int _index);
		@Override
		List<? extends CommodityPayout.CommodityPayoutBuilder> getCommodityPayout();
		ForwardPayout.ForwardPayoutBuilder getOrCreateForwardPayout(int _index);
		@Override
		List<? extends ForwardPayout.ForwardPayoutBuilder> getForwardPayout();
		FixedPricePayout.FixedPricePayoutBuilder getOrCreateFixedPricePayout(int _index);
		@Override
		List<? extends FixedPricePayout.FixedPricePayoutBuilder> getFixedPricePayout();
		SecurityPayout.SecurityPayoutBuilder getOrCreateSecurityPayout(int _index);
		@Override
		List<? extends SecurityPayout.SecurityPayoutBuilder> getSecurityPayout();
		Cashflow.CashflowBuilder getOrCreateCashflow(int _index);
		@Override
		List<? extends Cashflow.CashflowBuilder> getCashflow();
		PerformancePayout.PerformancePayoutBuilder getOrCreatePerformancePayout(int _index);
		@Override
		List<? extends PerformancePayout.PerformancePayoutBuilder> getPerformancePayout();
		AssetPayout.AssetPayoutBuilder getOrCreateAssetPayout(int _index);
		@Override
		List<? extends AssetPayout.AssetPayoutBuilder> getAssetPayout();
		MetaFields.MetaFieldsBuilder getOrCreateMeta();
		@Override
		MetaFields.MetaFieldsBuilder getMeta();
		Payout.PayoutBuilder addInterestRatePayout(InterestRatePayout interestRatePayout);
		Payout.PayoutBuilder addInterestRatePayout(InterestRatePayout interestRatePayout, int _idx);
		Payout.PayoutBuilder addInterestRatePayout(List<? extends InterestRatePayout> interestRatePayout);
		Payout.PayoutBuilder setInterestRatePayout(List<? extends InterestRatePayout> interestRatePayout);
		Payout.PayoutBuilder setCreditDefaultPayout(CreditDefaultPayout creditDefaultPayout);
		Payout.PayoutBuilder addOptionPayout(OptionPayout optionPayout);
		Payout.PayoutBuilder addOptionPayout(OptionPayout optionPayout, int _idx);
		Payout.PayoutBuilder addOptionPayout(List<? extends OptionPayout> optionPayout);
		Payout.PayoutBuilder setOptionPayout(List<? extends OptionPayout> optionPayout);
		Payout.PayoutBuilder addCommodityPayout(CommodityPayout commodityPayout);
		Payout.PayoutBuilder addCommodityPayout(CommodityPayout commodityPayout, int _idx);
		Payout.PayoutBuilder addCommodityPayout(List<? extends CommodityPayout> commodityPayout);
		Payout.PayoutBuilder setCommodityPayout(List<? extends CommodityPayout> commodityPayout);
		Payout.PayoutBuilder addForwardPayout(ForwardPayout forwardPayout);
		Payout.PayoutBuilder addForwardPayout(ForwardPayout forwardPayout, int _idx);
		Payout.PayoutBuilder addForwardPayout(List<? extends ForwardPayout> forwardPayout);
		Payout.PayoutBuilder setForwardPayout(List<? extends ForwardPayout> forwardPayout);
		Payout.PayoutBuilder addFixedPricePayout(FixedPricePayout fixedPricePayout);
		Payout.PayoutBuilder addFixedPricePayout(FixedPricePayout fixedPricePayout, int _idx);
		Payout.PayoutBuilder addFixedPricePayout(List<? extends FixedPricePayout> fixedPricePayout);
		Payout.PayoutBuilder setFixedPricePayout(List<? extends FixedPricePayout> fixedPricePayout);
		Payout.PayoutBuilder addSecurityPayout(SecurityPayout securityPayout);
		Payout.PayoutBuilder addSecurityPayout(SecurityPayout securityPayout, int _idx);
		Payout.PayoutBuilder addSecurityPayout(List<? extends SecurityPayout> securityPayout);
		Payout.PayoutBuilder setSecurityPayout(List<? extends SecurityPayout> securityPayout);
		Payout.PayoutBuilder addCashflow(Cashflow cashflow);
		Payout.PayoutBuilder addCashflow(Cashflow cashflow, int _idx);
		Payout.PayoutBuilder addCashflow(List<? extends Cashflow> cashflow);
		Payout.PayoutBuilder setCashflow(List<? extends Cashflow> cashflow);
		Payout.PayoutBuilder addPerformancePayout(PerformancePayout performancePayout);
		Payout.PayoutBuilder addPerformancePayout(PerformancePayout performancePayout, int _idx);
		Payout.PayoutBuilder addPerformancePayout(List<? extends PerformancePayout> performancePayout);
		Payout.PayoutBuilder setPerformancePayout(List<? extends PerformancePayout> performancePayout);
		Payout.PayoutBuilder addAssetPayout(AssetPayout assetPayout);
		Payout.PayoutBuilder addAssetPayout(AssetPayout assetPayout, int _idx);
		Payout.PayoutBuilder addAssetPayout(List<? extends AssetPayout> assetPayout);
		Payout.PayoutBuilder setAssetPayout(List<? extends AssetPayout> assetPayout);
		Payout.PayoutBuilder setMeta(MetaFields meta);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("interestRatePayout"), processor, InterestRatePayout.InterestRatePayoutBuilder.class, getInterestRatePayout());
			processRosetta(path.newSubPath("creditDefaultPayout"), processor, CreditDefaultPayout.CreditDefaultPayoutBuilder.class, getCreditDefaultPayout());
			processRosetta(path.newSubPath("optionPayout"), processor, OptionPayout.OptionPayoutBuilder.class, getOptionPayout());
			processRosetta(path.newSubPath("commodityPayout"), processor, CommodityPayout.CommodityPayoutBuilder.class, getCommodityPayout());
			processRosetta(path.newSubPath("forwardPayout"), processor, ForwardPayout.ForwardPayoutBuilder.class, getForwardPayout());
			processRosetta(path.newSubPath("fixedPricePayout"), processor, FixedPricePayout.FixedPricePayoutBuilder.class, getFixedPricePayout());
			processRosetta(path.newSubPath("securityPayout"), processor, SecurityPayout.SecurityPayoutBuilder.class, getSecurityPayout());
			processRosetta(path.newSubPath("cashflow"), processor, Cashflow.CashflowBuilder.class, getCashflow());
			processRosetta(path.newSubPath("performancePayout"), processor, PerformancePayout.PerformancePayoutBuilder.class, getPerformancePayout());
			processRosetta(path.newSubPath("assetPayout"), processor, AssetPayout.AssetPayoutBuilder.class, getAssetPayout());
			processRosetta(path.newSubPath("meta"), processor, MetaFields.MetaFieldsBuilder.class, getMeta());
		}
		

		Payout.PayoutBuilder prune();
	}

	/*********************** Immutable Implementation of Payout  ***********************/
	class PayoutImpl implements Payout {
		private final List<? extends InterestRatePayout> interestRatePayout;
		private final CreditDefaultPayout creditDefaultPayout;
		private final List<? extends OptionPayout> optionPayout;
		private final List<? extends CommodityPayout> commodityPayout;
		private final List<? extends ForwardPayout> forwardPayout;
		private final List<? extends FixedPricePayout> fixedPricePayout;
		private final List<? extends SecurityPayout> securityPayout;
		private final List<? extends Cashflow> cashflow;
		private final List<? extends PerformancePayout> performancePayout;
		private final List<? extends AssetPayout> assetPayout;
		private final MetaFields meta;
		
		protected PayoutImpl(Payout.PayoutBuilder builder) {
			this.interestRatePayout = ofNullable(builder.getInterestRatePayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.creditDefaultPayout = ofNullable(builder.getCreditDefaultPayout()).map(f->f.build()).orElse(null);
			this.optionPayout = ofNullable(builder.getOptionPayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.commodityPayout = ofNullable(builder.getCommodityPayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.forwardPayout = ofNullable(builder.getForwardPayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.fixedPricePayout = ofNullable(builder.getFixedPricePayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.securityPayout = ofNullable(builder.getSecurityPayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.cashflow = ofNullable(builder.getCashflow()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.performancePayout = ofNullable(builder.getPerformancePayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.assetPayout = ofNullable(builder.getAssetPayout()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.meta = ofNullable(builder.getMeta()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute("interestRatePayout")
		@RuneAttribute("interestRatePayout")
		public List<? extends InterestRatePayout> getInterestRatePayout() {
			return interestRatePayout;
		}
		
		@Override
		@RosettaAttribute("creditDefaultPayout")
		@RuneAttribute("creditDefaultPayout")
		public CreditDefaultPayout getCreditDefaultPayout() {
			return creditDefaultPayout;
		}
		
		@Override
		@RosettaAttribute("optionPayout")
		@RuneAttribute("optionPayout")
		public List<? extends OptionPayout> getOptionPayout() {
			return optionPayout;
		}
		
		@Override
		@RosettaAttribute("commodityPayout")
		@RuneAttribute("commodityPayout")
		public List<? extends CommodityPayout> getCommodityPayout() {
			return commodityPayout;
		}
		
		@Override
		@RosettaAttribute("forwardPayout")
		@RuneAttribute("forwardPayout")
		public List<? extends ForwardPayout> getForwardPayout() {
			return forwardPayout;
		}
		
		@Override
		@RosettaAttribute("fixedPricePayout")
		@RuneAttribute("fixedPricePayout")
		public List<? extends FixedPricePayout> getFixedPricePayout() {
			return fixedPricePayout;
		}
		
		@Override
		@RosettaAttribute("securityPayout")
		@RuneAttribute("securityPayout")
		public List<? extends SecurityPayout> getSecurityPayout() {
			return securityPayout;
		}
		
		@Override
		@RosettaAttribute("cashflow")
		@RuneAttribute("cashflow")
		public List<? extends Cashflow> getCashflow() {
			return cashflow;
		}
		
		@Override
		@RosettaAttribute("performancePayout")
		@RuneAttribute("performancePayout")
		public List<? extends PerformancePayout> getPerformancePayout() {
			return performancePayout;
		}
		
		@Override
		@RosettaAttribute("assetPayout")
		@RuneAttribute("assetPayout")
		public List<? extends AssetPayout> getAssetPayout() {
			return assetPayout;
		}
		
		@Override
		@RosettaAttribute("meta")
		@RuneAttribute("meta")
		@RuneMetaType
		public MetaFields getMeta() {
			return meta;
		}
		
		@Override
		public Payout build() {
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder toBuilder() {
			Payout.PayoutBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(Payout.PayoutBuilder builder) {
			ofNullable(getInterestRatePayout()).ifPresent(builder::setInterestRatePayout);
			ofNullable(getCreditDefaultPayout()).ifPresent(builder::setCreditDefaultPayout);
			ofNullable(getOptionPayout()).ifPresent(builder::setOptionPayout);
			ofNullable(getCommodityPayout()).ifPresent(builder::setCommodityPayout);
			ofNullable(getForwardPayout()).ifPresent(builder::setForwardPayout);
			ofNullable(getFixedPricePayout()).ifPresent(builder::setFixedPricePayout);
			ofNullable(getSecurityPayout()).ifPresent(builder::setSecurityPayout);
			ofNullable(getCashflow()).ifPresent(builder::setCashflow);
			ofNullable(getPerformancePayout()).ifPresent(builder::setPerformancePayout);
			ofNullable(getAssetPayout()).ifPresent(builder::setAssetPayout);
			ofNullable(getMeta()).ifPresent(builder::setMeta);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			Payout _that = getType().cast(o);
		
			if (!ListEquals.listEquals(interestRatePayout, _that.getInterestRatePayout())) return false;
			if (!Objects.equals(creditDefaultPayout, _that.getCreditDefaultPayout())) return false;
			if (!ListEquals.listEquals(optionPayout, _that.getOptionPayout())) return false;
			if (!ListEquals.listEquals(commodityPayout, _that.getCommodityPayout())) return false;
			if (!ListEquals.listEquals(forwardPayout, _that.getForwardPayout())) return false;
			if (!ListEquals.listEquals(fixedPricePayout, _that.getFixedPricePayout())) return false;
			if (!ListEquals.listEquals(securityPayout, _that.getSecurityPayout())) return false;
			if (!ListEquals.listEquals(cashflow, _that.getCashflow())) return false;
			if (!ListEquals.listEquals(performancePayout, _that.getPerformancePayout())) return false;
			if (!ListEquals.listEquals(assetPayout, _that.getAssetPayout())) return false;
			if (!Objects.equals(meta, _that.getMeta())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (interestRatePayout != null ? interestRatePayout.hashCode() : 0);
			_result = 31 * _result + (creditDefaultPayout != null ? creditDefaultPayout.hashCode() : 0);
			_result = 31 * _result + (optionPayout != null ? optionPayout.hashCode() : 0);
			_result = 31 * _result + (commodityPayout != null ? commodityPayout.hashCode() : 0);
			_result = 31 * _result + (forwardPayout != null ? forwardPayout.hashCode() : 0);
			_result = 31 * _result + (fixedPricePayout != null ? fixedPricePayout.hashCode() : 0);
			_result = 31 * _result + (securityPayout != null ? securityPayout.hashCode() : 0);
			_result = 31 * _result + (cashflow != null ? cashflow.hashCode() : 0);
			_result = 31 * _result + (performancePayout != null ? performancePayout.hashCode() : 0);
			_result = 31 * _result + (assetPayout != null ? assetPayout.hashCode() : 0);
			_result = 31 * _result + (meta != null ? meta.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "Payout {" +
				"interestRatePayout=" + this.interestRatePayout + ", " +
				"creditDefaultPayout=" + this.creditDefaultPayout + ", " +
				"optionPayout=" + this.optionPayout + ", " +
				"commodityPayout=" + this.commodityPayout + ", " +
				"forwardPayout=" + this.forwardPayout + ", " +
				"fixedPricePayout=" + this.fixedPricePayout + ", " +
				"securityPayout=" + this.securityPayout + ", " +
				"cashflow=" + this.cashflow + ", " +
				"performancePayout=" + this.performancePayout + ", " +
				"assetPayout=" + this.assetPayout + ", " +
				"meta=" + this.meta +
			'}';
		}
	}

	/*********************** Builder Implementation of Payout  ***********************/
	class PayoutBuilderImpl implements Payout.PayoutBuilder {
	
		protected List<InterestRatePayout.InterestRatePayoutBuilder> interestRatePayout = new ArrayList<>();
		protected CreditDefaultPayout.CreditDefaultPayoutBuilder creditDefaultPayout;
		protected List<OptionPayout.OptionPayoutBuilder> optionPayout = new ArrayList<>();
		protected List<CommodityPayout.CommodityPayoutBuilder> commodityPayout = new ArrayList<>();
		protected List<ForwardPayout.ForwardPayoutBuilder> forwardPayout = new ArrayList<>();
		protected List<FixedPricePayout.FixedPricePayoutBuilder> fixedPricePayout = new ArrayList<>();
		protected List<SecurityPayout.SecurityPayoutBuilder> securityPayout = new ArrayList<>();
		protected List<Cashflow.CashflowBuilder> cashflow = new ArrayList<>();
		protected List<PerformancePayout.PerformancePayoutBuilder> performancePayout = new ArrayList<>();
		protected List<AssetPayout.AssetPayoutBuilder> assetPayout = new ArrayList<>();
		protected MetaFields.MetaFieldsBuilder meta;
		
		@Override
		@RosettaAttribute("interestRatePayout")
		@RuneAttribute("interestRatePayout")
		public List<? extends InterestRatePayout.InterestRatePayoutBuilder> getInterestRatePayout() {
			return interestRatePayout;
		}
		
		@Override
		public InterestRatePayout.InterestRatePayoutBuilder getOrCreateInterestRatePayout(int _index) {
		
			if (interestRatePayout==null) {
				this.interestRatePayout = new ArrayList<>();
			}
			InterestRatePayout.InterestRatePayoutBuilder result;
			return getIndex(interestRatePayout, _index, () -> {
						InterestRatePayout.InterestRatePayoutBuilder newInterestRatePayout = InterestRatePayout.builder();
						return newInterestRatePayout;
					});
		}
		
		@Override
		@RosettaAttribute("creditDefaultPayout")
		@RuneAttribute("creditDefaultPayout")
		public CreditDefaultPayout.CreditDefaultPayoutBuilder getCreditDefaultPayout() {
			return creditDefaultPayout;
		}
		
		@Override
		public CreditDefaultPayout.CreditDefaultPayoutBuilder getOrCreateCreditDefaultPayout() {
			CreditDefaultPayout.CreditDefaultPayoutBuilder result;
			if (creditDefaultPayout!=null) {
				result = creditDefaultPayout;
			}
			else {
				result = creditDefaultPayout = CreditDefaultPayout.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("optionPayout")
		@RuneAttribute("optionPayout")
		public List<? extends OptionPayout.OptionPayoutBuilder> getOptionPayout() {
			return optionPayout;
		}
		
		@Override
		public OptionPayout.OptionPayoutBuilder getOrCreateOptionPayout(int _index) {
		
			if (optionPayout==null) {
				this.optionPayout = new ArrayList<>();
			}
			OptionPayout.OptionPayoutBuilder result;
			return getIndex(optionPayout, _index, () -> {
						OptionPayout.OptionPayoutBuilder newOptionPayout = OptionPayout.builder();
						return newOptionPayout;
					});
		}
		
		@Override
		@RosettaAttribute("commodityPayout")
		@RuneAttribute("commodityPayout")
		public List<? extends CommodityPayout.CommodityPayoutBuilder> getCommodityPayout() {
			return commodityPayout;
		}
		
		@Override
		public CommodityPayout.CommodityPayoutBuilder getOrCreateCommodityPayout(int _index) {
		
			if (commodityPayout==null) {
				this.commodityPayout = new ArrayList<>();
			}
			CommodityPayout.CommodityPayoutBuilder result;
			return getIndex(commodityPayout, _index, () -> {
						CommodityPayout.CommodityPayoutBuilder newCommodityPayout = CommodityPayout.builder();
						return newCommodityPayout;
					});
		}
		
		@Override
		@RosettaAttribute("forwardPayout")
		@RuneAttribute("forwardPayout")
		public List<? extends ForwardPayout.ForwardPayoutBuilder> getForwardPayout() {
			return forwardPayout;
		}
		
		@Override
		public ForwardPayout.ForwardPayoutBuilder getOrCreateForwardPayout(int _index) {
		
			if (forwardPayout==null) {
				this.forwardPayout = new ArrayList<>();
			}
			ForwardPayout.ForwardPayoutBuilder result;
			return getIndex(forwardPayout, _index, () -> {
						ForwardPayout.ForwardPayoutBuilder newForwardPayout = ForwardPayout.builder();
						return newForwardPayout;
					});
		}
		
		@Override
		@RosettaAttribute("fixedPricePayout")
		@RuneAttribute("fixedPricePayout")
		public List<? extends FixedPricePayout.FixedPricePayoutBuilder> getFixedPricePayout() {
			return fixedPricePayout;
		}
		
		@Override
		public FixedPricePayout.FixedPricePayoutBuilder getOrCreateFixedPricePayout(int _index) {
		
			if (fixedPricePayout==null) {
				this.fixedPricePayout = new ArrayList<>();
			}
			FixedPricePayout.FixedPricePayoutBuilder result;
			return getIndex(fixedPricePayout, _index, () -> {
						FixedPricePayout.FixedPricePayoutBuilder newFixedPricePayout = FixedPricePayout.builder();
						return newFixedPricePayout;
					});
		}
		
		@Override
		@RosettaAttribute("securityPayout")
		@RuneAttribute("securityPayout")
		public List<? extends SecurityPayout.SecurityPayoutBuilder> getSecurityPayout() {
			return securityPayout;
		}
		
		@Override
		public SecurityPayout.SecurityPayoutBuilder getOrCreateSecurityPayout(int _index) {
		
			if (securityPayout==null) {
				this.securityPayout = new ArrayList<>();
			}
			SecurityPayout.SecurityPayoutBuilder result;
			return getIndex(securityPayout, _index, () -> {
						SecurityPayout.SecurityPayoutBuilder newSecurityPayout = SecurityPayout.builder();
						return newSecurityPayout;
					});
		}
		
		@Override
		@RosettaAttribute("cashflow")
		@RuneAttribute("cashflow")
		public List<? extends Cashflow.CashflowBuilder> getCashflow() {
			return cashflow;
		}
		
		@Override
		public Cashflow.CashflowBuilder getOrCreateCashflow(int _index) {
		
			if (cashflow==null) {
				this.cashflow = new ArrayList<>();
			}
			Cashflow.CashflowBuilder result;
			return getIndex(cashflow, _index, () -> {
						Cashflow.CashflowBuilder newCashflow = Cashflow.builder();
						return newCashflow;
					});
		}
		
		@Override
		@RosettaAttribute("performancePayout")
		@RuneAttribute("performancePayout")
		public List<? extends PerformancePayout.PerformancePayoutBuilder> getPerformancePayout() {
			return performancePayout;
		}
		
		@Override
		public PerformancePayout.PerformancePayoutBuilder getOrCreatePerformancePayout(int _index) {
		
			if (performancePayout==null) {
				this.performancePayout = new ArrayList<>();
			}
			PerformancePayout.PerformancePayoutBuilder result;
			return getIndex(performancePayout, _index, () -> {
						PerformancePayout.PerformancePayoutBuilder newPerformancePayout = PerformancePayout.builder();
						return newPerformancePayout;
					});
		}
		
		@Override
		@RosettaAttribute("assetPayout")
		@RuneAttribute("assetPayout")
		public List<? extends AssetPayout.AssetPayoutBuilder> getAssetPayout() {
			return assetPayout;
		}
		
		@Override
		public AssetPayout.AssetPayoutBuilder getOrCreateAssetPayout(int _index) {
		
			if (assetPayout==null) {
				this.assetPayout = new ArrayList<>();
			}
			AssetPayout.AssetPayoutBuilder result;
			return getIndex(assetPayout, _index, () -> {
						AssetPayout.AssetPayoutBuilder newAssetPayout = AssetPayout.builder();
						return newAssetPayout;
					});
		}
		
		@Override
		@RosettaAttribute("meta")
		@RuneAttribute("meta")
		@RuneMetaType
		public MetaFields.MetaFieldsBuilder getMeta() {
			return meta;
		}
		
		@Override
		public MetaFields.MetaFieldsBuilder getOrCreateMeta() {
			MetaFields.MetaFieldsBuilder result;
			if (meta!=null) {
				result = meta;
			}
			else {
				result = meta = MetaFields.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("interestRatePayout")
		@RuneAttribute("interestRatePayout")
		public Payout.PayoutBuilder addInterestRatePayout(InterestRatePayout _interestRatePayout) {
			if (_interestRatePayout != null) {
				this.interestRatePayout.add(_interestRatePayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addInterestRatePayout(InterestRatePayout _interestRatePayout, int _idx) {
			getIndex(this.interestRatePayout, _idx, () -> _interestRatePayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addInterestRatePayout(List<? extends InterestRatePayout> interestRatePayouts) {
			if (interestRatePayouts != null) {
				for (final InterestRatePayout toAdd : interestRatePayouts) {
					this.interestRatePayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("interestRatePayout")
		public Payout.PayoutBuilder setInterestRatePayout(List<? extends InterestRatePayout> interestRatePayouts) {
			if (interestRatePayouts == null) {
				this.interestRatePayout = new ArrayList<>();
			} else {
				this.interestRatePayout = interestRatePayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("creditDefaultPayout")
		@RuneAttribute("creditDefaultPayout")
		public Payout.PayoutBuilder setCreditDefaultPayout(CreditDefaultPayout _creditDefaultPayout) {
			this.creditDefaultPayout = _creditDefaultPayout == null ? null : _creditDefaultPayout.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("optionPayout")
		@RuneAttribute("optionPayout")
		public Payout.PayoutBuilder addOptionPayout(OptionPayout _optionPayout) {
			if (_optionPayout != null) {
				this.optionPayout.add(_optionPayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addOptionPayout(OptionPayout _optionPayout, int _idx) {
			getIndex(this.optionPayout, _idx, () -> _optionPayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addOptionPayout(List<? extends OptionPayout> optionPayouts) {
			if (optionPayouts != null) {
				for (final OptionPayout toAdd : optionPayouts) {
					this.optionPayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("optionPayout")
		public Payout.PayoutBuilder setOptionPayout(List<? extends OptionPayout> optionPayouts) {
			if (optionPayouts == null) {
				this.optionPayout = new ArrayList<>();
			} else {
				this.optionPayout = optionPayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("commodityPayout")
		@RuneAttribute("commodityPayout")
		public Payout.PayoutBuilder addCommodityPayout(CommodityPayout _commodityPayout) {
			if (_commodityPayout != null) {
				this.commodityPayout.add(_commodityPayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addCommodityPayout(CommodityPayout _commodityPayout, int _idx) {
			getIndex(this.commodityPayout, _idx, () -> _commodityPayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addCommodityPayout(List<? extends CommodityPayout> commodityPayouts) {
			if (commodityPayouts != null) {
				for (final CommodityPayout toAdd : commodityPayouts) {
					this.commodityPayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("commodityPayout")
		public Payout.PayoutBuilder setCommodityPayout(List<? extends CommodityPayout> commodityPayouts) {
			if (commodityPayouts == null) {
				this.commodityPayout = new ArrayList<>();
			} else {
				this.commodityPayout = commodityPayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("forwardPayout")
		@RuneAttribute("forwardPayout")
		public Payout.PayoutBuilder addForwardPayout(ForwardPayout _forwardPayout) {
			if (_forwardPayout != null) {
				this.forwardPayout.add(_forwardPayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addForwardPayout(ForwardPayout _forwardPayout, int _idx) {
			getIndex(this.forwardPayout, _idx, () -> _forwardPayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addForwardPayout(List<? extends ForwardPayout> forwardPayouts) {
			if (forwardPayouts != null) {
				for (final ForwardPayout toAdd : forwardPayouts) {
					this.forwardPayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("forwardPayout")
		public Payout.PayoutBuilder setForwardPayout(List<? extends ForwardPayout> forwardPayouts) {
			if (forwardPayouts == null) {
				this.forwardPayout = new ArrayList<>();
			} else {
				this.forwardPayout = forwardPayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("fixedPricePayout")
		@RuneAttribute("fixedPricePayout")
		public Payout.PayoutBuilder addFixedPricePayout(FixedPricePayout _fixedPricePayout) {
			if (_fixedPricePayout != null) {
				this.fixedPricePayout.add(_fixedPricePayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addFixedPricePayout(FixedPricePayout _fixedPricePayout, int _idx) {
			getIndex(this.fixedPricePayout, _idx, () -> _fixedPricePayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addFixedPricePayout(List<? extends FixedPricePayout> fixedPricePayouts) {
			if (fixedPricePayouts != null) {
				for (final FixedPricePayout toAdd : fixedPricePayouts) {
					this.fixedPricePayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("fixedPricePayout")
		public Payout.PayoutBuilder setFixedPricePayout(List<? extends FixedPricePayout> fixedPricePayouts) {
			if (fixedPricePayouts == null) {
				this.fixedPricePayout = new ArrayList<>();
			} else {
				this.fixedPricePayout = fixedPricePayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("securityPayout")
		@RuneAttribute("securityPayout")
		public Payout.PayoutBuilder addSecurityPayout(SecurityPayout _securityPayout) {
			if (_securityPayout != null) {
				this.securityPayout.add(_securityPayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addSecurityPayout(SecurityPayout _securityPayout, int _idx) {
			getIndex(this.securityPayout, _idx, () -> _securityPayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addSecurityPayout(List<? extends SecurityPayout> securityPayouts) {
			if (securityPayouts != null) {
				for (final SecurityPayout toAdd : securityPayouts) {
					this.securityPayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("securityPayout")
		public Payout.PayoutBuilder setSecurityPayout(List<? extends SecurityPayout> securityPayouts) {
			if (securityPayouts == null) {
				this.securityPayout = new ArrayList<>();
			} else {
				this.securityPayout = securityPayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("cashflow")
		@RuneAttribute("cashflow")
		public Payout.PayoutBuilder addCashflow(Cashflow _cashflow) {
			if (_cashflow != null) {
				this.cashflow.add(_cashflow.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addCashflow(Cashflow _cashflow, int _idx) {
			getIndex(this.cashflow, _idx, () -> _cashflow.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addCashflow(List<? extends Cashflow> cashflows) {
			if (cashflows != null) {
				for (final Cashflow toAdd : cashflows) {
					this.cashflow.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("cashflow")
		public Payout.PayoutBuilder setCashflow(List<? extends Cashflow> cashflows) {
			if (cashflows == null) {
				this.cashflow = new ArrayList<>();
			} else {
				this.cashflow = cashflows.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("performancePayout")
		@RuneAttribute("performancePayout")
		public Payout.PayoutBuilder addPerformancePayout(PerformancePayout _performancePayout) {
			if (_performancePayout != null) {
				this.performancePayout.add(_performancePayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addPerformancePayout(PerformancePayout _performancePayout, int _idx) {
			getIndex(this.performancePayout, _idx, () -> _performancePayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addPerformancePayout(List<? extends PerformancePayout> performancePayouts) {
			if (performancePayouts != null) {
				for (final PerformancePayout toAdd : performancePayouts) {
					this.performancePayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("performancePayout")
		public Payout.PayoutBuilder setPerformancePayout(List<? extends PerformancePayout> performancePayouts) {
			if (performancePayouts == null) {
				this.performancePayout = new ArrayList<>();
			} else {
				this.performancePayout = performancePayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("assetPayout")
		@RuneAttribute("assetPayout")
		public Payout.PayoutBuilder addAssetPayout(AssetPayout _assetPayout) {
			if (_assetPayout != null) {
				this.assetPayout.add(_assetPayout.toBuilder());
			}
			return this;
		}
		
		@Override
		public Payout.PayoutBuilder addAssetPayout(AssetPayout _assetPayout, int _idx) {
			getIndex(this.assetPayout, _idx, () -> _assetPayout.toBuilder());
			return this;
		}
		
		@Override 
		public Payout.PayoutBuilder addAssetPayout(List<? extends AssetPayout> assetPayouts) {
			if (assetPayouts != null) {
				for (final AssetPayout toAdd : assetPayouts) {
					this.assetPayout.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("assetPayout")
		public Payout.PayoutBuilder setAssetPayout(List<? extends AssetPayout> assetPayouts) {
			if (assetPayouts == null) {
				this.assetPayout = new ArrayList<>();
			} else {
				this.assetPayout = assetPayouts.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("meta")
		@RuneAttribute("meta")
		@RuneMetaType
		public Payout.PayoutBuilder setMeta(MetaFields _meta) {
			this.meta = _meta == null ? null : _meta.toBuilder();
			return this;
		}
		
		@Override
		public Payout build() {
			return new Payout.PayoutImpl(this);
		}
		
		@Override
		public Payout.PayoutBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Payout.PayoutBuilder prune() {
			interestRatePayout = interestRatePayout.stream().filter(b->b!=null).<InterestRatePayout.InterestRatePayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (creditDefaultPayout!=null && !creditDefaultPayout.prune().hasData()) creditDefaultPayout = null;
			optionPayout = optionPayout.stream().filter(b->b!=null).<OptionPayout.OptionPayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			commodityPayout = commodityPayout.stream().filter(b->b!=null).<CommodityPayout.CommodityPayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			forwardPayout = forwardPayout.stream().filter(b->b!=null).<ForwardPayout.ForwardPayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			fixedPricePayout = fixedPricePayout.stream().filter(b->b!=null).<FixedPricePayout.FixedPricePayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			securityPayout = securityPayout.stream().filter(b->b!=null).<SecurityPayout.SecurityPayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			cashflow = cashflow.stream().filter(b->b!=null).<Cashflow.CashflowBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			performancePayout = performancePayout.stream().filter(b->b!=null).<PerformancePayout.PerformancePayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			assetPayout = assetPayout.stream().filter(b->b!=null).<AssetPayout.AssetPayoutBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (meta!=null && !meta.prune().hasData()) meta = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getInterestRatePayout()!=null && getInterestRatePayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getCreditDefaultPayout()!=null && getCreditDefaultPayout().hasData()) return true;
			if (getOptionPayout()!=null && getOptionPayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getCommodityPayout()!=null && getCommodityPayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getForwardPayout()!=null && getForwardPayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getFixedPricePayout()!=null && getFixedPricePayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getSecurityPayout()!=null && getSecurityPayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getCashflow()!=null && getCashflow().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getPerformancePayout()!=null && getPerformancePayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getAssetPayout()!=null && getAssetPayout().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Payout.PayoutBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			Payout.PayoutBuilder o = (Payout.PayoutBuilder) other;
			
			merger.mergeRosetta(getInterestRatePayout(), o.getInterestRatePayout(), this::getOrCreateInterestRatePayout);
			merger.mergeRosetta(getCreditDefaultPayout(), o.getCreditDefaultPayout(), this::setCreditDefaultPayout);
			merger.mergeRosetta(getOptionPayout(), o.getOptionPayout(), this::getOrCreateOptionPayout);
			merger.mergeRosetta(getCommodityPayout(), o.getCommodityPayout(), this::getOrCreateCommodityPayout);
			merger.mergeRosetta(getForwardPayout(), o.getForwardPayout(), this::getOrCreateForwardPayout);
			merger.mergeRosetta(getFixedPricePayout(), o.getFixedPricePayout(), this::getOrCreateFixedPricePayout);
			merger.mergeRosetta(getSecurityPayout(), o.getSecurityPayout(), this::getOrCreateSecurityPayout);
			merger.mergeRosetta(getCashflow(), o.getCashflow(), this::getOrCreateCashflow);
			merger.mergeRosetta(getPerformancePayout(), o.getPerformancePayout(), this::getOrCreatePerformancePayout);
			merger.mergeRosetta(getAssetPayout(), o.getAssetPayout(), this::getOrCreateAssetPayout);
			merger.mergeRosetta(getMeta(), o.getMeta(), this::setMeta);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			Payout _that = getType().cast(o);
		
			if (!ListEquals.listEquals(interestRatePayout, _that.getInterestRatePayout())) return false;
			if (!Objects.equals(creditDefaultPayout, _that.getCreditDefaultPayout())) return false;
			if (!ListEquals.listEquals(optionPayout, _that.getOptionPayout())) return false;
			if (!ListEquals.listEquals(commodityPayout, _that.getCommodityPayout())) return false;
			if (!ListEquals.listEquals(forwardPayout, _that.getForwardPayout())) return false;
			if (!ListEquals.listEquals(fixedPricePayout, _that.getFixedPricePayout())) return false;
			if (!ListEquals.listEquals(securityPayout, _that.getSecurityPayout())) return false;
			if (!ListEquals.listEquals(cashflow, _that.getCashflow())) return false;
			if (!ListEquals.listEquals(performancePayout, _that.getPerformancePayout())) return false;
			if (!ListEquals.listEquals(assetPayout, _that.getAssetPayout())) return false;
			if (!Objects.equals(meta, _that.getMeta())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (interestRatePayout != null ? interestRatePayout.hashCode() : 0);
			_result = 31 * _result + (creditDefaultPayout != null ? creditDefaultPayout.hashCode() : 0);
			_result = 31 * _result + (optionPayout != null ? optionPayout.hashCode() : 0);
			_result = 31 * _result + (commodityPayout != null ? commodityPayout.hashCode() : 0);
			_result = 31 * _result + (forwardPayout != null ? forwardPayout.hashCode() : 0);
			_result = 31 * _result + (fixedPricePayout != null ? fixedPricePayout.hashCode() : 0);
			_result = 31 * _result + (securityPayout != null ? securityPayout.hashCode() : 0);
			_result = 31 * _result + (cashflow != null ? cashflow.hashCode() : 0);
			_result = 31 * _result + (performancePayout != null ? performancePayout.hashCode() : 0);
			_result = 31 * _result + (assetPayout != null ? assetPayout.hashCode() : 0);
			_result = 31 * _result + (meta != null ? meta.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "PayoutBuilder {" +
				"interestRatePayout=" + this.interestRatePayout + ", " +
				"creditDefaultPayout=" + this.creditDefaultPayout + ", " +
				"optionPayout=" + this.optionPayout + ", " +
				"commodityPayout=" + this.commodityPayout + ", " +
				"forwardPayout=" + this.forwardPayout + ", " +
				"fixedPricePayout=" + this.fixedPricePayout + ", " +
				"securityPayout=" + this.securityPayout + ", " +
				"cashflow=" + this.cashflow + ", " +
				"performancePayout=" + this.performancePayout + ", " +
				"assetPayout=" + this.assetPayout + ", " +
				"meta=" + this.meta +
			'}';
		}
	}
}
