package cdm.product.template.validation.datarule;

import cdm.product.template.AssetPayout;
import cdm.product.template.DividendTerms;
import cdm.product.template.Duration;
import cdm.product.template.DurationTypeEnum;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("AssetPayoutDividendTermsValidation")
@ImplementedBy(AssetPayoutDividendTermsValidation.Default.class)
public interface AssetPayoutDividendTermsValidation extends Validator<AssetPayout> {
	
	String NAME = "AssetPayoutDividendTermsValidation";
	String DEFINITION = "if dividendTerms exists then durationType -> durationType = DurationTypeEnum -> Term";
	
	class Default implements AssetPayoutDividendTermsValidation {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, AssetPayout assetPayout) {
			ComparisonResult result = executeDataRule(assetPayout);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "AssetPayout", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "AssetPayout", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(AssetPayout assetPayout) {
			try {
				if (exists(MapperS.of(assetPayout).<DividendTerms>map("getDividendTerms", _assetPayout -> _assetPayout.getDividendTerms())).getOrDefault(false)) {
					return areEqual(MapperS.of(assetPayout).<Duration>map("getDurationType", _assetPayout -> _assetPayout.getDurationType()).<DurationTypeEnum>map("getDurationType", duration -> duration.getDurationType()), MapperS.of(DurationTypeEnum.TERM), CardinalityOperator.All);
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements AssetPayoutDividendTermsValidation {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, AssetPayout assetPayout) {
			return Collections.emptyList();
		}
	}
}
