package cdm.product.template.validation.datarule;

import cdm.base.staticdata.asset.common.Security;
import cdm.product.template.AssetPayout;
import cdm.product.template.Product;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("AssetPayoutProductMustBeSecurity")
@ImplementedBy(AssetPayoutProductMustBeSecurity.Default.class)
public interface AssetPayoutProductMustBeSecurity extends Validator<AssetPayout> {
	
	String NAME = "AssetPayoutProductMustBeSecurity";
	String DEFINITION = "securityInformation -> security exists";
	
	class Default implements AssetPayoutProductMustBeSecurity {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, AssetPayout assetPayout) {
			ComparisonResult result = executeDataRule(assetPayout);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "AssetPayout", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "AssetPayout", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(AssetPayout assetPayout) {
			try {
				return exists(MapperS.of(assetPayout).<Product>map("getSecurityInformation", _assetPayout -> _assetPayout.getSecurityInformation()).<Security>map("getSecurity", product -> product.getSecurity()));
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements AssetPayoutProductMustBeSecurity {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, AssetPayout assetPayout) {
			return Collections.emptyList();
		}
	}
}
