package cdm.product.template.validation.datarule;

import cdm.base.staticdata.asset.common.ProductTaxonomy;
import cdm.base.staticdata.asset.common.metafields.FieldWithMetaAssetClassEnum;
import cdm.product.template.ContractualProduct;
import cdm.product.template.EconomicTerms;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("ContractualProductPrimaryAssetClass")
@ImplementedBy(ContractualProductPrimaryAssetClass.Default.class)
public interface ContractualProductPrimaryAssetClass extends Validator<ContractualProduct> {
	
	String NAME = "ContractualProductPrimaryAssetClass";
	String DEFINITION = "if economicTerms -> nonStandardisedTerms = True then productTaxonomy -> primaryAssetClass exists";
	
	class Default implements ContractualProductPrimaryAssetClass {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, ContractualProduct contractualProduct) {
			ComparisonResult result = executeDataRule(contractualProduct);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "ContractualProduct", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "ContractualProduct", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(ContractualProduct contractualProduct) {
			try {
				if (areEqual(MapperS.of(contractualProduct).<EconomicTerms>map("getEconomicTerms", _contractualProduct -> _contractualProduct.getEconomicTerms()).<Boolean>map("getNonStandardisedTerms", economicTerms -> economicTerms.getNonStandardisedTerms()), MapperS.of(true), CardinalityOperator.All).getOrDefault(false)) {
					return exists(MapperS.of(contractualProduct).<ProductTaxonomy>mapC("getProductTaxonomy", _contractualProduct -> _contractualProduct.getProductTaxonomy()).<FieldWithMetaAssetClassEnum>map("getPrimaryAssetClass", productTaxonomy -> productTaxonomy.getPrimaryAssetClass()));
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements ContractualProductPrimaryAssetClass {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, ContractualProduct contractualProduct) {
			return Collections.emptyList();
		}
	}
}
