package cdm.product.template.validation.datarule;

import cdm.product.asset.InterestRatePayout;
import cdm.product.asset.StubValue;
import cdm.product.common.schedule.PaymentDates;
import cdm.product.common.schedule.StubPeriod;
import cdm.product.template.Payout;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.records.Date;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("PayoutLastRegularPaymentDate")
@ImplementedBy(PayoutLastRegularPaymentDate.Default.class)
public interface PayoutLastRegularPaymentDate extends Validator<Payout> {
	
	String NAME = "PayoutLastRegularPaymentDate";
	String DEFINITION = "if interestRatePayout -> paymentDates -> lastRegularPaymentDate exists and interestRatePayout -> stubPeriod -> finalStub exists then interestRatePayout count = 2";
	
	class Default implements PayoutLastRegularPaymentDate {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, Payout payout) {
			ComparisonResult result = executeDataRule(payout);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "Payout", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "Payout", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(Payout payout) {
			try {
				if (exists(MapperS.of(payout).<InterestRatePayout>mapC("getInterestRatePayout", _payout -> _payout.getInterestRatePayout()).<PaymentDates>map("getPaymentDates", interestRatePayout -> interestRatePayout.getPaymentDates()).<Date>map("getLastRegularPaymentDate", paymentDates -> paymentDates.getLastRegularPaymentDate())).and(exists(MapperS.of(payout).<InterestRatePayout>mapC("getInterestRatePayout", _payout -> _payout.getInterestRatePayout()).<StubPeriod>map("getStubPeriod", interestRatePayout -> interestRatePayout.getStubPeriod()).<StubValue>map("getFinalStub", stubPeriod -> stubPeriod.getFinalStub()))).getOrDefault(false)) {
					return areEqual(MapperS.of(MapperS.of(payout).<InterestRatePayout>mapC("getInterestRatePayout", _payout -> _payout.getInterestRatePayout()).resultCount()), MapperS.of(2), CardinalityOperator.All);
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements PayoutLastRegularPaymentDate {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, Payout payout) {
			return Collections.emptyList();
		}
	}
}
