package cdm.product.template.validation.datarule;

import cdm.observable.asset.TransactedPrice;
import cdm.product.asset.CreditDefaultPayout;
import cdm.product.asset.CreditIndexReferenceInformation;
import cdm.product.asset.GeneralTerms;
import cdm.product.template.Payout;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

/**
 * @version 5.24.0
 */
@RosettaDataRule("PayoutMarketPrice")
@ImplementedBy(PayoutMarketPrice.Default.class)
public interface PayoutMarketPrice extends Validator<Payout> {
	
	String NAME = "PayoutMarketPrice";
	String DEFINITION = "if creditDefaultPayout -> generalTerms -> indexReferenceInformation is absent then creditDefaultPayout -> transactedPrice -> marketFixedRate is absent and creditDefaultPayout -> transactedPrice -> marketPrice is absent";
	
	class Default implements PayoutMarketPrice {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, Payout payout) {
			ComparisonResult result = executeDataRule(payout);
			if (result.get()) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "Payout", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationType.DATA_RULE, "Payout", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(Payout payout) {
			try {
				if (notExists(MapperS.of(payout).<CreditDefaultPayout>map("getCreditDefaultPayout", _payout -> _payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<CreditIndexReferenceInformation>map("getIndexReferenceInformation", generalTerms -> generalTerms.getIndexReferenceInformation())).getOrDefault(false)) {
					return notExists(MapperS.of(payout).<CreditDefaultPayout>map("getCreditDefaultPayout", _payout -> _payout.getCreditDefaultPayout()).<TransactedPrice>map("getTransactedPrice", creditDefaultPayout -> creditDefaultPayout.getTransactedPrice()).<BigDecimal>map("getMarketFixedRate", transactedPrice -> transactedPrice.getMarketFixedRate())).and(notExists(MapperS.of(payout).<CreditDefaultPayout>map("getCreditDefaultPayout", _payout -> _payout.getCreditDefaultPayout()).<TransactedPrice>map("getTransactedPrice", creditDefaultPayout -> creditDefaultPayout.getTransactedPrice()).<BigDecimal>map("getMarketPrice", transactedPrice -> transactedPrice.getMarketPrice())));
				}
				return ComparisonResult.successEmptyOperand("");
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements PayoutMarketPrice {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, Payout payout) {
			return Collections.emptyList();
		}
	}
}
