package drr.regulation.common;

import cdm.event.common.CounterpartyPositionState;
import cdm.event.common.CounterpartyPositionState.CounterpartyPositionStateBuilder;
import cdm.event.common.TradeState;
import cdm.event.common.TradeState.TradeStateBuilder;
import cdm.event.workflow.WorkflowStep;
import cdm.event.workflow.WorkflowStep.WorkflowStepBuilder;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import drr.regulation.common.ReportableEvent;
import drr.regulation.common.ReportableEvent.ReportableEventBuilder;
import drr.regulation.common.ReportableEvent.ReportableEventBuilderImpl;
import drr.regulation.common.ReportableEvent.ReportableEventImpl;
import drr.regulation.common.ReportableInformation;
import drr.regulation.common.ReportableInformation.ReportableInformationBuilder;
import drr.regulation.common.meta.ReportableEventMeta;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * Specifies a workflowstep with enriched information required for reporting.
 * @version 6.0.0-dev.128
 */
@RosettaDataType(value="ReportableEvent", builder=ReportableEvent.ReportableEventBuilderImpl.class, version="6.0.0-dev.128")
@RuneDataType(value="ReportableEvent", model="drr", builder=ReportableEvent.ReportableEventBuilderImpl.class, version="6.0.0-dev.128")
public interface ReportableEvent extends RosettaModelObject {

	ReportableEventMeta metaData = new ReportableEventMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * The workflowstep that originated the reportable event.
	 */
	WorkflowStep getOriginatingWorkflowStep();
	/**
	 * The reportable trade decomposed from the originating workflow step when required.
	 */
	TradeState getReportableTrade();
	/**
	 * Additional information required for a reportable transaction, including the reporting regime.
	 */
	ReportableInformation getReportableInformation();
	/**
	 * The reportable position decomposed from the originating workflow step when required.
	 */
	CounterpartyPositionState getReportablePosition();

	/*********************** Build Methods  ***********************/
	ReportableEvent build();
	
	ReportableEvent.ReportableEventBuilder toBuilder();
	
	static ReportableEvent.ReportableEventBuilder builder() {
		return new ReportableEvent.ReportableEventBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportableEvent> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportableEvent> getType() {
		return ReportableEvent.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("originatingWorkflowStep"), processor, WorkflowStep.class, getOriginatingWorkflowStep());
		processRosetta(path.newSubPath("reportableTrade"), processor, TradeState.class, getReportableTrade());
		processRosetta(path.newSubPath("reportableInformation"), processor, ReportableInformation.class, getReportableInformation());
		processRosetta(path.newSubPath("reportablePosition"), processor, CounterpartyPositionState.class, getReportablePosition());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportableEventBuilder extends ReportableEvent, RosettaModelObjectBuilder {
		WorkflowStep.WorkflowStepBuilder getOrCreateOriginatingWorkflowStep();
		@Override
		WorkflowStep.WorkflowStepBuilder getOriginatingWorkflowStep();
		TradeState.TradeStateBuilder getOrCreateReportableTrade();
		@Override
		TradeState.TradeStateBuilder getReportableTrade();
		ReportableInformation.ReportableInformationBuilder getOrCreateReportableInformation();
		@Override
		ReportableInformation.ReportableInformationBuilder getReportableInformation();
		CounterpartyPositionState.CounterpartyPositionStateBuilder getOrCreateReportablePosition();
		@Override
		CounterpartyPositionState.CounterpartyPositionStateBuilder getReportablePosition();
		ReportableEvent.ReportableEventBuilder setOriginatingWorkflowStep(WorkflowStep originatingWorkflowStep);
		ReportableEvent.ReportableEventBuilder setReportableTrade(TradeState reportableTrade);
		ReportableEvent.ReportableEventBuilder setReportableInformation(ReportableInformation reportableInformation);
		ReportableEvent.ReportableEventBuilder setReportablePosition(CounterpartyPositionState reportablePosition);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("originatingWorkflowStep"), processor, WorkflowStep.WorkflowStepBuilder.class, getOriginatingWorkflowStep());
			processRosetta(path.newSubPath("reportableTrade"), processor, TradeState.TradeStateBuilder.class, getReportableTrade());
			processRosetta(path.newSubPath("reportableInformation"), processor, ReportableInformation.ReportableInformationBuilder.class, getReportableInformation());
			processRosetta(path.newSubPath("reportablePosition"), processor, CounterpartyPositionState.CounterpartyPositionStateBuilder.class, getReportablePosition());
		}
		

		ReportableEvent.ReportableEventBuilder prune();
	}

	/*********************** Immutable Implementation of ReportableEvent  ***********************/
	class ReportableEventImpl implements ReportableEvent {
		private final WorkflowStep originatingWorkflowStep;
		private final TradeState reportableTrade;
		private final ReportableInformation reportableInformation;
		private final CounterpartyPositionState reportablePosition;
		
		protected ReportableEventImpl(ReportableEvent.ReportableEventBuilder builder) {
			this.originatingWorkflowStep = ofNullable(builder.getOriginatingWorkflowStep()).map(f->f.build()).orElse(null);
			this.reportableTrade = ofNullable(builder.getReportableTrade()).map(f->f.build()).orElse(null);
			this.reportableInformation = ofNullable(builder.getReportableInformation()).map(f->f.build()).orElse(null);
			this.reportablePosition = ofNullable(builder.getReportablePosition()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute("originatingWorkflowStep")
		@RuneAttribute("originatingWorkflowStep")
		public WorkflowStep getOriginatingWorkflowStep() {
			return originatingWorkflowStep;
		}
		
		@Override
		@RosettaAttribute("reportableTrade")
		@RuneAttribute("reportableTrade")
		public TradeState getReportableTrade() {
			return reportableTrade;
		}
		
		@Override
		@RosettaAttribute("reportableInformation")
		@RuneAttribute("reportableInformation")
		public ReportableInformation getReportableInformation() {
			return reportableInformation;
		}
		
		@Override
		@RosettaAttribute("reportablePosition")
		@RuneAttribute("reportablePosition")
		public CounterpartyPositionState getReportablePosition() {
			return reportablePosition;
		}
		
		@Override
		public ReportableEvent build() {
			return this;
		}
		
		@Override
		public ReportableEvent.ReportableEventBuilder toBuilder() {
			ReportableEvent.ReportableEventBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportableEvent.ReportableEventBuilder builder) {
			ofNullable(getOriginatingWorkflowStep()).ifPresent(builder::setOriginatingWorkflowStep);
			ofNullable(getReportableTrade()).ifPresent(builder::setReportableTrade);
			ofNullable(getReportableInformation()).ifPresent(builder::setReportableInformation);
			ofNullable(getReportablePosition()).ifPresent(builder::setReportablePosition);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableEvent _that = getType().cast(o);
		
			if (!Objects.equals(originatingWorkflowStep, _that.getOriginatingWorkflowStep())) return false;
			if (!Objects.equals(reportableTrade, _that.getReportableTrade())) return false;
			if (!Objects.equals(reportableInformation, _that.getReportableInformation())) return false;
			if (!Objects.equals(reportablePosition, _that.getReportablePosition())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (originatingWorkflowStep != null ? originatingWorkflowStep.hashCode() : 0);
			_result = 31 * _result + (reportableTrade != null ? reportableTrade.hashCode() : 0);
			_result = 31 * _result + (reportableInformation != null ? reportableInformation.hashCode() : 0);
			_result = 31 * _result + (reportablePosition != null ? reportablePosition.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableEvent {" +
				"originatingWorkflowStep=" + this.originatingWorkflowStep + ", " +
				"reportableTrade=" + this.reportableTrade + ", " +
				"reportableInformation=" + this.reportableInformation + ", " +
				"reportablePosition=" + this.reportablePosition +
			'}';
		}
	}

	/*********************** Builder Implementation of ReportableEvent  ***********************/
	class ReportableEventBuilderImpl implements ReportableEvent.ReportableEventBuilder {
	
		protected WorkflowStep.WorkflowStepBuilder originatingWorkflowStep;
		protected TradeState.TradeStateBuilder reportableTrade;
		protected ReportableInformation.ReportableInformationBuilder reportableInformation;
		protected CounterpartyPositionState.CounterpartyPositionStateBuilder reportablePosition;
		
		@Override
		@RosettaAttribute("originatingWorkflowStep")
		@RuneAttribute("originatingWorkflowStep")
		public WorkflowStep.WorkflowStepBuilder getOriginatingWorkflowStep() {
			return originatingWorkflowStep;
		}
		
		@Override
		public WorkflowStep.WorkflowStepBuilder getOrCreateOriginatingWorkflowStep() {
			WorkflowStep.WorkflowStepBuilder result;
			if (originatingWorkflowStep!=null) {
				result = originatingWorkflowStep;
			}
			else {
				result = originatingWorkflowStep = WorkflowStep.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("reportableTrade")
		@RuneAttribute("reportableTrade")
		public TradeState.TradeStateBuilder getReportableTrade() {
			return reportableTrade;
		}
		
		@Override
		public TradeState.TradeStateBuilder getOrCreateReportableTrade() {
			TradeState.TradeStateBuilder result;
			if (reportableTrade!=null) {
				result = reportableTrade;
			}
			else {
				result = reportableTrade = TradeState.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("reportableInformation")
		@RuneAttribute("reportableInformation")
		public ReportableInformation.ReportableInformationBuilder getReportableInformation() {
			return reportableInformation;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder getOrCreateReportableInformation() {
			ReportableInformation.ReportableInformationBuilder result;
			if (reportableInformation!=null) {
				result = reportableInformation;
			}
			else {
				result = reportableInformation = ReportableInformation.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("reportablePosition")
		@RuneAttribute("reportablePosition")
		public CounterpartyPositionState.CounterpartyPositionStateBuilder getReportablePosition() {
			return reportablePosition;
		}
		
		@Override
		public CounterpartyPositionState.CounterpartyPositionStateBuilder getOrCreateReportablePosition() {
			CounterpartyPositionState.CounterpartyPositionStateBuilder result;
			if (reportablePosition!=null) {
				result = reportablePosition;
			}
			else {
				result = reportablePosition = CounterpartyPositionState.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("originatingWorkflowStep")
		@RuneAttribute("originatingWorkflowStep")
		public ReportableEvent.ReportableEventBuilder setOriginatingWorkflowStep(WorkflowStep _originatingWorkflowStep) {
			this.originatingWorkflowStep = _originatingWorkflowStep == null ? null : _originatingWorkflowStep.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("reportableTrade")
		@RuneAttribute("reportableTrade")
		public ReportableEvent.ReportableEventBuilder setReportableTrade(TradeState _reportableTrade) {
			this.reportableTrade = _reportableTrade == null ? null : _reportableTrade.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("reportableInformation")
		@RuneAttribute("reportableInformation")
		public ReportableEvent.ReportableEventBuilder setReportableInformation(ReportableInformation _reportableInformation) {
			this.reportableInformation = _reportableInformation == null ? null : _reportableInformation.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("reportablePosition")
		@RuneAttribute("reportablePosition")
		public ReportableEvent.ReportableEventBuilder setReportablePosition(CounterpartyPositionState _reportablePosition) {
			this.reportablePosition = _reportablePosition == null ? null : _reportablePosition.toBuilder();
			return this;
		}
		
		@Override
		public ReportableEvent build() {
			return new ReportableEvent.ReportableEventImpl(this);
		}
		
		@Override
		public ReportableEvent.ReportableEventBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableEvent.ReportableEventBuilder prune() {
			if (originatingWorkflowStep!=null && !originatingWorkflowStep.prune().hasData()) originatingWorkflowStep = null;
			if (reportableTrade!=null && !reportableTrade.prune().hasData()) reportableTrade = null;
			if (reportableInformation!=null && !reportableInformation.prune().hasData()) reportableInformation = null;
			if (reportablePosition!=null && !reportablePosition.prune().hasData()) reportablePosition = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getOriginatingWorkflowStep()!=null && getOriginatingWorkflowStep().hasData()) return true;
			if (getReportableTrade()!=null && getReportableTrade().hasData()) return true;
			if (getReportableInformation()!=null && getReportableInformation().hasData()) return true;
			if (getReportablePosition()!=null && getReportablePosition().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableEvent.ReportableEventBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			ReportableEvent.ReportableEventBuilder o = (ReportableEvent.ReportableEventBuilder) other;
			
			merger.mergeRosetta(getOriginatingWorkflowStep(), o.getOriginatingWorkflowStep(), this::setOriginatingWorkflowStep);
			merger.mergeRosetta(getReportableTrade(), o.getReportableTrade(), this::setReportableTrade);
			merger.mergeRosetta(getReportableInformation(), o.getReportableInformation(), this::setReportableInformation);
			merger.mergeRosetta(getReportablePosition(), o.getReportablePosition(), this::setReportablePosition);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableEvent _that = getType().cast(o);
		
			if (!Objects.equals(originatingWorkflowStep, _that.getOriginatingWorkflowStep())) return false;
			if (!Objects.equals(reportableTrade, _that.getReportableTrade())) return false;
			if (!Objects.equals(reportableInformation, _that.getReportableInformation())) return false;
			if (!Objects.equals(reportablePosition, _that.getReportablePosition())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (originatingWorkflowStep != null ? originatingWorkflowStep.hashCode() : 0);
			_result = 31 * _result + (reportableTrade != null ? reportableTrade.hashCode() : 0);
			_result = 31 * _result + (reportableInformation != null ? reportableInformation.hashCode() : 0);
			_result = 31 * _result + (reportablePosition != null ? reportablePosition.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableEventBuilder {" +
				"originatingWorkflowStep=" + this.originatingWorkflowStep + ", " +
				"reportableTrade=" + this.reportableTrade + ", " +
				"reportableInformation=" + this.reportableInformation + ", " +
				"reportablePosition=" + this.reportablePosition +
			'}';
		}
	}
}
