package drr.regulation.common.trade;

import cdm.base.staticdata.asset.common.ISOCountryCodeEnum;
import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RosettaIgnore;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import com.rosetta.util.ListEquals;
import drr.regulation.common.CommonAssetClass;
import drr.regulation.common.CommonContractType;
import drr.regulation.common.trade.CommonLeg;
import drr.regulation.common.trade.CommonLeg.CommonLegBuilder;
import drr.regulation.common.trade.CommonTransactionReport;
import drr.regulation.common.trade.CommonTransactionReport.CommonTransactionReportBuilder;
import drr.regulation.common.trade.CommonTransactionReport.CommonTransactionReportBuilderImpl;
import drr.regulation.common.trade.CommonTransactionReport.CommonTransactionReportImpl;
import drr.regulation.common.trade.basket.BasketConstituentsReport;
import drr.regulation.common.trade.basket.BasketConstituentsReport.BasketConstituentsReportBuilder;
import drr.regulation.common.trade.contract.UnderlyingIdentifier;
import drr.regulation.common.trade.contract.UnderlyingIdentifier.UnderlyingIdentifierBuilder;
import drr.regulation.common.trade.meta.CommonTransactionReportMeta;
import drr.regulation.common.trade.price.SingleOrUpperAndLowerBarrier;
import drr.regulation.common.trade.price.SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder;
import drr.standards.iosco.cde.CriticalDataElement;
import drr.standards.iosco.cde.CriticalDataElement.CriticalDataElementBuilder;
import drr.standards.iosco.cde.CriticalDataElement.CriticalDataElementImpl;
import drr.standards.iosco.cde.Leg;
import drr.standards.iosco.cde.payment.OtherPayment;
import drr.standards.iosco.cde.payment.OtherPayment.OtherPaymentBuilder;
import drr.standards.iosco.cde.price.PriceFormat;
import drr.standards.iosco.cde.price.PriceFormat.PriceFormatBuilder;
import drr.standards.iosco.cde.price.PriceNotationEnum;
import drr.standards.iosco.cde.price.PricePeriod;
import drr.standards.iosco.cde.price.PricePeriod.PricePeriodBuilder;
import drr.standards.iso.ActionTypeEnum;
import drr.standards.iso.ClearedEnum;
import drr.standards.iso.ConfirmationEnum;
import drr.standards.iso.DeliveryTypeEnum;
import drr.standards.iso.Direction1Enum;
import drr.standards.iso.Direction2Enum;
import drr.standards.iso.EmbeddedOptionTypeEnum;
import drr.standards.iso.EventTypeEnum;
import drr.standards.iso.OptionStyleEnum;
import drr.standards.iso.OptionTypeCode;
import drr.standards.iso.SeniorityEnum;
import drr.standards.iso.TradingCapacity7Code;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Common Transaction Report represents common attributes
 * @version 6.0.0-dev.128
 */
@RosettaDataType(value="CommonTransactionReport", builder=CommonTransactionReport.CommonTransactionReportBuilderImpl.class, version="6.0.0-dev.128")
@RuneDataType(value="CommonTransactionReport", model="drr", builder=CommonTransactionReport.CommonTransactionReportBuilderImpl.class, version="6.0.0-dev.128")
public interface CommonTransactionReport extends CriticalDataElement {

	CommonTransactionReportMeta metaData = new CommonTransactionReportMeta();

	/*********************** Getter Methods  ***********************/
	@Override
	CommonLeg getLeg1();
	@Override
	CommonLeg getLeg2();
	String getCounterparty2Name();
	String getExecutionAgentCounterparty1();
	String getExecutionAgentCounterparty2DTCC();
	String getReportSubmittingEntityID();
	String getUniqueTransactionIdentifier();
	String getUniqueTransactionIdentifierProprietary();
	String getPriorUTI();
	String getPriorUTIProprietary();
	String getUniqueProductIdentifier();
	CommonContractType getContractType();
	CommonAssetClass getAssetClass();
	ZonedDateTime getClearingTimestamp();
	String getPlatformIdentifier();
	Direction1Enum getDirection1();
	Direction2Enum getDirection2Leg1();
	Direction2Enum getDirection2Leg2();
	String getPackageIdentifier();
	SingleOrUpperAndLowerBarrier getBarrier();
	BigDecimal getExchangeRate();
	String getExchangeRateBasis();
	String getNameOfTheUnderlyingIndex();
	List<? extends BasketConstituentsReport> getBasketConstituents();
	UnderlyingIdentifier getUnderlying();
	String getQuantityUnitOfMeasureLeg1();
	String getQuantityUnitOfMeasureLeg2();
	Date getMaturityDateOfTheUnderlying();
	OptionTypeCode getOptionType();
	OptionStyleEnum getOptionStyle();
	DeliveryTypeEnum getDeliveryType();
	ISOCountryCodeEnum getCountryOfTheCounterparty2();
	ISOCountryCodeEnum getBookingLocation();
	ISOCountryCodeEnum getTraderLocation();
	TradingCapacity7Code getTradingCapacity();
	EmbeddedOptionTypeEnum getEmbeddedOptionType();
	SeniorityEnum getSeniority();
	Integer getSeries();
	BigDecimal getIndexFactor();
	String getTechnicalRecordId();

	/*********************** Build Methods  ***********************/
	CommonTransactionReport build();
	
	CommonTransactionReport.CommonTransactionReportBuilder toBuilder();
	
	static CommonTransactionReport.CommonTransactionReportBuilder builder() {
		return new CommonTransactionReport.CommonTransactionReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends CommonTransactionReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends CommonTransactionReport> getType() {
		return CommonTransactionReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("eventType"), EventTypeEnum.class, getEventType(), this);
		processor.processBasic(path.newSubPath("eventIdentifier"), String.class, getEventIdentifier(), this);
		processor.processBasic(path.newSubPath("effectiveDate"), Date.class, getEffectiveDate(), this);
		processor.processBasic(path.newSubPath("earlyTerminationDate"), Date.class, getEarlyTerminationDate(), this);
		processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("executionTimestamp"), ZonedDateTime.class, getExecutionTimestamp(), this);
		processor.processBasic(path.newSubPath("expirationDate"), Date.class, getExpirationDate(), this);
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
		processor.processBasic(path.newSubPath("beneficiary1"), String.class, getBeneficiary1(), this);
		processor.processBasic(path.newSubPath("beneficiary1IdentifierTypeIndicator"), Boolean.class, getBeneficiary1IdentifierTypeIndicator(), this);
		processor.processBasic(path.newSubPath("cleared"), ClearedEnum.class, getCleared(), this);
		processor.processBasic(path.newSubPath("centralCounterparty"), String.class, getCentralCounterparty(), this);
		processor.processBasic(path.newSubPath("clearingMember"), String.class, getClearingMember(), this);
		processor.processBasic(path.newSubPath("confirmed"), ConfirmationEnum.class, getConfirmed(), this);
		processor.processBasic(path.newSubPath("callAmount"), BigDecimal.class, getCallAmount(), this);
		processor.processBasic(path.newSubPath("putAmount"), BigDecimal.class, getPutAmount(), this);
		processor.processBasic(path.newSubPath("callCurrency"), ISOCurrencyCodeEnum.class, getCallCurrency(), this);
		processor.processBasic(path.newSubPath("putCurrency"), ISOCurrencyCodeEnum.class, getPutCurrency(), this);
		processRosetta(path.newSubPath("priceSchedule"), processor, PricePeriod.class, getPriceSchedule());
		processRosetta(path.newSubPath("strikePriceSchedule"), processor, PricePeriod.class, getStrikePriceSchedule());
		processRosetta(path.newSubPath("price"), processor, PriceFormat.class, getPrice());
		processor.processBasic(path.newSubPath("priceNotation"), PriceNotationEnum.class, getPriceNotation(), this);
		processor.processBasic(path.newSubPath("priceCurrency"), ISOCurrencyCodeEnum.class, getPriceCurrency(), this);
		processRosetta(path.newSubPath("packageTransactionPrice"), processor, PriceFormat.class, getPackageTransactionPrice());
		processor.processBasic(path.newSubPath("packageTransactionPriceNotation"), PriceNotationEnum.class, getPackageTransactionPriceNotation(), this);
		processor.processBasic(path.newSubPath("packageTransactionPriceCurrency"), ISOCurrencyCodeEnum.class, getPackageTransactionPriceCurrency(), this);
		processRosetta(path.newSubPath("packageTransactionSpread"), processor, PriceFormat.class, getPackageTransactionSpread());
		processor.processBasic(path.newSubPath("packageTransactionSpreadNotation"), PriceNotationEnum.class, getPackageTransactionSpreadNotation(), this);
		processor.processBasic(path.newSubPath("packageTransactionSpreadCurrency"), ISOCurrencyCodeEnum.class, getPackageTransactionSpreadCurrency(), this);
		processRosetta(path.newSubPath("strikePrice"), processor, PriceFormat.class, getStrikePrice());
		processor.processBasic(path.newSubPath("strikePriceNotation"), PriceNotationEnum.class, getStrikePriceNotation(), this);
		processor.processBasic(path.newSubPath("strikePriceCurrency"), String.class, getStrikePriceCurrency(), this);
		processor.processBasic(path.newSubPath("priceUnitOfMeasure"), String.class, getPriceUnitOfMeasure(), this);
		processor.processBasic(path.newSubPath("optionPremiumAmount"), BigDecimal.class, getOptionPremiumAmount(), this);
		processor.processBasic(path.newSubPath("optionPremiumCurrency"), ISOCurrencyCodeEnum.class, getOptionPremiumCurrency(), this);
		processor.processBasic(path.newSubPath("optionPremiumPaymentDate"), Date.class, getOptionPremiumPaymentDate(), this);
		processor.processBasic(path.newSubPath("cdSIndexAttachmentPoint"), BigDecimal.class, getCdSIndexAttachmentPoint(), this);
		processor.processBasic(path.newSubPath("cdSIndexDetachmentPoint"), BigDecimal.class, getCdSIndexDetachmentPoint(), this);
		processor.processBasic(path.newSubPath("collateralPortfolioIndicator"), Boolean.class, getCollateralPortfolioIndicator(), this);
		processor.processBasic(path.newSubPath("firstExerciseDate"), Date.class, getFirstExerciseDate(), this);
		processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
		processRosetta(path.newSubPath("otherPayment"), processor, OtherPayment.class, getOtherPayment());
		processRosetta(path.newSubPath("leg1"), processor, CommonLeg.class, getLeg1());
		processRosetta(path.newSubPath("leg2"), processor, CommonLeg.class, getLeg2());
		processor.processBasic(path.newSubPath("counterparty2Name"), String.class, getCounterparty2Name(), this);
		processor.processBasic(path.newSubPath("executionAgentCounterparty1"), String.class, getExecutionAgentCounterparty1(), this);
		processor.processBasic(path.newSubPath("executionAgentCounterparty2DTCC"), String.class, getExecutionAgentCounterparty2DTCC(), this);
		processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifierProprietary"), String.class, getUniqueTransactionIdentifierProprietary(), this);
		processor.processBasic(path.newSubPath("priorUTI"), String.class, getPriorUTI(), this);
		processor.processBasic(path.newSubPath("priorUTIProprietary"), String.class, getPriorUTIProprietary(), this);
		processor.processBasic(path.newSubPath("uniqueProductIdentifier"), String.class, getUniqueProductIdentifier(), this);
		processor.processBasic(path.newSubPath("contractType"), CommonContractType.class, getContractType(), this);
		processor.processBasic(path.newSubPath("assetClass"), CommonAssetClass.class, getAssetClass(), this);
		processor.processBasic(path.newSubPath("clearingTimestamp"), ZonedDateTime.class, getClearingTimestamp(), this);
		processor.processBasic(path.newSubPath("platformIdentifier"), String.class, getPlatformIdentifier(), this);
		processor.processBasic(path.newSubPath("direction1"), Direction1Enum.class, getDirection1(), this);
		processor.processBasic(path.newSubPath("direction2Leg1"), Direction2Enum.class, getDirection2Leg1(), this);
		processor.processBasic(path.newSubPath("direction2Leg2"), Direction2Enum.class, getDirection2Leg2(), this);
		processor.processBasic(path.newSubPath("packageIdentifier"), String.class, getPackageIdentifier(), this);
		processRosetta(path.newSubPath("barrier"), processor, SingleOrUpperAndLowerBarrier.class, getBarrier());
		processor.processBasic(path.newSubPath("exchangeRate"), BigDecimal.class, getExchangeRate(), this);
		processor.processBasic(path.newSubPath("exchangeRateBasis"), String.class, getExchangeRateBasis(), this);
		processor.processBasic(path.newSubPath("nameOfTheUnderlyingIndex"), String.class, getNameOfTheUnderlyingIndex(), this);
		processRosetta(path.newSubPath("basketConstituents"), processor, BasketConstituentsReport.class, getBasketConstituents());
		processRosetta(path.newSubPath("underlying"), processor, UnderlyingIdentifier.class, getUnderlying());
		processor.processBasic(path.newSubPath("quantityUnitOfMeasureLeg1"), String.class, getQuantityUnitOfMeasureLeg1(), this);
		processor.processBasic(path.newSubPath("quantityUnitOfMeasureLeg2"), String.class, getQuantityUnitOfMeasureLeg2(), this);
		processor.processBasic(path.newSubPath("maturityDateOfTheUnderlying"), Date.class, getMaturityDateOfTheUnderlying(), this);
		processor.processBasic(path.newSubPath("optionType"), OptionTypeCode.class, getOptionType(), this);
		processor.processBasic(path.newSubPath("optionStyle"), OptionStyleEnum.class, getOptionStyle(), this);
		processor.processBasic(path.newSubPath("deliveryType"), DeliveryTypeEnum.class, getDeliveryType(), this);
		processor.processBasic(path.newSubPath("countryOfTheCounterparty2"), ISOCountryCodeEnum.class, getCountryOfTheCounterparty2(), this);
		processor.processBasic(path.newSubPath("bookingLocation"), ISOCountryCodeEnum.class, getBookingLocation(), this);
		processor.processBasic(path.newSubPath("traderLocation"), ISOCountryCodeEnum.class, getTraderLocation(), this);
		processor.processBasic(path.newSubPath("tradingCapacity"), TradingCapacity7Code.class, getTradingCapacity(), this);
		processor.processBasic(path.newSubPath("embeddedOptionType"), EmbeddedOptionTypeEnum.class, getEmbeddedOptionType(), this);
		processor.processBasic(path.newSubPath("seniority"), SeniorityEnum.class, getSeniority(), this);
		processor.processBasic(path.newSubPath("series"), Integer.class, getSeries(), this);
		processor.processBasic(path.newSubPath("indexFactor"), BigDecimal.class, getIndexFactor(), this);
		processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface CommonTransactionReportBuilder extends CommonTransactionReport, CriticalDataElement.CriticalDataElementBuilder {
		CommonLeg.CommonLegBuilder getOrCreateLeg1();
		@Override
		CommonLeg.CommonLegBuilder getLeg1();
		CommonLeg.CommonLegBuilder getOrCreateLeg2();
		@Override
		CommonLeg.CommonLegBuilder getLeg2();
		SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder getOrCreateBarrier();
		@Override
		SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder getBarrier();
		BasketConstituentsReport.BasketConstituentsReportBuilder getOrCreateBasketConstituents(int _index);
		@Override
		List<? extends BasketConstituentsReport.BasketConstituentsReportBuilder> getBasketConstituents();
		UnderlyingIdentifier.UnderlyingIdentifierBuilder getOrCreateUnderlying();
		@Override
		UnderlyingIdentifier.UnderlyingIdentifierBuilder getUnderlying();
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setActionType(ActionTypeEnum actionType);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setEventType(EventTypeEnum eventType);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setEventIdentifier(String eventIdentifier);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setEffectiveDate(Date effectiveDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setEarlyTerminationDate(Date earlyTerminationDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setEventTimestamp(ZonedDateTime eventTimestamp);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setExecutionTimestamp(ZonedDateTime executionTimestamp);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setExpirationDate(Date expirationDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCounterparty1(String counterparty1);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2(String counterparty2);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2IdentifierType(Boolean counterparty2IdentifierType);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setBeneficiary1(String beneficiary1);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setBeneficiary1IdentifierTypeIndicator(Boolean beneficiary1IdentifierTypeIndicator);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCleared(ClearedEnum cleared);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCentralCounterparty(String centralCounterparty);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setClearingMember(String clearingMember);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setConfirmed(ConfirmationEnum confirmed);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCallAmount(BigDecimal callAmount);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPutAmount(BigDecimal putAmount);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCallCurrency(ISOCurrencyCodeEnum callCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPutCurrency(ISOCurrencyCodeEnum putCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(PricePeriod priceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(PricePeriod priceSchedule, int _idx);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(List<? extends PricePeriod> priceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPriceSchedule(List<? extends PricePeriod> priceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(PricePeriod strikePriceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(PricePeriod strikePriceSchedule, int _idx);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(List<? extends PricePeriod> strikePriceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceSchedule(List<? extends PricePeriod> strikePriceSchedule);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPrice(PriceFormat price);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPriceNotation(PriceNotationEnum priceNotation);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPriceCurrency(ISOCurrencyCodeEnum priceCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPrice(PriceFormat packageTransactionPrice);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPriceNotation(PriceNotationEnum packageTransactionPriceNotation);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPriceCurrency(ISOCurrencyCodeEnum packageTransactionPriceCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpread(PriceFormat packageTransactionSpread);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpreadNotation(PriceNotationEnum packageTransactionSpreadNotation);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpreadCurrency(ISOCurrencyCodeEnum packageTransactionSpreadCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setStrikePrice(PriceFormat strikePrice);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceNotation(PriceNotationEnum strikePriceNotation);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceCurrency(String strikePriceCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setPriceUnitOfMeasure(String priceUnitOfMeasure);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumAmount(BigDecimal optionPremiumAmount);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumCurrency(ISOCurrencyCodeEnum optionPremiumCurrency);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumPaymentDate(Date optionPremiumPaymentDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCdSIndexAttachmentPoint(BigDecimal cdSIndexAttachmentPoint);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCdSIndexDetachmentPoint(BigDecimal cdSIndexDetachmentPoint);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setCollateralPortfolioIndicator(Boolean collateralPortfolioIndicator);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setFirstExerciseDate(Date firstExerciseDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setFinalContractualSettlementDate(Date finalContractualSettlementDate);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(OtherPayment otherPayment);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(OtherPayment otherPayment, int _idx);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(List<? extends OtherPayment> otherPayment);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setOtherPayment(List<? extends OtherPayment> otherPayment);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setLeg1(Leg leg1);
		@Override
		CommonTransactionReport.CommonTransactionReportBuilder setLeg2(Leg leg2);
		CommonTransactionReport.CommonTransactionReportBuilder setLeg1(CommonLeg leg1);
		CommonTransactionReport.CommonTransactionReportBuilder setLeg2(CommonLeg leg2);
		CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2Name(String counterparty2Name);
		CommonTransactionReport.CommonTransactionReportBuilder setExecutionAgentCounterparty1(String executionAgentCounterparty1);
		CommonTransactionReport.CommonTransactionReportBuilder setExecutionAgentCounterparty2DTCC(String executionAgentCounterparty2DTCC);
		CommonTransactionReport.CommonTransactionReportBuilder setReportSubmittingEntityID(String reportSubmittingEntityID);
		CommonTransactionReport.CommonTransactionReportBuilder setUniqueTransactionIdentifier(String uniqueTransactionIdentifier);
		CommonTransactionReport.CommonTransactionReportBuilder setUniqueTransactionIdentifierProprietary(String uniqueTransactionIdentifierProprietary);
		CommonTransactionReport.CommonTransactionReportBuilder setPriorUTI(String priorUTI);
		CommonTransactionReport.CommonTransactionReportBuilder setPriorUTIProprietary(String priorUTIProprietary);
		CommonTransactionReport.CommonTransactionReportBuilder setUniqueProductIdentifier(String uniqueProductIdentifier);
		CommonTransactionReport.CommonTransactionReportBuilder setContractType(CommonContractType contractType);
		CommonTransactionReport.CommonTransactionReportBuilder setAssetClass(CommonAssetClass assetClass);
		CommonTransactionReport.CommonTransactionReportBuilder setClearingTimestamp(ZonedDateTime clearingTimestamp);
		CommonTransactionReport.CommonTransactionReportBuilder setPlatformIdentifier(String platformIdentifier);
		CommonTransactionReport.CommonTransactionReportBuilder setDirection1(Direction1Enum direction1);
		CommonTransactionReport.CommonTransactionReportBuilder setDirection2Leg1(Direction2Enum direction2Leg1);
		CommonTransactionReport.CommonTransactionReportBuilder setDirection2Leg2(Direction2Enum direction2Leg2);
		CommonTransactionReport.CommonTransactionReportBuilder setPackageIdentifier(String packageIdentifier);
		CommonTransactionReport.CommonTransactionReportBuilder setBarrier(SingleOrUpperAndLowerBarrier barrier);
		CommonTransactionReport.CommonTransactionReportBuilder setExchangeRate(BigDecimal exchangeRate);
		CommonTransactionReport.CommonTransactionReportBuilder setExchangeRateBasis(String exchangeRateBasis);
		CommonTransactionReport.CommonTransactionReportBuilder setNameOfTheUnderlyingIndex(String nameOfTheUnderlyingIndex);
		CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(BasketConstituentsReport basketConstituents);
		CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(BasketConstituentsReport basketConstituents, int _idx);
		CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(List<? extends BasketConstituentsReport> basketConstituents);
		CommonTransactionReport.CommonTransactionReportBuilder setBasketConstituents(List<? extends BasketConstituentsReport> basketConstituents);
		CommonTransactionReport.CommonTransactionReportBuilder setUnderlying(UnderlyingIdentifier underlying);
		CommonTransactionReport.CommonTransactionReportBuilder setQuantityUnitOfMeasureLeg1(String quantityUnitOfMeasureLeg1);
		CommonTransactionReport.CommonTransactionReportBuilder setQuantityUnitOfMeasureLeg2(String quantityUnitOfMeasureLeg2);
		CommonTransactionReport.CommonTransactionReportBuilder setMaturityDateOfTheUnderlying(Date maturityDateOfTheUnderlying);
		CommonTransactionReport.CommonTransactionReportBuilder setOptionType(OptionTypeCode optionType);
		CommonTransactionReport.CommonTransactionReportBuilder setOptionStyle(OptionStyleEnum optionStyle);
		CommonTransactionReport.CommonTransactionReportBuilder setDeliveryType(DeliveryTypeEnum deliveryType);
		CommonTransactionReport.CommonTransactionReportBuilder setCountryOfTheCounterparty2(ISOCountryCodeEnum countryOfTheCounterparty2);
		CommonTransactionReport.CommonTransactionReportBuilder setBookingLocation(ISOCountryCodeEnum bookingLocation);
		CommonTransactionReport.CommonTransactionReportBuilder setTraderLocation(ISOCountryCodeEnum traderLocation);
		CommonTransactionReport.CommonTransactionReportBuilder setTradingCapacity(TradingCapacity7Code tradingCapacity);
		CommonTransactionReport.CommonTransactionReportBuilder setEmbeddedOptionType(EmbeddedOptionTypeEnum embeddedOptionType);
		CommonTransactionReport.CommonTransactionReportBuilder setSeniority(SeniorityEnum seniority);
		CommonTransactionReport.CommonTransactionReportBuilder setSeries(Integer series);
		CommonTransactionReport.CommonTransactionReportBuilder setIndexFactor(BigDecimal indexFactor);
		CommonTransactionReport.CommonTransactionReportBuilder setTechnicalRecordId(String technicalRecordId);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("eventType"), EventTypeEnum.class, getEventType(), this);
			processor.processBasic(path.newSubPath("eventIdentifier"), String.class, getEventIdentifier(), this);
			processor.processBasic(path.newSubPath("effectiveDate"), Date.class, getEffectiveDate(), this);
			processor.processBasic(path.newSubPath("earlyTerminationDate"), Date.class, getEarlyTerminationDate(), this);
			processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("executionTimestamp"), ZonedDateTime.class, getExecutionTimestamp(), this);
			processor.processBasic(path.newSubPath("expirationDate"), Date.class, getExpirationDate(), this);
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
			processor.processBasic(path.newSubPath("beneficiary1"), String.class, getBeneficiary1(), this);
			processor.processBasic(path.newSubPath("beneficiary1IdentifierTypeIndicator"), Boolean.class, getBeneficiary1IdentifierTypeIndicator(), this);
			processor.processBasic(path.newSubPath("cleared"), ClearedEnum.class, getCleared(), this);
			processor.processBasic(path.newSubPath("centralCounterparty"), String.class, getCentralCounterparty(), this);
			processor.processBasic(path.newSubPath("clearingMember"), String.class, getClearingMember(), this);
			processor.processBasic(path.newSubPath("confirmed"), ConfirmationEnum.class, getConfirmed(), this);
			processor.processBasic(path.newSubPath("callAmount"), BigDecimal.class, getCallAmount(), this);
			processor.processBasic(path.newSubPath("putAmount"), BigDecimal.class, getPutAmount(), this);
			processor.processBasic(path.newSubPath("callCurrency"), ISOCurrencyCodeEnum.class, getCallCurrency(), this);
			processor.processBasic(path.newSubPath("putCurrency"), ISOCurrencyCodeEnum.class, getPutCurrency(), this);
			processRosetta(path.newSubPath("priceSchedule"), processor, PricePeriod.PricePeriodBuilder.class, getPriceSchedule());
			processRosetta(path.newSubPath("strikePriceSchedule"), processor, PricePeriod.PricePeriodBuilder.class, getStrikePriceSchedule());
			processRosetta(path.newSubPath("price"), processor, PriceFormat.PriceFormatBuilder.class, getPrice());
			processor.processBasic(path.newSubPath("priceNotation"), PriceNotationEnum.class, getPriceNotation(), this);
			processor.processBasic(path.newSubPath("priceCurrency"), ISOCurrencyCodeEnum.class, getPriceCurrency(), this);
			processRosetta(path.newSubPath("packageTransactionPrice"), processor, PriceFormat.PriceFormatBuilder.class, getPackageTransactionPrice());
			processor.processBasic(path.newSubPath("packageTransactionPriceNotation"), PriceNotationEnum.class, getPackageTransactionPriceNotation(), this);
			processor.processBasic(path.newSubPath("packageTransactionPriceCurrency"), ISOCurrencyCodeEnum.class, getPackageTransactionPriceCurrency(), this);
			processRosetta(path.newSubPath("packageTransactionSpread"), processor, PriceFormat.PriceFormatBuilder.class, getPackageTransactionSpread());
			processor.processBasic(path.newSubPath("packageTransactionSpreadNotation"), PriceNotationEnum.class, getPackageTransactionSpreadNotation(), this);
			processor.processBasic(path.newSubPath("packageTransactionSpreadCurrency"), ISOCurrencyCodeEnum.class, getPackageTransactionSpreadCurrency(), this);
			processRosetta(path.newSubPath("strikePrice"), processor, PriceFormat.PriceFormatBuilder.class, getStrikePrice());
			processor.processBasic(path.newSubPath("strikePriceNotation"), PriceNotationEnum.class, getStrikePriceNotation(), this);
			processor.processBasic(path.newSubPath("strikePriceCurrency"), String.class, getStrikePriceCurrency(), this);
			processor.processBasic(path.newSubPath("priceUnitOfMeasure"), String.class, getPriceUnitOfMeasure(), this);
			processor.processBasic(path.newSubPath("optionPremiumAmount"), BigDecimal.class, getOptionPremiumAmount(), this);
			processor.processBasic(path.newSubPath("optionPremiumCurrency"), ISOCurrencyCodeEnum.class, getOptionPremiumCurrency(), this);
			processor.processBasic(path.newSubPath("optionPremiumPaymentDate"), Date.class, getOptionPremiumPaymentDate(), this);
			processor.processBasic(path.newSubPath("cdSIndexAttachmentPoint"), BigDecimal.class, getCdSIndexAttachmentPoint(), this);
			processor.processBasic(path.newSubPath("cdSIndexDetachmentPoint"), BigDecimal.class, getCdSIndexDetachmentPoint(), this);
			processor.processBasic(path.newSubPath("collateralPortfolioIndicator"), Boolean.class, getCollateralPortfolioIndicator(), this);
			processor.processBasic(path.newSubPath("firstExerciseDate"), Date.class, getFirstExerciseDate(), this);
			processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
			processRosetta(path.newSubPath("otherPayment"), processor, OtherPayment.OtherPaymentBuilder.class, getOtherPayment());
			processRosetta(path.newSubPath("leg1"), processor, CommonLeg.CommonLegBuilder.class, getLeg1());
			processRosetta(path.newSubPath("leg2"), processor, CommonLeg.CommonLegBuilder.class, getLeg2());
			processor.processBasic(path.newSubPath("counterparty2Name"), String.class, getCounterparty2Name(), this);
			processor.processBasic(path.newSubPath("executionAgentCounterparty1"), String.class, getExecutionAgentCounterparty1(), this);
			processor.processBasic(path.newSubPath("executionAgentCounterparty2DTCC"), String.class, getExecutionAgentCounterparty2DTCC(), this);
			processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifierProprietary"), String.class, getUniqueTransactionIdentifierProprietary(), this);
			processor.processBasic(path.newSubPath("priorUTI"), String.class, getPriorUTI(), this);
			processor.processBasic(path.newSubPath("priorUTIProprietary"), String.class, getPriorUTIProprietary(), this);
			processor.processBasic(path.newSubPath("uniqueProductIdentifier"), String.class, getUniqueProductIdentifier(), this);
			processor.processBasic(path.newSubPath("contractType"), CommonContractType.class, getContractType(), this);
			processor.processBasic(path.newSubPath("assetClass"), CommonAssetClass.class, getAssetClass(), this);
			processor.processBasic(path.newSubPath("clearingTimestamp"), ZonedDateTime.class, getClearingTimestamp(), this);
			processor.processBasic(path.newSubPath("platformIdentifier"), String.class, getPlatformIdentifier(), this);
			processor.processBasic(path.newSubPath("direction1"), Direction1Enum.class, getDirection1(), this);
			processor.processBasic(path.newSubPath("direction2Leg1"), Direction2Enum.class, getDirection2Leg1(), this);
			processor.processBasic(path.newSubPath("direction2Leg2"), Direction2Enum.class, getDirection2Leg2(), this);
			processor.processBasic(path.newSubPath("packageIdentifier"), String.class, getPackageIdentifier(), this);
			processRosetta(path.newSubPath("barrier"), processor, SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder.class, getBarrier());
			processor.processBasic(path.newSubPath("exchangeRate"), BigDecimal.class, getExchangeRate(), this);
			processor.processBasic(path.newSubPath("exchangeRateBasis"), String.class, getExchangeRateBasis(), this);
			processor.processBasic(path.newSubPath("nameOfTheUnderlyingIndex"), String.class, getNameOfTheUnderlyingIndex(), this);
			processRosetta(path.newSubPath("basketConstituents"), processor, BasketConstituentsReport.BasketConstituentsReportBuilder.class, getBasketConstituents());
			processRosetta(path.newSubPath("underlying"), processor, UnderlyingIdentifier.UnderlyingIdentifierBuilder.class, getUnderlying());
			processor.processBasic(path.newSubPath("quantityUnitOfMeasureLeg1"), String.class, getQuantityUnitOfMeasureLeg1(), this);
			processor.processBasic(path.newSubPath("quantityUnitOfMeasureLeg2"), String.class, getQuantityUnitOfMeasureLeg2(), this);
			processor.processBasic(path.newSubPath("maturityDateOfTheUnderlying"), Date.class, getMaturityDateOfTheUnderlying(), this);
			processor.processBasic(path.newSubPath("optionType"), OptionTypeCode.class, getOptionType(), this);
			processor.processBasic(path.newSubPath("optionStyle"), OptionStyleEnum.class, getOptionStyle(), this);
			processor.processBasic(path.newSubPath("deliveryType"), DeliveryTypeEnum.class, getDeliveryType(), this);
			processor.processBasic(path.newSubPath("countryOfTheCounterparty2"), ISOCountryCodeEnum.class, getCountryOfTheCounterparty2(), this);
			processor.processBasic(path.newSubPath("bookingLocation"), ISOCountryCodeEnum.class, getBookingLocation(), this);
			processor.processBasic(path.newSubPath("traderLocation"), ISOCountryCodeEnum.class, getTraderLocation(), this);
			processor.processBasic(path.newSubPath("tradingCapacity"), TradingCapacity7Code.class, getTradingCapacity(), this);
			processor.processBasic(path.newSubPath("embeddedOptionType"), EmbeddedOptionTypeEnum.class, getEmbeddedOptionType(), this);
			processor.processBasic(path.newSubPath("seniority"), SeniorityEnum.class, getSeniority(), this);
			processor.processBasic(path.newSubPath("series"), Integer.class, getSeries(), this);
			processor.processBasic(path.newSubPath("indexFactor"), BigDecimal.class, getIndexFactor(), this);
			processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
		}
		

		CommonTransactionReport.CommonTransactionReportBuilder prune();
	}

	/*********************** Immutable Implementation of CommonTransactionReport  ***********************/
	class CommonTransactionReportImpl extends CriticalDataElement.CriticalDataElementImpl implements CommonTransactionReport {
		private final CommonLeg leg1;
		private final CommonLeg leg2;
		private final String counterparty2Name;
		private final String executionAgentCounterparty1;
		private final String executionAgentCounterparty2DTCC;
		private final String reportSubmittingEntityID;
		private final String uniqueTransactionIdentifier;
		private final String uniqueTransactionIdentifierProprietary;
		private final String priorUTI;
		private final String priorUTIProprietary;
		private final String uniqueProductIdentifier;
		private final CommonContractType contractType;
		private final CommonAssetClass assetClass;
		private final ZonedDateTime clearingTimestamp;
		private final String platformIdentifier;
		private final Direction1Enum direction1;
		private final Direction2Enum direction2Leg1;
		private final Direction2Enum direction2Leg2;
		private final String packageIdentifier;
		private final SingleOrUpperAndLowerBarrier barrier;
		private final BigDecimal exchangeRate;
		private final String exchangeRateBasis;
		private final String nameOfTheUnderlyingIndex;
		private final List<? extends BasketConstituentsReport> basketConstituents;
		private final UnderlyingIdentifier underlying;
		private final String quantityUnitOfMeasureLeg1;
		private final String quantityUnitOfMeasureLeg2;
		private final Date maturityDateOfTheUnderlying;
		private final OptionTypeCode optionType;
		private final OptionStyleEnum optionStyle;
		private final DeliveryTypeEnum deliveryType;
		private final ISOCountryCodeEnum countryOfTheCounterparty2;
		private final ISOCountryCodeEnum bookingLocation;
		private final ISOCountryCodeEnum traderLocation;
		private final TradingCapacity7Code tradingCapacity;
		private final EmbeddedOptionTypeEnum embeddedOptionType;
		private final SeniorityEnum seniority;
		private final Integer series;
		private final BigDecimal indexFactor;
		private final String technicalRecordId;
		
		protected CommonTransactionReportImpl(CommonTransactionReport.CommonTransactionReportBuilder builder) {
			super(builder);
			this.leg1 = ofNullable(builder.getLeg1()).map(f->f.build()).orElse(null);
			this.leg2 = ofNullable(builder.getLeg2()).map(f->f.build()).orElse(null);
			this.counterparty2Name = builder.getCounterparty2Name();
			this.executionAgentCounterparty1 = builder.getExecutionAgentCounterparty1();
			this.executionAgentCounterparty2DTCC = builder.getExecutionAgentCounterparty2DTCC();
			this.reportSubmittingEntityID = builder.getReportSubmittingEntityID();
			this.uniqueTransactionIdentifier = builder.getUniqueTransactionIdentifier();
			this.uniqueTransactionIdentifierProprietary = builder.getUniqueTransactionIdentifierProprietary();
			this.priorUTI = builder.getPriorUTI();
			this.priorUTIProprietary = builder.getPriorUTIProprietary();
			this.uniqueProductIdentifier = builder.getUniqueProductIdentifier();
			this.contractType = builder.getContractType();
			this.assetClass = builder.getAssetClass();
			this.clearingTimestamp = builder.getClearingTimestamp();
			this.platformIdentifier = builder.getPlatformIdentifier();
			this.direction1 = builder.getDirection1();
			this.direction2Leg1 = builder.getDirection2Leg1();
			this.direction2Leg2 = builder.getDirection2Leg2();
			this.packageIdentifier = builder.getPackageIdentifier();
			this.barrier = ofNullable(builder.getBarrier()).map(f->f.build()).orElse(null);
			this.exchangeRate = builder.getExchangeRate();
			this.exchangeRateBasis = builder.getExchangeRateBasis();
			this.nameOfTheUnderlyingIndex = builder.getNameOfTheUnderlyingIndex();
			this.basketConstituents = ofNullable(builder.getBasketConstituents()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.underlying = ofNullable(builder.getUnderlying()).map(f->f.build()).orElse(null);
			this.quantityUnitOfMeasureLeg1 = builder.getQuantityUnitOfMeasureLeg1();
			this.quantityUnitOfMeasureLeg2 = builder.getQuantityUnitOfMeasureLeg2();
			this.maturityDateOfTheUnderlying = builder.getMaturityDateOfTheUnderlying();
			this.optionType = builder.getOptionType();
			this.optionStyle = builder.getOptionStyle();
			this.deliveryType = builder.getDeliveryType();
			this.countryOfTheCounterparty2 = builder.getCountryOfTheCounterparty2();
			this.bookingLocation = builder.getBookingLocation();
			this.traderLocation = builder.getTraderLocation();
			this.tradingCapacity = builder.getTradingCapacity();
			this.embeddedOptionType = builder.getEmbeddedOptionType();
			this.seniority = builder.getSeniority();
			this.series = builder.getSeries();
			this.indexFactor = builder.getIndexFactor();
			this.technicalRecordId = builder.getTechnicalRecordId();
		}
		
		@Override
		@RosettaAttribute("leg1")
		@RuneAttribute("leg1")
		public CommonLeg getLeg1() {
			return leg1;
		}
		
		@Override
		@RosettaAttribute("leg2")
		@RuneAttribute("leg2")
		public CommonLeg getLeg2() {
			return leg2;
		}
		
		@Override
		@RosettaAttribute("counterparty2Name")
		@RuneAttribute("counterparty2Name")
		public String getCounterparty2Name() {
			return counterparty2Name;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty1")
		@RuneAttribute("executionAgentCounterparty1")
		public String getExecutionAgentCounterparty1() {
			return executionAgentCounterparty1;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty2DTCC")
		@RuneAttribute("executionAgentCounterparty2DTCC")
		public String getExecutionAgentCounterparty2DTCC() {
			return executionAgentCounterparty2DTCC;
		}
		
		@Override
		@RosettaAttribute("reportSubmittingEntityID")
		@RuneAttribute("reportSubmittingEntityID")
		public String getReportSubmittingEntityID() {
			return reportSubmittingEntityID;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifierProprietary")
		@RuneAttribute("uniqueTransactionIdentifierProprietary")
		public String getUniqueTransactionIdentifierProprietary() {
			return uniqueTransactionIdentifierProprietary;
		}
		
		@Override
		@RosettaAttribute("priorUTI")
		@RuneAttribute("priorUTI")
		public String getPriorUTI() {
			return priorUTI;
		}
		
		@Override
		@RosettaAttribute("priorUTIProprietary")
		@RuneAttribute("priorUTIProprietary")
		public String getPriorUTIProprietary() {
			return priorUTIProprietary;
		}
		
		@Override
		@RosettaAttribute("uniqueProductIdentifier")
		@RuneAttribute("uniqueProductIdentifier")
		public String getUniqueProductIdentifier() {
			return uniqueProductIdentifier;
		}
		
		@Override
		@RosettaAttribute("contractType")
		@RuneAttribute("contractType")
		public CommonContractType getContractType() {
			return contractType;
		}
		
		@Override
		@RosettaAttribute("assetClass")
		@RuneAttribute("assetClass")
		public CommonAssetClass getAssetClass() {
			return assetClass;
		}
		
		@Override
		@RosettaAttribute("clearingTimestamp")
		@RuneAttribute("clearingTimestamp")
		public ZonedDateTime getClearingTimestamp() {
			return clearingTimestamp;
		}
		
		@Override
		@RosettaAttribute("platformIdentifier")
		@RuneAttribute("platformIdentifier")
		public String getPlatformIdentifier() {
			return platformIdentifier;
		}
		
		@Override
		@RosettaAttribute("direction1")
		@RuneAttribute("direction1")
		public Direction1Enum getDirection1() {
			return direction1;
		}
		
		@Override
		@RosettaAttribute("direction2Leg1")
		@RuneAttribute("direction2Leg1")
		public Direction2Enum getDirection2Leg1() {
			return direction2Leg1;
		}
		
		@Override
		@RosettaAttribute("direction2Leg2")
		@RuneAttribute("direction2Leg2")
		public Direction2Enum getDirection2Leg2() {
			return direction2Leg2;
		}
		
		@Override
		@RosettaAttribute("packageIdentifier")
		@RuneAttribute("packageIdentifier")
		public String getPackageIdentifier() {
			return packageIdentifier;
		}
		
		@Override
		@RosettaAttribute("barrier")
		@RuneAttribute("barrier")
		public SingleOrUpperAndLowerBarrier getBarrier() {
			return barrier;
		}
		
		@Override
		@RosettaAttribute("exchangeRate")
		@RuneAttribute("exchangeRate")
		public BigDecimal getExchangeRate() {
			return exchangeRate;
		}
		
		@Override
		@RosettaAttribute("exchangeRateBasis")
		@RuneAttribute("exchangeRateBasis")
		public String getExchangeRateBasis() {
			return exchangeRateBasis;
		}
		
		@Override
		@RosettaAttribute("nameOfTheUnderlyingIndex")
		@RuneAttribute("nameOfTheUnderlyingIndex")
		public String getNameOfTheUnderlyingIndex() {
			return nameOfTheUnderlyingIndex;
		}
		
		@Override
		@RosettaAttribute("basketConstituents")
		@RuneAttribute("basketConstituents")
		public List<? extends BasketConstituentsReport> getBasketConstituents() {
			return basketConstituents;
		}
		
		@Override
		@RosettaAttribute("underlying")
		@RuneAttribute("underlying")
		public UnderlyingIdentifier getUnderlying() {
			return underlying;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg1")
		@RuneAttribute("quantityUnitOfMeasureLeg1")
		public String getQuantityUnitOfMeasureLeg1() {
			return quantityUnitOfMeasureLeg1;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg2")
		@RuneAttribute("quantityUnitOfMeasureLeg2")
		public String getQuantityUnitOfMeasureLeg2() {
			return quantityUnitOfMeasureLeg2;
		}
		
		@Override
		@RosettaAttribute("maturityDateOfTheUnderlying")
		@RuneAttribute("maturityDateOfTheUnderlying")
		public Date getMaturityDateOfTheUnderlying() {
			return maturityDateOfTheUnderlying;
		}
		
		@Override
		@RosettaAttribute("optionType")
		@RuneAttribute("optionType")
		public OptionTypeCode getOptionType() {
			return optionType;
		}
		
		@Override
		@RosettaAttribute("optionStyle")
		@RuneAttribute("optionStyle")
		public OptionStyleEnum getOptionStyle() {
			return optionStyle;
		}
		
		@Override
		@RosettaAttribute("deliveryType")
		@RuneAttribute("deliveryType")
		public DeliveryTypeEnum getDeliveryType() {
			return deliveryType;
		}
		
		@Override
		@RosettaAttribute("countryOfTheCounterparty2")
		@RuneAttribute("countryOfTheCounterparty2")
		public ISOCountryCodeEnum getCountryOfTheCounterparty2() {
			return countryOfTheCounterparty2;
		}
		
		@Override
		@RosettaAttribute("bookingLocation")
		@RuneAttribute("bookingLocation")
		public ISOCountryCodeEnum getBookingLocation() {
			return bookingLocation;
		}
		
		@Override
		@RosettaAttribute("traderLocation")
		@RuneAttribute("traderLocation")
		public ISOCountryCodeEnum getTraderLocation() {
			return traderLocation;
		}
		
		@Override
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		public TradingCapacity7Code getTradingCapacity() {
			return tradingCapacity;
		}
		
		@Override
		@RosettaAttribute("embeddedOptionType")
		@RuneAttribute("embeddedOptionType")
		public EmbeddedOptionTypeEnum getEmbeddedOptionType() {
			return embeddedOptionType;
		}
		
		@Override
		@RosettaAttribute("seniority")
		@RuneAttribute("seniority")
		public SeniorityEnum getSeniority() {
			return seniority;
		}
		
		@Override
		@RosettaAttribute("series")
		@RuneAttribute("series")
		public Integer getSeries() {
			return series;
		}
		
		@Override
		@RosettaAttribute("indexFactor")
		@RuneAttribute("indexFactor")
		public BigDecimal getIndexFactor() {
			return indexFactor;
		}
		
		@Override
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		public CommonTransactionReport build() {
			return this;
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder toBuilder() {
			CommonTransactionReport.CommonTransactionReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(CommonTransactionReport.CommonTransactionReportBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getLeg1()).ifPresent(builder::setLeg1);
			ofNullable(getLeg2()).ifPresent(builder::setLeg2);
			ofNullable(getCounterparty2Name()).ifPresent(builder::setCounterparty2Name);
			ofNullable(getExecutionAgentCounterparty1()).ifPresent(builder::setExecutionAgentCounterparty1);
			ofNullable(getExecutionAgentCounterparty2DTCC()).ifPresent(builder::setExecutionAgentCounterparty2DTCC);
			ofNullable(getReportSubmittingEntityID()).ifPresent(builder::setReportSubmittingEntityID);
			ofNullable(getUniqueTransactionIdentifier()).ifPresent(builder::setUniqueTransactionIdentifier);
			ofNullable(getUniqueTransactionIdentifierProprietary()).ifPresent(builder::setUniqueTransactionIdentifierProprietary);
			ofNullable(getPriorUTI()).ifPresent(builder::setPriorUTI);
			ofNullable(getPriorUTIProprietary()).ifPresent(builder::setPriorUTIProprietary);
			ofNullable(getUniqueProductIdentifier()).ifPresent(builder::setUniqueProductIdentifier);
			ofNullable(getContractType()).ifPresent(builder::setContractType);
			ofNullable(getAssetClass()).ifPresent(builder::setAssetClass);
			ofNullable(getClearingTimestamp()).ifPresent(builder::setClearingTimestamp);
			ofNullable(getPlatformIdentifier()).ifPresent(builder::setPlatformIdentifier);
			ofNullable(getDirection1()).ifPresent(builder::setDirection1);
			ofNullable(getDirection2Leg1()).ifPresent(builder::setDirection2Leg1);
			ofNullable(getDirection2Leg2()).ifPresent(builder::setDirection2Leg2);
			ofNullable(getPackageIdentifier()).ifPresent(builder::setPackageIdentifier);
			ofNullable(getBarrier()).ifPresent(builder::setBarrier);
			ofNullable(getExchangeRate()).ifPresent(builder::setExchangeRate);
			ofNullable(getExchangeRateBasis()).ifPresent(builder::setExchangeRateBasis);
			ofNullable(getNameOfTheUnderlyingIndex()).ifPresent(builder::setNameOfTheUnderlyingIndex);
			ofNullable(getBasketConstituents()).ifPresent(builder::setBasketConstituents);
			ofNullable(getUnderlying()).ifPresent(builder::setUnderlying);
			ofNullable(getQuantityUnitOfMeasureLeg1()).ifPresent(builder::setQuantityUnitOfMeasureLeg1);
			ofNullable(getQuantityUnitOfMeasureLeg2()).ifPresent(builder::setQuantityUnitOfMeasureLeg2);
			ofNullable(getMaturityDateOfTheUnderlying()).ifPresent(builder::setMaturityDateOfTheUnderlying);
			ofNullable(getOptionType()).ifPresent(builder::setOptionType);
			ofNullable(getOptionStyle()).ifPresent(builder::setOptionStyle);
			ofNullable(getDeliveryType()).ifPresent(builder::setDeliveryType);
			ofNullable(getCountryOfTheCounterparty2()).ifPresent(builder::setCountryOfTheCounterparty2);
			ofNullable(getBookingLocation()).ifPresent(builder::setBookingLocation);
			ofNullable(getTraderLocation()).ifPresent(builder::setTraderLocation);
			ofNullable(getTradingCapacity()).ifPresent(builder::setTradingCapacity);
			ofNullable(getEmbeddedOptionType()).ifPresent(builder::setEmbeddedOptionType);
			ofNullable(getSeniority()).ifPresent(builder::setSeniority);
			ofNullable(getSeries()).ifPresent(builder::setSeries);
			ofNullable(getIndexFactor()).ifPresent(builder::setIndexFactor);
			ofNullable(getTechnicalRecordId()).ifPresent(builder::setTechnicalRecordId);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			CommonTransactionReport _that = getType().cast(o);
		
			if (!Objects.equals(leg1, _that.getLeg1())) return false;
			if (!Objects.equals(leg2, _that.getLeg2())) return false;
			if (!Objects.equals(counterparty2Name, _that.getCounterparty2Name())) return false;
			if (!Objects.equals(executionAgentCounterparty1, _that.getExecutionAgentCounterparty1())) return false;
			if (!Objects.equals(executionAgentCounterparty2DTCC, _that.getExecutionAgentCounterparty2DTCC())) return false;
			if (!Objects.equals(reportSubmittingEntityID, _that.getReportSubmittingEntityID())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifierProprietary, _that.getUniqueTransactionIdentifierProprietary())) return false;
			if (!Objects.equals(priorUTI, _that.getPriorUTI())) return false;
			if (!Objects.equals(priorUTIProprietary, _that.getPriorUTIProprietary())) return false;
			if (!Objects.equals(uniqueProductIdentifier, _that.getUniqueProductIdentifier())) return false;
			if (!Objects.equals(contractType, _that.getContractType())) return false;
			if (!Objects.equals(assetClass, _that.getAssetClass())) return false;
			if (!Objects.equals(clearingTimestamp, _that.getClearingTimestamp())) return false;
			if (!Objects.equals(platformIdentifier, _that.getPlatformIdentifier())) return false;
			if (!Objects.equals(direction1, _that.getDirection1())) return false;
			if (!Objects.equals(direction2Leg1, _that.getDirection2Leg1())) return false;
			if (!Objects.equals(direction2Leg2, _that.getDirection2Leg2())) return false;
			if (!Objects.equals(packageIdentifier, _that.getPackageIdentifier())) return false;
			if (!Objects.equals(barrier, _that.getBarrier())) return false;
			if (!Objects.equals(exchangeRate, _that.getExchangeRate())) return false;
			if (!Objects.equals(exchangeRateBasis, _that.getExchangeRateBasis())) return false;
			if (!Objects.equals(nameOfTheUnderlyingIndex, _that.getNameOfTheUnderlyingIndex())) return false;
			if (!ListEquals.listEquals(basketConstituents, _that.getBasketConstituents())) return false;
			if (!Objects.equals(underlying, _that.getUnderlying())) return false;
			if (!Objects.equals(quantityUnitOfMeasureLeg1, _that.getQuantityUnitOfMeasureLeg1())) return false;
			if (!Objects.equals(quantityUnitOfMeasureLeg2, _that.getQuantityUnitOfMeasureLeg2())) return false;
			if (!Objects.equals(maturityDateOfTheUnderlying, _that.getMaturityDateOfTheUnderlying())) return false;
			if (!Objects.equals(optionType, _that.getOptionType())) return false;
			if (!Objects.equals(optionStyle, _that.getOptionStyle())) return false;
			if (!Objects.equals(deliveryType, _that.getDeliveryType())) return false;
			if (!Objects.equals(countryOfTheCounterparty2, _that.getCountryOfTheCounterparty2())) return false;
			if (!Objects.equals(bookingLocation, _that.getBookingLocation())) return false;
			if (!Objects.equals(traderLocation, _that.getTraderLocation())) return false;
			if (!Objects.equals(tradingCapacity, _that.getTradingCapacity())) return false;
			if (!Objects.equals(embeddedOptionType, _that.getEmbeddedOptionType())) return false;
			if (!Objects.equals(seniority, _that.getSeniority())) return false;
			if (!Objects.equals(series, _that.getSeries())) return false;
			if (!Objects.equals(indexFactor, _that.getIndexFactor())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (leg1 != null ? leg1.hashCode() : 0);
			_result = 31 * _result + (leg2 != null ? leg2.hashCode() : 0);
			_result = 31 * _result + (counterparty2Name != null ? counterparty2Name.hashCode() : 0);
			_result = 31 * _result + (executionAgentCounterparty1 != null ? executionAgentCounterparty1.hashCode() : 0);
			_result = 31 * _result + (executionAgentCounterparty2DTCC != null ? executionAgentCounterparty2DTCC.hashCode() : 0);
			_result = 31 * _result + (reportSubmittingEntityID != null ? reportSubmittingEntityID.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifierProprietary != null ? uniqueTransactionIdentifierProprietary.hashCode() : 0);
			_result = 31 * _result + (priorUTI != null ? priorUTI.hashCode() : 0);
			_result = 31 * _result + (priorUTIProprietary != null ? priorUTIProprietary.hashCode() : 0);
			_result = 31 * _result + (uniqueProductIdentifier != null ? uniqueProductIdentifier.hashCode() : 0);
			_result = 31 * _result + (contractType != null ? contractType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (assetClass != null ? assetClass.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (clearingTimestamp != null ? clearingTimestamp.hashCode() : 0);
			_result = 31 * _result + (platformIdentifier != null ? platformIdentifier.hashCode() : 0);
			_result = 31 * _result + (direction1 != null ? direction1.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (direction2Leg1 != null ? direction2Leg1.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (direction2Leg2 != null ? direction2Leg2.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageIdentifier != null ? packageIdentifier.hashCode() : 0);
			_result = 31 * _result + (barrier != null ? barrier.hashCode() : 0);
			_result = 31 * _result + (exchangeRate != null ? exchangeRate.hashCode() : 0);
			_result = 31 * _result + (exchangeRateBasis != null ? exchangeRateBasis.hashCode() : 0);
			_result = 31 * _result + (nameOfTheUnderlyingIndex != null ? nameOfTheUnderlyingIndex.hashCode() : 0);
			_result = 31 * _result + (basketConstituents != null ? basketConstituents.hashCode() : 0);
			_result = 31 * _result + (underlying != null ? underlying.hashCode() : 0);
			_result = 31 * _result + (quantityUnitOfMeasureLeg1 != null ? quantityUnitOfMeasureLeg1.hashCode() : 0);
			_result = 31 * _result + (quantityUnitOfMeasureLeg2 != null ? quantityUnitOfMeasureLeg2.hashCode() : 0);
			_result = 31 * _result + (maturityDateOfTheUnderlying != null ? maturityDateOfTheUnderlying.hashCode() : 0);
			_result = 31 * _result + (optionType != null ? optionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (optionStyle != null ? optionStyle.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (deliveryType != null ? deliveryType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (countryOfTheCounterparty2 != null ? countryOfTheCounterparty2.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (bookingLocation != null ? bookingLocation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (traderLocation != null ? traderLocation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradingCapacity != null ? tradingCapacity.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (embeddedOptionType != null ? embeddedOptionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (seniority != null ? seniority.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (series != null ? series.hashCode() : 0);
			_result = 31 * _result + (indexFactor != null ? indexFactor.hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CommonTransactionReport {" +
				"leg1=" + this.leg1 + ", " +
				"leg2=" + this.leg2 + ", " +
				"counterparty2Name=" + this.counterparty2Name + ", " +
				"executionAgentCounterparty1=" + this.executionAgentCounterparty1 + ", " +
				"executionAgentCounterparty2DTCC=" + this.executionAgentCounterparty2DTCC + ", " +
				"reportSubmittingEntityID=" + this.reportSubmittingEntityID + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"uniqueTransactionIdentifierProprietary=" + this.uniqueTransactionIdentifierProprietary + ", " +
				"priorUTI=" + this.priorUTI + ", " +
				"priorUTIProprietary=" + this.priorUTIProprietary + ", " +
				"uniqueProductIdentifier=" + this.uniqueProductIdentifier + ", " +
				"contractType=" + this.contractType + ", " +
				"assetClass=" + this.assetClass + ", " +
				"clearingTimestamp=" + this.clearingTimestamp + ", " +
				"platformIdentifier=" + this.platformIdentifier + ", " +
				"direction1=" + this.direction1 + ", " +
				"direction2Leg1=" + this.direction2Leg1 + ", " +
				"direction2Leg2=" + this.direction2Leg2 + ", " +
				"packageIdentifier=" + this.packageIdentifier + ", " +
				"barrier=" + this.barrier + ", " +
				"exchangeRate=" + this.exchangeRate + ", " +
				"exchangeRateBasis=" + this.exchangeRateBasis + ", " +
				"nameOfTheUnderlyingIndex=" + this.nameOfTheUnderlyingIndex + ", " +
				"basketConstituents=" + this.basketConstituents + ", " +
				"underlying=" + this.underlying + ", " +
				"quantityUnitOfMeasureLeg1=" + this.quantityUnitOfMeasureLeg1 + ", " +
				"quantityUnitOfMeasureLeg2=" + this.quantityUnitOfMeasureLeg2 + ", " +
				"maturityDateOfTheUnderlying=" + this.maturityDateOfTheUnderlying + ", " +
				"optionType=" + this.optionType + ", " +
				"optionStyle=" + this.optionStyle + ", " +
				"deliveryType=" + this.deliveryType + ", " +
				"countryOfTheCounterparty2=" + this.countryOfTheCounterparty2 + ", " +
				"bookingLocation=" + this.bookingLocation + ", " +
				"traderLocation=" + this.traderLocation + ", " +
				"tradingCapacity=" + this.tradingCapacity + ", " +
				"embeddedOptionType=" + this.embeddedOptionType + ", " +
				"seniority=" + this.seniority + ", " +
				"series=" + this.series + ", " +
				"indexFactor=" + this.indexFactor + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of CommonTransactionReport  ***********************/
	class CommonTransactionReportBuilderImpl implements CommonTransactionReport.CommonTransactionReportBuilder {
	
		protected ActionTypeEnum actionType;
		protected EventTypeEnum eventType;
		protected String eventIdentifier;
		protected Date effectiveDate;
		protected Date earlyTerminationDate;
		protected ZonedDateTime eventTimestamp;
		protected ZonedDateTime reportingTimestamp;
		protected ZonedDateTime executionTimestamp;
		protected Date expirationDate;
		protected String counterparty1;
		protected String counterparty2;
		protected Boolean counterparty2IdentifierType;
		protected String beneficiary1;
		protected Boolean beneficiary1IdentifierTypeIndicator;
		protected ClearedEnum cleared;
		protected String centralCounterparty;
		protected String clearingMember;
		protected ConfirmationEnum confirmed;
		protected BigDecimal callAmount;
		protected BigDecimal putAmount;
		protected ISOCurrencyCodeEnum callCurrency;
		protected ISOCurrencyCodeEnum putCurrency;
		protected List<PricePeriod.PricePeriodBuilder> priceSchedule = new ArrayList<>();
		protected List<PricePeriod.PricePeriodBuilder> strikePriceSchedule = new ArrayList<>();
		protected PriceFormat.PriceFormatBuilder price;
		protected PriceNotationEnum priceNotation;
		protected ISOCurrencyCodeEnum priceCurrency;
		protected PriceFormat.PriceFormatBuilder packageTransactionPrice;
		protected PriceNotationEnum packageTransactionPriceNotation;
		protected ISOCurrencyCodeEnum packageTransactionPriceCurrency;
		protected PriceFormat.PriceFormatBuilder packageTransactionSpread;
		protected PriceNotationEnum packageTransactionSpreadNotation;
		protected ISOCurrencyCodeEnum packageTransactionSpreadCurrency;
		protected PriceFormat.PriceFormatBuilder strikePrice;
		protected PriceNotationEnum strikePriceNotation;
		protected String strikePriceCurrency;
		protected String priceUnitOfMeasure;
		protected BigDecimal optionPremiumAmount;
		protected ISOCurrencyCodeEnum optionPremiumCurrency;
		protected Date optionPremiumPaymentDate;
		protected BigDecimal cdSIndexAttachmentPoint;
		protected BigDecimal cdSIndexDetachmentPoint;
		protected Boolean collateralPortfolioIndicator;
		protected Date firstExerciseDate;
		protected Date finalContractualSettlementDate;
		protected List<OtherPayment.OtherPaymentBuilder> otherPayment = new ArrayList<>();
		protected CommonLeg.CommonLegBuilder leg1;
		protected CommonLeg.CommonLegBuilder leg2;
		protected String counterparty2Name;
		protected String executionAgentCounterparty1;
		protected String executionAgentCounterparty2DTCC;
		protected String reportSubmittingEntityID;
		protected String uniqueTransactionIdentifier;
		protected String uniqueTransactionIdentifierProprietary;
		protected String priorUTI;
		protected String priorUTIProprietary;
		protected String uniqueProductIdentifier;
		protected CommonContractType contractType;
		protected CommonAssetClass assetClass;
		protected ZonedDateTime clearingTimestamp;
		protected String platformIdentifier;
		protected Direction1Enum direction1;
		protected Direction2Enum direction2Leg1;
		protected Direction2Enum direction2Leg2;
		protected String packageIdentifier;
		protected SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder barrier;
		protected BigDecimal exchangeRate;
		protected String exchangeRateBasis;
		protected String nameOfTheUnderlyingIndex;
		protected List<BasketConstituentsReport.BasketConstituentsReportBuilder> basketConstituents = new ArrayList<>();
		protected UnderlyingIdentifier.UnderlyingIdentifierBuilder underlying;
		protected String quantityUnitOfMeasureLeg1;
		protected String quantityUnitOfMeasureLeg2;
		protected Date maturityDateOfTheUnderlying;
		protected OptionTypeCode optionType;
		protected OptionStyleEnum optionStyle;
		protected DeliveryTypeEnum deliveryType;
		protected ISOCountryCodeEnum countryOfTheCounterparty2;
		protected ISOCountryCodeEnum bookingLocation;
		protected ISOCountryCodeEnum traderLocation;
		protected TradingCapacity7Code tradingCapacity;
		protected EmbeddedOptionTypeEnum embeddedOptionType;
		protected SeniorityEnum seniority;
		protected Integer series;
		protected BigDecimal indexFactor;
		protected String technicalRecordId;
		
		@Override
		@RosettaAttribute("actionType")
		@RuneAttribute("actionType")
		public ActionTypeEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute("eventType")
		@RuneAttribute("eventType")
		public EventTypeEnum getEventType() {
			return eventType;
		}
		
		@Override
		@RosettaAttribute("eventIdentifier")
		@RuneAttribute("eventIdentifier")
		public String getEventIdentifier() {
			return eventIdentifier;
		}
		
		@Override
		@RosettaAttribute("effectiveDate")
		@RuneAttribute("effectiveDate")
		public Date getEffectiveDate() {
			return effectiveDate;
		}
		
		@Override
		@RosettaAttribute("earlyTerminationDate")
		@RuneAttribute("earlyTerminationDate")
		public Date getEarlyTerminationDate() {
			return earlyTerminationDate;
		}
		
		@Override
		@RosettaAttribute("eventTimestamp")
		@RuneAttribute("eventTimestamp")
		public ZonedDateTime getEventTimestamp() {
			return eventTimestamp;
		}
		
		@Override
		@RosettaAttribute("reportingTimestamp")
		@RuneAttribute("reportingTimestamp")
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute("executionTimestamp")
		@RuneAttribute("executionTimestamp")
		public ZonedDateTime getExecutionTimestamp() {
			return executionTimestamp;
		}
		
		@Override
		@RosettaAttribute("expirationDate")
		@RuneAttribute("expirationDate")
		public Date getExpirationDate() {
			return expirationDate;
		}
		
		@Override
		@RosettaAttribute("counterparty1")
		@RuneAttribute("counterparty1")
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute("counterparty2")
		@RuneAttribute("counterparty2")
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute("counterparty2IdentifierType")
		@RuneAttribute("counterparty2IdentifierType")
		public Boolean getCounterparty2IdentifierType() {
			return counterparty2IdentifierType;
		}
		
		@Override
		@RosettaAttribute("beneficiary1")
		@RuneAttribute("beneficiary1")
		public String getBeneficiary1() {
			return beneficiary1;
		}
		
		@Override
		@RosettaAttribute("beneficiary1IdentifierTypeIndicator")
		@RuneAttribute("beneficiary1IdentifierTypeIndicator")
		public Boolean getBeneficiary1IdentifierTypeIndicator() {
			return beneficiary1IdentifierTypeIndicator;
		}
		
		@Override
		@RosettaAttribute("cleared")
		@RuneAttribute("cleared")
		public ClearedEnum getCleared() {
			return cleared;
		}
		
		@Override
		@RosettaAttribute("centralCounterparty")
		@RuneAttribute("centralCounterparty")
		public String getCentralCounterparty() {
			return centralCounterparty;
		}
		
		@Override
		@RosettaAttribute("clearingMember")
		@RuneAttribute("clearingMember")
		public String getClearingMember() {
			return clearingMember;
		}
		
		@Override
		@RosettaAttribute("confirmed")
		@RuneAttribute("confirmed")
		public ConfirmationEnum getConfirmed() {
			return confirmed;
		}
		
		@Override
		@RosettaAttribute("callAmount")
		@RuneAttribute("callAmount")
		public BigDecimal getCallAmount() {
			return callAmount;
		}
		
		@Override
		@RosettaAttribute("putAmount")
		@RuneAttribute("putAmount")
		public BigDecimal getPutAmount() {
			return putAmount;
		}
		
		@Override
		@RosettaAttribute("callCurrency")
		@RuneAttribute("callCurrency")
		public ISOCurrencyCodeEnum getCallCurrency() {
			return callCurrency;
		}
		
		@Override
		@RosettaAttribute("putCurrency")
		@RuneAttribute("putCurrency")
		public ISOCurrencyCodeEnum getPutCurrency() {
			return putCurrency;
		}
		
		@Override
		@RosettaAttribute("priceSchedule")
		@RuneAttribute("priceSchedule")
		public List<? extends PricePeriod.PricePeriodBuilder> getPriceSchedule() {
			return priceSchedule;
		}
		
		@Override
		public PricePeriod.PricePeriodBuilder getOrCreatePriceSchedule(int _index) {
		
			if (priceSchedule==null) {
				this.priceSchedule = new ArrayList<>();
			}
			PricePeriod.PricePeriodBuilder result;
			return getIndex(priceSchedule, _index, () -> {
						PricePeriod.PricePeriodBuilder newPriceSchedule = PricePeriod.builder();
						return newPriceSchedule;
					});
		}
		
		@Override
		@RosettaAttribute("strikePriceSchedule")
		@RuneAttribute("strikePriceSchedule")
		public List<? extends PricePeriod.PricePeriodBuilder> getStrikePriceSchedule() {
			return strikePriceSchedule;
		}
		
		@Override
		public PricePeriod.PricePeriodBuilder getOrCreateStrikePriceSchedule(int _index) {
		
			if (strikePriceSchedule==null) {
				this.strikePriceSchedule = new ArrayList<>();
			}
			PricePeriod.PricePeriodBuilder result;
			return getIndex(strikePriceSchedule, _index, () -> {
						PricePeriod.PricePeriodBuilder newStrikePriceSchedule = PricePeriod.builder();
						return newStrikePriceSchedule;
					});
		}
		
		@Override
		@RosettaAttribute("price")
		@RuneAttribute("price")
		public PriceFormat.PriceFormatBuilder getPrice() {
			return price;
		}
		
		@Override
		public PriceFormat.PriceFormatBuilder getOrCreatePrice() {
			PriceFormat.PriceFormatBuilder result;
			if (price!=null) {
				result = price;
			}
			else {
				result = price = PriceFormat.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("priceNotation")
		@RuneAttribute("priceNotation")
		public PriceNotationEnum getPriceNotation() {
			return priceNotation;
		}
		
		@Override
		@RosettaAttribute("priceCurrency")
		@RuneAttribute("priceCurrency")
		public ISOCurrencyCodeEnum getPriceCurrency() {
			return priceCurrency;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPrice")
		@RuneAttribute("packageTransactionPrice")
		public PriceFormat.PriceFormatBuilder getPackageTransactionPrice() {
			return packageTransactionPrice;
		}
		
		@Override
		public PriceFormat.PriceFormatBuilder getOrCreatePackageTransactionPrice() {
			PriceFormat.PriceFormatBuilder result;
			if (packageTransactionPrice!=null) {
				result = packageTransactionPrice;
			}
			else {
				result = packageTransactionPrice = PriceFormat.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPriceNotation")
		@RuneAttribute("packageTransactionPriceNotation")
		public PriceNotationEnum getPackageTransactionPriceNotation() {
			return packageTransactionPriceNotation;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPriceCurrency")
		@RuneAttribute("packageTransactionPriceCurrency")
		public ISOCurrencyCodeEnum getPackageTransactionPriceCurrency() {
			return packageTransactionPriceCurrency;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpread")
		@RuneAttribute("packageTransactionSpread")
		public PriceFormat.PriceFormatBuilder getPackageTransactionSpread() {
			return packageTransactionSpread;
		}
		
		@Override
		public PriceFormat.PriceFormatBuilder getOrCreatePackageTransactionSpread() {
			PriceFormat.PriceFormatBuilder result;
			if (packageTransactionSpread!=null) {
				result = packageTransactionSpread;
			}
			else {
				result = packageTransactionSpread = PriceFormat.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpreadNotation")
		@RuneAttribute("packageTransactionSpreadNotation")
		public PriceNotationEnum getPackageTransactionSpreadNotation() {
			return packageTransactionSpreadNotation;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpreadCurrency")
		@RuneAttribute("packageTransactionSpreadCurrency")
		public ISOCurrencyCodeEnum getPackageTransactionSpreadCurrency() {
			return packageTransactionSpreadCurrency;
		}
		
		@Override
		@RosettaAttribute("strikePrice")
		@RuneAttribute("strikePrice")
		public PriceFormat.PriceFormatBuilder getStrikePrice() {
			return strikePrice;
		}
		
		@Override
		public PriceFormat.PriceFormatBuilder getOrCreateStrikePrice() {
			PriceFormat.PriceFormatBuilder result;
			if (strikePrice!=null) {
				result = strikePrice;
			}
			else {
				result = strikePrice = PriceFormat.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("strikePriceNotation")
		@RuneAttribute("strikePriceNotation")
		public PriceNotationEnum getStrikePriceNotation() {
			return strikePriceNotation;
		}
		
		@Override
		@RosettaAttribute("strikePriceCurrency")
		@RuneAttribute("strikePriceCurrency")
		public String getStrikePriceCurrency() {
			return strikePriceCurrency;
		}
		
		@Override
		@RosettaAttribute("priceUnitOfMeasure")
		@RuneAttribute("priceUnitOfMeasure")
		public String getPriceUnitOfMeasure() {
			return priceUnitOfMeasure;
		}
		
		@Override
		@RosettaAttribute("optionPremiumAmount")
		@RuneAttribute("optionPremiumAmount")
		public BigDecimal getOptionPremiumAmount() {
			return optionPremiumAmount;
		}
		
		@Override
		@RosettaAttribute("optionPremiumCurrency")
		@RuneAttribute("optionPremiumCurrency")
		public ISOCurrencyCodeEnum getOptionPremiumCurrency() {
			return optionPremiumCurrency;
		}
		
		@Override
		@RosettaAttribute("optionPremiumPaymentDate")
		@RuneAttribute("optionPremiumPaymentDate")
		public Date getOptionPremiumPaymentDate() {
			return optionPremiumPaymentDate;
		}
		
		@Override
		@RosettaAttribute("cdSIndexAttachmentPoint")
		@RuneAttribute("cdSIndexAttachmentPoint")
		public BigDecimal getCdSIndexAttachmentPoint() {
			return cdSIndexAttachmentPoint;
		}
		
		@Override
		@RosettaAttribute("cdSIndexDetachmentPoint")
		@RuneAttribute("cdSIndexDetachmentPoint")
		public BigDecimal getCdSIndexDetachmentPoint() {
			return cdSIndexDetachmentPoint;
		}
		
		@Override
		@RosettaAttribute("collateralPortfolioIndicator")
		@RuneAttribute("collateralPortfolioIndicator")
		public Boolean getCollateralPortfolioIndicator() {
			return collateralPortfolioIndicator;
		}
		
		@Override
		@RosettaAttribute("firstExerciseDate")
		@RuneAttribute("firstExerciseDate")
		public Date getFirstExerciseDate() {
			return firstExerciseDate;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public Date getFinalContractualSettlementDate() {
			return finalContractualSettlementDate;
		}
		
		@Override
		@RosettaAttribute("otherPayment")
		@RuneAttribute("otherPayment")
		public List<? extends OtherPayment.OtherPaymentBuilder> getOtherPayment() {
			return otherPayment;
		}
		
		@Override
		public OtherPayment.OtherPaymentBuilder getOrCreateOtherPayment(int _index) {
		
			if (otherPayment==null) {
				this.otherPayment = new ArrayList<>();
			}
			OtherPayment.OtherPaymentBuilder result;
			return getIndex(otherPayment, _index, () -> {
						OtherPayment.OtherPaymentBuilder newOtherPayment = OtherPayment.builder();
						return newOtherPayment;
					});
		}
		
		@Override
		@RosettaAttribute("leg1")
		@RuneAttribute("leg1")
		public CommonLeg.CommonLegBuilder getLeg1() {
			return leg1;
		}
		
		@Override
		public CommonLeg.CommonLegBuilder getOrCreateLeg1() {
			CommonLeg.CommonLegBuilder result;
			if (leg1!=null) {
				result = leg1;
			}
			else {
				result = leg1 = CommonLeg.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("leg2")
		@RuneAttribute("leg2")
		public CommonLeg.CommonLegBuilder getLeg2() {
			return leg2;
		}
		
		@Override
		public CommonLeg.CommonLegBuilder getOrCreateLeg2() {
			CommonLeg.CommonLegBuilder result;
			if (leg2!=null) {
				result = leg2;
			}
			else {
				result = leg2 = CommonLeg.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("counterparty2Name")
		@RuneAttribute("counterparty2Name")
		public String getCounterparty2Name() {
			return counterparty2Name;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty1")
		@RuneAttribute("executionAgentCounterparty1")
		public String getExecutionAgentCounterparty1() {
			return executionAgentCounterparty1;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty2DTCC")
		@RuneAttribute("executionAgentCounterparty2DTCC")
		public String getExecutionAgentCounterparty2DTCC() {
			return executionAgentCounterparty2DTCC;
		}
		
		@Override
		@RosettaAttribute("reportSubmittingEntityID")
		@RuneAttribute("reportSubmittingEntityID")
		public String getReportSubmittingEntityID() {
			return reportSubmittingEntityID;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifierProprietary")
		@RuneAttribute("uniqueTransactionIdentifierProprietary")
		public String getUniqueTransactionIdentifierProprietary() {
			return uniqueTransactionIdentifierProprietary;
		}
		
		@Override
		@RosettaAttribute("priorUTI")
		@RuneAttribute("priorUTI")
		public String getPriorUTI() {
			return priorUTI;
		}
		
		@Override
		@RosettaAttribute("priorUTIProprietary")
		@RuneAttribute("priorUTIProprietary")
		public String getPriorUTIProprietary() {
			return priorUTIProprietary;
		}
		
		@Override
		@RosettaAttribute("uniqueProductIdentifier")
		@RuneAttribute("uniqueProductIdentifier")
		public String getUniqueProductIdentifier() {
			return uniqueProductIdentifier;
		}
		
		@Override
		@RosettaAttribute("contractType")
		@RuneAttribute("contractType")
		public CommonContractType getContractType() {
			return contractType;
		}
		
		@Override
		@RosettaAttribute("assetClass")
		@RuneAttribute("assetClass")
		public CommonAssetClass getAssetClass() {
			return assetClass;
		}
		
		@Override
		@RosettaAttribute("clearingTimestamp")
		@RuneAttribute("clearingTimestamp")
		public ZonedDateTime getClearingTimestamp() {
			return clearingTimestamp;
		}
		
		@Override
		@RosettaAttribute("platformIdentifier")
		@RuneAttribute("platformIdentifier")
		public String getPlatformIdentifier() {
			return platformIdentifier;
		}
		
		@Override
		@RosettaAttribute("direction1")
		@RuneAttribute("direction1")
		public Direction1Enum getDirection1() {
			return direction1;
		}
		
		@Override
		@RosettaAttribute("direction2Leg1")
		@RuneAttribute("direction2Leg1")
		public Direction2Enum getDirection2Leg1() {
			return direction2Leg1;
		}
		
		@Override
		@RosettaAttribute("direction2Leg2")
		@RuneAttribute("direction2Leg2")
		public Direction2Enum getDirection2Leg2() {
			return direction2Leg2;
		}
		
		@Override
		@RosettaAttribute("packageIdentifier")
		@RuneAttribute("packageIdentifier")
		public String getPackageIdentifier() {
			return packageIdentifier;
		}
		
		@Override
		@RosettaAttribute("barrier")
		@RuneAttribute("barrier")
		public SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder getBarrier() {
			return barrier;
		}
		
		@Override
		public SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder getOrCreateBarrier() {
			SingleOrUpperAndLowerBarrier.SingleOrUpperAndLowerBarrierBuilder result;
			if (barrier!=null) {
				result = barrier;
			}
			else {
				result = barrier = SingleOrUpperAndLowerBarrier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("exchangeRate")
		@RuneAttribute("exchangeRate")
		public BigDecimal getExchangeRate() {
			return exchangeRate;
		}
		
		@Override
		@RosettaAttribute("exchangeRateBasis")
		@RuneAttribute("exchangeRateBasis")
		public String getExchangeRateBasis() {
			return exchangeRateBasis;
		}
		
		@Override
		@RosettaAttribute("nameOfTheUnderlyingIndex")
		@RuneAttribute("nameOfTheUnderlyingIndex")
		public String getNameOfTheUnderlyingIndex() {
			return nameOfTheUnderlyingIndex;
		}
		
		@Override
		@RosettaAttribute("basketConstituents")
		@RuneAttribute("basketConstituents")
		public List<? extends BasketConstituentsReport.BasketConstituentsReportBuilder> getBasketConstituents() {
			return basketConstituents;
		}
		
		@Override
		public BasketConstituentsReport.BasketConstituentsReportBuilder getOrCreateBasketConstituents(int _index) {
		
			if (basketConstituents==null) {
				this.basketConstituents = new ArrayList<>();
			}
			BasketConstituentsReport.BasketConstituentsReportBuilder result;
			return getIndex(basketConstituents, _index, () -> {
						BasketConstituentsReport.BasketConstituentsReportBuilder newBasketConstituents = BasketConstituentsReport.builder();
						return newBasketConstituents;
					});
		}
		
		@Override
		@RosettaAttribute("underlying")
		@RuneAttribute("underlying")
		public UnderlyingIdentifier.UnderlyingIdentifierBuilder getUnderlying() {
			return underlying;
		}
		
		@Override
		public UnderlyingIdentifier.UnderlyingIdentifierBuilder getOrCreateUnderlying() {
			UnderlyingIdentifier.UnderlyingIdentifierBuilder result;
			if (underlying!=null) {
				result = underlying;
			}
			else {
				result = underlying = UnderlyingIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg1")
		@RuneAttribute("quantityUnitOfMeasureLeg1")
		public String getQuantityUnitOfMeasureLeg1() {
			return quantityUnitOfMeasureLeg1;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg2")
		@RuneAttribute("quantityUnitOfMeasureLeg2")
		public String getQuantityUnitOfMeasureLeg2() {
			return quantityUnitOfMeasureLeg2;
		}
		
		@Override
		@RosettaAttribute("maturityDateOfTheUnderlying")
		@RuneAttribute("maturityDateOfTheUnderlying")
		public Date getMaturityDateOfTheUnderlying() {
			return maturityDateOfTheUnderlying;
		}
		
		@Override
		@RosettaAttribute("optionType")
		@RuneAttribute("optionType")
		public OptionTypeCode getOptionType() {
			return optionType;
		}
		
		@Override
		@RosettaAttribute("optionStyle")
		@RuneAttribute("optionStyle")
		public OptionStyleEnum getOptionStyle() {
			return optionStyle;
		}
		
		@Override
		@RosettaAttribute("deliveryType")
		@RuneAttribute("deliveryType")
		public DeliveryTypeEnum getDeliveryType() {
			return deliveryType;
		}
		
		@Override
		@RosettaAttribute("countryOfTheCounterparty2")
		@RuneAttribute("countryOfTheCounterparty2")
		public ISOCountryCodeEnum getCountryOfTheCounterparty2() {
			return countryOfTheCounterparty2;
		}
		
		@Override
		@RosettaAttribute("bookingLocation")
		@RuneAttribute("bookingLocation")
		public ISOCountryCodeEnum getBookingLocation() {
			return bookingLocation;
		}
		
		@Override
		@RosettaAttribute("traderLocation")
		@RuneAttribute("traderLocation")
		public ISOCountryCodeEnum getTraderLocation() {
			return traderLocation;
		}
		
		@Override
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		public TradingCapacity7Code getTradingCapacity() {
			return tradingCapacity;
		}
		
		@Override
		@RosettaAttribute("embeddedOptionType")
		@RuneAttribute("embeddedOptionType")
		public EmbeddedOptionTypeEnum getEmbeddedOptionType() {
			return embeddedOptionType;
		}
		
		@Override
		@RosettaAttribute("seniority")
		@RuneAttribute("seniority")
		public SeniorityEnum getSeniority() {
			return seniority;
		}
		
		@Override
		@RosettaAttribute("series")
		@RuneAttribute("series")
		public Integer getSeries() {
			return series;
		}
		
		@Override
		@RosettaAttribute("indexFactor")
		@RuneAttribute("indexFactor")
		public BigDecimal getIndexFactor() {
			return indexFactor;
		}
		
		@Override
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		@RosettaAttribute("actionType")
		@RuneAttribute("actionType")
		public CommonTransactionReport.CommonTransactionReportBuilder setActionType(ActionTypeEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@Override
		@RosettaAttribute("eventType")
		@RuneAttribute("eventType")
		public CommonTransactionReport.CommonTransactionReportBuilder setEventType(EventTypeEnum _eventType) {
			this.eventType = _eventType == null ? null : _eventType;
			return this;
		}
		
		@Override
		@RosettaAttribute("eventIdentifier")
		@RuneAttribute("eventIdentifier")
		public CommonTransactionReport.CommonTransactionReportBuilder setEventIdentifier(String _eventIdentifier) {
			this.eventIdentifier = _eventIdentifier == null ? null : _eventIdentifier;
			return this;
		}
		
		@Override
		@RosettaAttribute("effectiveDate")
		@RuneAttribute("effectiveDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setEffectiveDate(Date _effectiveDate) {
			this.effectiveDate = _effectiveDate == null ? null : _effectiveDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("earlyTerminationDate")
		@RuneAttribute("earlyTerminationDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setEarlyTerminationDate(Date _earlyTerminationDate) {
			this.earlyTerminationDate = _earlyTerminationDate == null ? null : _earlyTerminationDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("eventTimestamp")
		@RuneAttribute("eventTimestamp")
		public CommonTransactionReport.CommonTransactionReportBuilder setEventTimestamp(ZonedDateTime _eventTimestamp) {
			this.eventTimestamp = _eventTimestamp == null ? null : _eventTimestamp;
			return this;
		}
		
		@Override
		@RosettaAttribute("reportingTimestamp")
		@RuneAttribute("reportingTimestamp")
		public CommonTransactionReport.CommonTransactionReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@Override
		@RosettaAttribute("executionTimestamp")
		@RuneAttribute("executionTimestamp")
		public CommonTransactionReport.CommonTransactionReportBuilder setExecutionTimestamp(ZonedDateTime _executionTimestamp) {
			this.executionTimestamp = _executionTimestamp == null ? null : _executionTimestamp;
			return this;
		}
		
		@Override
		@RosettaAttribute("expirationDate")
		@RuneAttribute("expirationDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setExpirationDate(Date _expirationDate) {
			this.expirationDate = _expirationDate == null ? null : _expirationDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("counterparty1")
		@RuneAttribute("counterparty1")
		public CommonTransactionReport.CommonTransactionReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@Override
		@RosettaAttribute("counterparty2")
		@RuneAttribute("counterparty2")
		public CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@Override
		@RosettaAttribute("counterparty2IdentifierType")
		@RuneAttribute("counterparty2IdentifierType")
		public CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2IdentifierType(Boolean _counterparty2IdentifierType) {
			this.counterparty2IdentifierType = _counterparty2IdentifierType == null ? null : _counterparty2IdentifierType;
			return this;
		}
		
		@Override
		@RosettaAttribute("beneficiary1")
		@RuneAttribute("beneficiary1")
		public CommonTransactionReport.CommonTransactionReportBuilder setBeneficiary1(String _beneficiary1) {
			this.beneficiary1 = _beneficiary1 == null ? null : _beneficiary1;
			return this;
		}
		
		@Override
		@RosettaAttribute("beneficiary1IdentifierTypeIndicator")
		@RuneAttribute("beneficiary1IdentifierTypeIndicator")
		public CommonTransactionReport.CommonTransactionReportBuilder setBeneficiary1IdentifierTypeIndicator(Boolean _beneficiary1IdentifierTypeIndicator) {
			this.beneficiary1IdentifierTypeIndicator = _beneficiary1IdentifierTypeIndicator == null ? null : _beneficiary1IdentifierTypeIndicator;
			return this;
		}
		
		@Override
		@RosettaAttribute("cleared")
		@RuneAttribute("cleared")
		public CommonTransactionReport.CommonTransactionReportBuilder setCleared(ClearedEnum _cleared) {
			this.cleared = _cleared == null ? null : _cleared;
			return this;
		}
		
		@Override
		@RosettaAttribute("centralCounterparty")
		@RuneAttribute("centralCounterparty")
		public CommonTransactionReport.CommonTransactionReportBuilder setCentralCounterparty(String _centralCounterparty) {
			this.centralCounterparty = _centralCounterparty == null ? null : _centralCounterparty;
			return this;
		}
		
		@Override
		@RosettaAttribute("clearingMember")
		@RuneAttribute("clearingMember")
		public CommonTransactionReport.CommonTransactionReportBuilder setClearingMember(String _clearingMember) {
			this.clearingMember = _clearingMember == null ? null : _clearingMember;
			return this;
		}
		
		@Override
		@RosettaAttribute("confirmed")
		@RuneAttribute("confirmed")
		public CommonTransactionReport.CommonTransactionReportBuilder setConfirmed(ConfirmationEnum _confirmed) {
			this.confirmed = _confirmed == null ? null : _confirmed;
			return this;
		}
		
		@Override
		@RosettaAttribute("callAmount")
		@RuneAttribute("callAmount")
		public CommonTransactionReport.CommonTransactionReportBuilder setCallAmount(BigDecimal _callAmount) {
			this.callAmount = _callAmount == null ? null : _callAmount;
			return this;
		}
		
		@Override
		@RosettaAttribute("putAmount")
		@RuneAttribute("putAmount")
		public CommonTransactionReport.CommonTransactionReportBuilder setPutAmount(BigDecimal _putAmount) {
			this.putAmount = _putAmount == null ? null : _putAmount;
			return this;
		}
		
		@Override
		@RosettaAttribute("callCurrency")
		@RuneAttribute("callCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setCallCurrency(ISOCurrencyCodeEnum _callCurrency) {
			this.callCurrency = _callCurrency == null ? null : _callCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("putCurrency")
		@RuneAttribute("putCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setPutCurrency(ISOCurrencyCodeEnum _putCurrency) {
			this.putCurrency = _putCurrency == null ? null : _putCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("priceSchedule")
		@RuneAttribute("priceSchedule")
		public CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(PricePeriod _priceSchedule) {
			if (_priceSchedule != null) {
				this.priceSchedule.add(_priceSchedule.toBuilder());
			}
			return this;
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(PricePeriod _priceSchedule, int _idx) {
			getIndex(this.priceSchedule, _idx, () -> _priceSchedule.toBuilder());
			return this;
		}
		
		@Override 
		public CommonTransactionReport.CommonTransactionReportBuilder addPriceSchedule(List<? extends PricePeriod> priceSchedules) {
			if (priceSchedules != null) {
				for (final PricePeriod toAdd : priceSchedules) {
					this.priceSchedule.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("priceSchedule")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriceSchedule(List<? extends PricePeriod> priceSchedules) {
			if (priceSchedules == null) {
				this.priceSchedule = new ArrayList<>();
			} else {
				this.priceSchedule = priceSchedules.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("strikePriceSchedule")
		@RuneAttribute("strikePriceSchedule")
		public CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(PricePeriod _strikePriceSchedule) {
			if (_strikePriceSchedule != null) {
				this.strikePriceSchedule.add(_strikePriceSchedule.toBuilder());
			}
			return this;
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(PricePeriod _strikePriceSchedule, int _idx) {
			getIndex(this.strikePriceSchedule, _idx, () -> _strikePriceSchedule.toBuilder());
			return this;
		}
		
		@Override 
		public CommonTransactionReport.CommonTransactionReportBuilder addStrikePriceSchedule(List<? extends PricePeriod> strikePriceSchedules) {
			if (strikePriceSchedules != null) {
				for (final PricePeriod toAdd : strikePriceSchedules) {
					this.strikePriceSchedule.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("strikePriceSchedule")
		public CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceSchedule(List<? extends PricePeriod> strikePriceSchedules) {
			if (strikePriceSchedules == null) {
				this.strikePriceSchedule = new ArrayList<>();
			} else {
				this.strikePriceSchedule = strikePriceSchedules.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("price")
		@RuneAttribute("price")
		public CommonTransactionReport.CommonTransactionReportBuilder setPrice(PriceFormat _price) {
			this.price = _price == null ? null : _price.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("priceNotation")
		@RuneAttribute("priceNotation")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriceNotation(PriceNotationEnum _priceNotation) {
			this.priceNotation = _priceNotation == null ? null : _priceNotation;
			return this;
		}
		
		@Override
		@RosettaAttribute("priceCurrency")
		@RuneAttribute("priceCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriceCurrency(ISOCurrencyCodeEnum _priceCurrency) {
			this.priceCurrency = _priceCurrency == null ? null : _priceCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPrice")
		@RuneAttribute("packageTransactionPrice")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPrice(PriceFormat _packageTransactionPrice) {
			this.packageTransactionPrice = _packageTransactionPrice == null ? null : _packageTransactionPrice.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPriceNotation")
		@RuneAttribute("packageTransactionPriceNotation")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPriceNotation(PriceNotationEnum _packageTransactionPriceNotation) {
			this.packageTransactionPriceNotation = _packageTransactionPriceNotation == null ? null : _packageTransactionPriceNotation;
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionPriceCurrency")
		@RuneAttribute("packageTransactionPriceCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionPriceCurrency(ISOCurrencyCodeEnum _packageTransactionPriceCurrency) {
			this.packageTransactionPriceCurrency = _packageTransactionPriceCurrency == null ? null : _packageTransactionPriceCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpread")
		@RuneAttribute("packageTransactionSpread")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpread(PriceFormat _packageTransactionSpread) {
			this.packageTransactionSpread = _packageTransactionSpread == null ? null : _packageTransactionSpread.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpreadNotation")
		@RuneAttribute("packageTransactionSpreadNotation")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpreadNotation(PriceNotationEnum _packageTransactionSpreadNotation) {
			this.packageTransactionSpreadNotation = _packageTransactionSpreadNotation == null ? null : _packageTransactionSpreadNotation;
			return this;
		}
		
		@Override
		@RosettaAttribute("packageTransactionSpreadCurrency")
		@RuneAttribute("packageTransactionSpreadCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageTransactionSpreadCurrency(ISOCurrencyCodeEnum _packageTransactionSpreadCurrency) {
			this.packageTransactionSpreadCurrency = _packageTransactionSpreadCurrency == null ? null : _packageTransactionSpreadCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("strikePrice")
		@RuneAttribute("strikePrice")
		public CommonTransactionReport.CommonTransactionReportBuilder setStrikePrice(PriceFormat _strikePrice) {
			this.strikePrice = _strikePrice == null ? null : _strikePrice.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("strikePriceNotation")
		@RuneAttribute("strikePriceNotation")
		public CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceNotation(PriceNotationEnum _strikePriceNotation) {
			this.strikePriceNotation = _strikePriceNotation == null ? null : _strikePriceNotation;
			return this;
		}
		
		@Override
		@RosettaAttribute("strikePriceCurrency")
		@RuneAttribute("strikePriceCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setStrikePriceCurrency(String _strikePriceCurrency) {
			this.strikePriceCurrency = _strikePriceCurrency == null ? null : _strikePriceCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("priceUnitOfMeasure")
		@RuneAttribute("priceUnitOfMeasure")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriceUnitOfMeasure(String _priceUnitOfMeasure) {
			this.priceUnitOfMeasure = _priceUnitOfMeasure == null ? null : _priceUnitOfMeasure;
			return this;
		}
		
		@Override
		@RosettaAttribute("optionPremiumAmount")
		@RuneAttribute("optionPremiumAmount")
		public CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumAmount(BigDecimal _optionPremiumAmount) {
			this.optionPremiumAmount = _optionPremiumAmount == null ? null : _optionPremiumAmount;
			return this;
		}
		
		@Override
		@RosettaAttribute("optionPremiumCurrency")
		@RuneAttribute("optionPremiumCurrency")
		public CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumCurrency(ISOCurrencyCodeEnum _optionPremiumCurrency) {
			this.optionPremiumCurrency = _optionPremiumCurrency == null ? null : _optionPremiumCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("optionPremiumPaymentDate")
		@RuneAttribute("optionPremiumPaymentDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setOptionPremiumPaymentDate(Date _optionPremiumPaymentDate) {
			this.optionPremiumPaymentDate = _optionPremiumPaymentDate == null ? null : _optionPremiumPaymentDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("cdSIndexAttachmentPoint")
		@RuneAttribute("cdSIndexAttachmentPoint")
		public CommonTransactionReport.CommonTransactionReportBuilder setCdSIndexAttachmentPoint(BigDecimal _cdSIndexAttachmentPoint) {
			this.cdSIndexAttachmentPoint = _cdSIndexAttachmentPoint == null ? null : _cdSIndexAttachmentPoint;
			return this;
		}
		
		@Override
		@RosettaAttribute("cdSIndexDetachmentPoint")
		@RuneAttribute("cdSIndexDetachmentPoint")
		public CommonTransactionReport.CommonTransactionReportBuilder setCdSIndexDetachmentPoint(BigDecimal _cdSIndexDetachmentPoint) {
			this.cdSIndexDetachmentPoint = _cdSIndexDetachmentPoint == null ? null : _cdSIndexDetachmentPoint;
			return this;
		}
		
		@Override
		@RosettaAttribute("collateralPortfolioIndicator")
		@RuneAttribute("collateralPortfolioIndicator")
		public CommonTransactionReport.CommonTransactionReportBuilder setCollateralPortfolioIndicator(Boolean _collateralPortfolioIndicator) {
			this.collateralPortfolioIndicator = _collateralPortfolioIndicator == null ? null : _collateralPortfolioIndicator;
			return this;
		}
		
		@Override
		@RosettaAttribute("firstExerciseDate")
		@RuneAttribute("firstExerciseDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setFirstExerciseDate(Date _firstExerciseDate) {
			this.firstExerciseDate = _firstExerciseDate == null ? null : _firstExerciseDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public CommonTransactionReport.CommonTransactionReportBuilder setFinalContractualSettlementDate(Date _finalContractualSettlementDate) {
			this.finalContractualSettlementDate = _finalContractualSettlementDate == null ? null : _finalContractualSettlementDate;
			return this;
		}
		
		@Override
		@RosettaAttribute("otherPayment")
		@RuneAttribute("otherPayment")
		public CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(OtherPayment _otherPayment) {
			if (_otherPayment != null) {
				this.otherPayment.add(_otherPayment.toBuilder());
			}
			return this;
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(OtherPayment _otherPayment, int _idx) {
			getIndex(this.otherPayment, _idx, () -> _otherPayment.toBuilder());
			return this;
		}
		
		@Override 
		public CommonTransactionReport.CommonTransactionReportBuilder addOtherPayment(List<? extends OtherPayment> otherPayments) {
			if (otherPayments != null) {
				for (final OtherPayment toAdd : otherPayments) {
					this.otherPayment.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("otherPayment")
		public CommonTransactionReport.CommonTransactionReportBuilder setOtherPayment(List<? extends OtherPayment> otherPayments) {
			if (otherPayments == null) {
				this.otherPayment = new ArrayList<>();
			} else {
				this.otherPayment = otherPayments.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("leg1")
		@RuneAttribute("leg1")
		public CommonTransactionReport.CommonTransactionReportBuilder setLeg1(CommonLeg _leg1) {
			this.leg1 = _leg1 == null ? null : _leg1.toBuilder();
			return this;
		}
		
		@Override
		@RosettaIgnore
		public CommonTransactionReport.CommonTransactionReportBuilder setLeg1(Leg _leg1) {
			final CommonLeg ifThenElseResult;
			if (_leg1 == null) {
				ifThenElseResult = null;
			} else {
				ifThenElseResult = _leg1 instanceof CommonLeg ? CommonLeg.class.cast(_leg1) : null;
			}
			return setLeg1(ifThenElseResult);
		}
		
		@Override
		@RosettaAttribute("leg2")
		@RuneAttribute("leg2")
		public CommonTransactionReport.CommonTransactionReportBuilder setLeg2(CommonLeg _leg2) {
			this.leg2 = _leg2 == null ? null : _leg2.toBuilder();
			return this;
		}
		
		@Override
		@RosettaIgnore
		public CommonTransactionReport.CommonTransactionReportBuilder setLeg2(Leg _leg2) {
			final CommonLeg ifThenElseResult;
			if (_leg2 == null) {
				ifThenElseResult = null;
			} else {
				ifThenElseResult = _leg2 instanceof CommonLeg ? CommonLeg.class.cast(_leg2) : null;
			}
			return setLeg2(ifThenElseResult);
		}
		
		@Override
		@RosettaAttribute("counterparty2Name")
		@RuneAttribute("counterparty2Name")
		public CommonTransactionReport.CommonTransactionReportBuilder setCounterparty2Name(String _counterparty2Name) {
			this.counterparty2Name = _counterparty2Name == null ? null : _counterparty2Name;
			return this;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty1")
		@RuneAttribute("executionAgentCounterparty1")
		public CommonTransactionReport.CommonTransactionReportBuilder setExecutionAgentCounterparty1(String _executionAgentCounterparty1) {
			this.executionAgentCounterparty1 = _executionAgentCounterparty1 == null ? null : _executionAgentCounterparty1;
			return this;
		}
		
		@Override
		@RosettaAttribute("executionAgentCounterparty2DTCC")
		@RuneAttribute("executionAgentCounterparty2DTCC")
		public CommonTransactionReport.CommonTransactionReportBuilder setExecutionAgentCounterparty2DTCC(String _executionAgentCounterparty2DTCC) {
			this.executionAgentCounterparty2DTCC = _executionAgentCounterparty2DTCC == null ? null : _executionAgentCounterparty2DTCC;
			return this;
		}
		
		@Override
		@RosettaAttribute("reportSubmittingEntityID")
		@RuneAttribute("reportSubmittingEntityID")
		public CommonTransactionReport.CommonTransactionReportBuilder setReportSubmittingEntityID(String _reportSubmittingEntityID) {
			this.reportSubmittingEntityID = _reportSubmittingEntityID == null ? null : _reportSubmittingEntityID;
			return this;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public CommonTransactionReport.CommonTransactionReportBuilder setUniqueTransactionIdentifier(String _uniqueTransactionIdentifier) {
			this.uniqueTransactionIdentifier = _uniqueTransactionIdentifier == null ? null : _uniqueTransactionIdentifier;
			return this;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifierProprietary")
		@RuneAttribute("uniqueTransactionIdentifierProprietary")
		public CommonTransactionReport.CommonTransactionReportBuilder setUniqueTransactionIdentifierProprietary(String _uniqueTransactionIdentifierProprietary) {
			this.uniqueTransactionIdentifierProprietary = _uniqueTransactionIdentifierProprietary == null ? null : _uniqueTransactionIdentifierProprietary;
			return this;
		}
		
		@Override
		@RosettaAttribute("priorUTI")
		@RuneAttribute("priorUTI")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriorUTI(String _priorUTI) {
			this.priorUTI = _priorUTI == null ? null : _priorUTI;
			return this;
		}
		
		@Override
		@RosettaAttribute("priorUTIProprietary")
		@RuneAttribute("priorUTIProprietary")
		public CommonTransactionReport.CommonTransactionReportBuilder setPriorUTIProprietary(String _priorUTIProprietary) {
			this.priorUTIProprietary = _priorUTIProprietary == null ? null : _priorUTIProprietary;
			return this;
		}
		
		@Override
		@RosettaAttribute("uniqueProductIdentifier")
		@RuneAttribute("uniqueProductIdentifier")
		public CommonTransactionReport.CommonTransactionReportBuilder setUniqueProductIdentifier(String _uniqueProductIdentifier) {
			this.uniqueProductIdentifier = _uniqueProductIdentifier == null ? null : _uniqueProductIdentifier;
			return this;
		}
		
		@Override
		@RosettaAttribute("contractType")
		@RuneAttribute("contractType")
		public CommonTransactionReport.CommonTransactionReportBuilder setContractType(CommonContractType _contractType) {
			this.contractType = _contractType == null ? null : _contractType;
			return this;
		}
		
		@Override
		@RosettaAttribute("assetClass")
		@RuneAttribute("assetClass")
		public CommonTransactionReport.CommonTransactionReportBuilder setAssetClass(CommonAssetClass _assetClass) {
			this.assetClass = _assetClass == null ? null : _assetClass;
			return this;
		}
		
		@Override
		@RosettaAttribute("clearingTimestamp")
		@RuneAttribute("clearingTimestamp")
		public CommonTransactionReport.CommonTransactionReportBuilder setClearingTimestamp(ZonedDateTime _clearingTimestamp) {
			this.clearingTimestamp = _clearingTimestamp == null ? null : _clearingTimestamp;
			return this;
		}
		
		@Override
		@RosettaAttribute("platformIdentifier")
		@RuneAttribute("platformIdentifier")
		public CommonTransactionReport.CommonTransactionReportBuilder setPlatformIdentifier(String _platformIdentifier) {
			this.platformIdentifier = _platformIdentifier == null ? null : _platformIdentifier;
			return this;
		}
		
		@Override
		@RosettaAttribute("direction1")
		@RuneAttribute("direction1")
		public CommonTransactionReport.CommonTransactionReportBuilder setDirection1(Direction1Enum _direction1) {
			this.direction1 = _direction1 == null ? null : _direction1;
			return this;
		}
		
		@Override
		@RosettaAttribute("direction2Leg1")
		@RuneAttribute("direction2Leg1")
		public CommonTransactionReport.CommonTransactionReportBuilder setDirection2Leg1(Direction2Enum _direction2Leg1) {
			this.direction2Leg1 = _direction2Leg1 == null ? null : _direction2Leg1;
			return this;
		}
		
		@Override
		@RosettaAttribute("direction2Leg2")
		@RuneAttribute("direction2Leg2")
		public CommonTransactionReport.CommonTransactionReportBuilder setDirection2Leg2(Direction2Enum _direction2Leg2) {
			this.direction2Leg2 = _direction2Leg2 == null ? null : _direction2Leg2;
			return this;
		}
		
		@Override
		@RosettaAttribute("packageIdentifier")
		@RuneAttribute("packageIdentifier")
		public CommonTransactionReport.CommonTransactionReportBuilder setPackageIdentifier(String _packageIdentifier) {
			this.packageIdentifier = _packageIdentifier == null ? null : _packageIdentifier;
			return this;
		}
		
		@Override
		@RosettaAttribute("barrier")
		@RuneAttribute("barrier")
		public CommonTransactionReport.CommonTransactionReportBuilder setBarrier(SingleOrUpperAndLowerBarrier _barrier) {
			this.barrier = _barrier == null ? null : _barrier.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("exchangeRate")
		@RuneAttribute("exchangeRate")
		public CommonTransactionReport.CommonTransactionReportBuilder setExchangeRate(BigDecimal _exchangeRate) {
			this.exchangeRate = _exchangeRate == null ? null : _exchangeRate;
			return this;
		}
		
		@Override
		@RosettaAttribute("exchangeRateBasis")
		@RuneAttribute("exchangeRateBasis")
		public CommonTransactionReport.CommonTransactionReportBuilder setExchangeRateBasis(String _exchangeRateBasis) {
			this.exchangeRateBasis = _exchangeRateBasis == null ? null : _exchangeRateBasis;
			return this;
		}
		
		@Override
		@RosettaAttribute("nameOfTheUnderlyingIndex")
		@RuneAttribute("nameOfTheUnderlyingIndex")
		public CommonTransactionReport.CommonTransactionReportBuilder setNameOfTheUnderlyingIndex(String _nameOfTheUnderlyingIndex) {
			this.nameOfTheUnderlyingIndex = _nameOfTheUnderlyingIndex == null ? null : _nameOfTheUnderlyingIndex;
			return this;
		}
		
		@Override
		@RosettaAttribute("basketConstituents")
		@RuneAttribute("basketConstituents")
		public CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(BasketConstituentsReport _basketConstituents) {
			if (_basketConstituents != null) {
				this.basketConstituents.add(_basketConstituents.toBuilder());
			}
			return this;
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(BasketConstituentsReport _basketConstituents, int _idx) {
			getIndex(this.basketConstituents, _idx, () -> _basketConstituents.toBuilder());
			return this;
		}
		
		@Override 
		public CommonTransactionReport.CommonTransactionReportBuilder addBasketConstituents(List<? extends BasketConstituentsReport> basketConstituentss) {
			if (basketConstituentss != null) {
				for (final BasketConstituentsReport toAdd : basketConstituentss) {
					this.basketConstituents.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("basketConstituents")
		public CommonTransactionReport.CommonTransactionReportBuilder setBasketConstituents(List<? extends BasketConstituentsReport> basketConstituentss) {
			if (basketConstituentss == null) {
				this.basketConstituents = new ArrayList<>();
			} else {
				this.basketConstituents = basketConstituentss.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("underlying")
		@RuneAttribute("underlying")
		public CommonTransactionReport.CommonTransactionReportBuilder setUnderlying(UnderlyingIdentifier _underlying) {
			this.underlying = _underlying == null ? null : _underlying.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg1")
		@RuneAttribute("quantityUnitOfMeasureLeg1")
		public CommonTransactionReport.CommonTransactionReportBuilder setQuantityUnitOfMeasureLeg1(String _quantityUnitOfMeasureLeg1) {
			this.quantityUnitOfMeasureLeg1 = _quantityUnitOfMeasureLeg1 == null ? null : _quantityUnitOfMeasureLeg1;
			return this;
		}
		
		@Override
		@RosettaAttribute("quantityUnitOfMeasureLeg2")
		@RuneAttribute("quantityUnitOfMeasureLeg2")
		public CommonTransactionReport.CommonTransactionReportBuilder setQuantityUnitOfMeasureLeg2(String _quantityUnitOfMeasureLeg2) {
			this.quantityUnitOfMeasureLeg2 = _quantityUnitOfMeasureLeg2 == null ? null : _quantityUnitOfMeasureLeg2;
			return this;
		}
		
		@Override
		@RosettaAttribute("maturityDateOfTheUnderlying")
		@RuneAttribute("maturityDateOfTheUnderlying")
		public CommonTransactionReport.CommonTransactionReportBuilder setMaturityDateOfTheUnderlying(Date _maturityDateOfTheUnderlying) {
			this.maturityDateOfTheUnderlying = _maturityDateOfTheUnderlying == null ? null : _maturityDateOfTheUnderlying;
			return this;
		}
		
		@Override
		@RosettaAttribute("optionType")
		@RuneAttribute("optionType")
		public CommonTransactionReport.CommonTransactionReportBuilder setOptionType(OptionTypeCode _optionType) {
			this.optionType = _optionType == null ? null : _optionType;
			return this;
		}
		
		@Override
		@RosettaAttribute("optionStyle")
		@RuneAttribute("optionStyle")
		public CommonTransactionReport.CommonTransactionReportBuilder setOptionStyle(OptionStyleEnum _optionStyle) {
			this.optionStyle = _optionStyle == null ? null : _optionStyle;
			return this;
		}
		
		@Override
		@RosettaAttribute("deliveryType")
		@RuneAttribute("deliveryType")
		public CommonTransactionReport.CommonTransactionReportBuilder setDeliveryType(DeliveryTypeEnum _deliveryType) {
			this.deliveryType = _deliveryType == null ? null : _deliveryType;
			return this;
		}
		
		@Override
		@RosettaAttribute("countryOfTheCounterparty2")
		@RuneAttribute("countryOfTheCounterparty2")
		public CommonTransactionReport.CommonTransactionReportBuilder setCountryOfTheCounterparty2(ISOCountryCodeEnum _countryOfTheCounterparty2) {
			this.countryOfTheCounterparty2 = _countryOfTheCounterparty2 == null ? null : _countryOfTheCounterparty2;
			return this;
		}
		
		@Override
		@RosettaAttribute("bookingLocation")
		@RuneAttribute("bookingLocation")
		public CommonTransactionReport.CommonTransactionReportBuilder setBookingLocation(ISOCountryCodeEnum _bookingLocation) {
			this.bookingLocation = _bookingLocation == null ? null : _bookingLocation;
			return this;
		}
		
		@Override
		@RosettaAttribute("traderLocation")
		@RuneAttribute("traderLocation")
		public CommonTransactionReport.CommonTransactionReportBuilder setTraderLocation(ISOCountryCodeEnum _traderLocation) {
			this.traderLocation = _traderLocation == null ? null : _traderLocation;
			return this;
		}
		
		@Override
		@RosettaAttribute("tradingCapacity")
		@RuneAttribute("tradingCapacity")
		public CommonTransactionReport.CommonTransactionReportBuilder setTradingCapacity(TradingCapacity7Code _tradingCapacity) {
			this.tradingCapacity = _tradingCapacity == null ? null : _tradingCapacity;
			return this;
		}
		
		@Override
		@RosettaAttribute("embeddedOptionType")
		@RuneAttribute("embeddedOptionType")
		public CommonTransactionReport.CommonTransactionReportBuilder setEmbeddedOptionType(EmbeddedOptionTypeEnum _embeddedOptionType) {
			this.embeddedOptionType = _embeddedOptionType == null ? null : _embeddedOptionType;
			return this;
		}
		
		@Override
		@RosettaAttribute("seniority")
		@RuneAttribute("seniority")
		public CommonTransactionReport.CommonTransactionReportBuilder setSeniority(SeniorityEnum _seniority) {
			this.seniority = _seniority == null ? null : _seniority;
			return this;
		}
		
		@Override
		@RosettaAttribute("series")
		@RuneAttribute("series")
		public CommonTransactionReport.CommonTransactionReportBuilder setSeries(Integer _series) {
			this.series = _series == null ? null : _series;
			return this;
		}
		
		@Override
		@RosettaAttribute("indexFactor")
		@RuneAttribute("indexFactor")
		public CommonTransactionReport.CommonTransactionReportBuilder setIndexFactor(BigDecimal _indexFactor) {
			this.indexFactor = _indexFactor == null ? null : _indexFactor;
			return this;
		}
		
		@Override
		@RosettaAttribute("technicalRecordId")
		@RuneAttribute("technicalRecordId")
		public CommonTransactionReport.CommonTransactionReportBuilder setTechnicalRecordId(String _technicalRecordId) {
			this.technicalRecordId = _technicalRecordId == null ? null : _technicalRecordId;
			return this;
		}
		
		@Override
		public CommonTransactionReport build() {
			return new CommonTransactionReport.CommonTransactionReportImpl(this);
		}
		
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder prune() {
			priceSchedule = priceSchedule.stream().filter(b->b!=null).<PricePeriod.PricePeriodBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			strikePriceSchedule = strikePriceSchedule.stream().filter(b->b!=null).<PricePeriod.PricePeriodBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (price!=null && !price.prune().hasData()) price = null;
			if (packageTransactionPrice!=null && !packageTransactionPrice.prune().hasData()) packageTransactionPrice = null;
			if (packageTransactionSpread!=null && !packageTransactionSpread.prune().hasData()) packageTransactionSpread = null;
			if (strikePrice!=null && !strikePrice.prune().hasData()) strikePrice = null;
			otherPayment = otherPayment.stream().filter(b->b!=null).<OtherPayment.OtherPaymentBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (leg1!=null && !leg1.prune().hasData()) leg1 = null;
			if (leg2!=null && !leg2.prune().hasData()) leg2 = null;
			if (barrier!=null && !barrier.prune().hasData()) barrier = null;
			basketConstituents = basketConstituents.stream().filter(b->b!=null).<BasketConstituentsReport.BasketConstituentsReportBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (underlying!=null && !underlying.prune().hasData()) underlying = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getActionType()!=null) return true;
			if (getEventType()!=null) return true;
			if (getEventIdentifier()!=null) return true;
			if (getEffectiveDate()!=null) return true;
			if (getEarlyTerminationDate()!=null) return true;
			if (getEventTimestamp()!=null) return true;
			if (getReportingTimestamp()!=null) return true;
			if (getExecutionTimestamp()!=null) return true;
			if (getExpirationDate()!=null) return true;
			if (getCounterparty1()!=null) return true;
			if (getCounterparty2()!=null) return true;
			if (getCounterparty2IdentifierType()!=null) return true;
			if (getBeneficiary1()!=null) return true;
			if (getBeneficiary1IdentifierTypeIndicator()!=null) return true;
			if (getCleared()!=null) return true;
			if (getCentralCounterparty()!=null) return true;
			if (getClearingMember()!=null) return true;
			if (getConfirmed()!=null) return true;
			if (getCallAmount()!=null) return true;
			if (getPutAmount()!=null) return true;
			if (getCallCurrency()!=null) return true;
			if (getPutCurrency()!=null) return true;
			if (getPriceSchedule()!=null && getPriceSchedule().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getStrikePriceSchedule()!=null && getStrikePriceSchedule().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getPrice()!=null && getPrice().hasData()) return true;
			if (getPriceNotation()!=null) return true;
			if (getPriceCurrency()!=null) return true;
			if (getPackageTransactionPrice()!=null && getPackageTransactionPrice().hasData()) return true;
			if (getPackageTransactionPriceNotation()!=null) return true;
			if (getPackageTransactionPriceCurrency()!=null) return true;
			if (getPackageTransactionSpread()!=null && getPackageTransactionSpread().hasData()) return true;
			if (getPackageTransactionSpreadNotation()!=null) return true;
			if (getPackageTransactionSpreadCurrency()!=null) return true;
			if (getStrikePrice()!=null && getStrikePrice().hasData()) return true;
			if (getStrikePriceNotation()!=null) return true;
			if (getStrikePriceCurrency()!=null) return true;
			if (getPriceUnitOfMeasure()!=null) return true;
			if (getOptionPremiumAmount()!=null) return true;
			if (getOptionPremiumCurrency()!=null) return true;
			if (getOptionPremiumPaymentDate()!=null) return true;
			if (getCdSIndexAttachmentPoint()!=null) return true;
			if (getCdSIndexDetachmentPoint()!=null) return true;
			if (getCollateralPortfolioIndicator()!=null) return true;
			if (getFirstExerciseDate()!=null) return true;
			if (getFinalContractualSettlementDate()!=null) return true;
			if (getOtherPayment()!=null && getOtherPayment().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getLeg1()!=null && getLeg1().hasData()) return true;
			if (getLeg2()!=null && getLeg2().hasData()) return true;
			if (getCounterparty2Name()!=null) return true;
			if (getExecutionAgentCounterparty1()!=null) return true;
			if (getExecutionAgentCounterparty2DTCC()!=null) return true;
			if (getReportSubmittingEntityID()!=null) return true;
			if (getUniqueTransactionIdentifier()!=null) return true;
			if (getUniqueTransactionIdentifierProprietary()!=null) return true;
			if (getPriorUTI()!=null) return true;
			if (getPriorUTIProprietary()!=null) return true;
			if (getUniqueProductIdentifier()!=null) return true;
			if (getContractType()!=null) return true;
			if (getAssetClass()!=null) return true;
			if (getClearingTimestamp()!=null) return true;
			if (getPlatformIdentifier()!=null) return true;
			if (getDirection1()!=null) return true;
			if (getDirection2Leg1()!=null) return true;
			if (getDirection2Leg2()!=null) return true;
			if (getPackageIdentifier()!=null) return true;
			if (getBarrier()!=null && getBarrier().hasData()) return true;
			if (getExchangeRate()!=null) return true;
			if (getExchangeRateBasis()!=null) return true;
			if (getNameOfTheUnderlyingIndex()!=null) return true;
			if (getBasketConstituents()!=null && getBasketConstituents().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getUnderlying()!=null && getUnderlying().hasData()) return true;
			if (getQuantityUnitOfMeasureLeg1()!=null) return true;
			if (getQuantityUnitOfMeasureLeg2()!=null) return true;
			if (getMaturityDateOfTheUnderlying()!=null) return true;
			if (getOptionType()!=null) return true;
			if (getOptionStyle()!=null) return true;
			if (getDeliveryType()!=null) return true;
			if (getCountryOfTheCounterparty2()!=null) return true;
			if (getBookingLocation()!=null) return true;
			if (getTraderLocation()!=null) return true;
			if (getTradingCapacity()!=null) return true;
			if (getEmbeddedOptionType()!=null) return true;
			if (getSeniority()!=null) return true;
			if (getSeries()!=null) return true;
			if (getIndexFactor()!=null) return true;
			if (getTechnicalRecordId()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CommonTransactionReport.CommonTransactionReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			CommonTransactionReport.CommonTransactionReportBuilder o = (CommonTransactionReport.CommonTransactionReportBuilder) other;
			
			merger.mergeRosetta(getPriceSchedule(), o.getPriceSchedule(), this::getOrCreatePriceSchedule);
			merger.mergeRosetta(getStrikePriceSchedule(), o.getStrikePriceSchedule(), this::getOrCreateStrikePriceSchedule);
			merger.mergeRosetta(getPrice(), o.getPrice(), this::setPrice);
			merger.mergeRosetta(getPackageTransactionPrice(), o.getPackageTransactionPrice(), this::setPackageTransactionPrice);
			merger.mergeRosetta(getPackageTransactionSpread(), o.getPackageTransactionSpread(), this::setPackageTransactionSpread);
			merger.mergeRosetta(getStrikePrice(), o.getStrikePrice(), this::setStrikePrice);
			merger.mergeRosetta(getOtherPayment(), o.getOtherPayment(), this::getOrCreateOtherPayment);
			merger.mergeRosetta(getLeg1(), o.getLeg1(), this::setLeg1);
			merger.mergeRosetta(getLeg2(), o.getLeg2(), this::setLeg2);
			merger.mergeRosetta(getBarrier(), o.getBarrier(), this::setBarrier);
			merger.mergeRosetta(getBasketConstituents(), o.getBasketConstituents(), this::getOrCreateBasketConstituents);
			merger.mergeRosetta(getUnderlying(), o.getUnderlying(), this::setUnderlying);
			
			merger.mergeBasic(getActionType(), o.getActionType(), this::setActionType);
			merger.mergeBasic(getEventType(), o.getEventType(), this::setEventType);
			merger.mergeBasic(getEventIdentifier(), o.getEventIdentifier(), this::setEventIdentifier);
			merger.mergeBasic(getEffectiveDate(), o.getEffectiveDate(), this::setEffectiveDate);
			merger.mergeBasic(getEarlyTerminationDate(), o.getEarlyTerminationDate(), this::setEarlyTerminationDate);
			merger.mergeBasic(getEventTimestamp(), o.getEventTimestamp(), this::setEventTimestamp);
			merger.mergeBasic(getReportingTimestamp(), o.getReportingTimestamp(), this::setReportingTimestamp);
			merger.mergeBasic(getExecutionTimestamp(), o.getExecutionTimestamp(), this::setExecutionTimestamp);
			merger.mergeBasic(getExpirationDate(), o.getExpirationDate(), this::setExpirationDate);
			merger.mergeBasic(getCounterparty1(), o.getCounterparty1(), this::setCounterparty1);
			merger.mergeBasic(getCounterparty2(), o.getCounterparty2(), this::setCounterparty2);
			merger.mergeBasic(getCounterparty2IdentifierType(), o.getCounterparty2IdentifierType(), this::setCounterparty2IdentifierType);
			merger.mergeBasic(getBeneficiary1(), o.getBeneficiary1(), this::setBeneficiary1);
			merger.mergeBasic(getBeneficiary1IdentifierTypeIndicator(), o.getBeneficiary1IdentifierTypeIndicator(), this::setBeneficiary1IdentifierTypeIndicator);
			merger.mergeBasic(getCleared(), o.getCleared(), this::setCleared);
			merger.mergeBasic(getCentralCounterparty(), o.getCentralCounterparty(), this::setCentralCounterparty);
			merger.mergeBasic(getClearingMember(), o.getClearingMember(), this::setClearingMember);
			merger.mergeBasic(getConfirmed(), o.getConfirmed(), this::setConfirmed);
			merger.mergeBasic(getCallAmount(), o.getCallAmount(), this::setCallAmount);
			merger.mergeBasic(getPutAmount(), o.getPutAmount(), this::setPutAmount);
			merger.mergeBasic(getCallCurrency(), o.getCallCurrency(), this::setCallCurrency);
			merger.mergeBasic(getPutCurrency(), o.getPutCurrency(), this::setPutCurrency);
			merger.mergeBasic(getPriceNotation(), o.getPriceNotation(), this::setPriceNotation);
			merger.mergeBasic(getPriceCurrency(), o.getPriceCurrency(), this::setPriceCurrency);
			merger.mergeBasic(getPackageTransactionPriceNotation(), o.getPackageTransactionPriceNotation(), this::setPackageTransactionPriceNotation);
			merger.mergeBasic(getPackageTransactionPriceCurrency(), o.getPackageTransactionPriceCurrency(), this::setPackageTransactionPriceCurrency);
			merger.mergeBasic(getPackageTransactionSpreadNotation(), o.getPackageTransactionSpreadNotation(), this::setPackageTransactionSpreadNotation);
			merger.mergeBasic(getPackageTransactionSpreadCurrency(), o.getPackageTransactionSpreadCurrency(), this::setPackageTransactionSpreadCurrency);
			merger.mergeBasic(getStrikePriceNotation(), o.getStrikePriceNotation(), this::setStrikePriceNotation);
			merger.mergeBasic(getStrikePriceCurrency(), o.getStrikePriceCurrency(), this::setStrikePriceCurrency);
			merger.mergeBasic(getPriceUnitOfMeasure(), o.getPriceUnitOfMeasure(), this::setPriceUnitOfMeasure);
			merger.mergeBasic(getOptionPremiumAmount(), o.getOptionPremiumAmount(), this::setOptionPremiumAmount);
			merger.mergeBasic(getOptionPremiumCurrency(), o.getOptionPremiumCurrency(), this::setOptionPremiumCurrency);
			merger.mergeBasic(getOptionPremiumPaymentDate(), o.getOptionPremiumPaymentDate(), this::setOptionPremiumPaymentDate);
			merger.mergeBasic(getCdSIndexAttachmentPoint(), o.getCdSIndexAttachmentPoint(), this::setCdSIndexAttachmentPoint);
			merger.mergeBasic(getCdSIndexDetachmentPoint(), o.getCdSIndexDetachmentPoint(), this::setCdSIndexDetachmentPoint);
			merger.mergeBasic(getCollateralPortfolioIndicator(), o.getCollateralPortfolioIndicator(), this::setCollateralPortfolioIndicator);
			merger.mergeBasic(getFirstExerciseDate(), o.getFirstExerciseDate(), this::setFirstExerciseDate);
			merger.mergeBasic(getFinalContractualSettlementDate(), o.getFinalContractualSettlementDate(), this::setFinalContractualSettlementDate);
			merger.mergeBasic(getCounterparty2Name(), o.getCounterparty2Name(), this::setCounterparty2Name);
			merger.mergeBasic(getExecutionAgentCounterparty1(), o.getExecutionAgentCounterparty1(), this::setExecutionAgentCounterparty1);
			merger.mergeBasic(getExecutionAgentCounterparty2DTCC(), o.getExecutionAgentCounterparty2DTCC(), this::setExecutionAgentCounterparty2DTCC);
			merger.mergeBasic(getReportSubmittingEntityID(), o.getReportSubmittingEntityID(), this::setReportSubmittingEntityID);
			merger.mergeBasic(getUniqueTransactionIdentifier(), o.getUniqueTransactionIdentifier(), this::setUniqueTransactionIdentifier);
			merger.mergeBasic(getUniqueTransactionIdentifierProprietary(), o.getUniqueTransactionIdentifierProprietary(), this::setUniqueTransactionIdentifierProprietary);
			merger.mergeBasic(getPriorUTI(), o.getPriorUTI(), this::setPriorUTI);
			merger.mergeBasic(getPriorUTIProprietary(), o.getPriorUTIProprietary(), this::setPriorUTIProprietary);
			merger.mergeBasic(getUniqueProductIdentifier(), o.getUniqueProductIdentifier(), this::setUniqueProductIdentifier);
			merger.mergeBasic(getContractType(), o.getContractType(), this::setContractType);
			merger.mergeBasic(getAssetClass(), o.getAssetClass(), this::setAssetClass);
			merger.mergeBasic(getClearingTimestamp(), o.getClearingTimestamp(), this::setClearingTimestamp);
			merger.mergeBasic(getPlatformIdentifier(), o.getPlatformIdentifier(), this::setPlatformIdentifier);
			merger.mergeBasic(getDirection1(), o.getDirection1(), this::setDirection1);
			merger.mergeBasic(getDirection2Leg1(), o.getDirection2Leg1(), this::setDirection2Leg1);
			merger.mergeBasic(getDirection2Leg2(), o.getDirection2Leg2(), this::setDirection2Leg2);
			merger.mergeBasic(getPackageIdentifier(), o.getPackageIdentifier(), this::setPackageIdentifier);
			merger.mergeBasic(getExchangeRate(), o.getExchangeRate(), this::setExchangeRate);
			merger.mergeBasic(getExchangeRateBasis(), o.getExchangeRateBasis(), this::setExchangeRateBasis);
			merger.mergeBasic(getNameOfTheUnderlyingIndex(), o.getNameOfTheUnderlyingIndex(), this::setNameOfTheUnderlyingIndex);
			merger.mergeBasic(getQuantityUnitOfMeasureLeg1(), o.getQuantityUnitOfMeasureLeg1(), this::setQuantityUnitOfMeasureLeg1);
			merger.mergeBasic(getQuantityUnitOfMeasureLeg2(), o.getQuantityUnitOfMeasureLeg2(), this::setQuantityUnitOfMeasureLeg2);
			merger.mergeBasic(getMaturityDateOfTheUnderlying(), o.getMaturityDateOfTheUnderlying(), this::setMaturityDateOfTheUnderlying);
			merger.mergeBasic(getOptionType(), o.getOptionType(), this::setOptionType);
			merger.mergeBasic(getOptionStyle(), o.getOptionStyle(), this::setOptionStyle);
			merger.mergeBasic(getDeliveryType(), o.getDeliveryType(), this::setDeliveryType);
			merger.mergeBasic(getCountryOfTheCounterparty2(), o.getCountryOfTheCounterparty2(), this::setCountryOfTheCounterparty2);
			merger.mergeBasic(getBookingLocation(), o.getBookingLocation(), this::setBookingLocation);
			merger.mergeBasic(getTraderLocation(), o.getTraderLocation(), this::setTraderLocation);
			merger.mergeBasic(getTradingCapacity(), o.getTradingCapacity(), this::setTradingCapacity);
			merger.mergeBasic(getEmbeddedOptionType(), o.getEmbeddedOptionType(), this::setEmbeddedOptionType);
			merger.mergeBasic(getSeniority(), o.getSeniority(), this::setSeniority);
			merger.mergeBasic(getSeries(), o.getSeries(), this::setSeries);
			merger.mergeBasic(getIndexFactor(), o.getIndexFactor(), this::setIndexFactor);
			merger.mergeBasic(getTechnicalRecordId(), o.getTechnicalRecordId(), this::setTechnicalRecordId);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CommonTransactionReport _that = getType().cast(o);
		
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(eventType, _that.getEventType())) return false;
			if (!Objects.equals(eventIdentifier, _that.getEventIdentifier())) return false;
			if (!Objects.equals(effectiveDate, _that.getEffectiveDate())) return false;
			if (!Objects.equals(earlyTerminationDate, _that.getEarlyTerminationDate())) return false;
			if (!Objects.equals(eventTimestamp, _that.getEventTimestamp())) return false;
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(executionTimestamp, _that.getExecutionTimestamp())) return false;
			if (!Objects.equals(expirationDate, _that.getExpirationDate())) return false;
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierType, _that.getCounterparty2IdentifierType())) return false;
			if (!Objects.equals(beneficiary1, _that.getBeneficiary1())) return false;
			if (!Objects.equals(beneficiary1IdentifierTypeIndicator, _that.getBeneficiary1IdentifierTypeIndicator())) return false;
			if (!Objects.equals(cleared, _that.getCleared())) return false;
			if (!Objects.equals(centralCounterparty, _that.getCentralCounterparty())) return false;
			if (!Objects.equals(clearingMember, _that.getClearingMember())) return false;
			if (!Objects.equals(confirmed, _that.getConfirmed())) return false;
			if (!Objects.equals(callAmount, _that.getCallAmount())) return false;
			if (!Objects.equals(putAmount, _that.getPutAmount())) return false;
			if (!Objects.equals(callCurrency, _that.getCallCurrency())) return false;
			if (!Objects.equals(putCurrency, _that.getPutCurrency())) return false;
			if (!ListEquals.listEquals(priceSchedule, _that.getPriceSchedule())) return false;
			if (!ListEquals.listEquals(strikePriceSchedule, _that.getStrikePriceSchedule())) return false;
			if (!Objects.equals(price, _that.getPrice())) return false;
			if (!Objects.equals(priceNotation, _that.getPriceNotation())) return false;
			if (!Objects.equals(priceCurrency, _that.getPriceCurrency())) return false;
			if (!Objects.equals(packageTransactionPrice, _that.getPackageTransactionPrice())) return false;
			if (!Objects.equals(packageTransactionPriceNotation, _that.getPackageTransactionPriceNotation())) return false;
			if (!Objects.equals(packageTransactionPriceCurrency, _that.getPackageTransactionPriceCurrency())) return false;
			if (!Objects.equals(packageTransactionSpread, _that.getPackageTransactionSpread())) return false;
			if (!Objects.equals(packageTransactionSpreadNotation, _that.getPackageTransactionSpreadNotation())) return false;
			if (!Objects.equals(packageTransactionSpreadCurrency, _that.getPackageTransactionSpreadCurrency())) return false;
			if (!Objects.equals(strikePrice, _that.getStrikePrice())) return false;
			if (!Objects.equals(strikePriceNotation, _that.getStrikePriceNotation())) return false;
			if (!Objects.equals(strikePriceCurrency, _that.getStrikePriceCurrency())) return false;
			if (!Objects.equals(priceUnitOfMeasure, _that.getPriceUnitOfMeasure())) return false;
			if (!Objects.equals(optionPremiumAmount, _that.getOptionPremiumAmount())) return false;
			if (!Objects.equals(optionPremiumCurrency, _that.getOptionPremiumCurrency())) return false;
			if (!Objects.equals(optionPremiumPaymentDate, _that.getOptionPremiumPaymentDate())) return false;
			if (!Objects.equals(cdSIndexAttachmentPoint, _that.getCdSIndexAttachmentPoint())) return false;
			if (!Objects.equals(cdSIndexDetachmentPoint, _that.getCdSIndexDetachmentPoint())) return false;
			if (!Objects.equals(collateralPortfolioIndicator, _that.getCollateralPortfolioIndicator())) return false;
			if (!Objects.equals(firstExerciseDate, _that.getFirstExerciseDate())) return false;
			if (!Objects.equals(finalContractualSettlementDate, _that.getFinalContractualSettlementDate())) return false;
			if (!ListEquals.listEquals(otherPayment, _that.getOtherPayment())) return false;
			if (!Objects.equals(leg1, _that.getLeg1())) return false;
			if (!Objects.equals(leg2, _that.getLeg2())) return false;
			if (!Objects.equals(counterparty2Name, _that.getCounterparty2Name())) return false;
			if (!Objects.equals(executionAgentCounterparty1, _that.getExecutionAgentCounterparty1())) return false;
			if (!Objects.equals(executionAgentCounterparty2DTCC, _that.getExecutionAgentCounterparty2DTCC())) return false;
			if (!Objects.equals(reportSubmittingEntityID, _that.getReportSubmittingEntityID())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifierProprietary, _that.getUniqueTransactionIdentifierProprietary())) return false;
			if (!Objects.equals(priorUTI, _that.getPriorUTI())) return false;
			if (!Objects.equals(priorUTIProprietary, _that.getPriorUTIProprietary())) return false;
			if (!Objects.equals(uniqueProductIdentifier, _that.getUniqueProductIdentifier())) return false;
			if (!Objects.equals(contractType, _that.getContractType())) return false;
			if (!Objects.equals(assetClass, _that.getAssetClass())) return false;
			if (!Objects.equals(clearingTimestamp, _that.getClearingTimestamp())) return false;
			if (!Objects.equals(platformIdentifier, _that.getPlatformIdentifier())) return false;
			if (!Objects.equals(direction1, _that.getDirection1())) return false;
			if (!Objects.equals(direction2Leg1, _that.getDirection2Leg1())) return false;
			if (!Objects.equals(direction2Leg2, _that.getDirection2Leg2())) return false;
			if (!Objects.equals(packageIdentifier, _that.getPackageIdentifier())) return false;
			if (!Objects.equals(barrier, _that.getBarrier())) return false;
			if (!Objects.equals(exchangeRate, _that.getExchangeRate())) return false;
			if (!Objects.equals(exchangeRateBasis, _that.getExchangeRateBasis())) return false;
			if (!Objects.equals(nameOfTheUnderlyingIndex, _that.getNameOfTheUnderlyingIndex())) return false;
			if (!ListEquals.listEquals(basketConstituents, _that.getBasketConstituents())) return false;
			if (!Objects.equals(underlying, _that.getUnderlying())) return false;
			if (!Objects.equals(quantityUnitOfMeasureLeg1, _that.getQuantityUnitOfMeasureLeg1())) return false;
			if (!Objects.equals(quantityUnitOfMeasureLeg2, _that.getQuantityUnitOfMeasureLeg2())) return false;
			if (!Objects.equals(maturityDateOfTheUnderlying, _that.getMaturityDateOfTheUnderlying())) return false;
			if (!Objects.equals(optionType, _that.getOptionType())) return false;
			if (!Objects.equals(optionStyle, _that.getOptionStyle())) return false;
			if (!Objects.equals(deliveryType, _that.getDeliveryType())) return false;
			if (!Objects.equals(countryOfTheCounterparty2, _that.getCountryOfTheCounterparty2())) return false;
			if (!Objects.equals(bookingLocation, _that.getBookingLocation())) return false;
			if (!Objects.equals(traderLocation, _that.getTraderLocation())) return false;
			if (!Objects.equals(tradingCapacity, _that.getTradingCapacity())) return false;
			if (!Objects.equals(embeddedOptionType, _that.getEmbeddedOptionType())) return false;
			if (!Objects.equals(seniority, _that.getSeniority())) return false;
			if (!Objects.equals(series, _that.getSeries())) return false;
			if (!Objects.equals(indexFactor, _that.getIndexFactor())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (eventType != null ? eventType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (eventIdentifier != null ? eventIdentifier.hashCode() : 0);
			_result = 31 * _result + (effectiveDate != null ? effectiveDate.hashCode() : 0);
			_result = 31 * _result + (earlyTerminationDate != null ? earlyTerminationDate.hashCode() : 0);
			_result = 31 * _result + (eventTimestamp != null ? eventTimestamp.hashCode() : 0);
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (executionTimestamp != null ? executionTimestamp.hashCode() : 0);
			_result = 31 * _result + (expirationDate != null ? expirationDate.hashCode() : 0);
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierType != null ? counterparty2IdentifierType.hashCode() : 0);
			_result = 31 * _result + (beneficiary1 != null ? beneficiary1.hashCode() : 0);
			_result = 31 * _result + (beneficiary1IdentifierTypeIndicator != null ? beneficiary1IdentifierTypeIndicator.hashCode() : 0);
			_result = 31 * _result + (cleared != null ? cleared.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (centralCounterparty != null ? centralCounterparty.hashCode() : 0);
			_result = 31 * _result + (clearingMember != null ? clearingMember.hashCode() : 0);
			_result = 31 * _result + (confirmed != null ? confirmed.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (callAmount != null ? callAmount.hashCode() : 0);
			_result = 31 * _result + (putAmount != null ? putAmount.hashCode() : 0);
			_result = 31 * _result + (callCurrency != null ? callCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (putCurrency != null ? putCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (priceSchedule != null ? priceSchedule.hashCode() : 0);
			_result = 31 * _result + (strikePriceSchedule != null ? strikePriceSchedule.hashCode() : 0);
			_result = 31 * _result + (price != null ? price.hashCode() : 0);
			_result = 31 * _result + (priceNotation != null ? priceNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (priceCurrency != null ? priceCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageTransactionPrice != null ? packageTransactionPrice.hashCode() : 0);
			_result = 31 * _result + (packageTransactionPriceNotation != null ? packageTransactionPriceNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageTransactionPriceCurrency != null ? packageTransactionPriceCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageTransactionSpread != null ? packageTransactionSpread.hashCode() : 0);
			_result = 31 * _result + (packageTransactionSpreadNotation != null ? packageTransactionSpreadNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageTransactionSpreadCurrency != null ? packageTransactionSpreadCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (strikePrice != null ? strikePrice.hashCode() : 0);
			_result = 31 * _result + (strikePriceNotation != null ? strikePriceNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (strikePriceCurrency != null ? strikePriceCurrency.hashCode() : 0);
			_result = 31 * _result + (priceUnitOfMeasure != null ? priceUnitOfMeasure.hashCode() : 0);
			_result = 31 * _result + (optionPremiumAmount != null ? optionPremiumAmount.hashCode() : 0);
			_result = 31 * _result + (optionPremiumCurrency != null ? optionPremiumCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (optionPremiumPaymentDate != null ? optionPremiumPaymentDate.hashCode() : 0);
			_result = 31 * _result + (cdSIndexAttachmentPoint != null ? cdSIndexAttachmentPoint.hashCode() : 0);
			_result = 31 * _result + (cdSIndexDetachmentPoint != null ? cdSIndexDetachmentPoint.hashCode() : 0);
			_result = 31 * _result + (collateralPortfolioIndicator != null ? collateralPortfolioIndicator.hashCode() : 0);
			_result = 31 * _result + (firstExerciseDate != null ? firstExerciseDate.hashCode() : 0);
			_result = 31 * _result + (finalContractualSettlementDate != null ? finalContractualSettlementDate.hashCode() : 0);
			_result = 31 * _result + (otherPayment != null ? otherPayment.hashCode() : 0);
			_result = 31 * _result + (leg1 != null ? leg1.hashCode() : 0);
			_result = 31 * _result + (leg2 != null ? leg2.hashCode() : 0);
			_result = 31 * _result + (counterparty2Name != null ? counterparty2Name.hashCode() : 0);
			_result = 31 * _result + (executionAgentCounterparty1 != null ? executionAgentCounterparty1.hashCode() : 0);
			_result = 31 * _result + (executionAgentCounterparty2DTCC != null ? executionAgentCounterparty2DTCC.hashCode() : 0);
			_result = 31 * _result + (reportSubmittingEntityID != null ? reportSubmittingEntityID.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifierProprietary != null ? uniqueTransactionIdentifierProprietary.hashCode() : 0);
			_result = 31 * _result + (priorUTI != null ? priorUTI.hashCode() : 0);
			_result = 31 * _result + (priorUTIProprietary != null ? priorUTIProprietary.hashCode() : 0);
			_result = 31 * _result + (uniqueProductIdentifier != null ? uniqueProductIdentifier.hashCode() : 0);
			_result = 31 * _result + (contractType != null ? contractType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (assetClass != null ? assetClass.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (clearingTimestamp != null ? clearingTimestamp.hashCode() : 0);
			_result = 31 * _result + (platformIdentifier != null ? platformIdentifier.hashCode() : 0);
			_result = 31 * _result + (direction1 != null ? direction1.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (direction2Leg1 != null ? direction2Leg1.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (direction2Leg2 != null ? direction2Leg2.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (packageIdentifier != null ? packageIdentifier.hashCode() : 0);
			_result = 31 * _result + (barrier != null ? barrier.hashCode() : 0);
			_result = 31 * _result + (exchangeRate != null ? exchangeRate.hashCode() : 0);
			_result = 31 * _result + (exchangeRateBasis != null ? exchangeRateBasis.hashCode() : 0);
			_result = 31 * _result + (nameOfTheUnderlyingIndex != null ? nameOfTheUnderlyingIndex.hashCode() : 0);
			_result = 31 * _result + (basketConstituents != null ? basketConstituents.hashCode() : 0);
			_result = 31 * _result + (underlying != null ? underlying.hashCode() : 0);
			_result = 31 * _result + (quantityUnitOfMeasureLeg1 != null ? quantityUnitOfMeasureLeg1.hashCode() : 0);
			_result = 31 * _result + (quantityUnitOfMeasureLeg2 != null ? quantityUnitOfMeasureLeg2.hashCode() : 0);
			_result = 31 * _result + (maturityDateOfTheUnderlying != null ? maturityDateOfTheUnderlying.hashCode() : 0);
			_result = 31 * _result + (optionType != null ? optionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (optionStyle != null ? optionStyle.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (deliveryType != null ? deliveryType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (countryOfTheCounterparty2 != null ? countryOfTheCounterparty2.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (bookingLocation != null ? bookingLocation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (traderLocation != null ? traderLocation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradingCapacity != null ? tradingCapacity.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (embeddedOptionType != null ? embeddedOptionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (seniority != null ? seniority.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (series != null ? series.hashCode() : 0);
			_result = 31 * _result + (indexFactor != null ? indexFactor.hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CommonTransactionReportBuilder {" +
				"actionType=" + this.actionType + ", " +
				"eventType=" + this.eventType + ", " +
				"eventIdentifier=" + this.eventIdentifier + ", " +
				"effectiveDate=" + this.effectiveDate + ", " +
				"earlyTerminationDate=" + this.earlyTerminationDate + ", " +
				"eventTimestamp=" + this.eventTimestamp + ", " +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"executionTimestamp=" + this.executionTimestamp + ", " +
				"expirationDate=" + this.expirationDate + ", " +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierType=" + this.counterparty2IdentifierType + ", " +
				"beneficiary1=" + this.beneficiary1 + ", " +
				"beneficiary1IdentifierTypeIndicator=" + this.beneficiary1IdentifierTypeIndicator + ", " +
				"cleared=" + this.cleared + ", " +
				"centralCounterparty=" + this.centralCounterparty + ", " +
				"clearingMember=" + this.clearingMember + ", " +
				"confirmed=" + this.confirmed + ", " +
				"callAmount=" + this.callAmount + ", " +
				"putAmount=" + this.putAmount + ", " +
				"callCurrency=" + this.callCurrency + ", " +
				"putCurrency=" + this.putCurrency + ", " +
				"priceSchedule=" + this.priceSchedule + ", " +
				"strikePriceSchedule=" + this.strikePriceSchedule + ", " +
				"price=" + this.price + ", " +
				"priceNotation=" + this.priceNotation + ", " +
				"priceCurrency=" + this.priceCurrency + ", " +
				"packageTransactionPrice=" + this.packageTransactionPrice + ", " +
				"packageTransactionPriceNotation=" + this.packageTransactionPriceNotation + ", " +
				"packageTransactionPriceCurrency=" + this.packageTransactionPriceCurrency + ", " +
				"packageTransactionSpread=" + this.packageTransactionSpread + ", " +
				"packageTransactionSpreadNotation=" + this.packageTransactionSpreadNotation + ", " +
				"packageTransactionSpreadCurrency=" + this.packageTransactionSpreadCurrency + ", " +
				"strikePrice=" + this.strikePrice + ", " +
				"strikePriceNotation=" + this.strikePriceNotation + ", " +
				"strikePriceCurrency=" + this.strikePriceCurrency + ", " +
				"priceUnitOfMeasure=" + this.priceUnitOfMeasure + ", " +
				"optionPremiumAmount=" + this.optionPremiumAmount + ", " +
				"optionPremiumCurrency=" + this.optionPremiumCurrency + ", " +
				"optionPremiumPaymentDate=" + this.optionPremiumPaymentDate + ", " +
				"cdSIndexAttachmentPoint=" + this.cdSIndexAttachmentPoint + ", " +
				"cdSIndexDetachmentPoint=" + this.cdSIndexDetachmentPoint + ", " +
				"collateralPortfolioIndicator=" + this.collateralPortfolioIndicator + ", " +
				"firstExerciseDate=" + this.firstExerciseDate + ", " +
				"finalContractualSettlementDate=" + this.finalContractualSettlementDate + ", " +
				"otherPayment=" + this.otherPayment + ", " +
				"leg1=" + this.leg1 + ", " +
				"leg2=" + this.leg2 + ", " +
				"counterparty2Name=" + this.counterparty2Name + ", " +
				"executionAgentCounterparty1=" + this.executionAgentCounterparty1 + ", " +
				"executionAgentCounterparty2DTCC=" + this.executionAgentCounterparty2DTCC + ", " +
				"reportSubmittingEntityID=" + this.reportSubmittingEntityID + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"uniqueTransactionIdentifierProprietary=" + this.uniqueTransactionIdentifierProprietary + ", " +
				"priorUTI=" + this.priorUTI + ", " +
				"priorUTIProprietary=" + this.priorUTIProprietary + ", " +
				"uniqueProductIdentifier=" + this.uniqueProductIdentifier + ", " +
				"contractType=" + this.contractType + ", " +
				"assetClass=" + this.assetClass + ", " +
				"clearingTimestamp=" + this.clearingTimestamp + ", " +
				"platformIdentifier=" + this.platformIdentifier + ", " +
				"direction1=" + this.direction1 + ", " +
				"direction2Leg1=" + this.direction2Leg1 + ", " +
				"direction2Leg2=" + this.direction2Leg2 + ", " +
				"packageIdentifier=" + this.packageIdentifier + ", " +
				"barrier=" + this.barrier + ", " +
				"exchangeRate=" + this.exchangeRate + ", " +
				"exchangeRateBasis=" + this.exchangeRateBasis + ", " +
				"nameOfTheUnderlyingIndex=" + this.nameOfTheUnderlyingIndex + ", " +
				"basketConstituents=" + this.basketConstituents + ", " +
				"underlying=" + this.underlying + ", " +
				"quantityUnitOfMeasureLeg1=" + this.quantityUnitOfMeasureLeg1 + ", " +
				"quantityUnitOfMeasureLeg2=" + this.quantityUnitOfMeasureLeg2 + ", " +
				"maturityDateOfTheUnderlying=" + this.maturityDateOfTheUnderlying + ", " +
				"optionType=" + this.optionType + ", " +
				"optionStyle=" + this.optionStyle + ", " +
				"deliveryType=" + this.deliveryType + ", " +
				"countryOfTheCounterparty2=" + this.countryOfTheCounterparty2 + ", " +
				"bookingLocation=" + this.bookingLocation + ", " +
				"traderLocation=" + this.traderLocation + ", " +
				"tradingCapacity=" + this.tradingCapacity + ", " +
				"embeddedOptionType=" + this.embeddedOptionType + ", " +
				"seniority=" + this.seniority + ", " +
				"series=" + this.series + ", " +
				"indexFactor=" + this.indexFactor + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}';
		}
	}
}
