package drr.regulation.common.trade.price.functions;

import cdm.observable.event.FeaturePayment;
import cdm.observable.event.Trigger;
import cdm.observable.event.TriggerEvent;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.metafields.FieldWithMetaString;
import drr.regulation.common.functions.ConvertNonISOToISOCurrency;
import drr.regulation.common.trade.price.BarrierPrice;
import drr.regulation.common.trade.price.BarrierPrice.BarrierPriceBuilder;
import drr.standards.iosco.cde.price.PriceFormat;
import drr.standards.iosco.cde.price.PriceNotationEnum;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(BarrierFromTriggerEvent.BarrierFromTriggerEventDefault.class)
public abstract class BarrierFromTriggerEvent implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected ConvertNonISOToISOCurrency convertNonISOToISOCurrency;

	/**
	* @param triggerEvent 
	* @return barrier 
	*/
	public BarrierPrice evaluate(TriggerEvent triggerEvent) {
		BarrierPrice.BarrierPriceBuilder barrierBuilder = doEvaluate(triggerEvent);
		
		final BarrierPrice barrier;
		if (barrierBuilder == null) {
			barrier = null;
		} else {
			barrier = barrierBuilder.build();
			objectValidator.validate(BarrierPrice.class, barrier);
		}
		
		return barrier;
	}

	protected abstract BarrierPrice.BarrierPriceBuilder doEvaluate(TriggerEvent triggerEvent);

	public static class BarrierFromTriggerEventDefault extends BarrierFromTriggerEvent {
		@Override
		protected BarrierPrice.BarrierPriceBuilder doEvaluate(TriggerEvent triggerEvent) {
			BarrierPrice.BarrierPriceBuilder barrier = BarrierPrice.builder();
			return assignOutput(barrier, triggerEvent);
		}
		
		protected BarrierPrice.BarrierPriceBuilder assignOutput(BarrierPrice.BarrierPriceBuilder barrier, TriggerEvent triggerEvent) {
			final PriceNotationEnum ifThenElseResult;
			if (exists(MapperS.of(triggerEvent).<Trigger>map("getTrigger", _triggerEvent -> _triggerEvent.getTrigger()).<BigDecimal>map("getLevel", trigger -> trigger.getLevel())).getOrDefault(false)) {
				ifThenElseResult = PriceNotationEnum.MONETARY;
			} else if (exists(MapperS.of(triggerEvent).<Trigger>map("getTrigger", _triggerEvent -> _triggerEvent.getTrigger()).<BigDecimal>map("getLevelPercentage", trigger -> trigger.getLevelPercentage())).getOrDefault(false)) {
				ifThenElseResult = PriceNotationEnum.DECIMAL;
			} else {
				ifThenElseResult = null;
			}
			barrier = toBuilder(BarrierPrice.builder()
				.setPrice(PriceFormat.builder()
					.setMonetary(MapperS.of(triggerEvent).<Trigger>map("getTrigger", _triggerEvent -> _triggerEvent.getTrigger()).<BigDecimal>map("getLevel", trigger -> trigger.getLevel()).get())
					.setDecimal(MapperS.of(triggerEvent).<Trigger>map("getTrigger", _triggerEvent -> _triggerEvent.getTrigger()).<BigDecimal>map("getLevelPercentage", trigger -> trigger.getLevelPercentage()).get())
					.build())
				.setNotation(ifThenElseResult)
				.setCurrency(MapperS.of(triggerEvent).<FeaturePayment>map("getFeaturePayment", _triggerEvent -> _triggerEvent.getFeaturePayment()).<FieldWithMetaString>map("getCurrency", featurePayment -> featurePayment.getCurrency())
					.mapSingleToItem(item -> {
						final FieldWithMetaString fieldWithMetaString = item.get();
						return MapperS.of(convertNonISOToISOCurrency.evaluate((fieldWithMetaString == null ? null : fieldWithMetaString.getValue())));
					}).get())
				.build());
			
			return Optional.ofNullable(barrier)
				.map(o -> o.prune())
				.orElse(null);
		}
	}
}
