package drr.regulation.csa.rewrite.trade.validation;

import com.google.common.collect.Lists;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.ValidationResult.ValidationType;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.csa.rewrite.trade.CSATransactionReport;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkNumber;
import static com.rosetta.model.lib.expression.ExpressionOperators.checkString;
import static com.rosetta.model.lib.validation.ValidationResult.failure;
import static com.rosetta.model.lib.validation.ValidationResult.success;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

public class CSATransactionReportTypeFormatValidator implements Validator<CSATransactionReport> {

	private List<ComparisonResult> getComparisonResults(CSATransactionReport o) {
		return Lists.<ComparisonResult>newArrayList(
				checkString("eventIdentifier", o.getEventIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("counterparty1", o.getCounterparty1(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("counterparty2", o.getCounterparty2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("beneficiary1", o.getBeneficiary1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("centralCounterparty", o.getCentralCounterparty(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("clearingMember", o.getClearingMember(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkNumber("callAmount", o.getCallAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("putAmount", o.getPutAmount(), of(25), of(5), empty(), empty()), 
				checkString("priceUnitOfMeasure", o.getPriceUnitOfMeasure(), 1, of(4), empty()), 
				checkNumber("optionPremiumAmount", o.getOptionPremiumAmount(), of(25), of(5), empty(), empty()), 
				checkNumber("cdSIndexAttachmentPoint", o.getCdSIndexAttachmentPoint(), of(11), of(10), empty(), empty()), 
				checkNumber("cdSIndexDetachmentPoint", o.getCdSIndexDetachmentPoint(), of(11), of(10), empty(), empty()), 
				checkString("counterparty2Name", o.getCounterparty2Name(), 1, of(150), empty()), 
				checkString("executionAgentCounterparty2DTCC", o.getExecutionAgentCounterparty2DTCC(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("reportSubmittingEntityID", o.getReportSubmittingEntityID(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("uniqueTransactionIdentifier", o.getUniqueTransactionIdentifier(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("uniqueTransactionIdentifierProprietary", o.getUniqueTransactionIdentifierProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("priorUTI", o.getPriorUTI(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("priorUTIProprietary", o.getPriorUTIProprietary(), 0, of(72), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,70}[a-zA-Z0-9]{1}"))), 
				checkString("platformIdentifier", o.getPlatformIdentifier(), 1, of(4), empty()), 
				checkString("packageIdentifier", o.getPackageIdentifier(), 1, of(100), of(Pattern.compile("[a-zA-Z0-9]{1,100}"))), 
				checkNumber("exchangeRate", o.getExchangeRate(), of(18), of(13), empty(), empty()), 
				checkString("exchangeRateBasis", o.getExchangeRateBasis(), 7, of(7), of(Pattern.compile("[A-Z]{3,3}/[A-Z]{3,3}"))), 
				checkString("nameOfTheUnderlyingIndex", o.getNameOfTheUnderlyingIndex(), 1, of(50), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_\\&]{1,50}"))), 
				checkString("quantityUnitOfMeasureLeg1", o.getQuantityUnitOfMeasureLeg1(), 1, of(4), empty()), 
				checkString("quantityUnitOfMeasureLeg2", o.getQuantityUnitOfMeasureLeg2(), 1, of(4), empty()), 
				checkNumber("series", o.getSeries(), of(5), of(0), empty(), empty()), 
				checkNumber("indexFactor", o.getIndexFactor(), of(11), of(10), empty(), empty()), 
				checkString("counterparty2IdentifierSource", o.getCounterparty2IdentifierSource(), 1, of(4), empty()), 
				checkString("buyerIdentifier", o.getBuyerIdentifier(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("sellerIdentifier", o.getSellerIdentifier(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("payerIdentifierLeg1", o.getPayerIdentifierLeg1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("payerIdentifierLeg2", o.getPayerIdentifierLeg2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("receiverIdentifierLeg1", o.getReceiverIdentifierLeg1(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("receiverIdentifierLeg2", o.getReceiverIdentifierLeg2(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("brokerIdentifier", o.getBrokerIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("countryAndProvinceOrTerritoryOfIndividual", o.getCountryAndProvinceOrTerritoryOfIndividual(), 1, of(5), empty()), 
				checkString("subsequentPositionUTI", o.getSubsequentPositionUTI(), 0, of(52), of(Pattern.compile("[a-zA-Z0-9]{1}[a-zA-Z0-9\\:.\\-_]{0,50}[a-zA-Z0-9]{1}"))), 
				checkString("priorUSI", o.getPriorUSI(), 1, of(42), of(Pattern.compile("[a-zA-Z0-9]{1,42}"))), 
				checkString("masterAgreementType", o.getMasterAgreementType(), 1, of(4), empty()), 
				checkNumber("masterAgreementVersion", o.getMasterAgreementVersion(), empty(), of(0), empty(), empty()), 
				checkString("clearingMemberIdentifier", o.getClearingMemberIdentifier(), 0, empty(), of(Pattern.compile("[A-Z0-9]{18,18}[0-9]{2,2}"))), 
				checkString("variationMarginCollateralPortfolioCode", o.getVariationMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("initialMarginCollateralPortfolioCode", o.getInitialMarginCollateralPortfolioCode(), 1, of(52), of(Pattern.compile("[a-zA-Z0-9]{1,52}"))), 
				checkString("customBasketCode", o.getCustomBasketCode(), 20, of(72), of(Pattern.compile("[a-zA-Z0-9]{1,72}"))), 
				checkString("underlierIDOtherLeg1", o.getUnderlierIDOtherLeg1(), 1, of(350), of(Pattern.compile(".{1,350}"))), 
				checkString("underlierIDOtherLeg2", o.getUnderlierIDOtherLeg2(), 1, of(350), of(Pattern.compile(".{1,350}"))), 
				checkString("underlierIDOtherSourceLeg1", o.getUnderlierIDOtherSourceLeg1(), 1, of(350), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_]{1,350}"))), 
				checkString("underlierIDOtherSourceLeg2", o.getUnderlierIDOtherSourceLeg2(), 1, of(350), of(Pattern.compile("[a-zA-Z0-9 \\:.\\-_]{1,350}"))), 
				checkString("underlyingAssetPriceSourceLeg1", o.getUnderlyingAssetPriceSourceLeg1(), 1, of(50), of(Pattern.compile(".{1,50}"))), 
				checkString("underlyingAssetPriceSourceLeg2", o.getUnderlyingAssetPriceSourceLeg2(), 1, of(50), of(Pattern.compile(".{1,50}")))
			);
	}

	@Override
	public List<ValidationResult<?>> getValidationResults(RosettaPath path, CSATransactionReport o) {
		return getComparisonResults(o)
			.stream()
			.map(res -> {
				if (!isNullOrEmpty(res.getError())) {
					return failure("CSATransactionReport", ValidationType.TYPE_FORMAT, "CSATransactionReport", path, "", res.getError());
				}
				return success("CSATransactionReport", ValidationType.TYPE_FORMAT, "CSATransactionReport", path, "");
			})
			.collect(toList());
	}

}
