package drr.standards.iosco.cde;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.util.ListEquals;
import drr.standards.iosco.cde.Leg;
import drr.standards.iosco.cde.Leg.LegBuilder;
import drr.standards.iosco.cde.Leg.LegBuilderImpl;
import drr.standards.iosco.cde.Leg.LegImpl;
import drr.standards.iosco.cde.meta.LegMeta;
import drr.standards.iosco.cde.payment.PeriodicPayment;
import drr.standards.iosco.cde.payment.PeriodicPayment.PeriodicPaymentBuilder;
import drr.standards.iosco.cde.price.PriceFormat;
import drr.standards.iosco.cde.price.PriceFormat.PriceFormatBuilder;
import drr.standards.iosco.cde.price.PriceNotationEnum;
import drr.standards.iosco.cde.quantity.NotionalPeriod;
import drr.standards.iosco.cde.quantity.NotionalPeriod.NotionalPeriodBuilder;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * @version 6.0.0-dev.128
 */
@RosettaDataType(value="Leg", builder=Leg.LegBuilderImpl.class, version="6.0.0-dev.128")
@RuneDataType(value="Leg", model="drr", builder=Leg.LegBuilderImpl.class, version="6.0.0-dev.128")
public interface Leg extends RosettaModelObject {

	LegMeta metaData = new LegMeta();

	/*********************** Getter Methods  ***********************/
	PeriodicPayment getPeriodicPayment();
	BigDecimal getNotionalAmount();
	/**
	 *
	 * effectiveDate
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "78.1"
	 *
	 * Provision Unadjusted date on which the associated notional amount of becomes effective.
	 *
	 *
	 * effectiveDate
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220811"
	 *
	 * Provision Model should contain a fall back for unadjustedDate when adjustedDate is only available. If an adjusted date is only provided then fields requiring an unadjusted date are left blank which will result in a NACK from the TR.  Functional rules should be updated to fall back on adjusted date if available.
	 *
	 *
	 * endDate
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "78.2"
	 *
	 * Provision Unadjusted end date of the notional amount (not applicable if the unadjusted end date of a given schedules period is back-to-back with the unadjusted effective date of the subsequent period).
	 *
	 *
	 * value
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "78.3"
	 *
	 * Provision Notional amount which becomes effective on the associated unadjusted effective date.
	 *
	 */
	List<? extends NotionalPeriod> getNotionalAmountSchedule();
	/**
	 *
	 * effectiveDate
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "80.1"
	 *
	 * Provision Unadjusted date on which the associated notional quantity of becomes effective.
	 *
	 *
	 * endDate
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "80.2"
	 *
	 * Provision Unadjusted end date of the notional quantity (not applicable if the unadjusted end date of a given schedules period is back-to-back with the unadjusted effective date of the subsequent period).
	 *
	 *
	 * value
	 * Body CPMI_IOSCO
	 * Corpus TechnicalGuidance CDE Harmonisation of Critical Data Elements (other than UTI and UPI) "The G20 Leaders agreed in 2009 that all over-the-counter (OTC) derivative transactions should be reported to trade repositories (TRs) to further the goals of improving transparency, mitigating systemic risk and preventing market abuse.1 Aggregation of the data being reported across TRs will help authorities to obtain a comprehensive view of the OTC derivatives market and its activity. Such aggregation is feasible if the work on standardisation and harmonisation of important data elements [is] completed.2 Since November 2014, the CPMI and IOSCO working group for the harmonisation of key OTC derivatives data elements (Harmonisation Group) has worked to develop global guidance regarding the definition, format and usage of key OTC derivatives data elements reported to TRs, including the Unique Transaction Identifier (UTI), the Unique Product Identifier (UPI) and other critical data elements. Technical Guidance on the Harmonisation of the Unique Transaction Identifier (UTI) was published in February 20173 and Technical Guidance on the Harmonisation of the Unique Product Identifier (UPI) was published in September 2017.4 The CPMI and IOSCO also published consultative reports on a first, a second and a third batch of critical data elements other than UTI and UPI in September 2015, October 2016 and June 2017, respectively" 
	 * section "2" * field "80.3"
	 *
	 * Provision Notional quantity which becomes effective on the associated unadjusted effective date.
	 *
	 */
	List<? extends NotionalPeriod> getNotionalQuantitySchedule();
	ISOCurrencyCodeEnum getNotionalCurrency();
	BigDecimal getTotalNotionalQuantity();
	BigDecimal getFixedRate();
	ISOCurrencyCodeEnum getSettlementCurrency();
	PriceFormat getSpread();
	PriceNotationEnum getSpreadNotation();
	ISOCurrencyCodeEnum getSpreadCurrency();

	/*********************** Build Methods  ***********************/
	Leg build();
	
	Leg.LegBuilder toBuilder();
	
	static Leg.LegBuilder builder() {
		return new Leg.LegBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends Leg> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends Leg> getType() {
		return Leg.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("periodicPayment"), processor, PeriodicPayment.class, getPeriodicPayment());
		processor.processBasic(path.newSubPath("notionalAmount"), BigDecimal.class, getNotionalAmount(), this);
		processRosetta(path.newSubPath("notionalAmountSchedule"), processor, NotionalPeriod.class, getNotionalAmountSchedule());
		processRosetta(path.newSubPath("notionalQuantitySchedule"), processor, NotionalPeriod.class, getNotionalQuantitySchedule());
		processor.processBasic(path.newSubPath("notionalCurrency"), ISOCurrencyCodeEnum.class, getNotionalCurrency(), this);
		processor.processBasic(path.newSubPath("totalNotionalQuantity"), BigDecimal.class, getTotalNotionalQuantity(), this);
		processor.processBasic(path.newSubPath("fixedRate"), BigDecimal.class, getFixedRate(), this);
		processor.processBasic(path.newSubPath("settlementCurrency"), ISOCurrencyCodeEnum.class, getSettlementCurrency(), this);
		processRosetta(path.newSubPath("spread"), processor, PriceFormat.class, getSpread());
		processor.processBasic(path.newSubPath("spreadNotation"), PriceNotationEnum.class, getSpreadNotation(), this);
		processor.processBasic(path.newSubPath("spreadCurrency"), ISOCurrencyCodeEnum.class, getSpreadCurrency(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface LegBuilder extends Leg, RosettaModelObjectBuilder {
		PeriodicPayment.PeriodicPaymentBuilder getOrCreatePeriodicPayment();
		@Override
		PeriodicPayment.PeriodicPaymentBuilder getPeriodicPayment();
		NotionalPeriod.NotionalPeriodBuilder getOrCreateNotionalAmountSchedule(int _index);
		@Override
		List<? extends NotionalPeriod.NotionalPeriodBuilder> getNotionalAmountSchedule();
		NotionalPeriod.NotionalPeriodBuilder getOrCreateNotionalQuantitySchedule(int _index);
		@Override
		List<? extends NotionalPeriod.NotionalPeriodBuilder> getNotionalQuantitySchedule();
		PriceFormat.PriceFormatBuilder getOrCreateSpread();
		@Override
		PriceFormat.PriceFormatBuilder getSpread();
		Leg.LegBuilder setPeriodicPayment(PeriodicPayment periodicPayment);
		Leg.LegBuilder setNotionalAmount(BigDecimal notionalAmount);
		Leg.LegBuilder addNotionalAmountSchedule(NotionalPeriod notionalAmountSchedule);
		Leg.LegBuilder addNotionalAmountSchedule(NotionalPeriod notionalAmountSchedule, int _idx);
		Leg.LegBuilder addNotionalAmountSchedule(List<? extends NotionalPeriod> notionalAmountSchedule);
		Leg.LegBuilder setNotionalAmountSchedule(List<? extends NotionalPeriod> notionalAmountSchedule);
		Leg.LegBuilder addNotionalQuantitySchedule(NotionalPeriod notionalQuantitySchedule);
		Leg.LegBuilder addNotionalQuantitySchedule(NotionalPeriod notionalQuantitySchedule, int _idx);
		Leg.LegBuilder addNotionalQuantitySchedule(List<? extends NotionalPeriod> notionalQuantitySchedule);
		Leg.LegBuilder setNotionalQuantitySchedule(List<? extends NotionalPeriod> notionalQuantitySchedule);
		Leg.LegBuilder setNotionalCurrency(ISOCurrencyCodeEnum notionalCurrency);
		Leg.LegBuilder setTotalNotionalQuantity(BigDecimal totalNotionalQuantity);
		Leg.LegBuilder setFixedRate(BigDecimal fixedRate);
		Leg.LegBuilder setSettlementCurrency(ISOCurrencyCodeEnum settlementCurrency);
		Leg.LegBuilder setSpread(PriceFormat spread);
		Leg.LegBuilder setSpreadNotation(PriceNotationEnum spreadNotation);
		Leg.LegBuilder setSpreadCurrency(ISOCurrencyCodeEnum spreadCurrency);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("periodicPayment"), processor, PeriodicPayment.PeriodicPaymentBuilder.class, getPeriodicPayment());
			processor.processBasic(path.newSubPath("notionalAmount"), BigDecimal.class, getNotionalAmount(), this);
			processRosetta(path.newSubPath("notionalAmountSchedule"), processor, NotionalPeriod.NotionalPeriodBuilder.class, getNotionalAmountSchedule());
			processRosetta(path.newSubPath("notionalQuantitySchedule"), processor, NotionalPeriod.NotionalPeriodBuilder.class, getNotionalQuantitySchedule());
			processor.processBasic(path.newSubPath("notionalCurrency"), ISOCurrencyCodeEnum.class, getNotionalCurrency(), this);
			processor.processBasic(path.newSubPath("totalNotionalQuantity"), BigDecimal.class, getTotalNotionalQuantity(), this);
			processor.processBasic(path.newSubPath("fixedRate"), BigDecimal.class, getFixedRate(), this);
			processor.processBasic(path.newSubPath("settlementCurrency"), ISOCurrencyCodeEnum.class, getSettlementCurrency(), this);
			processRosetta(path.newSubPath("spread"), processor, PriceFormat.PriceFormatBuilder.class, getSpread());
			processor.processBasic(path.newSubPath("spreadNotation"), PriceNotationEnum.class, getSpreadNotation(), this);
			processor.processBasic(path.newSubPath("spreadCurrency"), ISOCurrencyCodeEnum.class, getSpreadCurrency(), this);
		}
		

		Leg.LegBuilder prune();
	}

	/*********************** Immutable Implementation of Leg  ***********************/
	class LegImpl implements Leg {
		private final PeriodicPayment periodicPayment;
		private final BigDecimal notionalAmount;
		private final List<? extends NotionalPeriod> notionalAmountSchedule;
		private final List<? extends NotionalPeriod> notionalQuantitySchedule;
		private final ISOCurrencyCodeEnum notionalCurrency;
		private final BigDecimal totalNotionalQuantity;
		private final BigDecimal fixedRate;
		private final ISOCurrencyCodeEnum settlementCurrency;
		private final PriceFormat spread;
		private final PriceNotationEnum spreadNotation;
		private final ISOCurrencyCodeEnum spreadCurrency;
		
		protected LegImpl(Leg.LegBuilder builder) {
			this.periodicPayment = ofNullable(builder.getPeriodicPayment()).map(f->f.build()).orElse(null);
			this.notionalAmount = builder.getNotionalAmount();
			this.notionalAmountSchedule = ofNullable(builder.getNotionalAmountSchedule()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.notionalQuantitySchedule = ofNullable(builder.getNotionalQuantitySchedule()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.notionalCurrency = builder.getNotionalCurrency();
			this.totalNotionalQuantity = builder.getTotalNotionalQuantity();
			this.fixedRate = builder.getFixedRate();
			this.settlementCurrency = builder.getSettlementCurrency();
			this.spread = ofNullable(builder.getSpread()).map(f->f.build()).orElse(null);
			this.spreadNotation = builder.getSpreadNotation();
			this.spreadCurrency = builder.getSpreadCurrency();
		}
		
		@Override
		@RosettaAttribute("periodicPayment")
		@RuneAttribute("periodicPayment")
		public PeriodicPayment getPeriodicPayment() {
			return periodicPayment;
		}
		
		@Override
		@RosettaAttribute("notionalAmount")
		@RuneAttribute("notionalAmount")
		public BigDecimal getNotionalAmount() {
			return notionalAmount;
		}
		
		@Override
		@RosettaAttribute("notionalAmountSchedule")
		@RuneAttribute("notionalAmountSchedule")
		public List<? extends NotionalPeriod> getNotionalAmountSchedule() {
			return notionalAmountSchedule;
		}
		
		@Override
		@RosettaAttribute("notionalQuantitySchedule")
		@RuneAttribute("notionalQuantitySchedule")
		public List<? extends NotionalPeriod> getNotionalQuantitySchedule() {
			return notionalQuantitySchedule;
		}
		
		@Override
		@RosettaAttribute("notionalCurrency")
		@RuneAttribute("notionalCurrency")
		public ISOCurrencyCodeEnum getNotionalCurrency() {
			return notionalCurrency;
		}
		
		@Override
		@RosettaAttribute("totalNotionalQuantity")
		@RuneAttribute("totalNotionalQuantity")
		public BigDecimal getTotalNotionalQuantity() {
			return totalNotionalQuantity;
		}
		
		@Override
		@RosettaAttribute("fixedRate")
		@RuneAttribute("fixedRate")
		public BigDecimal getFixedRate() {
			return fixedRate;
		}
		
		@Override
		@RosettaAttribute("settlementCurrency")
		@RuneAttribute("settlementCurrency")
		public ISOCurrencyCodeEnum getSettlementCurrency() {
			return settlementCurrency;
		}
		
		@Override
		@RosettaAttribute("spread")
		@RuneAttribute("spread")
		public PriceFormat getSpread() {
			return spread;
		}
		
		@Override
		@RosettaAttribute("spreadNotation")
		@RuneAttribute("spreadNotation")
		public PriceNotationEnum getSpreadNotation() {
			return spreadNotation;
		}
		
		@Override
		@RosettaAttribute("spreadCurrency")
		@RuneAttribute("spreadCurrency")
		public ISOCurrencyCodeEnum getSpreadCurrency() {
			return spreadCurrency;
		}
		
		@Override
		public Leg build() {
			return this;
		}
		
		@Override
		public Leg.LegBuilder toBuilder() {
			Leg.LegBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(Leg.LegBuilder builder) {
			ofNullable(getPeriodicPayment()).ifPresent(builder::setPeriodicPayment);
			ofNullable(getNotionalAmount()).ifPresent(builder::setNotionalAmount);
			ofNullable(getNotionalAmountSchedule()).ifPresent(builder::setNotionalAmountSchedule);
			ofNullable(getNotionalQuantitySchedule()).ifPresent(builder::setNotionalQuantitySchedule);
			ofNullable(getNotionalCurrency()).ifPresent(builder::setNotionalCurrency);
			ofNullable(getTotalNotionalQuantity()).ifPresent(builder::setTotalNotionalQuantity);
			ofNullable(getFixedRate()).ifPresent(builder::setFixedRate);
			ofNullable(getSettlementCurrency()).ifPresent(builder::setSettlementCurrency);
			ofNullable(getSpread()).ifPresent(builder::setSpread);
			ofNullable(getSpreadNotation()).ifPresent(builder::setSpreadNotation);
			ofNullable(getSpreadCurrency()).ifPresent(builder::setSpreadCurrency);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			Leg _that = getType().cast(o);
		
			if (!Objects.equals(periodicPayment, _that.getPeriodicPayment())) return false;
			if (!Objects.equals(notionalAmount, _that.getNotionalAmount())) return false;
			if (!ListEquals.listEquals(notionalAmountSchedule, _that.getNotionalAmountSchedule())) return false;
			if (!ListEquals.listEquals(notionalQuantitySchedule, _that.getNotionalQuantitySchedule())) return false;
			if (!Objects.equals(notionalCurrency, _that.getNotionalCurrency())) return false;
			if (!Objects.equals(totalNotionalQuantity, _that.getTotalNotionalQuantity())) return false;
			if (!Objects.equals(fixedRate, _that.getFixedRate())) return false;
			if (!Objects.equals(settlementCurrency, _that.getSettlementCurrency())) return false;
			if (!Objects.equals(spread, _that.getSpread())) return false;
			if (!Objects.equals(spreadNotation, _that.getSpreadNotation())) return false;
			if (!Objects.equals(spreadCurrency, _that.getSpreadCurrency())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (periodicPayment != null ? periodicPayment.hashCode() : 0);
			_result = 31 * _result + (notionalAmount != null ? notionalAmount.hashCode() : 0);
			_result = 31 * _result + (notionalAmountSchedule != null ? notionalAmountSchedule.hashCode() : 0);
			_result = 31 * _result + (notionalQuantitySchedule != null ? notionalQuantitySchedule.hashCode() : 0);
			_result = 31 * _result + (notionalCurrency != null ? notionalCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (totalNotionalQuantity != null ? totalNotionalQuantity.hashCode() : 0);
			_result = 31 * _result + (fixedRate != null ? fixedRate.hashCode() : 0);
			_result = 31 * _result + (settlementCurrency != null ? settlementCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (spread != null ? spread.hashCode() : 0);
			_result = 31 * _result + (spreadNotation != null ? spreadNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (spreadCurrency != null ? spreadCurrency.getClass().getName().hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "Leg {" +
				"periodicPayment=" + this.periodicPayment + ", " +
				"notionalAmount=" + this.notionalAmount + ", " +
				"notionalAmountSchedule=" + this.notionalAmountSchedule + ", " +
				"notionalQuantitySchedule=" + this.notionalQuantitySchedule + ", " +
				"notionalCurrency=" + this.notionalCurrency + ", " +
				"totalNotionalQuantity=" + this.totalNotionalQuantity + ", " +
				"fixedRate=" + this.fixedRate + ", " +
				"settlementCurrency=" + this.settlementCurrency + ", " +
				"spread=" + this.spread + ", " +
				"spreadNotation=" + this.spreadNotation + ", " +
				"spreadCurrency=" + this.spreadCurrency +
			'}';
		}
	}

	/*********************** Builder Implementation of Leg  ***********************/
	class LegBuilderImpl implements Leg.LegBuilder {
	
		protected PeriodicPayment.PeriodicPaymentBuilder periodicPayment;
		protected BigDecimal notionalAmount;
		protected List<NotionalPeriod.NotionalPeriodBuilder> notionalAmountSchedule = new ArrayList<>();
		protected List<NotionalPeriod.NotionalPeriodBuilder> notionalQuantitySchedule = new ArrayList<>();
		protected ISOCurrencyCodeEnum notionalCurrency;
		protected BigDecimal totalNotionalQuantity;
		protected BigDecimal fixedRate;
		protected ISOCurrencyCodeEnum settlementCurrency;
		protected PriceFormat.PriceFormatBuilder spread;
		protected PriceNotationEnum spreadNotation;
		protected ISOCurrencyCodeEnum spreadCurrency;
		
		@Override
		@RosettaAttribute("periodicPayment")
		@RuneAttribute("periodicPayment")
		public PeriodicPayment.PeriodicPaymentBuilder getPeriodicPayment() {
			return periodicPayment;
		}
		
		@Override
		public PeriodicPayment.PeriodicPaymentBuilder getOrCreatePeriodicPayment() {
			PeriodicPayment.PeriodicPaymentBuilder result;
			if (periodicPayment!=null) {
				result = periodicPayment;
			}
			else {
				result = periodicPayment = PeriodicPayment.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("notionalAmount")
		@RuneAttribute("notionalAmount")
		public BigDecimal getNotionalAmount() {
			return notionalAmount;
		}
		
		@Override
		@RosettaAttribute("notionalAmountSchedule")
		@RuneAttribute("notionalAmountSchedule")
		public List<? extends NotionalPeriod.NotionalPeriodBuilder> getNotionalAmountSchedule() {
			return notionalAmountSchedule;
		}
		
		@Override
		public NotionalPeriod.NotionalPeriodBuilder getOrCreateNotionalAmountSchedule(int _index) {
		
			if (notionalAmountSchedule==null) {
				this.notionalAmountSchedule = new ArrayList<>();
			}
			NotionalPeriod.NotionalPeriodBuilder result;
			return getIndex(notionalAmountSchedule, _index, () -> {
						NotionalPeriod.NotionalPeriodBuilder newNotionalAmountSchedule = NotionalPeriod.builder();
						return newNotionalAmountSchedule;
					});
		}
		
		@Override
		@RosettaAttribute("notionalQuantitySchedule")
		@RuneAttribute("notionalQuantitySchedule")
		public List<? extends NotionalPeriod.NotionalPeriodBuilder> getNotionalQuantitySchedule() {
			return notionalQuantitySchedule;
		}
		
		@Override
		public NotionalPeriod.NotionalPeriodBuilder getOrCreateNotionalQuantitySchedule(int _index) {
		
			if (notionalQuantitySchedule==null) {
				this.notionalQuantitySchedule = new ArrayList<>();
			}
			NotionalPeriod.NotionalPeriodBuilder result;
			return getIndex(notionalQuantitySchedule, _index, () -> {
						NotionalPeriod.NotionalPeriodBuilder newNotionalQuantitySchedule = NotionalPeriod.builder();
						return newNotionalQuantitySchedule;
					});
		}
		
		@Override
		@RosettaAttribute("notionalCurrency")
		@RuneAttribute("notionalCurrency")
		public ISOCurrencyCodeEnum getNotionalCurrency() {
			return notionalCurrency;
		}
		
		@Override
		@RosettaAttribute("totalNotionalQuantity")
		@RuneAttribute("totalNotionalQuantity")
		public BigDecimal getTotalNotionalQuantity() {
			return totalNotionalQuantity;
		}
		
		@Override
		@RosettaAttribute("fixedRate")
		@RuneAttribute("fixedRate")
		public BigDecimal getFixedRate() {
			return fixedRate;
		}
		
		@Override
		@RosettaAttribute("settlementCurrency")
		@RuneAttribute("settlementCurrency")
		public ISOCurrencyCodeEnum getSettlementCurrency() {
			return settlementCurrency;
		}
		
		@Override
		@RosettaAttribute("spread")
		@RuneAttribute("spread")
		public PriceFormat.PriceFormatBuilder getSpread() {
			return spread;
		}
		
		@Override
		public PriceFormat.PriceFormatBuilder getOrCreateSpread() {
			PriceFormat.PriceFormatBuilder result;
			if (spread!=null) {
				result = spread;
			}
			else {
				result = spread = PriceFormat.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("spreadNotation")
		@RuneAttribute("spreadNotation")
		public PriceNotationEnum getSpreadNotation() {
			return spreadNotation;
		}
		
		@Override
		@RosettaAttribute("spreadCurrency")
		@RuneAttribute("spreadCurrency")
		public ISOCurrencyCodeEnum getSpreadCurrency() {
			return spreadCurrency;
		}
		
		@Override
		@RosettaAttribute("periodicPayment")
		@RuneAttribute("periodicPayment")
		public Leg.LegBuilder setPeriodicPayment(PeriodicPayment _periodicPayment) {
			this.periodicPayment = _periodicPayment == null ? null : _periodicPayment.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("notionalAmount")
		@RuneAttribute("notionalAmount")
		public Leg.LegBuilder setNotionalAmount(BigDecimal _notionalAmount) {
			this.notionalAmount = _notionalAmount == null ? null : _notionalAmount;
			return this;
		}
		
		@Override
		@RosettaAttribute("notionalAmountSchedule")
		@RuneAttribute("notionalAmountSchedule")
		public Leg.LegBuilder addNotionalAmountSchedule(NotionalPeriod _notionalAmountSchedule) {
			if (_notionalAmountSchedule != null) {
				this.notionalAmountSchedule.add(_notionalAmountSchedule.toBuilder());
			}
			return this;
		}
		
		@Override
		public Leg.LegBuilder addNotionalAmountSchedule(NotionalPeriod _notionalAmountSchedule, int _idx) {
			getIndex(this.notionalAmountSchedule, _idx, () -> _notionalAmountSchedule.toBuilder());
			return this;
		}
		
		@Override 
		public Leg.LegBuilder addNotionalAmountSchedule(List<? extends NotionalPeriod> notionalAmountSchedules) {
			if (notionalAmountSchedules != null) {
				for (final NotionalPeriod toAdd : notionalAmountSchedules) {
					this.notionalAmountSchedule.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("notionalAmountSchedule")
		public Leg.LegBuilder setNotionalAmountSchedule(List<? extends NotionalPeriod> notionalAmountSchedules) {
			if (notionalAmountSchedules == null) {
				this.notionalAmountSchedule = new ArrayList<>();
			} else {
				this.notionalAmountSchedule = notionalAmountSchedules.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("notionalQuantitySchedule")
		@RuneAttribute("notionalQuantitySchedule")
		public Leg.LegBuilder addNotionalQuantitySchedule(NotionalPeriod _notionalQuantitySchedule) {
			if (_notionalQuantitySchedule != null) {
				this.notionalQuantitySchedule.add(_notionalQuantitySchedule.toBuilder());
			}
			return this;
		}
		
		@Override
		public Leg.LegBuilder addNotionalQuantitySchedule(NotionalPeriod _notionalQuantitySchedule, int _idx) {
			getIndex(this.notionalQuantitySchedule, _idx, () -> _notionalQuantitySchedule.toBuilder());
			return this;
		}
		
		@Override 
		public Leg.LegBuilder addNotionalQuantitySchedule(List<? extends NotionalPeriod> notionalQuantitySchedules) {
			if (notionalQuantitySchedules != null) {
				for (final NotionalPeriod toAdd : notionalQuantitySchedules) {
					this.notionalQuantitySchedule.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@Override 
		@RuneAttribute("notionalQuantitySchedule")
		public Leg.LegBuilder setNotionalQuantitySchedule(List<? extends NotionalPeriod> notionalQuantitySchedules) {
			if (notionalQuantitySchedules == null) {
				this.notionalQuantitySchedule = new ArrayList<>();
			} else {
				this.notionalQuantitySchedule = notionalQuantitySchedules.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		@RosettaAttribute("notionalCurrency")
		@RuneAttribute("notionalCurrency")
		public Leg.LegBuilder setNotionalCurrency(ISOCurrencyCodeEnum _notionalCurrency) {
			this.notionalCurrency = _notionalCurrency == null ? null : _notionalCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("totalNotionalQuantity")
		@RuneAttribute("totalNotionalQuantity")
		public Leg.LegBuilder setTotalNotionalQuantity(BigDecimal _totalNotionalQuantity) {
			this.totalNotionalQuantity = _totalNotionalQuantity == null ? null : _totalNotionalQuantity;
			return this;
		}
		
		@Override
		@RosettaAttribute("fixedRate")
		@RuneAttribute("fixedRate")
		public Leg.LegBuilder setFixedRate(BigDecimal _fixedRate) {
			this.fixedRate = _fixedRate == null ? null : _fixedRate;
			return this;
		}
		
		@Override
		@RosettaAttribute("settlementCurrency")
		@RuneAttribute("settlementCurrency")
		public Leg.LegBuilder setSettlementCurrency(ISOCurrencyCodeEnum _settlementCurrency) {
			this.settlementCurrency = _settlementCurrency == null ? null : _settlementCurrency;
			return this;
		}
		
		@Override
		@RosettaAttribute("spread")
		@RuneAttribute("spread")
		public Leg.LegBuilder setSpread(PriceFormat _spread) {
			this.spread = _spread == null ? null : _spread.toBuilder();
			return this;
		}
		
		@Override
		@RosettaAttribute("spreadNotation")
		@RuneAttribute("spreadNotation")
		public Leg.LegBuilder setSpreadNotation(PriceNotationEnum _spreadNotation) {
			this.spreadNotation = _spreadNotation == null ? null : _spreadNotation;
			return this;
		}
		
		@Override
		@RosettaAttribute("spreadCurrency")
		@RuneAttribute("spreadCurrency")
		public Leg.LegBuilder setSpreadCurrency(ISOCurrencyCodeEnum _spreadCurrency) {
			this.spreadCurrency = _spreadCurrency == null ? null : _spreadCurrency;
			return this;
		}
		
		@Override
		public Leg build() {
			return new Leg.LegImpl(this);
		}
		
		@Override
		public Leg.LegBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Leg.LegBuilder prune() {
			if (periodicPayment!=null && !periodicPayment.prune().hasData()) periodicPayment = null;
			notionalAmountSchedule = notionalAmountSchedule.stream().filter(b->b!=null).<NotionalPeriod.NotionalPeriodBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			notionalQuantitySchedule = notionalQuantitySchedule.stream().filter(b->b!=null).<NotionalPeriod.NotionalPeriodBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (spread!=null && !spread.prune().hasData()) spread = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getPeriodicPayment()!=null && getPeriodicPayment().hasData()) return true;
			if (getNotionalAmount()!=null) return true;
			if (getNotionalAmountSchedule()!=null && getNotionalAmountSchedule().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getNotionalQuantitySchedule()!=null && getNotionalQuantitySchedule().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getNotionalCurrency()!=null) return true;
			if (getTotalNotionalQuantity()!=null) return true;
			if (getFixedRate()!=null) return true;
			if (getSettlementCurrency()!=null) return true;
			if (getSpread()!=null && getSpread().hasData()) return true;
			if (getSpreadNotation()!=null) return true;
			if (getSpreadCurrency()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public Leg.LegBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			Leg.LegBuilder o = (Leg.LegBuilder) other;
			
			merger.mergeRosetta(getPeriodicPayment(), o.getPeriodicPayment(), this::setPeriodicPayment);
			merger.mergeRosetta(getNotionalAmountSchedule(), o.getNotionalAmountSchedule(), this::getOrCreateNotionalAmountSchedule);
			merger.mergeRosetta(getNotionalQuantitySchedule(), o.getNotionalQuantitySchedule(), this::getOrCreateNotionalQuantitySchedule);
			merger.mergeRosetta(getSpread(), o.getSpread(), this::setSpread);
			
			merger.mergeBasic(getNotionalAmount(), o.getNotionalAmount(), this::setNotionalAmount);
			merger.mergeBasic(getNotionalCurrency(), o.getNotionalCurrency(), this::setNotionalCurrency);
			merger.mergeBasic(getTotalNotionalQuantity(), o.getTotalNotionalQuantity(), this::setTotalNotionalQuantity);
			merger.mergeBasic(getFixedRate(), o.getFixedRate(), this::setFixedRate);
			merger.mergeBasic(getSettlementCurrency(), o.getSettlementCurrency(), this::setSettlementCurrency);
			merger.mergeBasic(getSpreadNotation(), o.getSpreadNotation(), this::setSpreadNotation);
			merger.mergeBasic(getSpreadCurrency(), o.getSpreadCurrency(), this::setSpreadCurrency);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			Leg _that = getType().cast(o);
		
			if (!Objects.equals(periodicPayment, _that.getPeriodicPayment())) return false;
			if (!Objects.equals(notionalAmount, _that.getNotionalAmount())) return false;
			if (!ListEquals.listEquals(notionalAmountSchedule, _that.getNotionalAmountSchedule())) return false;
			if (!ListEquals.listEquals(notionalQuantitySchedule, _that.getNotionalQuantitySchedule())) return false;
			if (!Objects.equals(notionalCurrency, _that.getNotionalCurrency())) return false;
			if (!Objects.equals(totalNotionalQuantity, _that.getTotalNotionalQuantity())) return false;
			if (!Objects.equals(fixedRate, _that.getFixedRate())) return false;
			if (!Objects.equals(settlementCurrency, _that.getSettlementCurrency())) return false;
			if (!Objects.equals(spread, _that.getSpread())) return false;
			if (!Objects.equals(spreadNotation, _that.getSpreadNotation())) return false;
			if (!Objects.equals(spreadCurrency, _that.getSpreadCurrency())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (periodicPayment != null ? periodicPayment.hashCode() : 0);
			_result = 31 * _result + (notionalAmount != null ? notionalAmount.hashCode() : 0);
			_result = 31 * _result + (notionalAmountSchedule != null ? notionalAmountSchedule.hashCode() : 0);
			_result = 31 * _result + (notionalQuantitySchedule != null ? notionalQuantitySchedule.hashCode() : 0);
			_result = 31 * _result + (notionalCurrency != null ? notionalCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (totalNotionalQuantity != null ? totalNotionalQuantity.hashCode() : 0);
			_result = 31 * _result + (fixedRate != null ? fixedRate.hashCode() : 0);
			_result = 31 * _result + (settlementCurrency != null ? settlementCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (spread != null ? spread.hashCode() : 0);
			_result = 31 * _result + (spreadNotation != null ? spreadNotation.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (spreadCurrency != null ? spreadCurrency.getClass().getName().hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "LegBuilder {" +
				"periodicPayment=" + this.periodicPayment + ", " +
				"notionalAmount=" + this.notionalAmount + ", " +
				"notionalAmountSchedule=" + this.notionalAmountSchedule + ", " +
				"notionalQuantitySchedule=" + this.notionalQuantitySchedule + ", " +
				"notionalCurrency=" + this.notionalCurrency + ", " +
				"totalNotionalQuantity=" + this.totalNotionalQuantity + ", " +
				"fixedRate=" + this.fixedRate + ", " +
				"settlementCurrency=" + this.settlementCurrency + ", " +
				"spread=" + this.spread + ", " +
				"spreadNotation=" + this.spreadNotation + ", " +
				"spreadCurrency=" + this.spreadCurrency +
			'}';
		}
	}
}
