package drr.regulation.common.trade.functions;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperS;
import drr.regulation.common.trade.CommonLeg;
import drr.regulation.common.trade.payment.functions.PeriodicPaymentToCommonPeriodicPayment;
import drr.standards.iosco.cde.base.payment.PeriodicPayment;
import drr.standards.iosco.cde.base.price.PriceFormat;
import drr.standards.iosco.cde.base.price.PriceNotationEnum;
import drr.standards.iosco.cde.base.quantity.NotionalPeriod;
import drr.standards.iosco.cde.version3.Leg;
import drr.standards.iso.Direction2Enum;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;


@ImplementedBy(LegToCommonLeg.LegToCommonLegDefault.class)
public abstract class LegToCommonLeg implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected PeriodicPaymentToCommonPeriodicPayment periodicPaymentToCommonPeriodicPayment;

	/**
	* @param leg 
	* @return commonLeg 
	*/
	public CommonLeg evaluate(Leg leg) {
		CommonLeg.CommonLegBuilder commonLegBuilder = doEvaluate(leg);
		
		final CommonLeg commonLeg;
		if (commonLegBuilder == null) {
			commonLeg = null;
		} else {
			commonLeg = commonLegBuilder.build();
			objectValidator.validate(CommonLeg.class, commonLeg);
		}
		
		return commonLeg;
	}

	protected abstract CommonLeg.CommonLegBuilder doEvaluate(Leg leg);

	public static class LegToCommonLegDefault extends LegToCommonLeg {
		@Override
		protected CommonLeg.CommonLegBuilder doEvaluate(Leg leg) {
			CommonLeg.CommonLegBuilder commonLeg = CommonLeg.builder();
			return assignOutput(commonLeg, leg);
		}
		
		protected CommonLeg.CommonLegBuilder assignOutput(CommonLeg.CommonLegBuilder commonLeg, Leg leg) {
			commonLeg = toBuilder(CommonLeg.builder()
				.setPeriodicPayment(periodicPaymentToCommonPeriodicPayment.evaluate(MapperS.of(leg).<PeriodicPayment>map("getPeriodicPayment", _leg -> _leg.getPeriodicPayment()).get()))
				.setFixedRate(MapperS.of(leg).<BigDecimal>map("getFixedRate", _leg -> _leg.getFixedRate()).get())
				.setNotionalAmount(MapperS.of(leg).<BigDecimal>map("getNotionalAmount", _leg -> _leg.getNotionalAmount()).get())
				.setNotionalCurrency(MapperS.of(leg).<ISOCurrencyCodeEnum>map("getNotionalCurrency", _leg -> _leg.getNotionalCurrency()).get())
				.setNotionalAmountSchedule(MapperS.of(leg).<NotionalPeriod>mapC("getNotionalAmountSchedule", _leg -> _leg.getNotionalAmountSchedule()).getMulti())
				.setTotalNotionalQuantity(MapperS.of(leg).<BigDecimal>map("getTotalNotionalQuantity", _leg -> _leg.getTotalNotionalQuantity()).get())
				.setNotionalQuantitySchedule(MapperS.of(leg).<NotionalPeriod>mapC("getNotionalQuantitySchedule", _leg -> _leg.getNotionalQuantitySchedule()).getMulti())
				.setSettlementCurrency(MapperS.of(leg).<ISOCurrencyCodeEnum>map("getSettlementCurrency", _leg -> _leg.getSettlementCurrency()).get())
				.setSpread(MapperS.of(leg).<PriceFormat>map("getSpread", _leg -> _leg.getSpread()).get())
				.setSpreadNotation(MapperS.of(leg).<PriceNotationEnum>map("getSpreadNotation", _leg -> _leg.getSpreadNotation()).get())
				.setSpreadCurrency(MapperS.of(leg).<ISOCurrencyCodeEnum>map("getSpreadCurrency", _leg -> _leg.getSpreadCurrency()).get())
				.setQuantityUnitOfMeasure(MapperS.of(leg).<String>map("getQuantityUnitOfMeasure", _leg -> _leg.getQuantityUnitOfMeasure()).get())
				.setDirection2(MapperS.of(leg).<Direction2Enum>map("getDirection2", _leg -> _leg.getDirection2()).get())
				.build());
			
			return Optional.ofNullable(commonLeg)
				.map(o -> o.prune())
				.orElse(null);
		}
	}
}
