package drr.regulation.common.trade.basket.functions;

import cdm.base.math.NonNegativeQuantitySchedule;
import cdm.base.staticdata.asset.common.Index;
import cdm.base.staticdata.asset.common.Loan;
import cdm.base.staticdata.asset.common.Security;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaCommodity;
import cdm.event.common.Trade;
import cdm.product.asset.BasketReferenceInformation;
import cdm.product.asset.CommodityPayout;
import cdm.product.asset.CreditDefaultPayout;
import cdm.product.asset.GeneralTerms;
import cdm.product.asset.ReferenceObligation;
import cdm.product.asset.ReferencePair;
import cdm.product.asset.ReferencePool;
import cdm.product.asset.ReferencePoolItem;
import cdm.product.template.Basket;
import cdm.product.template.BasketConstituent;
import cdm.product.template.Payout;
import cdm.product.template.PerformancePayout;
import cdm.product.template.Product;
import cdm.product.template.TradableProduct;
import cdm.product.template.TradeLot;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import drr.regulation.common.functions.EconomicTermsForProduct;
import drr.regulation.common.functions.IsSingleCommodityPayoutProduct;
import drr.regulation.common.functions.ProductForTrade;
import drr.regulation.common.functions.UnderlierForProduct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GetBasketConstituents.GetBasketConstituentsDefault.class)
public abstract class GetBasketConstituents implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected EconomicTermsForProduct economicTermsForProduct;
	@Inject protected GetQuantityForConstituent getQuantityForConstituent;
	@Inject protected GetQuantityForReferencePoolItem getQuantityForReferencePoolItem;
	@Inject protected IsSingleCommodityPayoutProduct isSingleCommodityPayoutProduct;
	@Inject protected ProductForTrade productForTrade;
	@Inject protected UnderlierForProduct underlierForProduct;

	/**
	* @param trade 
	* @return result Single product identifier per basket constituent.
	*/
	public List<? extends BasketConstituent> evaluate(Trade trade) {
		List<BasketConstituent.BasketConstituentBuilder> resultBuilder = doEvaluate(trade);
		
		final List<? extends BasketConstituent> result;
		if (resultBuilder == null) {
			result = null;
		} else {
			result = resultBuilder.stream().map(BasketConstituent::build).collect(Collectors.toList());
			objectValidator.validate(BasketConstituent.class, result);
		}
		
		return result;
	}

	protected abstract List<BasketConstituent.BasketConstituentBuilder> doEvaluate(Trade trade);

	protected abstract MapperS<? extends Product> product(Trade trade);

	protected abstract MapperC<? extends TradeLot> tradeLots(Trade trade);

	protected abstract MapperC<? extends Product> basketConstituents(Trade trade);

	protected abstract MapperC<? extends ReferencePoolItem> referencePoolItems(Trade trade);

	public static class GetBasketConstituentsDefault extends GetBasketConstituents {
		@Override
		protected List<BasketConstituent.BasketConstituentBuilder> doEvaluate(Trade trade) {
			List<BasketConstituent.BasketConstituentBuilder> result = new ArrayList<>();
			return assignOutput(result, trade);
		}
		
		protected List<BasketConstituent.BasketConstituentBuilder> assignOutput(List<BasketConstituent.BasketConstituentBuilder> result, Trade trade) {
			final MapperC<? extends Product> thenArg0 = basketConstituents(trade);
			result.addAll(toBuilder(thenArg0
				.mapItem(item -> {
					final NonNegativeQuantitySchedule nonNegativeQuantitySchedule = getQuantityForConstituent.evaluate(item.get(), tradeLots(trade).getMulti());
					return MapperS.of(BasketConstituent.builder()
						.setQuantityValue((nonNegativeQuantitySchedule == null ? Collections.<NonNegativeQuantitySchedule>emptyList() : Collections.singletonList(nonNegativeQuantitySchedule)))
						.setSecurity(item.<Security>map("getSecurity", _product -> _product.getSecurity()).get())
						.setLoan(item.<Loan>map("getLoan", _product -> _product.getLoan()).get())
						.setCommodity(item.<ReferenceWithMetaCommodity>map("getCommodity", _product -> _product.getCommodity()).get())
						.setIndex(item.<Index>map("getIndex", _product -> _product.getIndex()).get())
						.build());
				}).getMulti()));
			
			final MapperC<? extends ReferencePoolItem> thenArg1 = referencePoolItems(trade);
			result.addAll(toBuilder(thenArg1
				.mapItem(item -> {
					final NonNegativeQuantitySchedule nonNegativeQuantitySchedule = getQuantityForReferencePoolItem.evaluate(item.get());
					return MapperS.of(BasketConstituent.builder()
						.setQuantityValue((nonNegativeQuantitySchedule == null ? Collections.<NonNegativeQuantitySchedule>emptyList() : Collections.singletonList(nonNegativeQuantitySchedule)))
						.setSecurity(item.<ReferencePair>map("getReferencePair", referencePoolItem -> referencePoolItem.getReferencePair()).<ReferenceObligation>map("getReferenceObligation", referencePair -> referencePair.getReferenceObligation()).<Security>map("getSecurity", referenceObligation -> referenceObligation.getSecurity()).get())
						.setLoan(item.<ReferencePair>map("getReferencePair", referencePoolItem -> referencePoolItem.getReferencePair()).<ReferenceObligation>map("getReferenceObligation", referencePair -> referencePair.getReferenceObligation()).<Loan>map("getLoan", referenceObligation -> referenceObligation.getLoan()).get())
						.build());
				}).getMulti()));
			
			return Optional.ofNullable(result)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
		
		@Override
		protected MapperS<? extends Product> product(Trade trade) {
			return MapperS.of(productForTrade.evaluate(trade));
		}
		
		@Override
		protected MapperC<? extends TradeLot> tradeLots(Trade trade) {
			return MapperS.of(trade).<TradableProduct>map("getTradableProduct", _trade -> _trade.getTradableProduct()).<TradeLot>mapC("getTradeLot", tradableProduct -> tradableProduct.getTradeLot());
		}
		
		@Override
		protected MapperC<? extends Product> basketConstituents(Trade trade) {
			if (exists(MapperS.of(underlierForProduct.evaluate(product(trade).get())).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).getOrDefault(false)) {
				return MapperS.of(underlierForProduct.evaluate(product(trade).get())).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<PerformancePayout>mapC("getPerformancePayout", payout -> payout.getPerformancePayout()).<Product>map("getUnderlier", performancePayout -> performancePayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<PerformancePayout>mapC("getPerformancePayout", payout -> payout.getPerformancePayout()).<Product>map("getUnderlier", performancePayout -> performancePayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CommodityPayout>mapC("getCommodityPayout", payout -> payout.getCommodityPayout()).<Product>map("getUnderlier", commodityPayout -> commodityPayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent())).and(areEqual(MapperS.of(isSingleCommodityPayoutProduct.evaluate(product(trade).get())), MapperS.of(false), CardinalityOperator.All)).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CommodityPayout>mapC("getCommodityPayout", payout -> payout.getCommodityPayout()).<Product>map("getUnderlier", commodityPayout -> commodityPayout.getUnderlier()).<Basket>map("getBasket", _product -> _product.getBasket()).<Product>mapC("getBasketConstituent", basket -> basket.getBasketConstituent());
			}
			return MapperC.<Product>ofNull();
		}
		
		@Override
		protected MapperC<? extends ReferencePoolItem> referencePoolItems(Trade trade) {
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<BasketReferenceInformation>map("getBasketReferenceInformation", generalTerms -> generalTerms.getBasketReferenceInformation()).<ReferencePool>map("getReferencePool", basketReferenceInformation -> basketReferenceInformation.getReferencePool()).<ReferencePoolItem>mapC("getReferencePoolItem", referencePool -> referencePool.getReferencePoolItem()).<ReferencePair>map("getReferencePair", referencePoolItem -> referencePoolItem.getReferencePair()).<ReferenceObligation>map("getReferenceObligation", referencePair -> referencePair.getReferenceObligation())).getOrDefault(false)) {
				return MapperS.of(economicTermsForProduct.evaluate(product(trade).get())).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<BasketReferenceInformation>map("getBasketReferenceInformation", generalTerms -> generalTerms.getBasketReferenceInformation()).<ReferencePool>map("getReferencePool", basketReferenceInformation -> basketReferenceInformation.getReferencePool()).<ReferencePoolItem>mapC("getReferencePoolItem", referencePool -> referencePool.getReferencePoolItem());
			}
			return MapperC.<ReferencePoolItem>ofNull();
		}
	}
}
