package drr.regulation.common.functions;

import cdm.base.staticdata.asset.common.ProductIdTypeEnum;
import cdm.base.staticdata.asset.common.ProductIdentifier;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(ExtractProductIdentifierBySource.ExtractProductIdentifierBySourceDefault.class)
public abstract class ExtractProductIdentifierBySource implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;

	/**
	* @param productIdentifiers 
	* @return productIdentifier 
	*/
	public ProductIdentifier evaluate(List<? extends ProductIdentifier> productIdentifiers) {
		ProductIdentifier.ProductIdentifierBuilder productIdentifierBuilder = doEvaluate(productIdentifiers);
		
		final ProductIdentifier productIdentifier;
		if (productIdentifierBuilder == null) {
			productIdentifier = null;
		} else {
			productIdentifier = productIdentifierBuilder.build();
			objectValidator.validate(ProductIdentifier.class, productIdentifier);
		}
		
		return productIdentifier;
	}

	protected abstract ProductIdentifier.ProductIdentifierBuilder doEvaluate(List<? extends ProductIdentifier> productIdentifiers);

	protected abstract MapperC<ProductIdTypeEnum> orderedSources(List<? extends ProductIdentifier> productIdentifiers);

	public static class ExtractProductIdentifierBySourceDefault extends ExtractProductIdentifierBySource {
		@Override
		protected ProductIdentifier.ProductIdentifierBuilder doEvaluate(List<? extends ProductIdentifier> productIdentifiers) {
			if (productIdentifiers == null) {
				productIdentifiers = Collections.emptyList();
			}
			ProductIdentifier.ProductIdentifierBuilder productIdentifier = ProductIdentifier.builder();
			return assignOutput(productIdentifier, productIdentifiers);
		}
		
		protected ProductIdentifier.ProductIdentifierBuilder assignOutput(ProductIdentifier.ProductIdentifierBuilder productIdentifier, List<? extends ProductIdentifier> productIdentifiers) {
			final MapperC<ProductIdentifier> thenArg = orderedSources(productIdentifiers)
				.mapItem(src -> {
					final MapperC<ProductIdentifier> _thenArg = MapperC.<ProductIdentifier>of(productIdentifiers)
						.filterItemNullSafe(item -> areEqual(item.<ProductIdTypeEnum>map("getSource", _productIdentifier -> _productIdentifier.getSource()), src, CardinalityOperator.All).get());
					return _thenArg
						.first();
				});
			productIdentifier = toBuilder(MapperS.of(thenArg
				.first().getOrDefault(MapperC.<ProductIdentifier>of(productIdentifiers)
				.first().get())).get());
			
			return Optional.ofNullable(productIdentifier)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected MapperC<ProductIdTypeEnum> orderedSources(List<? extends ProductIdentifier> productIdentifiers) {
			return MapperC.<ProductIdTypeEnum>of(MapperS.of(ProductIdTypeEnum.REDID), MapperS.of(ProductIdTypeEnum.CUSIP), MapperS.of(ProductIdTypeEnum.SEDOL), MapperS.of(ProductIdTypeEnum.RIC), MapperS.of(ProductIdTypeEnum.SICOVAM), MapperS.of(ProductIdTypeEnum.FIGI), MapperS.of(ProductIdTypeEnum.BBGTICKER), MapperS.of(ProductIdTypeEnum.NAME));
		}
	}
}
