package drr.regulation.common.validation.datarule;

import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.annotations.RosettaDataRule;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.validation.ValidationResult;
import com.rosetta.model.lib.validation.Validator;
import drr.regulation.common.RegimeNameEnum;
import drr.regulation.common.ReportingRegime;
import drr.regulation.common.ReportingRoleEnum;
import drr.regulation.common.metafields.FieldWithMetaRegimeNameEnum;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.rosetta.model.lib.expression.ExpressionOperatorsNullSafe.*;

/**
 * @version 6.21.2
 */
@RosettaDataRule("ReportingRegimeTradingCapacity")
@ImplementedBy(ReportingRegimeTradingCapacity.Default.class)
public interface ReportingRegimeTradingCapacity extends Validator<ReportingRegime> {
	
	String NAME = "ReportingRegimeTradingCapacity";
	String DEFINITION = "if regimeName = RegimeNameEnum -> MAS and reportingRole = ReportingRoleEnum -> ReportingParty then tradingCapacity = ReportingRoleEnum -> Agent or tradingCapacity = ReportingRoleEnum -> Principal";
	
	class Default implements ReportingRegimeTradingCapacity {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, ReportingRegime reportingRegime) {
			ComparisonResult result = executeDataRule(reportingRegime);
			if (result.getOrDefault(true)) {
				return Arrays.asList(ValidationResult.success(NAME, ValidationResult.ValidationType.DATA_RULE, "ReportingRegime", path, DEFINITION));
			}
			
			String failureMessage = result.getError();
			if (failureMessage == null || failureMessage.contains("Null") || failureMessage == "") {
				failureMessage = "Condition has failed.";
			}
			return Arrays.asList(ValidationResult.failure(NAME, ValidationResult.ValidationType.DATA_RULE, "ReportingRegime", path, DEFINITION, failureMessage));
		}
		
		private ComparisonResult executeDataRule(ReportingRegime reportingRegime) {
			try {
				if (areEqual(MapperS.of(reportingRegime).<FieldWithMetaRegimeNameEnum>map("getRegimeName", _reportingRegime -> _reportingRegime.getRegimeName()).<RegimeNameEnum>map("Type coercion", fieldWithMetaRegimeNameEnum -> fieldWithMetaRegimeNameEnum == null ? null : fieldWithMetaRegimeNameEnum.getValue()), MapperS.of(RegimeNameEnum.MAS), CardinalityOperator.All).andNullSafe(areEqual(MapperS.of(reportingRegime).<ReportingRoleEnum>map("getReportingRole", _reportingRegime -> _reportingRegime.getReportingRole()), MapperS.of(ReportingRoleEnum.REPORTING_PARTY), CardinalityOperator.All)).getOrDefault(false)) {
					return areEqual(MapperS.of(reportingRegime).<ReportingRoleEnum>map("getTradingCapacity", _reportingRegime -> _reportingRegime.getTradingCapacity()), MapperS.of(ReportingRoleEnum.AGENT), CardinalityOperator.All).orNullSafe(areEqual(MapperS.of(reportingRegime).<ReportingRoleEnum>map("getTradingCapacity", _reportingRegime -> _reportingRegime.getTradingCapacity()), MapperS.of(ReportingRoleEnum.PRINCIPAL), CardinalityOperator.All));
				}
				return ComparisonResult.ofEmpty();
			}
			catch (Exception ex) {
				return ComparisonResult.failure(ex.getMessage());
			}
		}
	}
	
	@SuppressWarnings("unused")
	class NoOp implements ReportingRegimeTradingCapacity {
	
		@Override
		public List<ValidationResult<?>> getValidationResults(RosettaPath path, ReportingRegime reportingRegime) {
			return Collections.emptyList();
		}
	}
}
