package drr.regulation.common;

import cdm.base.staticdata.asset.common.ProductIdentifier;
import cdm.base.staticdata.asset.common.metafields.ReferenceWithMetaProductIdentifier;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneScopedAttributeReference;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.util.ListEquals;
import drr.regulation.common.meta.ReportablePriceSourceMeta;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Provides the extraction of the Underlying Asset Price Source and correlate it with the identification of the trade products.
 * @version 6.25.3
 */
@RosettaDataType(value="ReportablePriceSource", builder=ReportablePriceSource.ReportablePriceSourceBuilderImpl.class, version="6.25.3")
@RuneDataType(value="ReportablePriceSource", model="drr", builder=ReportablePriceSource.ReportablePriceSourceBuilderImpl.class, version="6.25.3")
public interface ReportablePriceSource extends RosettaModelObject {

	ReportablePriceSourceMeta metaData = new ReportablePriceSourceMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Comprises an identifier and a source. The associated metadata key denotes the ability to associate a hash value to the ProductIdentifier instantiations for the purpose of model cross-referencing, in support of functionality such as the event effect and the lineage.
	 */
	List<? extends ReferenceWithMetaProductIdentifier> getProductIdentifier();
	/**
	 * Enables the identification of trade products with a specific Underlier ID (Other) that do not contain a productIdentifier and are, therefore, represented by a string (e.g., indexName).
	 */
	String getOtherId();
	/**
	 * For an underlying asset or benchmark not traded on a platform, the source of the price used to determine the value or level of the asset or benchmark.
	 */
	String getUnderlyingAssetPriceSource();

	/*********************** Build Methods  ***********************/
	ReportablePriceSource build();
	
	ReportablePriceSource.ReportablePriceSourceBuilder toBuilder();
	
	static ReportablePriceSource.ReportablePriceSourceBuilder builder() {
		return new ReportablePriceSource.ReportablePriceSourceBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportablePriceSource> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportablePriceSource> getType() {
		return ReportablePriceSource.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.class, getProductIdentifier());
		processor.processBasic(path.newSubPath("otherId"), String.class, getOtherId(), this);
		processor.processBasic(path.newSubPath("underlyingAssetPriceSource"), String.class, getUnderlyingAssetPriceSource(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportablePriceSourceBuilder extends ReportablePriceSource, RosettaModelObjectBuilder {
		ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder getOrCreateProductIdentifier(int index);
		@Override
		List<? extends ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder> getProductIdentifier();
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier productIdentifier, int idx);
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(ProductIdentifier productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(ProductIdentifier productIdentifier, int idx);
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifier);
		ReportablePriceSource.ReportablePriceSourceBuilder setOtherId(String otherId);
		ReportablePriceSource.ReportablePriceSourceBuilder setUnderlyingAssetPriceSource(String underlyingAssetPriceSource);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("productIdentifier"), processor, ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder.class, getProductIdentifier());
			processor.processBasic(path.newSubPath("otherId"), String.class, getOtherId(), this);
			processor.processBasic(path.newSubPath("underlyingAssetPriceSource"), String.class, getUnderlyingAssetPriceSource(), this);
		}
		

		ReportablePriceSource.ReportablePriceSourceBuilder prune();
	}

	/*********************** Immutable Implementation of ReportablePriceSource  ***********************/
	class ReportablePriceSourceImpl implements ReportablePriceSource {
		private final List<? extends ReferenceWithMetaProductIdentifier> productIdentifier;
		private final String otherId;
		private final String underlyingAssetPriceSource;
		
		protected ReportablePriceSourceImpl(ReportablePriceSource.ReportablePriceSourceBuilder builder) {
			this.productIdentifier = ofNullable(builder.getProductIdentifier()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.otherId = builder.getOtherId();
			this.underlyingAssetPriceSource = builder.getUnderlyingAssetPriceSource();
		}
		
		@Override
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public List<? extends ReferenceWithMetaProductIdentifier> getProductIdentifier() {
			return productIdentifier;
		}
		
		@Override
		@RosettaAttribute("otherId")
		@RuneAttribute("otherId")
		public String getOtherId() {
			return otherId;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		public String getUnderlyingAssetPriceSource() {
			return underlyingAssetPriceSource;
		}
		
		@Override
		public ReportablePriceSource build() {
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder toBuilder() {
			ReportablePriceSource.ReportablePriceSourceBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportablePriceSource.ReportablePriceSourceBuilder builder) {
			ofNullable(getProductIdentifier()).ifPresent(builder::setProductIdentifier);
			ofNullable(getOtherId()).ifPresent(builder::setOtherId);
			ofNullable(getUnderlyingAssetPriceSource()).ifPresent(builder::setUnderlyingAssetPriceSource);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportablePriceSource _that = getType().cast(o);
		
			if (!ListEquals.listEquals(productIdentifier, _that.getProductIdentifier())) return false;
			if (!Objects.equals(otherId, _that.getOtherId())) return false;
			if (!Objects.equals(underlyingAssetPriceSource, _that.getUnderlyingAssetPriceSource())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (productIdentifier != null ? productIdentifier.hashCode() : 0);
			_result = 31 * _result + (otherId != null ? otherId.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetPriceSource != null ? underlyingAssetPriceSource.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportablePriceSource {" +
				"productIdentifier=" + this.productIdentifier + ", " +
				"otherId=" + this.otherId + ", " +
				"underlyingAssetPriceSource=" + this.underlyingAssetPriceSource +
			'}';
		}
	}

	/*********************** Builder Implementation of ReportablePriceSource  ***********************/
	class ReportablePriceSourceBuilderImpl implements ReportablePriceSource.ReportablePriceSourceBuilder {
	
		protected List<ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder> productIdentifier = new ArrayList<>();
		protected String otherId;
		protected String underlyingAssetPriceSource;
		
		@Override
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		public List<? extends ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder> getProductIdentifier() {
			return productIdentifier;
		}
		
		@Override
		public ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder getOrCreateProductIdentifier(int index) {
			if (productIdentifier==null) {
				this.productIdentifier = new ArrayList<>();
			}
			return getIndex(productIdentifier, index, () -> {
						ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder newProductIdentifier = ReferenceWithMetaProductIdentifier.builder();
						return newProductIdentifier;
					});
		}
		
		@Override
		@RosettaAttribute("otherId")
		@RuneAttribute("otherId")
		public String getOtherId() {
			return otherId;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		public String getUnderlyingAssetPriceSource() {
			return underlyingAssetPriceSource;
		}
		
		@RosettaAttribute("productIdentifier")
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier) {
			if (_productIdentifier != null) {
				this.productIdentifier.add(_productIdentifier.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(ReferenceWithMetaProductIdentifier _productIdentifier, int idx) {
			getIndex(this.productIdentifier, idx, () -> _productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier) {
			this.getOrCreateProductIdentifier(-1).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(ProductIdentifier _productIdentifier, int idx) {
			this.getOrCreateProductIdentifier(idx).setValue(_productIdentifier.toBuilder());
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ReferenceWithMetaProductIdentifier toAdd : productIdentifiers) {
					this.productIdentifier.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("productIdentifier")
		@RuneScopedAttributeReference
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder setProductIdentifier(List<? extends ReferenceWithMetaProductIdentifier> productIdentifiers) {
			if (productIdentifiers == null) {
				this.productIdentifier = new ArrayList<>();
			} else {
				this.productIdentifier = productIdentifiers.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder addProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			if (productIdentifiers != null) {
				for (final ProductIdentifier toAdd : productIdentifiers) {
					this.addProductIdentifierValue(toAdd);
				}
			}
			return this;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder setProductIdentifierValue(List<? extends ProductIdentifier> productIdentifiers) {
			this.productIdentifier.clear();
			if (productIdentifiers != null) {
				productIdentifiers.forEach(this::addProductIdentifierValue);
			}
			return this;
		}
		
		@RosettaAttribute("otherId")
		@RuneAttribute("otherId")
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder setOtherId(String _otherId) {
			this.otherId = _otherId == null ? null : _otherId;
			return this;
		}
		
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder setUnderlyingAssetPriceSource(String _underlyingAssetPriceSource) {
			this.underlyingAssetPriceSource = _underlyingAssetPriceSource == null ? null : _underlyingAssetPriceSource;
			return this;
		}
		
		@Override
		public ReportablePriceSource build() {
			return new ReportablePriceSource.ReportablePriceSourceImpl(this);
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder prune() {
			productIdentifier = productIdentifier.stream().filter(b->b!=null).<ReferenceWithMetaProductIdentifier.ReferenceWithMetaProductIdentifierBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getProductIdentifier()!=null && getProductIdentifier().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getOtherId()!=null) return true;
			if (getUnderlyingAssetPriceSource()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			ReportablePriceSource.ReportablePriceSourceBuilder o = (ReportablePriceSource.ReportablePriceSourceBuilder) other;
			
			merger.mergeRosetta(getProductIdentifier(), o.getProductIdentifier(), this::getOrCreateProductIdentifier);
			
			merger.mergeBasic(getOtherId(), o.getOtherId(), this::setOtherId);
			merger.mergeBasic(getUnderlyingAssetPriceSource(), o.getUnderlyingAssetPriceSource(), this::setUnderlyingAssetPriceSource);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportablePriceSource _that = getType().cast(o);
		
			if (!ListEquals.listEquals(productIdentifier, _that.getProductIdentifier())) return false;
			if (!Objects.equals(otherId, _that.getOtherId())) return false;
			if (!Objects.equals(underlyingAssetPriceSource, _that.getUnderlyingAssetPriceSource())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (productIdentifier != null ? productIdentifier.hashCode() : 0);
			_result = 31 * _result + (otherId != null ? otherId.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetPriceSource != null ? underlyingAssetPriceSource.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportablePriceSourceBuilder {" +
				"productIdentifier=" + this.productIdentifier + ", " +
				"otherId=" + this.otherId + ", " +
				"underlyingAssetPriceSource=" + this.underlyingAssetPriceSource +
			'}';
		}
	}
}
