package drr.enrichment.upi.functions;

import cdm.base.staticdata.asset.common.DebtEconomics;
import cdm.base.staticdata.asset.common.DebtSeniorityEnum;
import cdm.base.staticdata.asset.common.DebtType;
import cdm.base.staticdata.asset.common.Security;
import cdm.product.asset.CreditDefaultPayout;
import cdm.product.asset.GeneralTerms;
import cdm.product.asset.ReferenceInformation;
import cdm.product.asset.ReferenceObligation;
import cdm.product.template.ContractualProduct;
import cdm.product.template.EconomicTerms;
import cdm.product.template.Payout;
import cdm.product.template.PerformancePayout;
import cdm.product.template.Product;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import drr.base.qualification.product.functions.IsCredit;
import drr.base.qualification.product.functions.IsCreditSwaption;
import drr.base.qualification.product.functions.IsTotalReturnSwapDebtUnderlier;
import drr.base.trade.functions.EconomicTermsForProduct;
import drr.base.trade.underlier.functions.UnderlierForProduct;
import drr.standards.iosco.upi.AnnaDsbDebtSeniorityEnum;
import drr.standards.iosco.upi.AnnaDsbUseCaseEnum;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperatorsNullSafe.*;

@ImplementedBy(DebtSeniority.DebtSeniorityDefault.class)
public abstract class DebtSeniority implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected EconomicTermsForProduct economicTermsForProduct;
	@Inject protected IsCredit isCredit;
	@Inject protected IsCreditSwaption isCreditSwaption;
	@Inject protected IsTotalReturnSwapDebtUnderlier isTotalReturnSwapDebtUnderlier;
	@Inject protected UnderlierForProduct underlierForProduct;

	/**
	* @param product 
	* @param useCase 
	* @return result 
	*/
	public AnnaDsbDebtSeniorityEnum evaluate(Product product, AnnaDsbUseCaseEnum useCase) {
		AnnaDsbDebtSeniorityEnum result = doEvaluate(product, useCase);
		
		return result;
	}

	protected abstract AnnaDsbDebtSeniorityEnum doEvaluate(Product product, AnnaDsbUseCaseEnum useCase);

	protected abstract MapperS<? extends EconomicTerms> economicTerms(Product product, AnnaDsbUseCaseEnum useCase);

	protected abstract MapperS<? extends ReferenceInformation> referenceInformation(Product product, AnnaDsbUseCaseEnum useCase);

	protected abstract MapperC<DebtSeniorityEnum> debtSeniority(Product product, AnnaDsbUseCaseEnum useCase);

	public static class DebtSeniorityDefault extends DebtSeniority {
		@Override
		protected AnnaDsbDebtSeniorityEnum doEvaluate(Product product, AnnaDsbUseCaseEnum useCase) {
			AnnaDsbDebtSeniorityEnum result = null;
			return assignOutput(result, product, useCase);
		}
		
		protected AnnaDsbDebtSeniorityEnum assignOutput(AnnaDsbDebtSeniorityEnum result, Product product, AnnaDsbUseCaseEnum useCase) {
			if (areEqual(debtSeniority(product, useCase), MapperS.of(DebtSeniorityEnum.SENIOR), CardinalityOperator.Any).getOrDefault(false)) {
				result = AnnaDsbDebtSeniorityEnum.SNDB;
			} else if (areEqual(debtSeniority(product, useCase), MapperS.of(DebtSeniorityEnum.SUBORDINATED), CardinalityOperator.Any).getOrDefault(false)) {
				result = AnnaDsbDebtSeniorityEnum.SBOD;
			} else if (areEqual(debtSeniority(product, useCase), MapperS.of(DebtSeniorityEnum.SECURED), CardinalityOperator.Any).getOrDefault(false)) {
				result = AnnaDsbDebtSeniorityEnum.MZZD;
			} else if (areEqual(MapperS.of(useCase), MapperS.of(AnnaDsbUseCaseEnum.NON_STANDARD), CardinalityOperator.All).orNullSafe(areEqual(MapperS.of(useCase), MapperS.of(AnnaDsbUseCaseEnum.TOTAL_RETURN_SWAP), CardinalityOperator.All)).getOrDefault(false)) {
				result = AnnaDsbDebtSeniorityEnum.SNDB;
			} else {
				result = AnnaDsbDebtSeniorityEnum.SNDB;
			}
			
			return result;
		}
		
		@Override
		protected MapperS<? extends EconomicTerms> economicTerms(Product product, AnnaDsbUseCaseEnum useCase) {
			return MapperS.of(product).<ContractualProduct>map("getContractualProduct", _product -> _product.getContractualProduct()).<EconomicTerms>map("getEconomicTerms", contractualProduct -> contractualProduct.getEconomicTerms());
		}
		
		@Override
		protected MapperS<? extends ReferenceInformation> referenceInformation(Product product, AnnaDsbUseCaseEnum useCase) {
			final Boolean boolean0 = isCredit.evaluate(product);
			if ((boolean0 == null ? false : boolean0)) {
				return MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", _economicTerms -> _economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<ReferenceInformation>map("getReferenceInformation", generalTerms -> generalTerms.getReferenceInformation());
			}
			final Boolean boolean1 = isCreditSwaption.evaluate(product);
			if ((boolean1 == null ? false : boolean1)) {
				return MapperS.of(economicTermsForProduct.evaluate(underlierForProduct.evaluate(product))).<Payout>map("getPayout", _economicTerms -> _economicTerms.getPayout()).<CreditDefaultPayout>map("getCreditDefaultPayout", payout -> payout.getCreditDefaultPayout()).<GeneralTerms>map("getGeneralTerms", creditDefaultPayout -> creditDefaultPayout.getGeneralTerms()).<ReferenceInformation>map("getReferenceInformation", generalTerms -> generalTerms.getReferenceInformation());
			}
			return MapperS.<ReferenceInformation>ofNull();
		}
		
		@Override
		protected MapperC<DebtSeniorityEnum> debtSeniority(Product product, AnnaDsbUseCaseEnum useCase) {
			if (exists(referenceInformation(product, useCase)).getOrDefault(false)) {
				return referenceInformation(product, useCase).<ReferenceObligation>mapC("getReferenceObligation", _referenceInformation -> _referenceInformation.getReferenceObligation()).<Security>map("getSecurity", referenceObligation -> referenceObligation.getSecurity()).<DebtType>map("getDebtType", security -> security.getDebtType()).<DebtEconomics>mapC("getDebtEconomics", debtType -> debtType.getDebtEconomics()).<DebtSeniorityEnum>map("getDebtSeniority", debtEconomics -> debtEconomics.getDebtSeniority());
			}
			final Boolean _boolean = isTotalReturnSwapDebtUnderlier.evaluate(product);
			if ((_boolean == null ? false : _boolean)) {
				return MapperS.of(MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", _economicTerms -> _economicTerms.getPayout()).<PerformancePayout>mapC("getPerformancePayout", payout -> payout.getPerformancePayout()).get()).<Product>map("getUnderlier", performancePayout -> performancePayout.getUnderlier()).<Security>map("getSecurity", _product -> _product.getSecurity()).<DebtType>map("getDebtType", security -> security.getDebtType()).<DebtEconomics>mapC("getDebtEconomics", debtType -> debtType.getDebtEconomics()).<DebtSeniorityEnum>map("getDebtSeniority", debtEconomics -> debtEconomics.getDebtSeniority());
			}
			return MapperC.<DebtSeniorityEnum>ofNull();
		}
	}
}
