package drr.regulation.esma.emir.refit.margin;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RosettaIgnore;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.annotations.RuneIgnore;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import drr.regulation.common.margin.CommonMarginReport;
import drr.regulation.esma.emir.refit.margin.meta.ESMAEMIRMarginReportMeta;
import drr.standards.iso.CollateralisationType3Code;
import drr.standards.iso.MarginActionEnum;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.101
 */
@RosettaDataType(value="ESMAEMIRMarginReport", builder=ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilderImpl.class, version="7.0.0-dev.101")
@RuneDataType(value="ESMAEMIRMarginReport", model="drr", builder=ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilderImpl.class, version="7.0.0-dev.101")
public interface ESMAEMIRMarginReport extends CommonMarginReport {

	ESMAEMIRMarginReportMeta metaData = new ESMAEMIRMarginReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body ESMA
	 * Corpus Regulation EMIR 648/2012 "Regulation (EU) No 648/2012 of the European Parliament and of the Council of 4 July 2012 on OTC derivatives, central counterparties and trade repositories Text with EEA relevance"  * Corpus Dissemination Margin   
	 * table "3" * dataElement "29" * field "Event date"
	 *
	 * Provision Date on which the reportable event relating to the derivative contract and captured by the report took place. In the case of collateral update - the date for which the information contained in the report is provided.
	 *
	 */
	@Override
	Date getEventDate();
	/**
	 *
	 * Body ESMA
	 * Corpus Regulation EMIR 648/2012 "Regulation (EU) No 648/2012 of the European Parliament and of the Council of 4 July 2012 on OTC derivatives, central counterparties and trade repositories Text with EEA relevance"  * Corpus CommissionDelegatedRegulation RTS 2022/1855 "Commission Delegated Regulation (EU) No 2022/1855 of 10 June 2022 supplementing Regulation (EU) No 648/2012 of the European Parliament and of the Council on OTC derivatives, central counterparties and trade repositories with regard to regulatory technical standards specifying the minimum details of the data to be reported to trade repositories and the type of reports to be used." 
	 * table "3" * dataElement "3" * field "Entity responsible for reporting"
	 *
	 * Provision Where a financial counterparty is solely responsible, and legally liable, for reporting on behalf of both counterparties in accordance with Article 9(1a) of Regulation (EU) No 648/2012 and the non-financial counterparty does not decide to report itself the details of its OTC derivative contracts with the financial counterparty, the unique code identifying that financial counterparty. Where a management company is responsible, and legally liable, for reporting on behalf of an Undertaking for Collective Investment in Transferable Securities (UCITS) in accordance with Article 9(1b) of that Regulation, the unique code identifying that management company. Where an Alternative Investment Fund Manager (AIFM) is responsible, and legally liable, for reporting on behalf of an Alternative Investment Fund (AIF) in accordance with Article 9(1c) of that Regulation, the unique code identifying that AIFM. Where an authorised entity that is responsible for managing and acting on behalf of an IORP is responsible, and legally liable, for reporting on its behalf in accordance with Article 9(1d) of that Regulation, the unique code identifying that entity.
	             This field is applicable only to OTC derivatives.
	 *
	 */
	String getEntityResponsibleForReporting();
	ESMAEMIRNonReportableCollateralData getNonReportable();

	/*********************** Build Methods  ***********************/
	ESMAEMIRMarginReport build();
	
	ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder toBuilder();
	
	static ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder builder() {
		return new ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ESMAEMIRMarginReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ESMAEMIRMarginReport> getType() {
		return ESMAEMIRMarginReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("collateralPortfolioCode"), String.class, getCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("collateralPortfolioIndicator"), Boolean.class, getCollateralPortfolioIndicator(), this);
		processor.processBasic(path.newSubPath("uti"), String.class, getUti(), this);
		processor.processBasic(path.newSubPath("collateralisationCategory"), CollateralisationType3Code.class, getCollateralisationCategory(), this);
		processor.processBasic(path.newSubPath("initialMarginPostedByTheCounterparty1PreHaircut"), BigDecimal.class, getInitialMarginPostedByTheCounterparty1PreHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginPostedByTheCounterparty1PostHaircut"), BigDecimal.class, getInitialMarginPostedByTheCounterparty1PostHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginPostedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getInitialMarginPostedByCounterparty1Currency(), this);
		processor.processBasic(path.newSubPath("variationMarginPostedByTheCounterparty1PreHaircut"), BigDecimal.class, getVariationMarginPostedByTheCounterparty1PreHaircut(), this);
		processor.processBasic(path.newSubPath("variationMarginPostedByTheCounterparty1PostHaircut"), BigDecimal.class, getVariationMarginPostedByTheCounterparty1PostHaircut(), this);
		processor.processBasic(path.newSubPath("variationMarginPostedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getVariationMarginPostedByCounterparty1Currency(), this);
		processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1PreHaircut"), BigDecimal.class, getInitialMarginCollectedByCounterparty1PreHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1PostHaircut"), BigDecimal.class, getInitialMarginCollectedByCounterparty1PostHaircut(), this);
		processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getInitialMarginCollectedByCounterparty1Currency(), this);
		processor.processBasic(path.newSubPath("variationMarginCollectedByTheCounterparty1PreHaircut"), BigDecimal.class, getVariationMarginCollectedByTheCounterparty1PreHaircut(), this);
		processor.processBasic(path.newSubPath("variationMarginCollectedByTheCounterparty1PostHaircut"), BigDecimal.class, getVariationMarginCollectedByTheCounterparty1PostHaircut(), this);
		processor.processBasic(path.newSubPath("variationMarginCollectedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getVariationMarginCollectedByCounterparty1Currency(), this);
		processor.processBasic(path.newSubPath("excessCollateralPostedByTheCounterparty1"), BigDecimal.class, getExcessCollateralPostedByTheCounterparty1(), this);
		processor.processBasic(path.newSubPath("currencyOfTheExcessCollateralPosted"), String.class, getCurrencyOfTheExcessCollateralPosted(), this);
		processor.processBasic(path.newSubPath("excessCollateralCollectedByTheCounterparty1"), BigDecimal.class, getExcessCollateralCollectedByTheCounterparty1(), this);
		processor.processBasic(path.newSubPath("currencyOfTheExcessCollateralCollected"), String.class, getCurrencyOfTheExcessCollateralCollected(), this);
		processor.processBasic(path.newSubPath("collateralTimestamp"), ZonedDateTime.class, getCollateralTimestamp(), this);
		processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
		processor.processBasic(path.newSubPath("actionType"), MarginActionEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
		processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
		processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
		processRosetta(path.newSubPath("nonReportable"), processor, ESMAEMIRNonReportableCollateralData.class, getNonReportable());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ESMAEMIRMarginReportBuilder extends ESMAEMIRMarginReport, CommonMarginReport.CommonMarginReportBuilder {
		ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder getOrCreateNonReportable();
		@Override
		ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder getNonReportable();
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty1(String counterparty1);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty2IdentifierType(Boolean counterparty2IdentifierType);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty2(String counterparty2);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralPortfolioCode(String collateralPortfolioCode);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralPortfolioIndicator(Boolean collateralPortfolioIndicator);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setUti(String uti);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralisationCategory(CollateralisationType3Code collateralisationCategory);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByTheCounterparty1PreHaircut(BigDecimal initialMarginPostedByTheCounterparty1PreHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByTheCounterparty1PostHaircut(BigDecimal initialMarginPostedByTheCounterparty1PostHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByCounterparty1Currency(ISOCurrencyCodeEnum initialMarginPostedByCounterparty1Currency);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByTheCounterparty1PreHaircut(BigDecimal variationMarginPostedByTheCounterparty1PreHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByTheCounterparty1PostHaircut(BigDecimal variationMarginPostedByTheCounterparty1PostHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByCounterparty1Currency(ISOCurrencyCodeEnum variationMarginPostedByCounterparty1Currency);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1PreHaircut(BigDecimal initialMarginCollectedByCounterparty1PreHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1PostHaircut(BigDecimal initialMarginCollectedByCounterparty1PostHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1Currency(ISOCurrencyCodeEnum initialMarginCollectedByCounterparty1Currency);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByTheCounterparty1PreHaircut(BigDecimal variationMarginCollectedByTheCounterparty1PreHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByTheCounterparty1PostHaircut(BigDecimal variationMarginCollectedByTheCounterparty1PostHaircut);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByCounterparty1Currency(ISOCurrencyCodeEnum variationMarginCollectedByCounterparty1Currency);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setExcessCollateralPostedByTheCounterparty1(BigDecimal excessCollateralPostedByTheCounterparty1);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCurrencyOfTheExcessCollateralPosted(String currencyOfTheExcessCollateralPosted);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setExcessCollateralCollectedByTheCounterparty1(BigDecimal excessCollateralCollectedByTheCounterparty1);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCurrencyOfTheExcessCollateralCollected(String currencyOfTheExcessCollateralCollected);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralTimestamp(ZonedDateTime collateralTimestamp);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollateralPortfolioCode(String variationMarginCollateralPortfolioCode);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollateralPortfolioCode(String initialMarginCollateralPortfolioCode);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setReportSubmittingEntityID(String reportSubmittingEntityID);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setActionType(MarginActionEnum actionType);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventDate(Date eventDate);
		@Override
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventTimestamp(ZonedDateTime eventTimestamp);
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventDateOverriddenAsDate(Date eventDate);
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEntityResponsibleForReporting(String entityResponsibleForReporting);
		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setNonReportable(ESMAEMIRNonReportableCollateralData nonReportable);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("collateralPortfolioCode"), String.class, getCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("collateralPortfolioIndicator"), Boolean.class, getCollateralPortfolioIndicator(), this);
			processor.processBasic(path.newSubPath("uti"), String.class, getUti(), this);
			processor.processBasic(path.newSubPath("collateralisationCategory"), CollateralisationType3Code.class, getCollateralisationCategory(), this);
			processor.processBasic(path.newSubPath("initialMarginPostedByTheCounterparty1PreHaircut"), BigDecimal.class, getInitialMarginPostedByTheCounterparty1PreHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginPostedByTheCounterparty1PostHaircut"), BigDecimal.class, getInitialMarginPostedByTheCounterparty1PostHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginPostedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getInitialMarginPostedByCounterparty1Currency(), this);
			processor.processBasic(path.newSubPath("variationMarginPostedByTheCounterparty1PreHaircut"), BigDecimal.class, getVariationMarginPostedByTheCounterparty1PreHaircut(), this);
			processor.processBasic(path.newSubPath("variationMarginPostedByTheCounterparty1PostHaircut"), BigDecimal.class, getVariationMarginPostedByTheCounterparty1PostHaircut(), this);
			processor.processBasic(path.newSubPath("variationMarginPostedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getVariationMarginPostedByCounterparty1Currency(), this);
			processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1PreHaircut"), BigDecimal.class, getInitialMarginCollectedByCounterparty1PreHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1PostHaircut"), BigDecimal.class, getInitialMarginCollectedByCounterparty1PostHaircut(), this);
			processor.processBasic(path.newSubPath("initialMarginCollectedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getInitialMarginCollectedByCounterparty1Currency(), this);
			processor.processBasic(path.newSubPath("variationMarginCollectedByTheCounterparty1PreHaircut"), BigDecimal.class, getVariationMarginCollectedByTheCounterparty1PreHaircut(), this);
			processor.processBasic(path.newSubPath("variationMarginCollectedByTheCounterparty1PostHaircut"), BigDecimal.class, getVariationMarginCollectedByTheCounterparty1PostHaircut(), this);
			processor.processBasic(path.newSubPath("variationMarginCollectedByCounterparty1Currency"), ISOCurrencyCodeEnum.class, getVariationMarginCollectedByCounterparty1Currency(), this);
			processor.processBasic(path.newSubPath("excessCollateralPostedByTheCounterparty1"), BigDecimal.class, getExcessCollateralPostedByTheCounterparty1(), this);
			processor.processBasic(path.newSubPath("currencyOfTheExcessCollateralPosted"), String.class, getCurrencyOfTheExcessCollateralPosted(), this);
			processor.processBasic(path.newSubPath("excessCollateralCollectedByTheCounterparty1"), BigDecimal.class, getExcessCollateralCollectedByTheCounterparty1(), this);
			processor.processBasic(path.newSubPath("currencyOfTheExcessCollateralCollected"), String.class, getCurrencyOfTheExcessCollateralCollected(), this);
			processor.processBasic(path.newSubPath("collateralTimestamp"), ZonedDateTime.class, getCollateralTimestamp(), this);
			processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
			processor.processBasic(path.newSubPath("actionType"), MarginActionEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
			processor.processBasic(path.newSubPath("eventTimestamp"), ZonedDateTime.class, getEventTimestamp(), this);
			processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
			processRosetta(path.newSubPath("nonReportable"), processor, ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder.class, getNonReportable());
		}
		

		ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder prune();
	}

	/*********************** Immutable Implementation of ESMAEMIRMarginReport  ***********************/
	class ESMAEMIRMarginReportImpl extends CommonMarginReport.CommonMarginReportImpl implements ESMAEMIRMarginReport {
		private final Date eventDate;
		private final String entityResponsibleForReporting;
		private final ESMAEMIRNonReportableCollateralData nonReportable;
		
		protected ESMAEMIRMarginReportImpl(ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder builder) {
			super(builder);
			this.eventDate = builder.getEventDate();
			this.entityResponsibleForReporting = builder.getEntityResponsibleForReporting();
			this.nonReportable = ofNullable(builder.getNonReportable()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		public Date getEventDate() {
			return eventDate;
		}
		
		@Override
		@RosettaAttribute("entityResponsibleForReporting")
		@RuneAttribute("entityResponsibleForReporting")
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		public ESMAEMIRNonReportableCollateralData getNonReportable() {
			return nonReportable;
		}
		
		@Override
		public ESMAEMIRMarginReport build() {
			return this;
		}
		
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder toBuilder() {
			ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getEventDate()).ifPresent(builder::setEventDateOverriddenAsDate);
			ofNullable(getEntityResponsibleForReporting()).ifPresent(builder::setEntityResponsibleForReporting);
			ofNullable(getNonReportable()).ifPresent(builder::setNonReportable);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ESMAEMIRMarginReport _that = getType().cast(o);
		
			if (!Objects.equals(eventDate, _that.getEventDate())) return false;
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(nonReportable, _that.getNonReportable())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (eventDate != null ? eventDate.hashCode() : 0);
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (nonReportable != null ? nonReportable.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ESMAEMIRMarginReport {" +
				"eventDate=" + this.eventDate + ", " +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"nonReportable=" + this.nonReportable +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of ESMAEMIRMarginReport  ***********************/
	class ESMAEMIRMarginReportBuilderImpl extends CommonMarginReport.CommonMarginReportBuilderImpl implements ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder {
	
		protected Date eventDate;
		protected String entityResponsibleForReporting;
		protected ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder nonReportable;
		
		@Override
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		public Date getEventDate() {
			return eventDate;
		}
		
		@Override
		@RosettaAttribute("entityResponsibleForReporting")
		@RuneAttribute("entityResponsibleForReporting")
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		public ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder getNonReportable() {
			return nonReportable;
		}
		
		@Override
		public ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder getOrCreateNonReportable() {
			ESMAEMIRNonReportableCollateralData.ESMAEMIRNonReportableCollateralDataBuilder result;
			if (nonReportable!=null) {
				result = nonReportable;
			}
			else {
				result = nonReportable = ESMAEMIRNonReportableCollateralData.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute("counterparty2IdentifierType")
		@RuneAttribute("counterparty2IdentifierType")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty2IdentifierType(Boolean _counterparty2IdentifierType) {
			this.counterparty2IdentifierType = _counterparty2IdentifierType == null ? null : _counterparty2IdentifierType;
			return this;
		}
		
		@RosettaAttribute("counterparty2")
		@RuneAttribute("counterparty2")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute("collateralPortfolioCode")
		@RuneAttribute("collateralPortfolioCode")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralPortfolioCode(String _collateralPortfolioCode) {
			this.collateralPortfolioCode = _collateralPortfolioCode == null ? null : _collateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute("collateralPortfolioIndicator")
		@RuneAttribute("collateralPortfolioIndicator")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralPortfolioIndicator(Boolean _collateralPortfolioIndicator) {
			this.collateralPortfolioIndicator = _collateralPortfolioIndicator == null ? null : _collateralPortfolioIndicator;
			return this;
		}
		
		@RosettaAttribute("uti")
		@RuneAttribute("uti")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setUti(String _uti) {
			this.uti = _uti == null ? null : _uti;
			return this;
		}
		
		@RosettaAttribute("collateralisationCategory")
		@RuneAttribute("collateralisationCategory")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralisationCategory(CollateralisationType3Code _collateralisationCategory) {
			this.collateralisationCategory = _collateralisationCategory == null ? null : _collateralisationCategory;
			return this;
		}
		
		@RosettaAttribute("initialMarginPostedByTheCounterparty1PreHaircut")
		@RuneAttribute("initialMarginPostedByTheCounterparty1PreHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByTheCounterparty1PreHaircut(BigDecimal _initialMarginPostedByTheCounterparty1PreHaircut) {
			this.initialMarginPostedByTheCounterparty1PreHaircut = _initialMarginPostedByTheCounterparty1PreHaircut == null ? null : _initialMarginPostedByTheCounterparty1PreHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginPostedByTheCounterparty1PostHaircut")
		@RuneAttribute("initialMarginPostedByTheCounterparty1PostHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByTheCounterparty1PostHaircut(BigDecimal _initialMarginPostedByTheCounterparty1PostHaircut) {
			this.initialMarginPostedByTheCounterparty1PostHaircut = _initialMarginPostedByTheCounterparty1PostHaircut == null ? null : _initialMarginPostedByTheCounterparty1PostHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginPostedByCounterparty1Currency")
		@RuneAttribute("initialMarginPostedByCounterparty1Currency")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginPostedByCounterparty1Currency(ISOCurrencyCodeEnum _initialMarginPostedByCounterparty1Currency) {
			this.initialMarginPostedByCounterparty1Currency = _initialMarginPostedByCounterparty1Currency == null ? null : _initialMarginPostedByCounterparty1Currency;
			return this;
		}
		
		@RosettaAttribute("variationMarginPostedByTheCounterparty1PreHaircut")
		@RuneAttribute("variationMarginPostedByTheCounterparty1PreHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByTheCounterparty1PreHaircut(BigDecimal _variationMarginPostedByTheCounterparty1PreHaircut) {
			this.variationMarginPostedByTheCounterparty1PreHaircut = _variationMarginPostedByTheCounterparty1PreHaircut == null ? null : _variationMarginPostedByTheCounterparty1PreHaircut;
			return this;
		}
		
		@RosettaAttribute("variationMarginPostedByTheCounterparty1PostHaircut")
		@RuneAttribute("variationMarginPostedByTheCounterparty1PostHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByTheCounterparty1PostHaircut(BigDecimal _variationMarginPostedByTheCounterparty1PostHaircut) {
			this.variationMarginPostedByTheCounterparty1PostHaircut = _variationMarginPostedByTheCounterparty1PostHaircut == null ? null : _variationMarginPostedByTheCounterparty1PostHaircut;
			return this;
		}
		
		@RosettaAttribute("variationMarginPostedByCounterparty1Currency")
		@RuneAttribute("variationMarginPostedByCounterparty1Currency")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginPostedByCounterparty1Currency(ISOCurrencyCodeEnum _variationMarginPostedByCounterparty1Currency) {
			this.variationMarginPostedByCounterparty1Currency = _variationMarginPostedByCounterparty1Currency == null ? null : _variationMarginPostedByCounterparty1Currency;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollectedByCounterparty1PreHaircut")
		@RuneAttribute("initialMarginCollectedByCounterparty1PreHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1PreHaircut(BigDecimal _initialMarginCollectedByCounterparty1PreHaircut) {
			this.initialMarginCollectedByCounterparty1PreHaircut = _initialMarginCollectedByCounterparty1PreHaircut == null ? null : _initialMarginCollectedByCounterparty1PreHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollectedByCounterparty1PostHaircut")
		@RuneAttribute("initialMarginCollectedByCounterparty1PostHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1PostHaircut(BigDecimal _initialMarginCollectedByCounterparty1PostHaircut) {
			this.initialMarginCollectedByCounterparty1PostHaircut = _initialMarginCollectedByCounterparty1PostHaircut == null ? null : _initialMarginCollectedByCounterparty1PostHaircut;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollectedByCounterparty1Currency")
		@RuneAttribute("initialMarginCollectedByCounterparty1Currency")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollectedByCounterparty1Currency(ISOCurrencyCodeEnum _initialMarginCollectedByCounterparty1Currency) {
			this.initialMarginCollectedByCounterparty1Currency = _initialMarginCollectedByCounterparty1Currency == null ? null : _initialMarginCollectedByCounterparty1Currency;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollectedByTheCounterparty1PreHaircut")
		@RuneAttribute("variationMarginCollectedByTheCounterparty1PreHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByTheCounterparty1PreHaircut(BigDecimal _variationMarginCollectedByTheCounterparty1PreHaircut) {
			this.variationMarginCollectedByTheCounterparty1PreHaircut = _variationMarginCollectedByTheCounterparty1PreHaircut == null ? null : _variationMarginCollectedByTheCounterparty1PreHaircut;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollectedByTheCounterparty1PostHaircut")
		@RuneAttribute("variationMarginCollectedByTheCounterparty1PostHaircut")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByTheCounterparty1PostHaircut(BigDecimal _variationMarginCollectedByTheCounterparty1PostHaircut) {
			this.variationMarginCollectedByTheCounterparty1PostHaircut = _variationMarginCollectedByTheCounterparty1PostHaircut == null ? null : _variationMarginCollectedByTheCounterparty1PostHaircut;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollectedByCounterparty1Currency")
		@RuneAttribute("variationMarginCollectedByCounterparty1Currency")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollectedByCounterparty1Currency(ISOCurrencyCodeEnum _variationMarginCollectedByCounterparty1Currency) {
			this.variationMarginCollectedByCounterparty1Currency = _variationMarginCollectedByCounterparty1Currency == null ? null : _variationMarginCollectedByCounterparty1Currency;
			return this;
		}
		
		@RosettaAttribute("excessCollateralPostedByTheCounterparty1")
		@RuneAttribute("excessCollateralPostedByTheCounterparty1")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setExcessCollateralPostedByTheCounterparty1(BigDecimal _excessCollateralPostedByTheCounterparty1) {
			this.excessCollateralPostedByTheCounterparty1 = _excessCollateralPostedByTheCounterparty1 == null ? null : _excessCollateralPostedByTheCounterparty1;
			return this;
		}
		
		@RosettaAttribute("currencyOfTheExcessCollateralPosted")
		@RuneAttribute("currencyOfTheExcessCollateralPosted")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCurrencyOfTheExcessCollateralPosted(String _currencyOfTheExcessCollateralPosted) {
			this.currencyOfTheExcessCollateralPosted = _currencyOfTheExcessCollateralPosted == null ? null : _currencyOfTheExcessCollateralPosted;
			return this;
		}
		
		@RosettaAttribute("excessCollateralCollectedByTheCounterparty1")
		@RuneAttribute("excessCollateralCollectedByTheCounterparty1")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setExcessCollateralCollectedByTheCounterparty1(BigDecimal _excessCollateralCollectedByTheCounterparty1) {
			this.excessCollateralCollectedByTheCounterparty1 = _excessCollateralCollectedByTheCounterparty1 == null ? null : _excessCollateralCollectedByTheCounterparty1;
			return this;
		}
		
		@RosettaAttribute("currencyOfTheExcessCollateralCollected")
		@RuneAttribute("currencyOfTheExcessCollateralCollected")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCurrencyOfTheExcessCollateralCollected(String _currencyOfTheExcessCollateralCollected) {
			this.currencyOfTheExcessCollateralCollected = _currencyOfTheExcessCollateralCollected == null ? null : _currencyOfTheExcessCollateralCollected;
			return this;
		}
		
		@RosettaAttribute("collateralTimestamp")
		@RuneAttribute("collateralTimestamp")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setCollateralTimestamp(ZonedDateTime _collateralTimestamp) {
			this.collateralTimestamp = _collateralTimestamp == null ? null : _collateralTimestamp;
			return this;
		}
		
		@RosettaAttribute("variationMarginCollateralPortfolioCode")
		@RuneAttribute("variationMarginCollateralPortfolioCode")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setVariationMarginCollateralPortfolioCode(String _variationMarginCollateralPortfolioCode) {
			this.variationMarginCollateralPortfolioCode = _variationMarginCollateralPortfolioCode == null ? null : _variationMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute("initialMarginCollateralPortfolioCode")
		@RuneAttribute("initialMarginCollateralPortfolioCode")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setInitialMarginCollateralPortfolioCode(String _initialMarginCollateralPortfolioCode) {
			this.initialMarginCollateralPortfolioCode = _initialMarginCollateralPortfolioCode == null ? null : _initialMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute(value="reportSubmittingEntityID", isRequired=true)
		@RuneAttribute(value="reportSubmittingEntityID", isRequired=true)
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setReportSubmittingEntityID(String _reportSubmittingEntityID) {
			this.reportSubmittingEntityID = _reportSubmittingEntityID == null ? null : _reportSubmittingEntityID;
			return this;
		}
		
		@RosettaAttribute("actionType")
		@RuneAttribute("actionType")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setActionType(MarginActionEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventDateOverriddenAsDate(Date _eventDate) {
			this.eventDate = _eventDate == null ? null : _eventDate;
			return this;
		}
		
		@RosettaIgnore
		@RuneIgnore
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventDate(Date _eventDate) {
			return setEventDateOverriddenAsDate(_eventDate);
		}
		
		@RosettaAttribute("eventTimestamp")
		@RuneAttribute("eventTimestamp")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEventTimestamp(ZonedDateTime _eventTimestamp) {
			this.eventTimestamp = _eventTimestamp == null ? null : _eventTimestamp;
			return this;
		}
		
		@RosettaAttribute("entityResponsibleForReporting")
		@RuneAttribute("entityResponsibleForReporting")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setEntityResponsibleForReporting(String _entityResponsibleForReporting) {
			this.entityResponsibleForReporting = _entityResponsibleForReporting == null ? null : _entityResponsibleForReporting;
			return this;
		}
		
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder setNonReportable(ESMAEMIRNonReportableCollateralData _nonReportable) {
			this.nonReportable = _nonReportable == null ? null : _nonReportable.toBuilder();
			return this;
		}
		
		@Override
		public ESMAEMIRMarginReport build() {
			return new ESMAEMIRMarginReport.ESMAEMIRMarginReportImpl(this);
		}
		
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder prune() {
			super.prune();
			if (nonReportable!=null && !nonReportable.prune().hasData()) nonReportable = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getEventDate()!=null) return true;
			if (getEntityResponsibleForReporting()!=null) return true;
			if (getNonReportable()!=null && getNonReportable().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder o = (ESMAEMIRMarginReport.ESMAEMIRMarginReportBuilder) other;
			
			merger.mergeRosetta(getNonReportable(), o.getNonReportable(), this::setNonReportable);
			
			merger.mergeBasic(getEventDate(), o.getEventDate(), this::setEventDateOverriddenAsDate);
			merger.mergeBasic(getEntityResponsibleForReporting(), o.getEntityResponsibleForReporting(), this::setEntityResponsibleForReporting);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ESMAEMIRMarginReport _that = getType().cast(o);
		
			if (!Objects.equals(eventDate, _that.getEventDate())) return false;
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(nonReportable, _that.getNonReportable())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (eventDate != null ? eventDate.hashCode() : 0);
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (nonReportable != null ? nonReportable.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ESMAEMIRMarginReportBuilder {" +
				"eventDate=" + this.eventDate + ", " +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"nonReportable=" + this.nonReportable +
			'}' + " " + super.toString();
		}
	}
}
