package drr.base.trade.price.functions;

import cdm.base.datetime.DateRange;
import cdm.base.math.DatedValue;
import cdm.observable.asset.PriceSchedule;
import cdm.product.template.CalculationSchedule;
import cdm.product.template.SchedulePeriod;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.records.Date;
import drr.base.trade.price.PriceNotationEnum;
import drr.base.trade.price.PricePeriod;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GetReportableSchedulePeriod.GetReportableSchedulePeriodDefault.class)
public abstract class GetReportableSchedulePeriod implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected PriceFormatFromNotation priceFormatFromNotation;

	/**
	* @param priceSchedule 
	* @param calculationSchedule 
	* @param notation 
	* @return pricePeriod 
	*/
	public List<? extends PricePeriod> evaluate(PriceSchedule priceSchedule, CalculationSchedule calculationSchedule, PriceNotationEnum notation) {
		List<PricePeriod.PricePeriodBuilder> pricePeriodBuilder = doEvaluate(priceSchedule, calculationSchedule, notation);
		
		final List<? extends PricePeriod> pricePeriod;
		if (pricePeriodBuilder == null) {
			pricePeriod = null;
		} else {
			pricePeriod = pricePeriodBuilder.stream().map(PricePeriod::build).collect(Collectors.toList());
			objectValidator.validate(PricePeriod.class, pricePeriod);
		}
		
		return pricePeriod;
	}

	protected abstract List<PricePeriod.PricePeriodBuilder> doEvaluate(PriceSchedule priceSchedule, CalculationSchedule calculationSchedule, PriceNotationEnum notation);

	public static class GetReportableSchedulePeriodDefault extends GetReportableSchedulePeriod {
		@Override
		protected List<PricePeriod.PricePeriodBuilder> doEvaluate(PriceSchedule priceSchedule, CalculationSchedule calculationSchedule, PriceNotationEnum notation) {
			List<PricePeriod.PricePeriodBuilder> pricePeriod = new ArrayList<>();
			return assignOutput(pricePeriod, priceSchedule, calculationSchedule, notation);
		}
		
		protected List<PricePeriod.PricePeriodBuilder> assignOutput(List<PricePeriod.PricePeriodBuilder> pricePeriod, PriceSchedule priceSchedule, CalculationSchedule calculationSchedule, PriceNotationEnum notation) {
			final MapperC<PricePeriod> thenArg = MapperS.of(priceSchedule).<DatedValue>mapC("getDatedValue", _priceSchedule -> _priceSchedule.getDatedValue())
				.mapItem(priceSchedulePeriod -> {
					final MapperC<SchedulePeriod> thenArg0;
					if (exists(MapperS.of(calculationSchedule)).getOrDefault(false)) {
						thenArg0 = MapperS.of(calculationSchedule).<SchedulePeriod>mapC("getSchedulePeriod", _calculationSchedule -> _calculationSchedule.getSchedulePeriod())
							.filterItemNullSafe(customPeriod -> areEqual(priceSchedulePeriod.<Date>map("getDate", datedValue -> datedValue.getDate()), customPeriod.<DateRange>map("getCalculationPeriod", schedulePeriod -> schedulePeriod.getCalculationPeriod()).<Date>map("getStartDate", dateRange -> dateRange.getStartDate()), CardinalityOperator.All).get());
					} else {
						thenArg0 = MapperC.<SchedulePeriod>ofNull();
					}
					final MapperC<Date> thenArg1 = thenArg0
						.mapItem(item -> item.<DateRange>map("getCalculationPeriod", schedulePeriod -> schedulePeriod.getCalculationPeriod()).<Date>map("getEndDate", dateRange -> dateRange.getEndDate()));
					return MapperS.of(PricePeriod.builder()
						.setPrice(priceFormatFromNotation.evaluate(priceSchedulePeriod.<BigDecimal>map("getValue", datedValue -> datedValue.getValue()).get(), notation))
						.setEffectiveDate(priceSchedulePeriod.<Date>map("getDate", datedValue -> datedValue.getDate()).get())
						.setEndDate(MapperS.of(thenArg1.get()).get())
						.build());
				});
			pricePeriod.addAll(toBuilder(thenArg
				.sort(item -> item.<Date>map("getEffectiveDate", _pricePeriod -> _pricePeriod.getEffectiveDate())).getMulti()));
			
			return Optional.ofNullable(pricePeriod)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
