package drr.base.trade.underlier.functions;

import cdm.product.template.ForwardPayout;
import cdm.product.template.OptionPayout;
import cdm.product.template.Payout;
import cdm.product.template.Product;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperS;
import drr.base.trade.functions.EconomicTermsForProduct;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(UnderlierForProduct.UnderlierForProductDefault.class)
public abstract class UnderlierForProduct implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected EconomicTermsForProduct economicTermsForProduct;

	/**
	* @param product 
	* @return underlierProduct 
	*/
	public Product evaluate(Product product) {
		Product.ProductBuilder underlierProductBuilder = doEvaluate(product);
		
		final Product underlierProduct;
		if (underlierProductBuilder == null) {
			underlierProduct = null;
		} else {
			underlierProduct = underlierProductBuilder.build();
			objectValidator.validate(Product.class, underlierProduct);
		}
		
		return underlierProduct;
	}

	protected abstract Product.ProductBuilder doEvaluate(Product product);

	public static class UnderlierForProductDefault extends UnderlierForProduct {
		@Override
		protected Product.ProductBuilder doEvaluate(Product product) {
			Product.ProductBuilder underlierProduct = Product.builder();
			return assignOutput(underlierProduct, product);
		}
		
		protected Product.ProductBuilder assignOutput(Product.ProductBuilder underlierProduct, Product product) {
			if (exists(MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<OptionPayout>mapC("getOptionPayout", payout -> payout.getOptionPayout())).getOrDefault(false)) {
				underlierProduct = toBuilder(MapperS.of(MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<OptionPayout>mapC("getOptionPayout", payout -> payout.getOptionPayout()).get()).<Product>map("getUnderlier", optionPayout -> optionPayout.getUnderlier()).get());
			} else if (exists(MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<ForwardPayout>mapC("getForwardPayout", payout -> payout.getForwardPayout())).getOrDefault(false)) {
				underlierProduct = toBuilder(MapperS.of(MapperS.of(economicTermsForProduct.evaluate(product)).<Payout>map("getPayout", economicTerms -> economicTerms.getPayout()).<ForwardPayout>mapC("getForwardPayout", payout -> payout.getForwardPayout()).get()).<Product>map("getUnderlier", forwardPayout -> forwardPayout.getUnderlier()).get());
			} else {
				underlierProduct = null;
			}
			
			return Optional.ofNullable(underlierProduct)
				.map(o -> o.prune())
				.orElse(null);
		}
	}
}
