package drr.regulation.cftc.rewrite.valuation;

import cdm.base.staticdata.asset.common.AssetClassEnum;
import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import cdm.base.staticdata.party.PartyIdentifierTypeEnum;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import com.rosetta.util.ListEquals;
import drr.regulation.cftc.rewrite.valuation.meta.CFTCValuationReportMeta;
import drr.regulation.common.SupervisoryBodyEnum;
import drr.standards.iso.ActionTypeEnum;
import drr.standards.iso.ValuationType1Code;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.17
 */
@RosettaDataType(value="CFTCValuationReport", builder=CFTCValuationReport.CFTCValuationReportBuilderImpl.class, version="7.0.0-dev.17")
@RuneDataType(value="CFTCValuationReport", model="drr", builder=CFTCValuationReport.CFTCValuationReportBuilderImpl.class, version="7.0.0-dev.17")
public interface CFTCValuationReport extends RosettaModelObject {

	CFTCValuationReportMeta metaData = new CFTCValuationReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "13" * field "Counterparty 1 (reporting counterparty)"
	 *
	 * Provision Identifier of the counterparty to an OTC derivative transaction who is fulfilling its reporting obligation via the report in question. In jurisdictions where both parties must report the transaction, the identifier of Counterparty 1 always identifies the reporting counterparty. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund, and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "13" * field "Counterparty 1 (reporting counterparty)" * footnote "24"
	 *
	 * Provision Throughout this Technical Specification, references to 'OTC derivatives' in CDE data elements should be read to mean swaps.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "13" * field "Counterparty 1 (reporting counterparty)" * footnote "25"
	 *
	 * Provision  ROC Statement  Individuals Acting in a Business Capacity, ROC Statement - Individuals Acting in a Business Capacity.
	 *
	 */
	String getCounterparty1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "14" * field "Counterparty 2"
	 *
	 * Provision Identifier of the second counterparty to an OTC derivative transaction. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund, and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "26"
	 *
	 * Provision In the case of a swap transaction for pre-allocated block executed by a fund manager on behalf of a fund, the fund manager is reported as the counterparty.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "27"
	 *
	 * Provision Only one counterparty should be reported. In cases where multiple counterparties are legally responsible as the second counterparty (such as an obligated group, for example), report only one of the counterparties and use the
	             same counterparty for all continuation data and lifecycle events.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "28"
	 *
	 * Provision ROC Statement  Individuals Acting in a Business Capacity, ROC Statement - Individuals Acting in a Business Capacity
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "14" * field "Counterparty 2" * footnote "29"
	 *
	 * Provision Throughout this Technical Specification, Privacy Law Identifier (PLI) has been added as an allowable value for all Counterparty 2 (#14) associated data elements
	 *
	 */
	String getCounterparty2();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "15" * field "Counterparty 2 Identifier Source"
	 *
	 * Provision Source used to identify the Counterparty 2.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "15" * field "Counterparty 2 Identifier Source" * footnote "30"
	 *
	 * Provision Throughout this Technical Specification, for references to 'Privacy Law Identifiers', refer to DMO Letter No. 17-16, http://www.cftc.gov/idc/groups/public/@lrlettergeneral/documents/letter/17-16.pdf
	 *
	 */
	String getCounterparty2IdentifierSource();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "22" * field "Submitter Identifier"
	 *
	 * Provision Identifier of the entity submitting the data to the swap data repository (SDR). The Submitter identifier will be the same as the reporting counterparty or swap execution facility (SEF), unless they use a third-party service provider to submit the data to SDR in which case, report the identifier of the third-party service provider.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Check whether trade is executed in a SEF first. If it is, SEF is obliged to be the Submitter. Then check for Reporting Party and Data Submitter.
	 *
	 */
	String getSubmitterIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "26" * field "Action Type"
	 *
	 * Provision Type of action taken on the swap transaction or type of end-of-day reporting. Actions may include, but are not limited to, new, modify, correct, error, terminate, revive, transfer out, valuation, and collateral.
	                 - New: An action that reports a new swap transaction. It applies to the first message relating to a new USI or UTI.
	                 - Modify: An action that modifies the state of a previously submitted transaction (e.g., credit event) or changes a term of a previously submitted transaction due to a newly negotiated modification (amendment) or updates previously missing information (e.g., post price swap). It does not include correction of a previous transaction.
	                 - Correct: An action that corrects erroneous data of a previously submitted transaction.
	                 - Error: An action of cancellation of a wrongly submitted entire transaction in case it never came into existence or was not subject to part 43/part 45 reporting requirements but was reported erroneously.
	                 - Terminate: An action that closes an existing transaction because of a new event (e.g., Compression, Novation). This does not apply to transactions that terminate at contractual maturity date.
	                 - Revive: An action that reinstates a swap transaction that was reported as error or terminated by mistake.
	                 - Transfer out: An action that transfers swap transaction from one SDR to another SDR (change of swap data repository).
	                 - Valuation: An update to valuation data. There will be no corresponding Event type.
	                 - Collateral: An update to collateral margin data. There will be no corresponding Event type. Refer to appendix F for event model sample scenarios
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "26" * field "Action Type" * footnote "33"
	 *
	 * Provision Only one Action type value is allowed per submission. Multiple Action type values should not be submitted in one transaction report. For example, if a data element needs to be corrected on a previously submitted transaction that is getting terminated, the Correct (CORR) value should be submitted as a separate submission prior to the submission of the Terminate (TERM) transaction.
	 *
	 */
	ActionTypeEnum getActionType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "87" * field "Unique Product Identifier"
	 *
	 * Provision A unique set of characters that represents a particular OTC derivative. The Commission will designate a UPI pursuant to part 45.7.
	 				   Note: A Unique product identifier short name (D4), defined as, 'When the Commission designates a UPI pursuant to part 45, a humanly readable description made available by the UPI issuer corresponding to the UPI'.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "87" * field "Unique Product Identifier" * footnote "68"
	 *
	 * Provision Until the Commission designates a UPI for the commodity asset class pursuant to part 45, continue reporting product data elements according to the internal product identifier or product description used by the swap data repository to which the swap is reported. When the Commission designates a UPI for the commodity asset class pursuant to part 45, report the UPI.
	 *
	 */
	String getUniqueProductIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "97" * field "Reporting Timestamp"
	 *
	 * Provision Date and time of the submission of the report to the trade repository
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "97" * field "Reporting Timestamp" * footnote "74"
	 *
	 * Provision Reporting timestamp (#97) is recorded and reported by the submitter.
	 *
	 */
	ZonedDateTime getReportingTimestamp();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "102" * field "Unique Swap Identifier (USI)"
	 *
	 * Provision The USI is a unique identifier assigned to all swap transactions which identifies the transaction (the swap and its counterparties) uniquely throughout its duration. It consists of a namespace and a transaction identifier.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "102" * field "Unique Swap Identifier (USI)" * footnote "76"
	 *
	 * Provision Throughout this Technical Specification, references to 'CFTC USI Data Standard' should refer to the USI Data Standard, https://www.cftc.gov/sites/default/files/idc/groups/public/@swaps/documents/dfsubmission/usidatastandards100112.pdf
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Rule does not model UTI > USI priority defined in CFTC specification.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Firms do expect to report USI/UTI in both formats: as a single identifier field as well as using Issuer + identifier. The functional model needs to handle both scenarios.
	 *
	 */
	String getUniqueSwapIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "103" * field "Unique Transaction Identifier (UTI)"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle and used for all recordkeeping and all swap data reporting pursuant to para 45.5. A UTI is comprised of the LEI of the generating entity and a unique alphanumeric code.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Rule does not model UTI > USI priority defined in CFTC specification.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Firms do expect to report USI/UTI in both formats: as a single identifier field as well as using Issuer + identifier. The functional model needs to handle both scenarios.
	 *
	 */
	String getUniqueTransactionIdentifier();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "106" * field "Next Floating Reference Reset Date"
	 *
	 * Provision The nearest date in the future that the floating reference resets on.
	 *
	 */
	Date getNextFloatingReferenceResetDateLeg1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "106" * field "Next Floating Reference Reset Date"
	 *
	 * Provision The nearest date in the future that the floating reference resets on.
	 *
	 */
	Date getNextFloatingReferenceResetDateLeg2();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "107" * field "Last Floating Reference Value"
	 *
	 * Provision The most recent sampling of the value of the floating reference for to determine cash flow. Ties to Last floating reference reset date data element.
	 *
	 */
	BigDecimal getLastFloatingReferenceValueLeg1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "107" * field "Last Floating Reference Value"
	 *
	 * Provision The most recent sampling of the value of the floating reference for to determine cash flow. Ties to Last floating reference reset date data element.
	 *
	 */
	BigDecimal getLastFloatingReferenceValueLeg2();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "108" * field "Last floating reference reset date"
	 *
	 * Provision The date of the most recent sampling of the floating reference to determine cash flow. Ties to Last floating reference value data element.
	 *
	 */
	Date getLastFloatingReferenceResetDateLeg1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "108" * field "Last floating reference reset date"
	 *
	 * Provision The date of the most recent sampling of the floating reference to determine cash flow. Ties to Last floating reference value data element.
	 *
	 */
	Date getLastFloatingReferenceResetDateLeg2();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "109" * field "Delta"
	 *
	 * Provision The ratio of the absolute change in price of an OTC derivative transaction to the change in price of the underlier, at the time a new transaction is reported or when a change in the notional amount is reported.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "109" * field "Delta" * footnote "77"
	 *
	 * Provision Delta must be reported daily regardless of whether there is a change in the value since the last reporting.
	 *
	 */
	BigDecimal getDelta();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "110" * field "Valuation Amount"
	 *
	 * Provision Current value of the outstanding contract. Valuation amount is expressed as the exit cost of the contract or components of the contract, i.e., the price that would be received to sell the contract (in the market in an orderly transaction at the valuation date).
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "110" * field "Valuation Amount" * footnote "78"
	 *
	 * Provision Valuation amount must be reported daily regardless of whether there is a change in the value since the last reporting.
	 *
	 */
	BigDecimal getValuationAmount();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "111" * field "Valuation Currency"
	 *
	 * Provision Currency in which the valuation amount is denominated.
	 *
	 */
	ISOCurrencyCodeEnum getValuationCurrency();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "112" * field "Valuation Method"
	 *
	 * Provision Source and method used for the valuation of the transaction by the reporting counterparty. If at least one valuation input is used that is classified as mark-to-model in appendix D, then the whole valuation is classified as mark-to-model. If only inputs are used that are classified as mark-to-market in appendix D, then the whole valuation is classified as mark-to-market.
	 *
	 */
	ValuationType1Code getValuationMethod();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "113" * field "Valuation Timestamp"
	 *
	 * Provision Date and time of the last valuation marked to market, provided by the central counterparty (CCP) or calculated using the current or last available market price of the inputs. If for example a currency exchange rate is the basis for a transactions valuation, then the valuation timestamp reflects the moment in time that exchange rate was current.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "113" * field "Valuation Timestamp" * footnote "79"
	 *
	 * Provision Reported by the derivatives clearing organization (DCO) for cleared swaps and by the swap dealer for uncleared swaps
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "113" * field "Valuation Timestamp" * footnote "80"
	 *
	 * Provision The timestamp portion is not required to be represented for Valuation timestamp. The format must be reported as YYYY-MM-DD.
	 *
	 */
	ZonedDateTime getValuationTimestamp();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code"
	 *
	 * Provision If collateral is reported on a portfolio basis, a unique code assigned by the reporting counterparty to the portfolio that tracks the aggregate initial margin of a set of open swap transactions. This data element is not applicable if the collateralisation was performed on a transaction level basis, or if there is no collateral agreement, or if no collateral is posted or received. The portfolio code is required for both collateral reporting and valuation reporting in order to link the 2 data sets.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code" * footnote "82"
	 *
	 * Provision If collateralization was performed on a transaction level basis, 'TRANSACTION-LEVEL' is accepted. “NOTAPPLICABLE” is accepted if (i) collateralization was performed on a portfolio basis and there is no IM portfolio code, or (ii) it is a submission from a DCO. 
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "116" * field "Initial Margin Collateral Portfolio Code" * footnote "83"
	 *
	 * Provision  The M in the Part 43/45 Asset Class column is for collateral and valuation reporting only.
	 *
	 */
	String getInitialMarginCollateralPortfolioCode();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code"
	 *
	 * Provision If collateral is reported on a portfolio basis, a unique code assigned by the reporting counterparty to the portfolio that tracks the aggregate variation margin related to a set of open swap transactions. This data element is not applicable if the collateralisation was performed on a transaction level basis, or if there is no collateral agreement, or if no collateral is posted or received. The portfolio code is required for both collateral reporting and valuation reporting in order to link the 2 data sets.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code" * footnote "90"
	 *
	 * Provision If collateralization was performed on a transaction level basis, 'TRANSACTION-LEVEL' is accepted. “NOTAPPLICABLE” is accepted if (i) collateralization was performed on a portfolio basis and there is no VM portfolio code, or (ii) it is a submission from a DCO.
	 *
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation   
	 * appendix "1" * dataElement "124" * field "Variation Margin Collateral Portfolio Code" * footnote "91"
	 *
	 * Provision The 'M' in the 'Part 43/45 Asset Class' column is for collateral and valuation reporting only.
	 *
	 */
	String getVariationMarginCollateralPortfolioCode();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - ID Type"
	 *
	 * Provision GTR required Control Field.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup CanadianDataAndReporting ISDA Data and Reporting Canadian Working Group "This distribution list is responsible for addressing industry compliance and implementation issues related to trade reporting under Canadian regulatory requirements across provinces. The group has been the driving force for work such as the Multi-lateral non-dealer Canadian reporting agreement, regulatory response letters, and reporting discussions with the OSC, MSC, Quebec, BCSC and ASC." 
	 * date "20250508" * field "Trade Party 1 - ID Type"
	 *
	 * Provision Reporting firms on the WG call said they would be using an LEI. There is only a possible use case for crypto trades reporting by SEFs; however, DRR is not going to model this case until is demanded by firms.
	 *
	 */
	PartyIdentifierTypeEnum getTradeParty1IDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Submitting Party - ID Type"
	 *
	 * Provision GTR required Control Field.
	 *
	 */
	PartyIdentifierTypeEnum getSubmittingPartyIDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "USI ID Prefix"
	 *
	 * Provision The USI is a unique identifier assigned to all swap transactions which identifies the transaction (the swap and its counterparties) uniquely throughout its duration. It consists of a namespace and a transaction identifier.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250327" * field "USI ID Prefix"
	 *
	 * Provision The group agreed on not considering USI ID Prefix due to it comes from an old format and it had been agreed to only consider one field, that is USI ID, considering the concatenation of both fields. Thus, USI ID is the only field that needs to be populated.
	 *
	 */
	String getUsiIDPrefix();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "UTI ID Prefix"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle and used for all recordkeeping and all swap data reporting pursuant to §45.5. A UTI is comprised of the LEI of the generating entity and a unique alphanumeric code.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250327" * field "UTI ID Prefix"
	 *
	 * Provision The group agreed on not considering UTI ID Prefix due to it comes from an old format and it had been agreed to only consider one field, that is UTI ID, considering the concatenation of both fields. Thus, UTI ID is the only field that needs to be populated.
	 *
	 */
	String getUtiIDPrefix();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Submitted For Party"
	 *
	 * Provision Need field for GTR Processing.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250507"
	 *
	 * Provision Agreed to default the output to the LEI of the reporting party.
	 *
	 */
	String getSubmittedForParty();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - Reporting Destination"
	 *
	 * Provision Needed to report Jurisdiction.
	 *
	 */
	List<SupervisoryBodyEnum> getTradeParty1ReportingDestination();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 2 - Reporting Destination"
	 *
	 * Provision Needed to report Jurisdiction.
	 *
	 */
	List<SupervisoryBodyEnum> getTradeParty2ReportingDestination();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Primary Asset Class"
	 *
	 * Provision UPI RDL: Indicates whether the asset, benchmark or another derivatives contract underlying a derivatives contract is, or references, an equity, rate, credit, commodity or foreign exchange asset.
	 *
	 */
	AssetClassEnum getPrimaryAssetClass();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Comment 1"
	 *
	 * Provision For CSV submissions if  populated with an * the entire line is treated as comment.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250507" * field "Comment 1"
	 *
	 * Provision Empty because CSV submission is still not supported by current DRR model. This rule will be revisited once Harmonized CSV projection is included.
	 *
	 */
	String getComment1();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Message ID"
	 *
	 * Provision Purpose of this field is to allow firms to submit a unique ID that will allow them to tie in specific submissions/modifications. For example if 5 modifications are submitted on a specific PET, this will allow the firm to ensure they receive responses for all 5 modifications.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup EUUKNAPeerReviewGroup ISDA EU/UK/North America Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended EU/UK EMIR and North America transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20250604" * field "Message ID"
	 *
	 * Provision Each Message ID must be unique, irrespective of the report type. This unique identification is achieved by concatenating with the Reporting Timestamp field.
	 *
	 */
	String getMessageID();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Message Type"
	 *
	 * Provision Need to indicate what message is being submitted to GTR.
	 *
	 */
	String getMessageType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 2 - Execution Agent ID"
	 *
	 * Provision To allow submitter to name an execution agent to view the report data. 
	 *
	 */
	String getTradeParty2ExecutionAgentID();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 2 - Execution Agent ID Type"
	 *
	 * Provision To allow submitter to name an execution agent to view the report data. 
	 *
	 */
	PartyIdentifierTypeEnum getTradeParty2ExecutionAgentIDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - Execution Agent ID"
	 *
	 * Provision To allow submitter to name an execution agent to view the report data. 
	 *
	 */
	String getTradeParty1ExecutionAgentID();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - Execution Agent ID Type"
	 *
	 * Provision To allow submitter to name an execution agent to view the report data. 
	 *
	 */
	PartyIdentifierTypeEnum getTradeParty1ExecutionAgentIDType();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Trade Party 1 - Transaction ID"
	 *
	 * Provision Internal trade identifier for Counterparty 1 to map to their internal system.
	 *
	 */
	String getTradeParty1TransactionID();
	/**
	 *
	 * Body CFTC
	 * Corpus Dissemination Valuation    * Corpus Specifications DTCC_Specs DTCC Specs "Document providing the message specifications required for inbound message for DTCC for CFTC." 
	 * field "Version"
	 *
	 * Provision Memo field. Used to identify the asset class and template type of submission.
	 *
	 */
	String getVersion();

	/*********************** Build Methods  ***********************/
	CFTCValuationReport build();
	
	CFTCValuationReport.CFTCValuationReportBuilder toBuilder();
	
	static CFTCValuationReport.CFTCValuationReportBuilder builder() {
		return new CFTCValuationReport.CFTCValuationReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends CFTCValuationReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends CFTCValuationReport> getType() {
		return CFTCValuationReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
		processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
		processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("uniqueProductIdentifier"), String.class, getUniqueProductIdentifier(), this);
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("uniqueSwapIdentifier"), String.class, getUniqueSwapIdentifier(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
		processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg1"), Date.class, getNextFloatingReferenceResetDateLeg1(), this);
		processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg2"), Date.class, getNextFloatingReferenceResetDateLeg2(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg1"), BigDecimal.class, getLastFloatingReferenceValueLeg1(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg2"), BigDecimal.class, getLastFloatingReferenceValueLeg2(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg1"), Date.class, getLastFloatingReferenceResetDateLeg1(), this);
		processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg2"), Date.class, getLastFloatingReferenceResetDateLeg2(), this);
		processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
		processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
		processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
		processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
		processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
		processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
		processor.processBasic(path.newSubPath("tradeParty1IDType"), PartyIdentifierTypeEnum.class, getTradeParty1IDType(), this);
		processor.processBasic(path.newSubPath("submittingPartyIDType"), PartyIdentifierTypeEnum.class, getSubmittingPartyIDType(), this);
		processor.processBasic(path.newSubPath("usiIDPrefix"), String.class, getUsiIDPrefix(), this);
		processor.processBasic(path.newSubPath("utiIDPrefix"), String.class, getUtiIDPrefix(), this);
		processor.processBasic(path.newSubPath("submittedForParty"), String.class, getSubmittedForParty(), this);
		processor.processBasic(path.newSubPath("tradeParty1ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty1ReportingDestination(), this);
		processor.processBasic(path.newSubPath("tradeParty2ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty2ReportingDestination(), this);
		processor.processBasic(path.newSubPath("primaryAssetClass"), AssetClassEnum.class, getPrimaryAssetClass(), this);
		processor.processBasic(path.newSubPath("comment1"), String.class, getComment1(), this);
		processor.processBasic(path.newSubPath("messageID"), String.class, getMessageID(), this);
		processor.processBasic(path.newSubPath("messageType"), String.class, getMessageType(), this);
		processor.processBasic(path.newSubPath("tradeParty2ExecutionAgentID"), String.class, getTradeParty2ExecutionAgentID(), this);
		processor.processBasic(path.newSubPath("tradeParty2ExecutionAgentIDType"), PartyIdentifierTypeEnum.class, getTradeParty2ExecutionAgentIDType(), this);
		processor.processBasic(path.newSubPath("tradeParty1ExecutionAgentID"), String.class, getTradeParty1ExecutionAgentID(), this);
		processor.processBasic(path.newSubPath("tradeParty1ExecutionAgentIDType"), PartyIdentifierTypeEnum.class, getTradeParty1ExecutionAgentIDType(), this);
		processor.processBasic(path.newSubPath("tradeParty1TransactionID"), String.class, getTradeParty1TransactionID(), this);
		processor.processBasic(path.newSubPath("version"), String.class, getVersion(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface CFTCValuationReportBuilder extends CFTCValuationReport, RosettaModelObjectBuilder {
		CFTCValuationReport.CFTCValuationReportBuilder setCounterparty1(String counterparty1);
		CFTCValuationReport.CFTCValuationReportBuilder setCounterparty2(String counterparty2);
		CFTCValuationReport.CFTCValuationReportBuilder setCounterparty2IdentifierSource(String counterparty2IdentifierSource);
		CFTCValuationReport.CFTCValuationReportBuilder setSubmitterIdentifier(String submitterIdentifier);
		CFTCValuationReport.CFTCValuationReportBuilder setActionType(ActionTypeEnum actionType);
		CFTCValuationReport.CFTCValuationReportBuilder setUniqueProductIdentifier(String uniqueProductIdentifier);
		CFTCValuationReport.CFTCValuationReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		CFTCValuationReport.CFTCValuationReportBuilder setUniqueSwapIdentifier(String uniqueSwapIdentifier);
		CFTCValuationReport.CFTCValuationReportBuilder setUniqueTransactionIdentifier(String uniqueTransactionIdentifier);
		CFTCValuationReport.CFTCValuationReportBuilder setNextFloatingReferenceResetDateLeg1(Date nextFloatingReferenceResetDateLeg1);
		CFTCValuationReport.CFTCValuationReportBuilder setNextFloatingReferenceResetDateLeg2(Date nextFloatingReferenceResetDateLeg2);
		CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceValueLeg1(BigDecimal lastFloatingReferenceValueLeg1);
		CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceValueLeg2(BigDecimal lastFloatingReferenceValueLeg2);
		CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceResetDateLeg1(Date lastFloatingReferenceResetDateLeg1);
		CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceResetDateLeg2(Date lastFloatingReferenceResetDateLeg2);
		CFTCValuationReport.CFTCValuationReportBuilder setDelta(BigDecimal delta);
		CFTCValuationReport.CFTCValuationReportBuilder setValuationAmount(BigDecimal valuationAmount);
		CFTCValuationReport.CFTCValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum valuationCurrency);
		CFTCValuationReport.CFTCValuationReportBuilder setValuationMethod(ValuationType1Code valuationMethod);
		CFTCValuationReport.CFTCValuationReportBuilder setValuationTimestamp(ZonedDateTime valuationTimestamp);
		CFTCValuationReport.CFTCValuationReportBuilder setInitialMarginCollateralPortfolioCode(String initialMarginCollateralPortfolioCode);
		CFTCValuationReport.CFTCValuationReportBuilder setVariationMarginCollateralPortfolioCode(String variationMarginCollateralPortfolioCode);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1IDType(PartyIdentifierTypeEnum tradeParty1IDType);
		CFTCValuationReport.CFTCValuationReportBuilder setSubmittingPartyIDType(PartyIdentifierTypeEnum submittingPartyIDType);
		CFTCValuationReport.CFTCValuationReportBuilder setUsiIDPrefix(String usiIDPrefix);
		CFTCValuationReport.CFTCValuationReportBuilder setUtiIDPrefix(String utiIDPrefix);
		CFTCValuationReport.CFTCValuationReportBuilder setSubmittedForParty(String submittedForParty);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum tradeParty1ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum tradeParty1ReportingDestination, int idx);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(SupervisoryBodyEnum tradeParty2ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(SupervisoryBodyEnum tradeParty2ReportingDestination, int idx);
		CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(List<SupervisoryBodyEnum> tradeParty2ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ReportingDestination(List<SupervisoryBodyEnum> tradeParty2ReportingDestination);
		CFTCValuationReport.CFTCValuationReportBuilder setPrimaryAssetClass(AssetClassEnum primaryAssetClass);
		CFTCValuationReport.CFTCValuationReportBuilder setComment1(String comment1);
		CFTCValuationReport.CFTCValuationReportBuilder setMessageID(String messageID);
		CFTCValuationReport.CFTCValuationReportBuilder setMessageType(String messageType);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ExecutionAgentID(String tradeParty2ExecutionAgentID);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ExecutionAgentIDType(PartyIdentifierTypeEnum tradeParty2ExecutionAgentIDType);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ExecutionAgentID(String tradeParty1ExecutionAgentID);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ExecutionAgentIDType(PartyIdentifierTypeEnum tradeParty1ExecutionAgentIDType);
		CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1TransactionID(String tradeParty1TransactionID);
		CFTCValuationReport.CFTCValuationReportBuilder setVersion(String version);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierSource"), String.class, getCounterparty2IdentifierSource(), this);
			processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
			processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("uniqueProductIdentifier"), String.class, getUniqueProductIdentifier(), this);
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("uniqueSwapIdentifier"), String.class, getUniqueSwapIdentifier(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
			processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg1"), Date.class, getNextFloatingReferenceResetDateLeg1(), this);
			processor.processBasic(path.newSubPath("nextFloatingReferenceResetDateLeg2"), Date.class, getNextFloatingReferenceResetDateLeg2(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg1"), BigDecimal.class, getLastFloatingReferenceValueLeg1(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceValueLeg2"), BigDecimal.class, getLastFloatingReferenceValueLeg2(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg1"), Date.class, getLastFloatingReferenceResetDateLeg1(), this);
			processor.processBasic(path.newSubPath("lastFloatingReferenceResetDateLeg2"), Date.class, getLastFloatingReferenceResetDateLeg2(), this);
			processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
			processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
			processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
			processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
			processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
			processor.processBasic(path.newSubPath("initialMarginCollateralPortfolioCode"), String.class, getInitialMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("variationMarginCollateralPortfolioCode"), String.class, getVariationMarginCollateralPortfolioCode(), this);
			processor.processBasic(path.newSubPath("tradeParty1IDType"), PartyIdentifierTypeEnum.class, getTradeParty1IDType(), this);
			processor.processBasic(path.newSubPath("submittingPartyIDType"), PartyIdentifierTypeEnum.class, getSubmittingPartyIDType(), this);
			processor.processBasic(path.newSubPath("usiIDPrefix"), String.class, getUsiIDPrefix(), this);
			processor.processBasic(path.newSubPath("utiIDPrefix"), String.class, getUtiIDPrefix(), this);
			processor.processBasic(path.newSubPath("submittedForParty"), String.class, getSubmittedForParty(), this);
			processor.processBasic(path.newSubPath("tradeParty1ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty1ReportingDestination(), this);
			processor.processBasic(path.newSubPath("tradeParty2ReportingDestination"), SupervisoryBodyEnum.class, getTradeParty2ReportingDestination(), this);
			processor.processBasic(path.newSubPath("primaryAssetClass"), AssetClassEnum.class, getPrimaryAssetClass(), this);
			processor.processBasic(path.newSubPath("comment1"), String.class, getComment1(), this);
			processor.processBasic(path.newSubPath("messageID"), String.class, getMessageID(), this);
			processor.processBasic(path.newSubPath("messageType"), String.class, getMessageType(), this);
			processor.processBasic(path.newSubPath("tradeParty2ExecutionAgentID"), String.class, getTradeParty2ExecutionAgentID(), this);
			processor.processBasic(path.newSubPath("tradeParty2ExecutionAgentIDType"), PartyIdentifierTypeEnum.class, getTradeParty2ExecutionAgentIDType(), this);
			processor.processBasic(path.newSubPath("tradeParty1ExecutionAgentID"), String.class, getTradeParty1ExecutionAgentID(), this);
			processor.processBasic(path.newSubPath("tradeParty1ExecutionAgentIDType"), PartyIdentifierTypeEnum.class, getTradeParty1ExecutionAgentIDType(), this);
			processor.processBasic(path.newSubPath("tradeParty1TransactionID"), String.class, getTradeParty1TransactionID(), this);
			processor.processBasic(path.newSubPath("version"), String.class, getVersion(), this);
		}
		

		CFTCValuationReport.CFTCValuationReportBuilder prune();
	}

	/*********************** Immutable Implementation of CFTCValuationReport  ***********************/
	class CFTCValuationReportImpl implements CFTCValuationReport {
		private final String counterparty1;
		private final String counterparty2;
		private final String counterparty2IdentifierSource;
		private final String submitterIdentifier;
		private final ActionTypeEnum actionType;
		private final String uniqueProductIdentifier;
		private final ZonedDateTime reportingTimestamp;
		private final String uniqueSwapIdentifier;
		private final String uniqueTransactionIdentifier;
		private final Date nextFloatingReferenceResetDateLeg1;
		private final Date nextFloatingReferenceResetDateLeg2;
		private final BigDecimal lastFloatingReferenceValueLeg1;
		private final BigDecimal lastFloatingReferenceValueLeg2;
		private final Date lastFloatingReferenceResetDateLeg1;
		private final Date lastFloatingReferenceResetDateLeg2;
		private final BigDecimal delta;
		private final BigDecimal valuationAmount;
		private final ISOCurrencyCodeEnum valuationCurrency;
		private final ValuationType1Code valuationMethod;
		private final ZonedDateTime valuationTimestamp;
		private final String initialMarginCollateralPortfolioCode;
		private final String variationMarginCollateralPortfolioCode;
		private final PartyIdentifierTypeEnum tradeParty1IDType;
		private final PartyIdentifierTypeEnum submittingPartyIDType;
		private final String usiIDPrefix;
		private final String utiIDPrefix;
		private final String submittedForParty;
		private final List<SupervisoryBodyEnum> tradeParty1ReportingDestination;
		private final List<SupervisoryBodyEnum> tradeParty2ReportingDestination;
		private final AssetClassEnum primaryAssetClass;
		private final String comment1;
		private final String messageID;
		private final String messageType;
		private final String tradeParty2ExecutionAgentID;
		private final PartyIdentifierTypeEnum tradeParty2ExecutionAgentIDType;
		private final String tradeParty1ExecutionAgentID;
		private final PartyIdentifierTypeEnum tradeParty1ExecutionAgentIDType;
		private final String tradeParty1TransactionID;
		private final String version;
		
		protected CFTCValuationReportImpl(CFTCValuationReport.CFTCValuationReportBuilder builder) {
			this.counterparty1 = builder.getCounterparty1();
			this.counterparty2 = builder.getCounterparty2();
			this.counterparty2IdentifierSource = builder.getCounterparty2IdentifierSource();
			this.submitterIdentifier = builder.getSubmitterIdentifier();
			this.actionType = builder.getActionType();
			this.uniqueProductIdentifier = builder.getUniqueProductIdentifier();
			this.reportingTimestamp = builder.getReportingTimestamp();
			this.uniqueSwapIdentifier = builder.getUniqueSwapIdentifier();
			this.uniqueTransactionIdentifier = builder.getUniqueTransactionIdentifier();
			this.nextFloatingReferenceResetDateLeg1 = builder.getNextFloatingReferenceResetDateLeg1();
			this.nextFloatingReferenceResetDateLeg2 = builder.getNextFloatingReferenceResetDateLeg2();
			this.lastFloatingReferenceValueLeg1 = builder.getLastFloatingReferenceValueLeg1();
			this.lastFloatingReferenceValueLeg2 = builder.getLastFloatingReferenceValueLeg2();
			this.lastFloatingReferenceResetDateLeg1 = builder.getLastFloatingReferenceResetDateLeg1();
			this.lastFloatingReferenceResetDateLeg2 = builder.getLastFloatingReferenceResetDateLeg2();
			this.delta = builder.getDelta();
			this.valuationAmount = builder.getValuationAmount();
			this.valuationCurrency = builder.getValuationCurrency();
			this.valuationMethod = builder.getValuationMethod();
			this.valuationTimestamp = builder.getValuationTimestamp();
			this.initialMarginCollateralPortfolioCode = builder.getInitialMarginCollateralPortfolioCode();
			this.variationMarginCollateralPortfolioCode = builder.getVariationMarginCollateralPortfolioCode();
			this.tradeParty1IDType = builder.getTradeParty1IDType();
			this.submittingPartyIDType = builder.getSubmittingPartyIDType();
			this.usiIDPrefix = builder.getUsiIDPrefix();
			this.utiIDPrefix = builder.getUtiIDPrefix();
			this.submittedForParty = builder.getSubmittedForParty();
			this.tradeParty1ReportingDestination = ofNullable(builder.getTradeParty1ReportingDestination()).filter(_l->!_l.isEmpty()).map(ImmutableList::copyOf).orElse(null);
			this.tradeParty2ReportingDestination = ofNullable(builder.getTradeParty2ReportingDestination()).filter(_l->!_l.isEmpty()).map(ImmutableList::copyOf).orElse(null);
			this.primaryAssetClass = builder.getPrimaryAssetClass();
			this.comment1 = builder.getComment1();
			this.messageID = builder.getMessageID();
			this.messageType = builder.getMessageType();
			this.tradeParty2ExecutionAgentID = builder.getTradeParty2ExecutionAgentID();
			this.tradeParty2ExecutionAgentIDType = builder.getTradeParty2ExecutionAgentIDType();
			this.tradeParty1ExecutionAgentID = builder.getTradeParty1ExecutionAgentID();
			this.tradeParty1ExecutionAgentIDType = builder.getTradeParty1ExecutionAgentIDType();
			this.tradeParty1TransactionID = builder.getTradeParty1TransactionID();
			this.version = builder.getVersion();
		}
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		public String getCounterparty2IdentifierSource() {
			return counterparty2IdentifierSource;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public ActionTypeEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="uniqueProductIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueProductIdentifier", isRequired=true)
		public String getUniqueProductIdentifier() {
			return uniqueProductIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute("uniqueSwapIdentifier")
		@RuneAttribute("uniqueSwapIdentifier")
		public String getUniqueSwapIdentifier() {
			return uniqueSwapIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("nextFloatingReferenceResetDateLeg1")
		@RuneAttribute("nextFloatingReferenceResetDateLeg1")
		public Date getNextFloatingReferenceResetDateLeg1() {
			return nextFloatingReferenceResetDateLeg1;
		}
		
		@Override
		@RosettaAttribute("nextFloatingReferenceResetDateLeg2")
		@RuneAttribute("nextFloatingReferenceResetDateLeg2")
		public Date getNextFloatingReferenceResetDateLeg2() {
			return nextFloatingReferenceResetDateLeg2;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceValueLeg1")
		@RuneAttribute("lastFloatingReferenceValueLeg1")
		public BigDecimal getLastFloatingReferenceValueLeg1() {
			return lastFloatingReferenceValueLeg1;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceValueLeg2")
		@RuneAttribute("lastFloatingReferenceValueLeg2")
		public BigDecimal getLastFloatingReferenceValueLeg2() {
			return lastFloatingReferenceValueLeg2;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceResetDateLeg1")
		@RuneAttribute("lastFloatingReferenceResetDateLeg1")
		public Date getLastFloatingReferenceResetDateLeg1() {
			return lastFloatingReferenceResetDateLeg1;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceResetDateLeg2")
		@RuneAttribute("lastFloatingReferenceResetDateLeg2")
		public Date getLastFloatingReferenceResetDateLeg2() {
			return lastFloatingReferenceResetDateLeg2;
		}
		
		@Override
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		public BigDecimal getDelta() {
			return delta;
		}
		
		@Override
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		public BigDecimal getValuationAmount() {
			return valuationAmount;
		}
		
		@Override
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		public ISOCurrencyCodeEnum getValuationCurrency() {
			return valuationCurrency;
		}
		
		@Override
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		public ValuationType1Code getValuationMethod() {
			return valuationMethod;
		}
		
		@Override
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		public ZonedDateTime getValuationTimestamp() {
			return valuationTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		public String getInitialMarginCollateralPortfolioCode() {
			return initialMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		public String getVariationMarginCollateralPortfolioCode() {
			return variationMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		public PartyIdentifierTypeEnum getTradeParty1IDType() {
			return tradeParty1IDType;
		}
		
		@Override
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		public PartyIdentifierTypeEnum getSubmittingPartyIDType() {
			return submittingPartyIDType;
		}
		
		@Override
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		public String getUsiIDPrefix() {
			return usiIDPrefix;
		}
		
		@Override
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		public String getUtiIDPrefix() {
			return utiIDPrefix;
		}
		
		@Override
		@RosettaAttribute(value="submittedForParty", isRequired=true)
		@RuneAttribute(value="submittedForParty", isRequired=true)
		public String getSubmittedForParty() {
			return submittedForParty;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty1ReportingDestination() {
			return tradeParty1ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty2ReportingDestination() {
			return tradeParty2ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="primaryAssetClass", isRequired=true)
		@RuneAttribute(value="primaryAssetClass", isRequired=true)
		public AssetClassEnum getPrimaryAssetClass() {
			return primaryAssetClass;
		}
		
		@Override
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		public String getComment1() {
			return comment1;
		}
		
		@Override
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		public String getMessageID() {
			return messageID;
		}
		
		@Override
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		public String getMessageType() {
			return messageType;
		}
		
		@Override
		@RosettaAttribute("tradeParty2ExecutionAgentID")
		@RuneAttribute("tradeParty2ExecutionAgentID")
		public String getTradeParty2ExecutionAgentID() {
			return tradeParty2ExecutionAgentID;
		}
		
		@Override
		@RosettaAttribute("tradeParty2ExecutionAgentIDType")
		@RuneAttribute("tradeParty2ExecutionAgentIDType")
		public PartyIdentifierTypeEnum getTradeParty2ExecutionAgentIDType() {
			return tradeParty2ExecutionAgentIDType;
		}
		
		@Override
		@RosettaAttribute("tradeParty1ExecutionAgentID")
		@RuneAttribute("tradeParty1ExecutionAgentID")
		public String getTradeParty1ExecutionAgentID() {
			return tradeParty1ExecutionAgentID;
		}
		
		@Override
		@RosettaAttribute("tradeParty1ExecutionAgentIDType")
		@RuneAttribute("tradeParty1ExecutionAgentIDType")
		public PartyIdentifierTypeEnum getTradeParty1ExecutionAgentIDType() {
			return tradeParty1ExecutionAgentIDType;
		}
		
		@Override
		@RosettaAttribute("tradeParty1TransactionID")
		@RuneAttribute("tradeParty1TransactionID")
		public String getTradeParty1TransactionID() {
			return tradeParty1TransactionID;
		}
		
		@Override
		@RosettaAttribute("version")
		@RuneAttribute("version")
		public String getVersion() {
			return version;
		}
		
		@Override
		public CFTCValuationReport build() {
			return this;
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder toBuilder() {
			CFTCValuationReport.CFTCValuationReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(CFTCValuationReport.CFTCValuationReportBuilder builder) {
			ofNullable(getCounterparty1()).ifPresent(builder::setCounterparty1);
			ofNullable(getCounterparty2()).ifPresent(builder::setCounterparty2);
			ofNullable(getCounterparty2IdentifierSource()).ifPresent(builder::setCounterparty2IdentifierSource);
			ofNullable(getSubmitterIdentifier()).ifPresent(builder::setSubmitterIdentifier);
			ofNullable(getActionType()).ifPresent(builder::setActionType);
			ofNullable(getUniqueProductIdentifier()).ifPresent(builder::setUniqueProductIdentifier);
			ofNullable(getReportingTimestamp()).ifPresent(builder::setReportingTimestamp);
			ofNullable(getUniqueSwapIdentifier()).ifPresent(builder::setUniqueSwapIdentifier);
			ofNullable(getUniqueTransactionIdentifier()).ifPresent(builder::setUniqueTransactionIdentifier);
			ofNullable(getNextFloatingReferenceResetDateLeg1()).ifPresent(builder::setNextFloatingReferenceResetDateLeg1);
			ofNullable(getNextFloatingReferenceResetDateLeg2()).ifPresent(builder::setNextFloatingReferenceResetDateLeg2);
			ofNullable(getLastFloatingReferenceValueLeg1()).ifPresent(builder::setLastFloatingReferenceValueLeg1);
			ofNullable(getLastFloatingReferenceValueLeg2()).ifPresent(builder::setLastFloatingReferenceValueLeg2);
			ofNullable(getLastFloatingReferenceResetDateLeg1()).ifPresent(builder::setLastFloatingReferenceResetDateLeg1);
			ofNullable(getLastFloatingReferenceResetDateLeg2()).ifPresent(builder::setLastFloatingReferenceResetDateLeg2);
			ofNullable(getDelta()).ifPresent(builder::setDelta);
			ofNullable(getValuationAmount()).ifPresent(builder::setValuationAmount);
			ofNullable(getValuationCurrency()).ifPresent(builder::setValuationCurrency);
			ofNullable(getValuationMethod()).ifPresent(builder::setValuationMethod);
			ofNullable(getValuationTimestamp()).ifPresent(builder::setValuationTimestamp);
			ofNullable(getInitialMarginCollateralPortfolioCode()).ifPresent(builder::setInitialMarginCollateralPortfolioCode);
			ofNullable(getVariationMarginCollateralPortfolioCode()).ifPresent(builder::setVariationMarginCollateralPortfolioCode);
			ofNullable(getTradeParty1IDType()).ifPresent(builder::setTradeParty1IDType);
			ofNullable(getSubmittingPartyIDType()).ifPresent(builder::setSubmittingPartyIDType);
			ofNullable(getUsiIDPrefix()).ifPresent(builder::setUsiIDPrefix);
			ofNullable(getUtiIDPrefix()).ifPresent(builder::setUtiIDPrefix);
			ofNullable(getSubmittedForParty()).ifPresent(builder::setSubmittedForParty);
			ofNullable(getTradeParty1ReportingDestination()).ifPresent(builder::setTradeParty1ReportingDestination);
			ofNullable(getTradeParty2ReportingDestination()).ifPresent(builder::setTradeParty2ReportingDestination);
			ofNullable(getPrimaryAssetClass()).ifPresent(builder::setPrimaryAssetClass);
			ofNullable(getComment1()).ifPresent(builder::setComment1);
			ofNullable(getMessageID()).ifPresent(builder::setMessageID);
			ofNullable(getMessageType()).ifPresent(builder::setMessageType);
			ofNullable(getTradeParty2ExecutionAgentID()).ifPresent(builder::setTradeParty2ExecutionAgentID);
			ofNullable(getTradeParty2ExecutionAgentIDType()).ifPresent(builder::setTradeParty2ExecutionAgentIDType);
			ofNullable(getTradeParty1ExecutionAgentID()).ifPresent(builder::setTradeParty1ExecutionAgentID);
			ofNullable(getTradeParty1ExecutionAgentIDType()).ifPresent(builder::setTradeParty1ExecutionAgentIDType);
			ofNullable(getTradeParty1TransactionID()).ifPresent(builder::setTradeParty1TransactionID);
			ofNullable(getVersion()).ifPresent(builder::setVersion);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CFTCValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierSource, _that.getCounterparty2IdentifierSource())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(uniqueProductIdentifier, _that.getUniqueProductIdentifier())) return false;
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(uniqueSwapIdentifier, _that.getUniqueSwapIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(nextFloatingReferenceResetDateLeg1, _that.getNextFloatingReferenceResetDateLeg1())) return false;
			if (!Objects.equals(nextFloatingReferenceResetDateLeg2, _that.getNextFloatingReferenceResetDateLeg2())) return false;
			if (!Objects.equals(lastFloatingReferenceValueLeg1, _that.getLastFloatingReferenceValueLeg1())) return false;
			if (!Objects.equals(lastFloatingReferenceValueLeg2, _that.getLastFloatingReferenceValueLeg2())) return false;
			if (!Objects.equals(lastFloatingReferenceResetDateLeg1, _that.getLastFloatingReferenceResetDateLeg1())) return false;
			if (!Objects.equals(lastFloatingReferenceResetDateLeg2, _that.getLastFloatingReferenceResetDateLeg2())) return false;
			if (!Objects.equals(delta, _that.getDelta())) return false;
			if (!Objects.equals(valuationAmount, _that.getValuationAmount())) return false;
			if (!Objects.equals(valuationCurrency, _that.getValuationCurrency())) return false;
			if (!Objects.equals(valuationMethod, _that.getValuationMethod())) return false;
			if (!Objects.equals(valuationTimestamp, _that.getValuationTimestamp())) return false;
			if (!Objects.equals(initialMarginCollateralPortfolioCode, _that.getInitialMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(variationMarginCollateralPortfolioCode, _that.getVariationMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(tradeParty1IDType, _that.getTradeParty1IDType())) return false;
			if (!Objects.equals(submittingPartyIDType, _that.getSubmittingPartyIDType())) return false;
			if (!Objects.equals(usiIDPrefix, _that.getUsiIDPrefix())) return false;
			if (!Objects.equals(utiIDPrefix, _that.getUtiIDPrefix())) return false;
			if (!Objects.equals(submittedForParty, _that.getSubmittedForParty())) return false;
			if (!ListEquals.listEquals(tradeParty1ReportingDestination, _that.getTradeParty1ReportingDestination())) return false;
			if (!ListEquals.listEquals(tradeParty2ReportingDestination, _that.getTradeParty2ReportingDestination())) return false;
			if (!Objects.equals(primaryAssetClass, _that.getPrimaryAssetClass())) return false;
			if (!Objects.equals(comment1, _that.getComment1())) return false;
			if (!Objects.equals(messageID, _that.getMessageID())) return false;
			if (!Objects.equals(messageType, _that.getMessageType())) return false;
			if (!Objects.equals(tradeParty2ExecutionAgentID, _that.getTradeParty2ExecutionAgentID())) return false;
			if (!Objects.equals(tradeParty2ExecutionAgentIDType, _that.getTradeParty2ExecutionAgentIDType())) return false;
			if (!Objects.equals(tradeParty1ExecutionAgentID, _that.getTradeParty1ExecutionAgentID())) return false;
			if (!Objects.equals(tradeParty1ExecutionAgentIDType, _that.getTradeParty1ExecutionAgentIDType())) return false;
			if (!Objects.equals(tradeParty1TransactionID, _that.getTradeParty1TransactionID())) return false;
			if (!Objects.equals(version, _that.getVersion())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierSource != null ? counterparty2IdentifierSource.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (uniqueProductIdentifier != null ? uniqueProductIdentifier.hashCode() : 0);
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (uniqueSwapIdentifier != null ? uniqueSwapIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (nextFloatingReferenceResetDateLeg1 != null ? nextFloatingReferenceResetDateLeg1.hashCode() : 0);
			_result = 31 * _result + (nextFloatingReferenceResetDateLeg2 != null ? nextFloatingReferenceResetDateLeg2.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceValueLeg1 != null ? lastFloatingReferenceValueLeg1.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceValueLeg2 != null ? lastFloatingReferenceValueLeg2.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceResetDateLeg1 != null ? lastFloatingReferenceResetDateLeg1.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceResetDateLeg2 != null ? lastFloatingReferenceResetDateLeg2.hashCode() : 0);
			_result = 31 * _result + (delta != null ? delta.hashCode() : 0);
			_result = 31 * _result + (valuationAmount != null ? valuationAmount.hashCode() : 0);
			_result = 31 * _result + (valuationCurrency != null ? valuationCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationMethod != null ? valuationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationTimestamp != null ? valuationTimestamp.hashCode() : 0);
			_result = 31 * _result + (initialMarginCollateralPortfolioCode != null ? initialMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (variationMarginCollateralPortfolioCode != null ? variationMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (tradeParty1IDType != null ? tradeParty1IDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (submittingPartyIDType != null ? submittingPartyIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (usiIDPrefix != null ? usiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (utiIDPrefix != null ? utiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (submittedForParty != null ? submittedForParty.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ReportingDestination != null ? tradeParty1ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (tradeParty2ReportingDestination != null ? tradeParty2ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (primaryAssetClass != null ? primaryAssetClass.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (comment1 != null ? comment1.hashCode() : 0);
			_result = 31 * _result + (messageID != null ? messageID.hashCode() : 0);
			_result = 31 * _result + (messageType != null ? messageType.hashCode() : 0);
			_result = 31 * _result + (tradeParty2ExecutionAgentID != null ? tradeParty2ExecutionAgentID.hashCode() : 0);
			_result = 31 * _result + (tradeParty2ExecutionAgentIDType != null ? tradeParty2ExecutionAgentIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1ExecutionAgentID != null ? tradeParty1ExecutionAgentID.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ExecutionAgentIDType != null ? tradeParty1ExecutionAgentIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1TransactionID != null ? tradeParty1TransactionID.hashCode() : 0);
			_result = 31 * _result + (version != null ? version.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CFTCValuationReport {" +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierSource=" + this.counterparty2IdentifierSource + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"actionType=" + this.actionType + ", " +
				"uniqueProductIdentifier=" + this.uniqueProductIdentifier + ", " +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"uniqueSwapIdentifier=" + this.uniqueSwapIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"nextFloatingReferenceResetDateLeg1=" + this.nextFloatingReferenceResetDateLeg1 + ", " +
				"nextFloatingReferenceResetDateLeg2=" + this.nextFloatingReferenceResetDateLeg2 + ", " +
				"lastFloatingReferenceValueLeg1=" + this.lastFloatingReferenceValueLeg1 + ", " +
				"lastFloatingReferenceValueLeg2=" + this.lastFloatingReferenceValueLeg2 + ", " +
				"lastFloatingReferenceResetDateLeg1=" + this.lastFloatingReferenceResetDateLeg1 + ", " +
				"lastFloatingReferenceResetDateLeg2=" + this.lastFloatingReferenceResetDateLeg2 + ", " +
				"delta=" + this.delta + ", " +
				"valuationAmount=" + this.valuationAmount + ", " +
				"valuationCurrency=" + this.valuationCurrency + ", " +
				"valuationMethod=" + this.valuationMethod + ", " +
				"valuationTimestamp=" + this.valuationTimestamp + ", " +
				"initialMarginCollateralPortfolioCode=" + this.initialMarginCollateralPortfolioCode + ", " +
				"variationMarginCollateralPortfolioCode=" + this.variationMarginCollateralPortfolioCode + ", " +
				"tradeParty1IDType=" + this.tradeParty1IDType + ", " +
				"submittingPartyIDType=" + this.submittingPartyIDType + ", " +
				"usiIDPrefix=" + this.usiIDPrefix + ", " +
				"utiIDPrefix=" + this.utiIDPrefix + ", " +
				"submittedForParty=" + this.submittedForParty + ", " +
				"tradeParty1ReportingDestination=" + this.tradeParty1ReportingDestination + ", " +
				"tradeParty2ReportingDestination=" + this.tradeParty2ReportingDestination + ", " +
				"primaryAssetClass=" + this.primaryAssetClass + ", " +
				"comment1=" + this.comment1 + ", " +
				"messageID=" + this.messageID + ", " +
				"messageType=" + this.messageType + ", " +
				"tradeParty2ExecutionAgentID=" + this.tradeParty2ExecutionAgentID + ", " +
				"tradeParty2ExecutionAgentIDType=" + this.tradeParty2ExecutionAgentIDType + ", " +
				"tradeParty1ExecutionAgentID=" + this.tradeParty1ExecutionAgentID + ", " +
				"tradeParty1ExecutionAgentIDType=" + this.tradeParty1ExecutionAgentIDType + ", " +
				"tradeParty1TransactionID=" + this.tradeParty1TransactionID + ", " +
				"version=" + this.version +
			'}';
		}
	}

	/*********************** Builder Implementation of CFTCValuationReport  ***********************/
	class CFTCValuationReportBuilderImpl implements CFTCValuationReport.CFTCValuationReportBuilder {
	
		protected String counterparty1;
		protected String counterparty2;
		protected String counterparty2IdentifierSource;
		protected String submitterIdentifier;
		protected ActionTypeEnum actionType;
		protected String uniqueProductIdentifier;
		protected ZonedDateTime reportingTimestamp;
		protected String uniqueSwapIdentifier;
		protected String uniqueTransactionIdentifier;
		protected Date nextFloatingReferenceResetDateLeg1;
		protected Date nextFloatingReferenceResetDateLeg2;
		protected BigDecimal lastFloatingReferenceValueLeg1;
		protected BigDecimal lastFloatingReferenceValueLeg2;
		protected Date lastFloatingReferenceResetDateLeg1;
		protected Date lastFloatingReferenceResetDateLeg2;
		protected BigDecimal delta;
		protected BigDecimal valuationAmount;
		protected ISOCurrencyCodeEnum valuationCurrency;
		protected ValuationType1Code valuationMethod;
		protected ZonedDateTime valuationTimestamp;
		protected String initialMarginCollateralPortfolioCode;
		protected String variationMarginCollateralPortfolioCode;
		protected PartyIdentifierTypeEnum tradeParty1IDType;
		protected PartyIdentifierTypeEnum submittingPartyIDType;
		protected String usiIDPrefix;
		protected String utiIDPrefix;
		protected String submittedForParty;
		protected List<SupervisoryBodyEnum> tradeParty1ReportingDestination = new ArrayList<>();
		protected List<SupervisoryBodyEnum> tradeParty2ReportingDestination = new ArrayList<>();
		protected AssetClassEnum primaryAssetClass;
		protected String comment1;
		protected String messageID;
		protected String messageType;
		protected String tradeParty2ExecutionAgentID;
		protected PartyIdentifierTypeEnum tradeParty2ExecutionAgentIDType;
		protected String tradeParty1ExecutionAgentID;
		protected PartyIdentifierTypeEnum tradeParty1ExecutionAgentIDType;
		protected String tradeParty1TransactionID;
		protected String version;
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		public String getCounterparty2IdentifierSource() {
			return counterparty2IdentifierSource;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public ActionTypeEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="uniqueProductIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueProductIdentifier", isRequired=true)
		public String getUniqueProductIdentifier() {
			return uniqueProductIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute("uniqueSwapIdentifier")
		@RuneAttribute("uniqueSwapIdentifier")
		public String getUniqueSwapIdentifier() {
			return uniqueSwapIdentifier;
		}
		
		@Override
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("nextFloatingReferenceResetDateLeg1")
		@RuneAttribute("nextFloatingReferenceResetDateLeg1")
		public Date getNextFloatingReferenceResetDateLeg1() {
			return nextFloatingReferenceResetDateLeg1;
		}
		
		@Override
		@RosettaAttribute("nextFloatingReferenceResetDateLeg2")
		@RuneAttribute("nextFloatingReferenceResetDateLeg2")
		public Date getNextFloatingReferenceResetDateLeg2() {
			return nextFloatingReferenceResetDateLeg2;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceValueLeg1")
		@RuneAttribute("lastFloatingReferenceValueLeg1")
		public BigDecimal getLastFloatingReferenceValueLeg1() {
			return lastFloatingReferenceValueLeg1;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceValueLeg2")
		@RuneAttribute("lastFloatingReferenceValueLeg2")
		public BigDecimal getLastFloatingReferenceValueLeg2() {
			return lastFloatingReferenceValueLeg2;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceResetDateLeg1")
		@RuneAttribute("lastFloatingReferenceResetDateLeg1")
		public Date getLastFloatingReferenceResetDateLeg1() {
			return lastFloatingReferenceResetDateLeg1;
		}
		
		@Override
		@RosettaAttribute("lastFloatingReferenceResetDateLeg2")
		@RuneAttribute("lastFloatingReferenceResetDateLeg2")
		public Date getLastFloatingReferenceResetDateLeg2() {
			return lastFloatingReferenceResetDateLeg2;
		}
		
		@Override
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		public BigDecimal getDelta() {
			return delta;
		}
		
		@Override
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		public BigDecimal getValuationAmount() {
			return valuationAmount;
		}
		
		@Override
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		public ISOCurrencyCodeEnum getValuationCurrency() {
			return valuationCurrency;
		}
		
		@Override
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		public ValuationType1Code getValuationMethod() {
			return valuationMethod;
		}
		
		@Override
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		public ZonedDateTime getValuationTimestamp() {
			return valuationTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		public String getInitialMarginCollateralPortfolioCode() {
			return initialMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		public String getVariationMarginCollateralPortfolioCode() {
			return variationMarginCollateralPortfolioCode;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		public PartyIdentifierTypeEnum getTradeParty1IDType() {
			return tradeParty1IDType;
		}
		
		@Override
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		public PartyIdentifierTypeEnum getSubmittingPartyIDType() {
			return submittingPartyIDType;
		}
		
		@Override
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		public String getUsiIDPrefix() {
			return usiIDPrefix;
		}
		
		@Override
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		public String getUtiIDPrefix() {
			return utiIDPrefix;
		}
		
		@Override
		@RosettaAttribute(value="submittedForParty", isRequired=true)
		@RuneAttribute(value="submittedForParty", isRequired=true)
		public String getSubmittedForParty() {
			return submittedForParty;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty1ReportingDestination() {
			return tradeParty1ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		public List<SupervisoryBodyEnum> getTradeParty2ReportingDestination() {
			return tradeParty2ReportingDestination;
		}
		
		@Override
		@RosettaAttribute(value="primaryAssetClass", isRequired=true)
		@RuneAttribute(value="primaryAssetClass", isRequired=true)
		public AssetClassEnum getPrimaryAssetClass() {
			return primaryAssetClass;
		}
		
		@Override
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		public String getComment1() {
			return comment1;
		}
		
		@Override
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		public String getMessageID() {
			return messageID;
		}
		
		@Override
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		public String getMessageType() {
			return messageType;
		}
		
		@Override
		@RosettaAttribute("tradeParty2ExecutionAgentID")
		@RuneAttribute("tradeParty2ExecutionAgentID")
		public String getTradeParty2ExecutionAgentID() {
			return tradeParty2ExecutionAgentID;
		}
		
		@Override
		@RosettaAttribute("tradeParty2ExecutionAgentIDType")
		@RuneAttribute("tradeParty2ExecutionAgentIDType")
		public PartyIdentifierTypeEnum getTradeParty2ExecutionAgentIDType() {
			return tradeParty2ExecutionAgentIDType;
		}
		
		@Override
		@RosettaAttribute("tradeParty1ExecutionAgentID")
		@RuneAttribute("tradeParty1ExecutionAgentID")
		public String getTradeParty1ExecutionAgentID() {
			return tradeParty1ExecutionAgentID;
		}
		
		@Override
		@RosettaAttribute("tradeParty1ExecutionAgentIDType")
		@RuneAttribute("tradeParty1ExecutionAgentIDType")
		public PartyIdentifierTypeEnum getTradeParty1ExecutionAgentIDType() {
			return tradeParty1ExecutionAgentIDType;
		}
		
		@Override
		@RosettaAttribute("tradeParty1TransactionID")
		@RuneAttribute("tradeParty1TransactionID")
		public String getTradeParty1TransactionID() {
			return tradeParty1TransactionID;
		}
		
		@Override
		@RosettaAttribute("version")
		@RuneAttribute("version")
		public String getVersion() {
			return version;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierSource", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setCounterparty2IdentifierSource(String _counterparty2IdentifierSource) {
			this.counterparty2IdentifierSource = _counterparty2IdentifierSource == null ? null : _counterparty2IdentifierSource;
			return this;
		}
		
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setSubmitterIdentifier(String _submitterIdentifier) {
			this.submitterIdentifier = _submitterIdentifier == null ? null : _submitterIdentifier;
			return this;
		}
		
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setActionType(ActionTypeEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute(value="uniqueProductIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueProductIdentifier", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setUniqueProductIdentifier(String _uniqueProductIdentifier) {
			this.uniqueProductIdentifier = _uniqueProductIdentifier == null ? null : _uniqueProductIdentifier;
			return this;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute("uniqueSwapIdentifier")
		@RuneAttribute("uniqueSwapIdentifier")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setUniqueSwapIdentifier(String _uniqueSwapIdentifier) {
			this.uniqueSwapIdentifier = _uniqueSwapIdentifier == null ? null : _uniqueSwapIdentifier;
			return this;
		}
		
		@RosettaAttribute("uniqueTransactionIdentifier")
		@RuneAttribute("uniqueTransactionIdentifier")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setUniqueTransactionIdentifier(String _uniqueTransactionIdentifier) {
			this.uniqueTransactionIdentifier = _uniqueTransactionIdentifier == null ? null : _uniqueTransactionIdentifier;
			return this;
		}
		
		@RosettaAttribute("nextFloatingReferenceResetDateLeg1")
		@RuneAttribute("nextFloatingReferenceResetDateLeg1")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setNextFloatingReferenceResetDateLeg1(Date _nextFloatingReferenceResetDateLeg1) {
			this.nextFloatingReferenceResetDateLeg1 = _nextFloatingReferenceResetDateLeg1 == null ? null : _nextFloatingReferenceResetDateLeg1;
			return this;
		}
		
		@RosettaAttribute("nextFloatingReferenceResetDateLeg2")
		@RuneAttribute("nextFloatingReferenceResetDateLeg2")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setNextFloatingReferenceResetDateLeg2(Date _nextFloatingReferenceResetDateLeg2) {
			this.nextFloatingReferenceResetDateLeg2 = _nextFloatingReferenceResetDateLeg2 == null ? null : _nextFloatingReferenceResetDateLeg2;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceValueLeg1")
		@RuneAttribute("lastFloatingReferenceValueLeg1")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceValueLeg1(BigDecimal _lastFloatingReferenceValueLeg1) {
			this.lastFloatingReferenceValueLeg1 = _lastFloatingReferenceValueLeg1 == null ? null : _lastFloatingReferenceValueLeg1;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceValueLeg2")
		@RuneAttribute("lastFloatingReferenceValueLeg2")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceValueLeg2(BigDecimal _lastFloatingReferenceValueLeg2) {
			this.lastFloatingReferenceValueLeg2 = _lastFloatingReferenceValueLeg2 == null ? null : _lastFloatingReferenceValueLeg2;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceResetDateLeg1")
		@RuneAttribute("lastFloatingReferenceResetDateLeg1")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceResetDateLeg1(Date _lastFloatingReferenceResetDateLeg1) {
			this.lastFloatingReferenceResetDateLeg1 = _lastFloatingReferenceResetDateLeg1 == null ? null : _lastFloatingReferenceResetDateLeg1;
			return this;
		}
		
		@RosettaAttribute("lastFloatingReferenceResetDateLeg2")
		@RuneAttribute("lastFloatingReferenceResetDateLeg2")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setLastFloatingReferenceResetDateLeg2(Date _lastFloatingReferenceResetDateLeg2) {
			this.lastFloatingReferenceResetDateLeg2 = _lastFloatingReferenceResetDateLeg2 == null ? null : _lastFloatingReferenceResetDateLeg2;
			return this;
		}
		
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setDelta(BigDecimal _delta) {
			this.delta = _delta == null ? null : _delta;
			return this;
		}
		
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setValuationAmount(BigDecimal _valuationAmount) {
			this.valuationAmount = _valuationAmount == null ? null : _valuationAmount;
			return this;
		}
		
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum _valuationCurrency) {
			this.valuationCurrency = _valuationCurrency == null ? null : _valuationCurrency;
			return this;
		}
		
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setValuationMethod(ValuationType1Code _valuationMethod) {
			this.valuationMethod = _valuationMethod == null ? null : _valuationMethod;
			return this;
		}
		
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setValuationTimestamp(ZonedDateTime _valuationTimestamp) {
			this.valuationTimestamp = _valuationTimestamp == null ? null : _valuationTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="initialMarginCollateralPortfolioCode", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setInitialMarginCollateralPortfolioCode(String _initialMarginCollateralPortfolioCode) {
			this.initialMarginCollateralPortfolioCode = _initialMarginCollateralPortfolioCode == null ? null : _initialMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		@RuneAttribute(value="variationMarginCollateralPortfolioCode", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setVariationMarginCollateralPortfolioCode(String _variationMarginCollateralPortfolioCode) {
			this.variationMarginCollateralPortfolioCode = _variationMarginCollateralPortfolioCode == null ? null : _variationMarginCollateralPortfolioCode;
			return this;
		}
		
		@RosettaAttribute(value="tradeParty1IDType", isRequired=true)
		@RuneAttribute(value="tradeParty1IDType", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1IDType(PartyIdentifierTypeEnum _tradeParty1IDType) {
			this.tradeParty1IDType = _tradeParty1IDType == null ? null : _tradeParty1IDType;
			return this;
		}
		
		@RosettaAttribute(value="submittingPartyIDType", isRequired=true)
		@RuneAttribute(value="submittingPartyIDType", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setSubmittingPartyIDType(PartyIdentifierTypeEnum _submittingPartyIDType) {
			this.submittingPartyIDType = _submittingPartyIDType == null ? null : _submittingPartyIDType;
			return this;
		}
		
		@RosettaAttribute("usiIDPrefix")
		@RuneAttribute("usiIDPrefix")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setUsiIDPrefix(String _usiIDPrefix) {
			this.usiIDPrefix = _usiIDPrefix == null ? null : _usiIDPrefix;
			return this;
		}
		
		@RosettaAttribute("utiIDPrefix")
		@RuneAttribute("utiIDPrefix")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setUtiIDPrefix(String _utiIDPrefix) {
			this.utiIDPrefix = _utiIDPrefix == null ? null : _utiIDPrefix;
			return this;
		}
		
		@RosettaAttribute(value="submittedForParty", isRequired=true)
		@RuneAttribute(value="submittedForParty", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setSubmittedForParty(String _submittedForParty) {
			this.submittedForParty = _submittedForParty == null ? null : _submittedForParty;
			return this;
		}
		
		@RosettaAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty1ReportingDestination", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum _tradeParty1ReportingDestination) {
			if (_tradeParty1ReportingDestination != null) {
				this.tradeParty1ReportingDestination.add(_tradeParty1ReportingDestination);
			}
			return this;
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(SupervisoryBodyEnum _tradeParty1ReportingDestination, int idx) {
			getIndex(this.tradeParty1ReportingDestination, idx, () -> _tradeParty1ReportingDestination);
			return this;
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestinations) {
			if (tradeParty1ReportingDestinations != null) {
				for (final SupervisoryBodyEnum toAdd : tradeParty1ReportingDestinations) {
					this.tradeParty1ReportingDestination.add(toAdd);
				}
			}
			return this;
		}
		
		@RuneAttribute("tradeParty1ReportingDestination")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ReportingDestination(List<SupervisoryBodyEnum> tradeParty1ReportingDestinations) {
			if (tradeParty1ReportingDestinations == null) {
				this.tradeParty1ReportingDestination = new ArrayList<>();
			} else {
				this.tradeParty1ReportingDestination = tradeParty1ReportingDestinations.stream()
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		@RuneAttribute(value="tradeParty2ReportingDestination", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(SupervisoryBodyEnum _tradeParty2ReportingDestination) {
			if (_tradeParty2ReportingDestination != null) {
				this.tradeParty2ReportingDestination.add(_tradeParty2ReportingDestination);
			}
			return this;
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(SupervisoryBodyEnum _tradeParty2ReportingDestination, int idx) {
			getIndex(this.tradeParty2ReportingDestination, idx, () -> _tradeParty2ReportingDestination);
			return this;
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder addTradeParty2ReportingDestination(List<SupervisoryBodyEnum> tradeParty2ReportingDestinations) {
			if (tradeParty2ReportingDestinations != null) {
				for (final SupervisoryBodyEnum toAdd : tradeParty2ReportingDestinations) {
					this.tradeParty2ReportingDestination.add(toAdd);
				}
			}
			return this;
		}
		
		@RuneAttribute("tradeParty2ReportingDestination")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ReportingDestination(List<SupervisoryBodyEnum> tradeParty2ReportingDestinations) {
			if (tradeParty2ReportingDestinations == null) {
				this.tradeParty2ReportingDestination = new ArrayList<>();
			} else {
				this.tradeParty2ReportingDestination = tradeParty2ReportingDestinations.stream()
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute(value="primaryAssetClass", isRequired=true)
		@RuneAttribute(value="primaryAssetClass", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setPrimaryAssetClass(AssetClassEnum _primaryAssetClass) {
			this.primaryAssetClass = _primaryAssetClass == null ? null : _primaryAssetClass;
			return this;
		}
		
		@RosettaAttribute(value="comment1", isRequired=true)
		@RuneAttribute(value="comment1", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setComment1(String _comment1) {
			this.comment1 = _comment1 == null ? null : _comment1;
			return this;
		}
		
		@RosettaAttribute(value="messageID", isRequired=true)
		@RuneAttribute(value="messageID", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setMessageID(String _messageID) {
			this.messageID = _messageID == null ? null : _messageID;
			return this;
		}
		
		@RosettaAttribute(value="messageType", isRequired=true)
		@RuneAttribute(value="messageType", isRequired=true)
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setMessageType(String _messageType) {
			this.messageType = _messageType == null ? null : _messageType;
			return this;
		}
		
		@RosettaAttribute("tradeParty2ExecutionAgentID")
		@RuneAttribute("tradeParty2ExecutionAgentID")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ExecutionAgentID(String _tradeParty2ExecutionAgentID) {
			this.tradeParty2ExecutionAgentID = _tradeParty2ExecutionAgentID == null ? null : _tradeParty2ExecutionAgentID;
			return this;
		}
		
		@RosettaAttribute("tradeParty2ExecutionAgentIDType")
		@RuneAttribute("tradeParty2ExecutionAgentIDType")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty2ExecutionAgentIDType(PartyIdentifierTypeEnum _tradeParty2ExecutionAgentIDType) {
			this.tradeParty2ExecutionAgentIDType = _tradeParty2ExecutionAgentIDType == null ? null : _tradeParty2ExecutionAgentIDType;
			return this;
		}
		
		@RosettaAttribute("tradeParty1ExecutionAgentID")
		@RuneAttribute("tradeParty1ExecutionAgentID")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ExecutionAgentID(String _tradeParty1ExecutionAgentID) {
			this.tradeParty1ExecutionAgentID = _tradeParty1ExecutionAgentID == null ? null : _tradeParty1ExecutionAgentID;
			return this;
		}
		
		@RosettaAttribute("tradeParty1ExecutionAgentIDType")
		@RuneAttribute("tradeParty1ExecutionAgentIDType")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1ExecutionAgentIDType(PartyIdentifierTypeEnum _tradeParty1ExecutionAgentIDType) {
			this.tradeParty1ExecutionAgentIDType = _tradeParty1ExecutionAgentIDType == null ? null : _tradeParty1ExecutionAgentIDType;
			return this;
		}
		
		@RosettaAttribute("tradeParty1TransactionID")
		@RuneAttribute("tradeParty1TransactionID")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setTradeParty1TransactionID(String _tradeParty1TransactionID) {
			this.tradeParty1TransactionID = _tradeParty1TransactionID == null ? null : _tradeParty1TransactionID;
			return this;
		}
		
		@RosettaAttribute("version")
		@RuneAttribute("version")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder setVersion(String _version) {
			this.version = _version == null ? null : _version;
			return this;
		}
		
		@Override
		public CFTCValuationReport build() {
			return new CFTCValuationReport.CFTCValuationReportImpl(this);
		}
		
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder prune() {
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getCounterparty1()!=null) return true;
			if (getCounterparty2()!=null) return true;
			if (getCounterparty2IdentifierSource()!=null) return true;
			if (getSubmitterIdentifier()!=null) return true;
			if (getActionType()!=null) return true;
			if (getUniqueProductIdentifier()!=null) return true;
			if (getReportingTimestamp()!=null) return true;
			if (getUniqueSwapIdentifier()!=null) return true;
			if (getUniqueTransactionIdentifier()!=null) return true;
			if (getNextFloatingReferenceResetDateLeg1()!=null) return true;
			if (getNextFloatingReferenceResetDateLeg2()!=null) return true;
			if (getLastFloatingReferenceValueLeg1()!=null) return true;
			if (getLastFloatingReferenceValueLeg2()!=null) return true;
			if (getLastFloatingReferenceResetDateLeg1()!=null) return true;
			if (getLastFloatingReferenceResetDateLeg2()!=null) return true;
			if (getDelta()!=null) return true;
			if (getValuationAmount()!=null) return true;
			if (getValuationCurrency()!=null) return true;
			if (getValuationMethod()!=null) return true;
			if (getValuationTimestamp()!=null) return true;
			if (getInitialMarginCollateralPortfolioCode()!=null) return true;
			if (getVariationMarginCollateralPortfolioCode()!=null) return true;
			if (getTradeParty1IDType()!=null) return true;
			if (getSubmittingPartyIDType()!=null) return true;
			if (getUsiIDPrefix()!=null) return true;
			if (getUtiIDPrefix()!=null) return true;
			if (getSubmittedForParty()!=null) return true;
			if (getTradeParty1ReportingDestination()!=null && !getTradeParty1ReportingDestination().isEmpty()) return true;
			if (getTradeParty2ReportingDestination()!=null && !getTradeParty2ReportingDestination().isEmpty()) return true;
			if (getPrimaryAssetClass()!=null) return true;
			if (getComment1()!=null) return true;
			if (getMessageID()!=null) return true;
			if (getMessageType()!=null) return true;
			if (getTradeParty2ExecutionAgentID()!=null) return true;
			if (getTradeParty2ExecutionAgentIDType()!=null) return true;
			if (getTradeParty1ExecutionAgentID()!=null) return true;
			if (getTradeParty1ExecutionAgentIDType()!=null) return true;
			if (getTradeParty1TransactionID()!=null) return true;
			if (getVersion()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public CFTCValuationReport.CFTCValuationReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			CFTCValuationReport.CFTCValuationReportBuilder o = (CFTCValuationReport.CFTCValuationReportBuilder) other;
			
			
			merger.mergeBasic(getCounterparty1(), o.getCounterparty1(), this::setCounterparty1);
			merger.mergeBasic(getCounterparty2(), o.getCounterparty2(), this::setCounterparty2);
			merger.mergeBasic(getCounterparty2IdentifierSource(), o.getCounterparty2IdentifierSource(), this::setCounterparty2IdentifierSource);
			merger.mergeBasic(getSubmitterIdentifier(), o.getSubmitterIdentifier(), this::setSubmitterIdentifier);
			merger.mergeBasic(getActionType(), o.getActionType(), this::setActionType);
			merger.mergeBasic(getUniqueProductIdentifier(), o.getUniqueProductIdentifier(), this::setUniqueProductIdentifier);
			merger.mergeBasic(getReportingTimestamp(), o.getReportingTimestamp(), this::setReportingTimestamp);
			merger.mergeBasic(getUniqueSwapIdentifier(), o.getUniqueSwapIdentifier(), this::setUniqueSwapIdentifier);
			merger.mergeBasic(getUniqueTransactionIdentifier(), o.getUniqueTransactionIdentifier(), this::setUniqueTransactionIdentifier);
			merger.mergeBasic(getNextFloatingReferenceResetDateLeg1(), o.getNextFloatingReferenceResetDateLeg1(), this::setNextFloatingReferenceResetDateLeg1);
			merger.mergeBasic(getNextFloatingReferenceResetDateLeg2(), o.getNextFloatingReferenceResetDateLeg2(), this::setNextFloatingReferenceResetDateLeg2);
			merger.mergeBasic(getLastFloatingReferenceValueLeg1(), o.getLastFloatingReferenceValueLeg1(), this::setLastFloatingReferenceValueLeg1);
			merger.mergeBasic(getLastFloatingReferenceValueLeg2(), o.getLastFloatingReferenceValueLeg2(), this::setLastFloatingReferenceValueLeg2);
			merger.mergeBasic(getLastFloatingReferenceResetDateLeg1(), o.getLastFloatingReferenceResetDateLeg1(), this::setLastFloatingReferenceResetDateLeg1);
			merger.mergeBasic(getLastFloatingReferenceResetDateLeg2(), o.getLastFloatingReferenceResetDateLeg2(), this::setLastFloatingReferenceResetDateLeg2);
			merger.mergeBasic(getDelta(), o.getDelta(), this::setDelta);
			merger.mergeBasic(getValuationAmount(), o.getValuationAmount(), this::setValuationAmount);
			merger.mergeBasic(getValuationCurrency(), o.getValuationCurrency(), this::setValuationCurrency);
			merger.mergeBasic(getValuationMethod(), o.getValuationMethod(), this::setValuationMethod);
			merger.mergeBasic(getValuationTimestamp(), o.getValuationTimestamp(), this::setValuationTimestamp);
			merger.mergeBasic(getInitialMarginCollateralPortfolioCode(), o.getInitialMarginCollateralPortfolioCode(), this::setInitialMarginCollateralPortfolioCode);
			merger.mergeBasic(getVariationMarginCollateralPortfolioCode(), o.getVariationMarginCollateralPortfolioCode(), this::setVariationMarginCollateralPortfolioCode);
			merger.mergeBasic(getTradeParty1IDType(), o.getTradeParty1IDType(), this::setTradeParty1IDType);
			merger.mergeBasic(getSubmittingPartyIDType(), o.getSubmittingPartyIDType(), this::setSubmittingPartyIDType);
			merger.mergeBasic(getUsiIDPrefix(), o.getUsiIDPrefix(), this::setUsiIDPrefix);
			merger.mergeBasic(getUtiIDPrefix(), o.getUtiIDPrefix(), this::setUtiIDPrefix);
			merger.mergeBasic(getSubmittedForParty(), o.getSubmittedForParty(), this::setSubmittedForParty);
			merger.mergeBasic(getTradeParty1ReportingDestination(), o.getTradeParty1ReportingDestination(), (Consumer<SupervisoryBodyEnum>) this::addTradeParty1ReportingDestination);
			merger.mergeBasic(getTradeParty2ReportingDestination(), o.getTradeParty2ReportingDestination(), (Consumer<SupervisoryBodyEnum>) this::addTradeParty2ReportingDestination);
			merger.mergeBasic(getPrimaryAssetClass(), o.getPrimaryAssetClass(), this::setPrimaryAssetClass);
			merger.mergeBasic(getComment1(), o.getComment1(), this::setComment1);
			merger.mergeBasic(getMessageID(), o.getMessageID(), this::setMessageID);
			merger.mergeBasic(getMessageType(), o.getMessageType(), this::setMessageType);
			merger.mergeBasic(getTradeParty2ExecutionAgentID(), o.getTradeParty2ExecutionAgentID(), this::setTradeParty2ExecutionAgentID);
			merger.mergeBasic(getTradeParty2ExecutionAgentIDType(), o.getTradeParty2ExecutionAgentIDType(), this::setTradeParty2ExecutionAgentIDType);
			merger.mergeBasic(getTradeParty1ExecutionAgentID(), o.getTradeParty1ExecutionAgentID(), this::setTradeParty1ExecutionAgentID);
			merger.mergeBasic(getTradeParty1ExecutionAgentIDType(), o.getTradeParty1ExecutionAgentIDType(), this::setTradeParty1ExecutionAgentIDType);
			merger.mergeBasic(getTradeParty1TransactionID(), o.getTradeParty1TransactionID(), this::setTradeParty1TransactionID);
			merger.mergeBasic(getVersion(), o.getVersion(), this::setVersion);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			CFTCValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierSource, _that.getCounterparty2IdentifierSource())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(uniqueProductIdentifier, _that.getUniqueProductIdentifier())) return false;
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(uniqueSwapIdentifier, _that.getUniqueSwapIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(nextFloatingReferenceResetDateLeg1, _that.getNextFloatingReferenceResetDateLeg1())) return false;
			if (!Objects.equals(nextFloatingReferenceResetDateLeg2, _that.getNextFloatingReferenceResetDateLeg2())) return false;
			if (!Objects.equals(lastFloatingReferenceValueLeg1, _that.getLastFloatingReferenceValueLeg1())) return false;
			if (!Objects.equals(lastFloatingReferenceValueLeg2, _that.getLastFloatingReferenceValueLeg2())) return false;
			if (!Objects.equals(lastFloatingReferenceResetDateLeg1, _that.getLastFloatingReferenceResetDateLeg1())) return false;
			if (!Objects.equals(lastFloatingReferenceResetDateLeg2, _that.getLastFloatingReferenceResetDateLeg2())) return false;
			if (!Objects.equals(delta, _that.getDelta())) return false;
			if (!Objects.equals(valuationAmount, _that.getValuationAmount())) return false;
			if (!Objects.equals(valuationCurrency, _that.getValuationCurrency())) return false;
			if (!Objects.equals(valuationMethod, _that.getValuationMethod())) return false;
			if (!Objects.equals(valuationTimestamp, _that.getValuationTimestamp())) return false;
			if (!Objects.equals(initialMarginCollateralPortfolioCode, _that.getInitialMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(variationMarginCollateralPortfolioCode, _that.getVariationMarginCollateralPortfolioCode())) return false;
			if (!Objects.equals(tradeParty1IDType, _that.getTradeParty1IDType())) return false;
			if (!Objects.equals(submittingPartyIDType, _that.getSubmittingPartyIDType())) return false;
			if (!Objects.equals(usiIDPrefix, _that.getUsiIDPrefix())) return false;
			if (!Objects.equals(utiIDPrefix, _that.getUtiIDPrefix())) return false;
			if (!Objects.equals(submittedForParty, _that.getSubmittedForParty())) return false;
			if (!ListEquals.listEquals(tradeParty1ReportingDestination, _that.getTradeParty1ReportingDestination())) return false;
			if (!ListEquals.listEquals(tradeParty2ReportingDestination, _that.getTradeParty2ReportingDestination())) return false;
			if (!Objects.equals(primaryAssetClass, _that.getPrimaryAssetClass())) return false;
			if (!Objects.equals(comment1, _that.getComment1())) return false;
			if (!Objects.equals(messageID, _that.getMessageID())) return false;
			if (!Objects.equals(messageType, _that.getMessageType())) return false;
			if (!Objects.equals(tradeParty2ExecutionAgentID, _that.getTradeParty2ExecutionAgentID())) return false;
			if (!Objects.equals(tradeParty2ExecutionAgentIDType, _that.getTradeParty2ExecutionAgentIDType())) return false;
			if (!Objects.equals(tradeParty1ExecutionAgentID, _that.getTradeParty1ExecutionAgentID())) return false;
			if (!Objects.equals(tradeParty1ExecutionAgentIDType, _that.getTradeParty1ExecutionAgentIDType())) return false;
			if (!Objects.equals(tradeParty1TransactionID, _that.getTradeParty1TransactionID())) return false;
			if (!Objects.equals(version, _that.getVersion())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierSource != null ? counterparty2IdentifierSource.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (uniqueProductIdentifier != null ? uniqueProductIdentifier.hashCode() : 0);
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (uniqueSwapIdentifier != null ? uniqueSwapIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (nextFloatingReferenceResetDateLeg1 != null ? nextFloatingReferenceResetDateLeg1.hashCode() : 0);
			_result = 31 * _result + (nextFloatingReferenceResetDateLeg2 != null ? nextFloatingReferenceResetDateLeg2.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceValueLeg1 != null ? lastFloatingReferenceValueLeg1.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceValueLeg2 != null ? lastFloatingReferenceValueLeg2.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceResetDateLeg1 != null ? lastFloatingReferenceResetDateLeg1.hashCode() : 0);
			_result = 31 * _result + (lastFloatingReferenceResetDateLeg2 != null ? lastFloatingReferenceResetDateLeg2.hashCode() : 0);
			_result = 31 * _result + (delta != null ? delta.hashCode() : 0);
			_result = 31 * _result + (valuationAmount != null ? valuationAmount.hashCode() : 0);
			_result = 31 * _result + (valuationCurrency != null ? valuationCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationMethod != null ? valuationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationTimestamp != null ? valuationTimestamp.hashCode() : 0);
			_result = 31 * _result + (initialMarginCollateralPortfolioCode != null ? initialMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (variationMarginCollateralPortfolioCode != null ? variationMarginCollateralPortfolioCode.hashCode() : 0);
			_result = 31 * _result + (tradeParty1IDType != null ? tradeParty1IDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (submittingPartyIDType != null ? submittingPartyIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (usiIDPrefix != null ? usiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (utiIDPrefix != null ? utiIDPrefix.hashCode() : 0);
			_result = 31 * _result + (submittedForParty != null ? submittedForParty.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ReportingDestination != null ? tradeParty1ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (tradeParty2ReportingDestination != null ? tradeParty2ReportingDestination.stream().map(Object::getClass).map(Class::getName).mapToInt(String::hashCode).sum() : 0);
			_result = 31 * _result + (primaryAssetClass != null ? primaryAssetClass.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (comment1 != null ? comment1.hashCode() : 0);
			_result = 31 * _result + (messageID != null ? messageID.hashCode() : 0);
			_result = 31 * _result + (messageType != null ? messageType.hashCode() : 0);
			_result = 31 * _result + (tradeParty2ExecutionAgentID != null ? tradeParty2ExecutionAgentID.hashCode() : 0);
			_result = 31 * _result + (tradeParty2ExecutionAgentIDType != null ? tradeParty2ExecutionAgentIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1ExecutionAgentID != null ? tradeParty1ExecutionAgentID.hashCode() : 0);
			_result = 31 * _result + (tradeParty1ExecutionAgentIDType != null ? tradeParty1ExecutionAgentIDType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (tradeParty1TransactionID != null ? tradeParty1TransactionID.hashCode() : 0);
			_result = 31 * _result + (version != null ? version.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "CFTCValuationReportBuilder {" +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierSource=" + this.counterparty2IdentifierSource + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"actionType=" + this.actionType + ", " +
				"uniqueProductIdentifier=" + this.uniqueProductIdentifier + ", " +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"uniqueSwapIdentifier=" + this.uniqueSwapIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"nextFloatingReferenceResetDateLeg1=" + this.nextFloatingReferenceResetDateLeg1 + ", " +
				"nextFloatingReferenceResetDateLeg2=" + this.nextFloatingReferenceResetDateLeg2 + ", " +
				"lastFloatingReferenceValueLeg1=" + this.lastFloatingReferenceValueLeg1 + ", " +
				"lastFloatingReferenceValueLeg2=" + this.lastFloatingReferenceValueLeg2 + ", " +
				"lastFloatingReferenceResetDateLeg1=" + this.lastFloatingReferenceResetDateLeg1 + ", " +
				"lastFloatingReferenceResetDateLeg2=" + this.lastFloatingReferenceResetDateLeg2 + ", " +
				"delta=" + this.delta + ", " +
				"valuationAmount=" + this.valuationAmount + ", " +
				"valuationCurrency=" + this.valuationCurrency + ", " +
				"valuationMethod=" + this.valuationMethod + ", " +
				"valuationTimestamp=" + this.valuationTimestamp + ", " +
				"initialMarginCollateralPortfolioCode=" + this.initialMarginCollateralPortfolioCode + ", " +
				"variationMarginCollateralPortfolioCode=" + this.variationMarginCollateralPortfolioCode + ", " +
				"tradeParty1IDType=" + this.tradeParty1IDType + ", " +
				"submittingPartyIDType=" + this.submittingPartyIDType + ", " +
				"usiIDPrefix=" + this.usiIDPrefix + ", " +
				"utiIDPrefix=" + this.utiIDPrefix + ", " +
				"submittedForParty=" + this.submittedForParty + ", " +
				"tradeParty1ReportingDestination=" + this.tradeParty1ReportingDestination + ", " +
				"tradeParty2ReportingDestination=" + this.tradeParty2ReportingDestination + ", " +
				"primaryAssetClass=" + this.primaryAssetClass + ", " +
				"comment1=" + this.comment1 + ", " +
				"messageID=" + this.messageID + ", " +
				"messageType=" + this.messageType + ", " +
				"tradeParty2ExecutionAgentID=" + this.tradeParty2ExecutionAgentID + ", " +
				"tradeParty2ExecutionAgentIDType=" + this.tradeParty2ExecutionAgentIDType + ", " +
				"tradeParty1ExecutionAgentID=" + this.tradeParty1ExecutionAgentID + ", " +
				"tradeParty1ExecutionAgentIDType=" + this.tradeParty1ExecutionAgentIDType + ", " +
				"tradeParty1TransactionID=" + this.tradeParty1TransactionID + ", " +
				"version=" + this.version +
			'}';
		}
	}
}
