package drr.regulation.common;

import cdm.event.common.CounterpartyPositionState;
import cdm.event.common.TradeState;
import cdm.event.workflow.WorkflowStep;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import drr.base.trade.ReportableEventBase;
import drr.regulation.common.meta.ReportableEventMeta;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * Specifies a workflowstep with enriched information required for reporting.
 * @version 7.0.0-dev.17
 */
@RosettaDataType(value="ReportableEvent", builder=ReportableEvent.ReportableEventBuilderImpl.class, version="7.0.0-dev.17")
@RuneDataType(value="ReportableEvent", model="drr", builder=ReportableEvent.ReportableEventBuilderImpl.class, version="7.0.0-dev.17")
public interface ReportableEvent extends ReportableEventBase {

	ReportableEventMeta metaData = new ReportableEventMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Additional information required for a reportable transaction, including the reporting regime.
	 */
	ReportableInformation getReportableInformation();

	/*********************** Build Methods  ***********************/
	ReportableEvent build();
	
	ReportableEvent.ReportableEventBuilder toBuilder();
	
	static ReportableEvent.ReportableEventBuilder builder() {
		return new ReportableEvent.ReportableEventBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportableEvent> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportableEvent> getType() {
		return ReportableEvent.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processRosetta(path.newSubPath("originatingWorkflowStep"), processor, WorkflowStep.class, getOriginatingWorkflowStep());
		processRosetta(path.newSubPath("reportableTrade"), processor, TradeState.class, getReportableTrade());
		processRosetta(path.newSubPath("reportablePosition"), processor, CounterpartyPositionState.class, getReportablePosition());
		processRosetta(path.newSubPath("reportableInformation"), processor, ReportableInformation.class, getReportableInformation());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportableEventBuilder extends ReportableEvent, ReportableEventBase.ReportableEventBaseBuilder {
		ReportableInformation.ReportableInformationBuilder getOrCreateReportableInformation();
		@Override
		ReportableInformation.ReportableInformationBuilder getReportableInformation();
		@Override
		ReportableEvent.ReportableEventBuilder setOriginatingWorkflowStep(WorkflowStep originatingWorkflowStep);
		@Override
		ReportableEvent.ReportableEventBuilder setReportableTrade(TradeState reportableTrade);
		@Override
		ReportableEvent.ReportableEventBuilder setReportablePosition(CounterpartyPositionState reportablePosition);
		ReportableEvent.ReportableEventBuilder setReportableInformation(ReportableInformation reportableInformation);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processRosetta(path.newSubPath("originatingWorkflowStep"), processor, WorkflowStep.WorkflowStepBuilder.class, getOriginatingWorkflowStep());
			processRosetta(path.newSubPath("reportableTrade"), processor, TradeState.TradeStateBuilder.class, getReportableTrade());
			processRosetta(path.newSubPath("reportablePosition"), processor, CounterpartyPositionState.CounterpartyPositionStateBuilder.class, getReportablePosition());
			processRosetta(path.newSubPath("reportableInformation"), processor, ReportableInformation.ReportableInformationBuilder.class, getReportableInformation());
		}
		

		ReportableEvent.ReportableEventBuilder prune();
	}

	/*********************** Immutable Implementation of ReportableEvent  ***********************/
	class ReportableEventImpl extends ReportableEventBase.ReportableEventBaseImpl implements ReportableEvent {
		private final ReportableInformation reportableInformation;
		
		protected ReportableEventImpl(ReportableEvent.ReportableEventBuilder builder) {
			super(builder);
			this.reportableInformation = ofNullable(builder.getReportableInformation()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="reportableInformation", isRequired=true)
		@RuneAttribute(value="reportableInformation", isRequired=true)
		public ReportableInformation getReportableInformation() {
			return reportableInformation;
		}
		
		@Override
		public ReportableEvent build() {
			return this;
		}
		
		@Override
		public ReportableEvent.ReportableEventBuilder toBuilder() {
			ReportableEvent.ReportableEventBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportableEvent.ReportableEventBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getReportableInformation()).ifPresent(builder::setReportableInformation);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ReportableEvent _that = getType().cast(o);
		
			if (!Objects.equals(reportableInformation, _that.getReportableInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (reportableInformation != null ? reportableInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableEvent {" +
				"reportableInformation=" + this.reportableInformation +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of ReportableEvent  ***********************/
	class ReportableEventBuilderImpl extends ReportableEventBase.ReportableEventBaseBuilderImpl implements ReportableEvent.ReportableEventBuilder {
	
		protected ReportableInformation.ReportableInformationBuilder reportableInformation;
		
		@Override
		@RosettaAttribute(value="reportableInformation", isRequired=true)
		@RuneAttribute(value="reportableInformation", isRequired=true)
		public ReportableInformation.ReportableInformationBuilder getReportableInformation() {
			return reportableInformation;
		}
		
		@Override
		public ReportableInformation.ReportableInformationBuilder getOrCreateReportableInformation() {
			ReportableInformation.ReportableInformationBuilder result;
			if (reportableInformation!=null) {
				result = reportableInformation;
			}
			else {
				result = reportableInformation = ReportableInformation.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute(value="originatingWorkflowStep", isRequired=true)
		@RuneAttribute(value="originatingWorkflowStep", isRequired=true)
		@Override
		public ReportableEvent.ReportableEventBuilder setOriginatingWorkflowStep(WorkflowStep _originatingWorkflowStep) {
			this.originatingWorkflowStep = _originatingWorkflowStep == null ? null : _originatingWorkflowStep.toBuilder();
			return this;
		}
		
		@RosettaAttribute("reportableTrade")
		@RuneAttribute("reportableTrade")
		@Override
		public ReportableEvent.ReportableEventBuilder setReportableTrade(TradeState _reportableTrade) {
			this.reportableTrade = _reportableTrade == null ? null : _reportableTrade.toBuilder();
			return this;
		}
		
		@RosettaAttribute("reportablePosition")
		@RuneAttribute("reportablePosition")
		@Override
		public ReportableEvent.ReportableEventBuilder setReportablePosition(CounterpartyPositionState _reportablePosition) {
			this.reportablePosition = _reportablePosition == null ? null : _reportablePosition.toBuilder();
			return this;
		}
		
		@RosettaAttribute(value="reportableInformation", isRequired=true)
		@RuneAttribute(value="reportableInformation", isRequired=true)
		@Override
		public ReportableEvent.ReportableEventBuilder setReportableInformation(ReportableInformation _reportableInformation) {
			this.reportableInformation = _reportableInformation == null ? null : _reportableInformation.toBuilder();
			return this;
		}
		
		@Override
		public ReportableEvent build() {
			return new ReportableEvent.ReportableEventImpl(this);
		}
		
		@Override
		public ReportableEvent.ReportableEventBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableEvent.ReportableEventBuilder prune() {
			super.prune();
			if (reportableInformation!=null && !reportableInformation.prune().hasData()) reportableInformation = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getReportableInformation()!=null && getReportableInformation().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableEvent.ReportableEventBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			ReportableEvent.ReportableEventBuilder o = (ReportableEvent.ReportableEventBuilder) other;
			
			merger.mergeRosetta(getReportableInformation(), o.getReportableInformation(), this::setReportableInformation);
			
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ReportableEvent _that = getType().cast(o);
		
			if (!Objects.equals(reportableInformation, _that.getReportableInformation())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (reportableInformation != null ? reportableInformation.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableEventBuilder {" +
				"reportableInformation=" + this.reportableInformation +
			'}' + " " + super.toString();
		}
	}
}
