package drr.regulation.csa.rewrite.functions;

import cdm.base.staticdata.asset.common.ISOCountryCodeEnum;
import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.MapperS;
import drr.base.trade.price.PriceFormat;
import drr.base.trade.price.PriceNotationEnum;
import drr.base.trade.quantity.NotionalPeriod;
import drr.regulation.common.trade.CommonLeg;
import drr.regulation.common.trade.payment.CommonPeriodicPayment;
import drr.regulation.csa.rewrite.trade.CSALeg;
import drr.standards.iso.FrequencyPeriodEnum;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.Optional;
import javax.inject.Inject;


@ImplementedBy(CSALegEnrichment.CSALegEnrichmentDefault.class)
public abstract class CSALegEnrichment implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;

	/**
	* @param legEnrichment 
	* @param quantityUnitOfMeasure 
	* @param underlierIDOther 
	* @param underlierIDOtherSource 
	* @param underlyingAssetTradingPlatformIdentifier 
	* @param underlyingAssetPriceSource 
	* @param cryptoAssetUnderlyingIndicator 
	* @param payerIdentifier 
	* @param receiverIdentifier 
	* @param settlementLocation 
	* @param commodityUnderlyerID 
	* @param commodityInstrumentID 
	* @param floatingRateIndex 
	* @return csaEnrichedLeg 
	*/
	public CSALeg evaluate(CommonLeg legEnrichment, String quantityUnitOfMeasure, String underlierIDOther, String underlierIDOtherSource, String underlyingAssetTradingPlatformIdentifier, String underlyingAssetPriceSource, Boolean cryptoAssetUnderlyingIndicator, String payerIdentifier, String receiverIdentifier, ISOCountryCodeEnum settlementLocation, String commodityUnderlyerID, String commodityInstrumentID, String floatingRateIndex) {
		CSALeg.CSALegBuilder csaEnrichedLegBuilder = doEvaluate(legEnrichment, quantityUnitOfMeasure, underlierIDOther, underlierIDOtherSource, underlyingAssetTradingPlatformIdentifier, underlyingAssetPriceSource, cryptoAssetUnderlyingIndicator, payerIdentifier, receiverIdentifier, settlementLocation, commodityUnderlyerID, commodityInstrumentID, floatingRateIndex);
		
		final CSALeg csaEnrichedLeg;
		if (csaEnrichedLegBuilder == null) {
			csaEnrichedLeg = null;
		} else {
			csaEnrichedLeg = csaEnrichedLegBuilder.build();
			objectValidator.validate(CSALeg.class, csaEnrichedLeg);
		}
		
		return csaEnrichedLeg;
	}

	protected abstract CSALeg.CSALegBuilder doEvaluate(CommonLeg legEnrichment, String quantityUnitOfMeasure, String underlierIDOther, String underlierIDOtherSource, String underlyingAssetTradingPlatformIdentifier, String underlyingAssetPriceSource, Boolean cryptoAssetUnderlyingIndicator, String payerIdentifier, String receiverIdentifier, ISOCountryCodeEnum settlementLocation, String commodityUnderlyerID, String commodityInstrumentID, String floatingRateIndex);

	public static class CSALegEnrichmentDefault extends CSALegEnrichment {
		@Override
		protected CSALeg.CSALegBuilder doEvaluate(CommonLeg legEnrichment, String quantityUnitOfMeasure, String underlierIDOther, String underlierIDOtherSource, String underlyingAssetTradingPlatformIdentifier, String underlyingAssetPriceSource, Boolean cryptoAssetUnderlyingIndicator, String payerIdentifier, String receiverIdentifier, ISOCountryCodeEnum settlementLocation, String commodityUnderlyerID, String commodityInstrumentID, String floatingRateIndex) {
			CSALeg.CSALegBuilder csaEnrichedLeg = CSALeg.builder();
			return assignOutput(csaEnrichedLeg, legEnrichment, quantityUnitOfMeasure, underlierIDOther, underlierIDOtherSource, underlyingAssetTradingPlatformIdentifier, underlyingAssetPriceSource, cryptoAssetUnderlyingIndicator, payerIdentifier, receiverIdentifier, settlementLocation, commodityUnderlyerID, commodityInstrumentID, floatingRateIndex);
		}
		
		protected CSALeg.CSALegBuilder assignOutput(CSALeg.CSALegBuilder csaEnrichedLeg, CommonLeg legEnrichment, String quantityUnitOfMeasure, String underlierIDOther, String underlierIDOtherSource, String underlyingAssetTradingPlatformIdentifier, String underlyingAssetPriceSource, Boolean cryptoAssetUnderlyingIndicator, String payerIdentifier, String receiverIdentifier, ISOCountryCodeEnum settlementLocation, String commodityUnderlyerID, String commodityInstrumentID, String floatingRateIndex) {
			csaEnrichedLeg
				.setPeriodicPayment(MapperS.of(legEnrichment).<CommonPeriodicPayment>map("getPeriodicPayment", commonLeg -> commonLeg.getPeriodicPayment()).get());
			
			csaEnrichedLeg
				.setNotionalAmountSchedule(MapperS.of(legEnrichment).<NotionalPeriod>mapC("getNotionalAmountSchedule", commonLeg -> commonLeg.getNotionalAmountSchedule()).getMulti());
			
			csaEnrichedLeg
				.setNotionalQuantitySchedule(MapperS.of(legEnrichment).<NotionalPeriod>mapC("getNotionalQuantitySchedule", commonLeg -> commonLeg.getNotionalQuantitySchedule()).getMulti());
			
			csaEnrichedLeg
				.setNotionalCurrency(MapperS.of(legEnrichment).<ISOCurrencyCodeEnum>map("getNotionalCurrency", commonLeg -> commonLeg.getNotionalCurrency()).get());
			
			csaEnrichedLeg
				.setFixedRate(MapperS.of(legEnrichment).<BigDecimal>map("getFixedRate", commonLeg -> commonLeg.getFixedRate()).get());
			
			csaEnrichedLeg
				.setSettlementCurrency(MapperS.of(legEnrichment).<ISOCurrencyCodeEnum>map("getSettlementCurrency", commonLeg -> commonLeg.getSettlementCurrency()).get());
			
			csaEnrichedLeg
				.setSpread(MapperS.of(legEnrichment).<PriceFormat>map("getSpread", commonLeg -> commonLeg.getSpread()).get());
			
			csaEnrichedLeg
				.setSpreadNotation(MapperS.of(legEnrichment).<PriceNotationEnum>map("getSpreadNotation", commonLeg -> commonLeg.getSpreadNotation()).get());
			
			csaEnrichedLeg
				.setSpreadCurrency(MapperS.of(legEnrichment).<ISOCurrencyCodeEnum>map("getSpreadCurrency", commonLeg -> commonLeg.getSpreadCurrency()).get());
			
			csaEnrichedLeg
				.setDirection2(null);
			
			csaEnrichedLeg
				.setNotionalQuantity(MapperS.of(legEnrichment).<BigDecimal>map("getNotionalQuantity", commonLeg -> commonLeg.getNotionalQuantity()).get());
			
			csaEnrichedLeg
				.setFixingDate(MapperS.of(legEnrichment).<ZonedDateTime>map("getFixingDate", commonLeg -> commonLeg.getFixingDate()).get());
			
			csaEnrichedLeg
				.setQuantityFrequency(MapperS.of(legEnrichment).<FrequencyPeriodEnum>map("getQuantityFrequency", commonLeg -> commonLeg.getQuantityFrequency()).get());
			
			csaEnrichedLeg
				.setQuantityFrequencyMultiplier(MapperS.of(legEnrichment).<Integer>map("getQuantityFrequencyMultiplier", commonLeg -> commonLeg.getQuantityFrequencyMultiplier()).get());
			
			csaEnrichedLeg
				.setPayerIdentifier(payerIdentifier);
			
			csaEnrichedLeg
				.setReceiverIdentifier(receiverIdentifier);
			
			csaEnrichedLeg
				.setSettlementLocation(settlementLocation);
			
			csaEnrichedLeg
				.setUnderlierIDOther(underlierIDOther);
			
			csaEnrichedLeg
				.setUnderlierIDOtherSource(underlierIDOtherSource);
			
			csaEnrichedLeg
				.setUnderlyingAssetTradingPlatformIdentifier(underlyingAssetTradingPlatformIdentifier);
			
			csaEnrichedLeg
				.setCryptoAssetUnderlyingIndicator(cryptoAssetUnderlyingIndicator);
			
			csaEnrichedLeg
				.setQuantityUnitOfMeasure(quantityUnitOfMeasure);
			
			csaEnrichedLeg
				.setTotalNotionalQuantity(MapperS.of(legEnrichment).<BigDecimal>map("getTotalNotionalQuantity", commonLeg -> commonLeg.getTotalNotionalQuantity()).get());
			
			csaEnrichedLeg
				.setNotionalAmount(MapperS.of(legEnrichment).<BigDecimal>map("getNotionalAmount", commonLeg -> commonLeg.getNotionalAmount()).get());
			
			csaEnrichedLeg
				.setCommodityUnderlyerID(commodityUnderlyerID);
			
			csaEnrichedLeg
				.setCommodityInstrumentID(commodityInstrumentID);
			
			csaEnrichedLeg
				.setFloatingRateIndex(floatingRateIndex);
			
			return Optional.ofNullable(csaEnrichedLeg)
				.map(o -> o.prune())
				.orElse(null);
		}
	}
}
