package drr.regulation.esma.emir.refit.valuation;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RosettaIgnore;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import drr.regulation.common.valuation.CommonValuationReport;
import drr.regulation.esma.emir.refit.valuation.meta.ESMAValuationReportMeta;
import drr.standards.iso.ActionTypeEnum;
import drr.standards.iso.ReportLevelEnum;
import drr.standards.iso.ValuationType1Code;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.17
 */
@RosettaDataType(value="ESMAValuationReport", builder=ESMAValuationReport.ESMAValuationReportBuilderImpl.class, version="7.0.0-dev.17")
@RuneDataType(value="ESMAValuationReport", model="drr", builder=ESMAValuationReport.ESMAValuationReportBuilderImpl.class, version="7.0.0-dev.17")
public interface ESMAValuationReport extends CommonValuationReport {

	ESMAValuationReportMeta metaData = new ESMAValuationReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body ESMA
	 * Corpus Regulation EMIR 648/2012 "Regulation (EU) No 648/2012 of the European Parliament and of the Council of 4 July 2012 on OTC derivatives, central counterparties and trade repositories Text with EEA relevance"  * Corpus CommissionDelegatedRegulation RTS 2022/1855 "Commission Delegated Regulation (EU) No 2022/1855 of 10 June 2022 supplementing Regulation (EU) No 648/2012 of the European Parliament and of the Council on OTC derivatives, central counterparties and trade repositories with regard to regulatory technical standards specifying the minimum details of the data to be reported to trade repositories and the type of reports to be used."  * Corpus Dissemination Valuation   
	 * table "2" * dataElement "153" * field "Event date"
	 *
	 * Provision Date on which the reportable event relating to the derivative contract and captured by the report took place or, in case of a modification when the modification become effective.
	 *
	 */
	@Override
	Date getEventDate();
	NonReportable getNonReportable();

	/*********************** Build Methods  ***********************/
	ESMAValuationReport build();
	
	ESMAValuationReport.ESMAValuationReportBuilder toBuilder();
	
	static ESMAValuationReport.ESMAValuationReportBuilder builder() {
		return new ESMAValuationReport.ESMAValuationReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ESMAValuationReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ESMAValuationReport> getType() {
		return ESMAValuationReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
		processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
		processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
		processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
		processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
		processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("level"), ReportLevelEnum.class, getLevel(), this);
		processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
		processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
		processor.processBasic(path.newSubPath("uti"), String.class, getUti(), this);
		processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
		processRosetta(path.newSubPath("nonReportable"), processor, NonReportable.class, getNonReportable());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ESMAValuationReportBuilder extends ESMAValuationReport, CommonValuationReport.CommonValuationReportBuilder {
		NonReportable.NonReportableBuilder getOrCreateNonReportable();
		@Override
		NonReportable.NonReportableBuilder getNonReportable();
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setCounterparty1(String counterparty1);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setCounterparty2IdentifierType(Boolean counterparty2IdentifierType);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setCounterparty2(String counterparty2);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setValuationAmount(BigDecimal valuationAmount);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum valuationCurrency);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setValuationTimestamp(ZonedDateTime valuationTimestamp);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setValuationMethod(ValuationType1Code valuationMethod);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setDelta(BigDecimal delta);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setActionType(ActionTypeEnum actionType);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setLevel(ReportLevelEnum level);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setReportSubmittingEntityID(String reportSubmittingEntityID);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setEntityResponsibleForReporting(String entityResponsibleForReporting);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setUti(String uti);
		@Override
		ESMAValuationReport.ESMAValuationReportBuilder setEventDate(Date eventDate);
		ESMAValuationReport.ESMAValuationReportBuilder setEventDateOverriddenAsDate(Date eventDate);
		ESMAValuationReport.ESMAValuationReportBuilder setNonReportable(NonReportable nonReportable);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
			processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
			processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
			processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
			processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
			processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("level"), ReportLevelEnum.class, getLevel(), this);
			processor.processBasic(path.newSubPath("reportSubmittingEntityID"), String.class, getReportSubmittingEntityID(), this);
			processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
			processor.processBasic(path.newSubPath("uti"), String.class, getUti(), this);
			processor.processBasic(path.newSubPath("eventDate"), Date.class, getEventDate(), this);
			processRosetta(path.newSubPath("nonReportable"), processor, NonReportable.NonReportableBuilder.class, getNonReportable());
		}
		

		ESMAValuationReport.ESMAValuationReportBuilder prune();
	}

	/*********************** Immutable Implementation of ESMAValuationReport  ***********************/
	class ESMAValuationReportImpl extends CommonValuationReport.CommonValuationReportImpl implements ESMAValuationReport {
		private final Date eventDate;
		private final NonReportable nonReportable;
		
		protected ESMAValuationReportImpl(ESMAValuationReport.ESMAValuationReportBuilder builder) {
			super(builder);
			this.eventDate = builder.getEventDate();
			this.nonReportable = ofNullable(builder.getNonReportable()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		public Date getEventDate() {
			return eventDate;
		}
		
		@Override
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		public NonReportable getNonReportable() {
			return nonReportable;
		}
		
		@Override
		public ESMAValuationReport build() {
			return this;
		}
		
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder toBuilder() {
			ESMAValuationReport.ESMAValuationReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ESMAValuationReport.ESMAValuationReportBuilder builder) {
			super.setBuilderFields(builder);
			ofNullable(getEventDate()).ifPresent(builder::setEventDateOverriddenAsDate);
			ofNullable(getNonReportable()).ifPresent(builder::setNonReportable);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ESMAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(eventDate, _that.getEventDate())) return false;
			if (!Objects.equals(nonReportable, _that.getNonReportable())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (eventDate != null ? eventDate.hashCode() : 0);
			_result = 31 * _result + (nonReportable != null ? nonReportable.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ESMAValuationReport {" +
				"eventDate=" + this.eventDate + ", " +
				"nonReportable=" + this.nonReportable +
			'}' + " " + super.toString();
		}
	}

	/*********************** Builder Implementation of ESMAValuationReport  ***********************/
	class ESMAValuationReportBuilderImpl extends CommonValuationReport.CommonValuationReportBuilderImpl implements ESMAValuationReport.ESMAValuationReportBuilder {
	
		protected Date eventDate;
		protected NonReportable.NonReportableBuilder nonReportable;
		
		@Override
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		public Date getEventDate() {
			return eventDate;
		}
		
		@Override
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		public NonReportable.NonReportableBuilder getNonReportable() {
			return nonReportable;
		}
		
		@Override
		public NonReportable.NonReportableBuilder getOrCreateNonReportable() {
			NonReportable.NonReportableBuilder result;
			if (nonReportable!=null) {
				result = nonReportable;
			}
			else {
				result = nonReportable = NonReportable.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setCounterparty2IdentifierType(Boolean _counterparty2IdentifierType) {
			this.counterparty2IdentifierType = _counterparty2IdentifierType == null ? null : _counterparty2IdentifierType;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setValuationAmount(BigDecimal _valuationAmount) {
			this.valuationAmount = _valuationAmount == null ? null : _valuationAmount;
			return this;
		}
		
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum _valuationCurrency) {
			this.valuationCurrency = _valuationCurrency == null ? null : _valuationCurrency;
			return this;
		}
		
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setValuationTimestamp(ZonedDateTime _valuationTimestamp) {
			this.valuationTimestamp = _valuationTimestamp == null ? null : _valuationTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setValuationMethod(ValuationType1Code _valuationMethod) {
			this.valuationMethod = _valuationMethod == null ? null : _valuationMethod;
			return this;
		}
		
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setDelta(BigDecimal _delta) {
			this.delta = _delta == null ? null : _delta;
			return this;
		}
		
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setActionType(ActionTypeEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute(value="level", isRequired=true)
		@RuneAttribute(value="level", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setLevel(ReportLevelEnum _level) {
			this.level = _level == null ? null : _level;
			return this;
		}
		
		@RosettaAttribute(value="reportSubmittingEntityID", isRequired=true)
		@RuneAttribute(value="reportSubmittingEntityID", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setReportSubmittingEntityID(String _reportSubmittingEntityID) {
			this.reportSubmittingEntityID = _reportSubmittingEntityID == null ? null : _reportSubmittingEntityID;
			return this;
		}
		
		@RosettaAttribute("entityResponsibleForReporting")
		@RuneAttribute("entityResponsibleForReporting")
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setEntityResponsibleForReporting(String _entityResponsibleForReporting) {
			this.entityResponsibleForReporting = _entityResponsibleForReporting == null ? null : _entityResponsibleForReporting;
			return this;
		}
		
		@RosettaAttribute(value="uti", isRequired=true)
		@RuneAttribute(value="uti", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setUti(String _uti) {
			this.uti = _uti == null ? null : _uti;
			return this;
		}
		
		@RosettaAttribute(value="eventDate", isRequired=true)
		@RuneAttribute(value="eventDate", isRequired=true)
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setEventDateOverriddenAsDate(Date _eventDate) {
			this.eventDate = _eventDate == null ? null : _eventDate;
			return this;
		}
		
		@RosettaIgnore
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setEventDate(Date _eventDate) {
			return setEventDateOverriddenAsDate(_eventDate);
		}
		
		@RosettaAttribute("nonReportable")
		@RuneAttribute("nonReportable")
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder setNonReportable(NonReportable _nonReportable) {
			this.nonReportable = _nonReportable == null ? null : _nonReportable.toBuilder();
			return this;
		}
		
		@Override
		public ESMAValuationReport build() {
			return new ESMAValuationReport.ESMAValuationReportImpl(this);
		}
		
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder prune() {
			super.prune();
			if (nonReportable!=null && !nonReportable.prune().hasData()) nonReportable = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (super.hasData()) return true;
			if (getEventDate()!=null) return true;
			if (getNonReportable()!=null && getNonReportable().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ESMAValuationReport.ESMAValuationReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			super.merge(other, merger);
			ESMAValuationReport.ESMAValuationReportBuilder o = (ESMAValuationReport.ESMAValuationReportBuilder) other;
			
			merger.mergeRosetta(getNonReportable(), o.getNonReportable(), this::setNonReportable);
			
			merger.mergeBasic(getEventDate(), o.getEventDate(), this::setEventDateOverriddenAsDate);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
			if (!super.equals(o)) return false;
		
			ESMAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(eventDate, _that.getEventDate())) return false;
			if (!Objects.equals(nonReportable, _that.getNonReportable())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = super.hashCode();
			_result = 31 * _result + (eventDate != null ? eventDate.hashCode() : 0);
			_result = 31 * _result + (nonReportable != null ? nonReportable.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ESMAValuationReportBuilder {" +
				"eventDate=" + this.eventDate + ", " +
				"nonReportable=" + this.nonReportable +
			'}' + " " + super.toString();
		}
	}
}
