package drr.regulation.hkma.rewrite.trade.reports;

import cdm.event.common.Trade;
import cdm.product.template.BasketConstituent;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.reports.ReportFunction;
import drr.base.trade.basket.BasketConstituentsReport;
import drr.base.trade.functions.TradeForEvent;
import drr.regulation.common.TransactionReportInstruction;
import drr.regulation.common.trade.basket.functions.GetBasketConstituents;
import drr.regulation.hkma.rewrite.trade.functions.IsAllowableActionForHKMA;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;


@ImplementedBy(BasketConstituentsRule.BasketConstituentsRuleDefault.class)
public abstract class BasketConstituentsRule implements ReportFunction<TransactionReportInstruction, List<? extends BasketConstituentsReport>> {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected BasketConstituentIdentifierRule basketConstituentIdentifierRule;
	@Inject protected BasketConstituentIdentifierSourceRule basketConstituentIdentifierSourceRule;
	@Inject protected BasketConstituentNumberOfUnitsRule basketConstituentNumberOfUnitsRule;
	@Inject protected BasketConstituentUnitOfMeasureRule basketConstituentUnitOfMeasureRule;
	@Inject protected GetBasketConstituents getBasketConstituents;
	@Inject protected IsAllowableActionForHKMA isAllowableActionForHKMA;
	@Inject protected TradeForEvent tradeForEvent;

	/**
	* @param input 
	* @return output 
	*/
	@Override
	public List<? extends BasketConstituentsReport> evaluate(TransactionReportInstruction input) {
		List<BasketConstituentsReport.BasketConstituentsReportBuilder> outputBuilder = doEvaluate(input);
		
		final List<? extends BasketConstituentsReport> output;
		if (outputBuilder == null) {
			output = null;
		} else {
			output = outputBuilder.stream().map(BasketConstituentsReport::build).collect(Collectors.toList());
			objectValidator.validate(BasketConstituentsReport.class, output);
		}
		
		return output;
	}

	protected abstract List<BasketConstituentsReport.BasketConstituentsReportBuilder> doEvaluate(TransactionReportInstruction input);

	public static class BasketConstituentsRuleDefault extends BasketConstituentsRule {
		@Override
		protected List<BasketConstituentsReport.BasketConstituentsReportBuilder> doEvaluate(TransactionReportInstruction input) {
			List<BasketConstituentsReport.BasketConstituentsReportBuilder> output = new ArrayList<>();
			return assignOutput(output, input);
		}
		
		protected List<BasketConstituentsReport.BasketConstituentsReportBuilder> assignOutput(List<BasketConstituentsReport.BasketConstituentsReportBuilder> output, TransactionReportInstruction input) {
			final MapperS<TransactionReportInstruction> thenArg0 = MapperS.of(input)
				.filterSingleNullSafe(item -> isAllowableActionForHKMA.evaluate(item.get()));
			final MapperS<Trade> thenArg1 = thenArg0
				.mapSingleToItem(item -> MapperS.of(tradeForEvent.evaluate(item.get())));
			final MapperC<BasketConstituent> thenArg2 = thenArg1
				.mapSingleToList(item -> MapperC.<BasketConstituent>of(getBasketConstituents.evaluate(item.get())));
			output = toBuilder(thenArg2
				.mapItem(item -> MapperS.of(BasketConstituentsReport.builder()
					.setIdentifier(basketConstituentIdentifierRule.evaluate(item.get()))
					.setSource(basketConstituentIdentifierSourceRule.evaluate(item.get()))
					.setNumberOfUnits(basketConstituentNumberOfUnitsRule.evaluate(item.get()))
					.setUnitOfMeasure(basketConstituentUnitOfMeasureRule.evaluate(item.get()))
					.build())).getMulti());
			
			return Optional.ofNullable(output)
				.map(o -> o.stream().map(i -> i.prune()).collect(Collectors.toList()))
				.orElse(null);
		}
	}
}
