package drr.regulation.jfsa.rewrite.valuation;

import cdm.base.staticdata.asset.common.ISOCurrencyCodeEnum;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import drr.regulation.jfsa.rewrite.valuation.meta.JFSAValuationReportMeta;
import drr.standards.iso.ActionTypeEnum;
import drr.standards.iso.ValuationType1Code;
import java.math.BigDecimal;
import java.time.ZonedDateTime;
import java.util.Objects;

import static java.util.Optional.ofNullable;

/**
 * @version 7.0.0-dev.17
 */
@RosettaDataType(value="JFSAValuationReport", builder=JFSAValuationReport.JFSAValuationReportBuilderImpl.class, version="7.0.0-dev.17")
@RuneDataType(value="JFSAValuationReport", model="drr", builder=JFSAValuationReport.JFSAValuationReportBuilderImpl.class, version="7.0.0-dev.17")
public interface JFSAValuationReport extends RosettaModelObject {

	JFSAValuationReportMeta metaData = new JFSAValuationReportMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "4" * field "Reporting Timestamp"
	 *
	 * Provision Date and time of the submission of the report to the trade repository
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "4" * field "Reporting timestamp - Remark"
	 *
	 * Provision System input time, data generation time, and system connection time are to be considered as acceptable values. In addition, if determining the time up to the seconds is impossible due to system restrictions, setting the seconds to 00 shall be acceptable. If it can be determined that the reporting date and time are the same as the trade date, then Trade Date, may be used as a substitute value. (If different, substitution will not be permitted.)
	 *
	 */
	ZonedDateTime getReportingTimestamp();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "6" * field "Entity Responsible for Reporting"
	 *
	 * Provision The identifier of the financial institution that is the reporting obligation.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "6" * field "Entity Responsible for Reporting - Remark"
	 *
	 * Provision Reporting party of the trades will be determined based on the identifier reported for 'Entity responsible for reporting' and by 'Submitter identifier'. Examples of who the reporting party is:
	             [Case 1] Where a party to a transaction is other than a fund (normal case):
	             'Entity responsible for reporting' = Financial institution with reporting obligation and involved in the transaction as Counterparty 1
	             'Counterparty 1 (reporting counterparty)' = Same as above
	             'Submitter identifier' = Same as above
	             [Case 2] Where a trading party is a fund:
	             'Entity responsible for reporting' = Trust Bank LEI
	             'Counterparty 1 (reporting counterparty)' = Fund LEI
	             'Submitter identifier' = Trust bank LEI 
	             [Case 3] Reporting party is an agent, etc. (where a regional bank requests the Trust Bank, etc. to report):
	             'Entity responsible for reporting' = Financial institution with reporting obligation and involved in the transaction as Counterparty 1
	             'Counterparty 1 (reporting counterparty)' = Same as above
	             'Submitter identifier' = LEI of the agent acting as the service provider for reporting
	 *
	 */
	String getEntityResponsibleForReporting();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "7" * field "Counterparty 1 (reporting counterparty)"
	 *
	 * Provision Identifier of the counterparty to an OTC derivative transaction who is fulfilling its reporting obligation via the report in question. In jurisdictions where both parties must report the transaction, the identifier of Counterparty 1 always identifies the reporting counterparty. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "7" * field "Counterparty 1 (reporting counterparty) - Remark"
	 *
	 * Provision Reporting using branch/international branch LEI is not allowed.
	 *
	 */
	String getCounterparty1();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "8" * field "Counterparty 2"
	 *
	 * Provision Identifier of the second counterparty to an OTC derivative transaction. In the case of an allocated derivative transaction executed by a fund manager on behalf of a fund, the fund, and not the fund manager is reported as the counterparty.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "8" * field "Counterparty 2 - Remark"
	 *
	 * Provision If LEI cannot be obtained, a tentative LEI is permitted. Tentative LEI must have a unique alphanumeric string to ensure the party can be uniquely identified.
	             Examples for assigning tentative LEI: [In the case of financial institutions] Tentative LEI = Party LEI + Counterparty BIC Code
	             [In the case of corporations and individuals] Tentative LEI = Trading Party LEI + Unique Alphanumeric Code (Control number assigned by the financial institution, etc.)
	             If a fund LEI is yet to be obtained, the Trust Bank must share a tentative LEI for the fund (Trust Bank LEI + Unique Alphanumeric Code) with its counterparty financial institution.
	             Enter the LEI of the clearing organization if the company transacts with a clearing organization prescribed in JFSA Notification No.105.
	             Reporting using branch/international branch LEI is not allowed.
	 *
	 */
	String getCounterparty2();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "9" * field "Counterparty 2 Identifier Type"
	 *
	 * Provision Indicator of whether LEI was used to identify the Counterparty 2.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "9" * field "Counterparty 2 Identifier Type - Remark"
	 *
	 * Provision Under the CDE Guidance, legal entity is to be reported as True and individual as False. However, for cases where a tentative LEI is used to report for a legal entity, it should be reported as False.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220307"
	 *
	 * Provision Peer review recommnended to detect private individual party representations (instead of LEI scheme).
	 *
	 */
	Boolean getCounterparty2IdentifierType();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "12" * field "Submitter Identifier"
	 *
	 * Provision Identifier of the entity submitting the data to the swap data repository (SDR). The Submitter identifier will be the same as the reporting counterparty, unless they use a third-party service provider to submit the data to SDR in which case, report the identifier of the third-party service provider.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "12" * field "Submitter Identifier - Remark"
	 *
	 * Provision See 'Entity responsible for reporting'. Applicable for central counterparty and financial institutions. If reporting is conducted using a platform, then the platform is deemed to be the submitter.
	 *
	 *
	 * Body ISDA
	 * Corpus WorkingGroup PeerReviewGroup ISDA EMIR/CFTC Digital Regulatory Reporting Peer Review "Industry group which works collaboratively to mutualize representation of amended CFTC and EMIR transaction reporting requirements in Digital Regulatory Reporting (DRR) - a digitized, single standard set of open-source machine-executable code using the Common Domain Model (CDM)." 
	 * date "20220309"
	 *
	 * Provision Check whether trade is executed in a SEF first. If it is, SEF is obliged to be the Submitter. Then check for Reporting Party and Data Submitter.
	 *
	 */
	String getSubmitterIdentifier();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "13" * field "New SDR Identifier"
	 *
	 * Provision Identifier of the new swap data repository where the swap transaction is transferred to.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "13" * field "New SDR Identifier - Remark"
	 *
	 * Provision If unapplicable, leave it blank.
	 *
	 */
	String getNewSDRIdentifier();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "25" * field "UTI"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle.
	 *
	 */
	String getUniqueTransactionIdentifier();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "25" * field "UTI Proprietary"
	 *
	 * Provision A unique identifier assigned to all swap transactions which identifies the swap uniquely throughout its lifecycle. (Trade identifier expressed in a proprietary notation)
	 *
	 */
	String getUtiProprietaryID();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "39" * field "Valuation Amount"
	 *
	 * Provision Current value of the outstanding contract without applying any valuation adjustments.
	                 Valuation amount is expressed as the exit cost of the contract or components of the contract, i.e. the price that would be received to sell the contract (in the market in an orderly transaction at the valuation date).
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "39" * field "Valuation Amount - Remark"
	 *
	 * Provision Valuation amount does not take into account the various valuation adjustments such as CVA (Credit Valuation Adjustment). Amount to be reported is to be the preadjusted valuation amount.
	 *
	 */
	BigDecimal getValuationAmount();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "40" * field "Valuation Currency"
	 *
	 * Provision Currency in which the valuation amount is denominated.
	 *
	 */
	ISOCurrencyCodeEnum getValuationCurrency();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "41" * field "Valuation Timestamp"
	 *
	 * Provision Date and time of the last valuation marked to market, provided by the central counterparty (CCP) or calculated using the current or last available market price of the inputs. If for example a currency exchange rate is the basis for a transactions valuation, then the valuation timestamp reflects the moment in time that exchange rate was current.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "41" * field "Valuation Timestamp - Remark"
	 *
	 * Provision Valuation time period specified by each financial institution is permitted.
	 *
	 */
	ZonedDateTime getValuationTimestamp();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "42" * field "Valuation Method"
	 *
	 * Provision Source and method used for the valuation of the transaction by the reporting counterparty.
	             If at least one valuation input is used that is classified as mark-to-model in appendix D of CFTC Technical Specification, then the whole valuation is classified as mark-to-model.
	             If only inputs are used that are classified as mark-to-market in appendix D of CFTC Technical Specification then the whole valuation is classified as mark-to-market.
	 *
	 */
	ValuationType1Code getValuationMethod();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "43" * field "Delta"
	 *
	 * Provision The ratio of the change in the price of an OTC derivative transaction to the change in the price of the underlier, at the time a new transaction is reported or when a change in the notional amount is reported.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "43" * field "Delta"
	 *
	 * Provision Reporting (including any amendments made on an existing trade), is required from April 7, 2025.
	 *
	 */
	BigDecimal getDelta();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "101" * field "Action Type"
	 *
	 * Provision Type of action taken on the swap transaction or type of end-of-day reporting.
	             Actions may include, but are not limited to, new, modify, correct, error, terminate, revive, transfer out, valuation, and collateral.
	             New: An action that reports a new swap transaction. It applies to the first message relating to a new UTI.
	             Modify: An action that modifies the state of a previously submitted transaction (e.g., credit event) or changes a term of a previously submitted transaction due to a newly negotiated modification (amendment) or updates previously missing information (e.g., post price swap). It does not include correction of a previous transaction.
	             Correct: An action that corrects erroneous data of a previously submitted transaction.
	             Error: An action of cancellation of a wrongly submitted entire transaction in case it never came into existence or was not subject to the reporting requirements but was reported erroneously, or a cancellation of duplicate report.
	             Terminate: An action that closes an existing transaction because of a new event (e.g., Compression, Novation). This does not apply to transactions that terminate at contractual maturity date.
	             Revive: An action that reinstates a swap transaction that was reported as error or terminated by mistake.
	             Transfer out: An action that transfers swap transaction from one SDR to another SDR (change of swap data repository).
	             Valuation: An update to valuation data. There will be no corresponding Event type.
	             Collateral: An update to collateral margin data. There will be no corresponding Event type.
	 *
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "101" * field "Action Type - Remark"
	 *
	 * Provision Status for reporting items such as 'LEI' and 'Confirm' could change during contract period. For those updates which does not involve contractual changes or error corrections, use Action Type: Modify.
	             If New, Correct, or Modify occurs on the same day as the valuation for a same transaction, report the correction and valuation separately.
	             [Specific example] (Specific example 1) Where an equity option transaction is knocked-out (currently reported as modify with a notional amount of 0)
	             Report 'ACTION TYPE' as 'Terminate (TERM)' and 'EVENT TYPE' as 'Early Termination (EART)'.
	             (Specific Example 2) Where a customer cancels an equity option transaction (currently reported as cancel) Report 'ACTION TYPE' as 'Terminate (TERM)' and 'EVENT TYPE' as 'Early Termination (EART)'.
	             (Specific example 3) Exercise of equity option transactions (previously excluded from reporting) Report 'ACTION TYPE' as 'Terminate (TERM)' 'EVENT TYPE' as 'Exercise (EXER)'.
	             In cases where the rights are exercised, it is expectd to be reported as 'Terminate' and 'Exercise'.
	 *
	 */
	ActionTypeEnum getActionType();
	/**
	 *
	 * Body JFSA
	 * Corpus Dissemination Valuation   
	 * dataElement "[Internal]" * field "Technical Record Id (ISO)"
	 *
	 * Provision Unique technical identification of the original data for which the status is provided for the BDR. Not sent to JFSA.
	 *
	 */
	String getTechnicalRecordId();

	/*********************** Build Methods  ***********************/
	JFSAValuationReport build();
	
	JFSAValuationReport.JFSAValuationReportBuilder toBuilder();
	
	static JFSAValuationReport.JFSAValuationReportBuilder builder() {
		return new JFSAValuationReport.JFSAValuationReportBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends JFSAValuationReport> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends JFSAValuationReport> getType() {
		return JFSAValuationReport.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
		processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
		processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
		processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
		processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
		processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
		processor.processBasic(path.newSubPath("newSDRIdentifier"), String.class, getNewSDRIdentifier(), this);
		processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
		processor.processBasic(path.newSubPath("utiProprietaryID"), String.class, getUtiProprietaryID(), this);
		processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
		processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
		processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
		processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
		processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
		processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
		processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
	}
	

	/*********************** Builder Interface  ***********************/
	interface JFSAValuationReportBuilder extends JFSAValuationReport, RosettaModelObjectBuilder {
		JFSAValuationReport.JFSAValuationReportBuilder setReportingTimestamp(ZonedDateTime reportingTimestamp);
		JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReporting(String entityResponsibleForReporting);
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty1(String counterparty1);
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2(String counterparty2);
		JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierType(Boolean counterparty2IdentifierType);
		JFSAValuationReport.JFSAValuationReportBuilder setSubmitterIdentifier(String submitterIdentifier);
		JFSAValuationReport.JFSAValuationReportBuilder setNewSDRIdentifier(String newSDRIdentifier);
		JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifier(String uniqueTransactionIdentifier);
		JFSAValuationReport.JFSAValuationReportBuilder setUtiProprietaryID(String utiProprietaryID);
		JFSAValuationReport.JFSAValuationReportBuilder setValuationAmount(BigDecimal valuationAmount);
		JFSAValuationReport.JFSAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum valuationCurrency);
		JFSAValuationReport.JFSAValuationReportBuilder setValuationTimestamp(ZonedDateTime valuationTimestamp);
		JFSAValuationReport.JFSAValuationReportBuilder setValuationMethod(ValuationType1Code valuationMethod);
		JFSAValuationReport.JFSAValuationReportBuilder setDelta(BigDecimal delta);
		JFSAValuationReport.JFSAValuationReportBuilder setActionType(ActionTypeEnum actionType);
		JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordId(String technicalRecordId);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("reportingTimestamp"), ZonedDateTime.class, getReportingTimestamp(), this);
			processor.processBasic(path.newSubPath("entityResponsibleForReporting"), String.class, getEntityResponsibleForReporting(), this);
			processor.processBasic(path.newSubPath("counterparty1"), String.class, getCounterparty1(), this);
			processor.processBasic(path.newSubPath("counterparty2"), String.class, getCounterparty2(), this);
			processor.processBasic(path.newSubPath("counterparty2IdentifierType"), Boolean.class, getCounterparty2IdentifierType(), this);
			processor.processBasic(path.newSubPath("submitterIdentifier"), String.class, getSubmitterIdentifier(), this);
			processor.processBasic(path.newSubPath("newSDRIdentifier"), String.class, getNewSDRIdentifier(), this);
			processor.processBasic(path.newSubPath("uniqueTransactionIdentifier"), String.class, getUniqueTransactionIdentifier(), this);
			processor.processBasic(path.newSubPath("utiProprietaryID"), String.class, getUtiProprietaryID(), this);
			processor.processBasic(path.newSubPath("valuationAmount"), BigDecimal.class, getValuationAmount(), this);
			processor.processBasic(path.newSubPath("valuationCurrency"), ISOCurrencyCodeEnum.class, getValuationCurrency(), this);
			processor.processBasic(path.newSubPath("valuationTimestamp"), ZonedDateTime.class, getValuationTimestamp(), this);
			processor.processBasic(path.newSubPath("valuationMethod"), ValuationType1Code.class, getValuationMethod(), this);
			processor.processBasic(path.newSubPath("delta"), BigDecimal.class, getDelta(), this);
			processor.processBasic(path.newSubPath("actionType"), ActionTypeEnum.class, getActionType(), this);
			processor.processBasic(path.newSubPath("technicalRecordId"), String.class, getTechnicalRecordId(), this);
		}
		

		JFSAValuationReport.JFSAValuationReportBuilder prune();
	}

	/*********************** Immutable Implementation of JFSAValuationReport  ***********************/
	class JFSAValuationReportImpl implements JFSAValuationReport {
		private final ZonedDateTime reportingTimestamp;
		private final String entityResponsibleForReporting;
		private final String counterparty1;
		private final String counterparty2;
		private final Boolean counterparty2IdentifierType;
		private final String submitterIdentifier;
		private final String newSDRIdentifier;
		private final String uniqueTransactionIdentifier;
		private final String utiProprietaryID;
		private final BigDecimal valuationAmount;
		private final ISOCurrencyCodeEnum valuationCurrency;
		private final ZonedDateTime valuationTimestamp;
		private final ValuationType1Code valuationMethod;
		private final BigDecimal delta;
		private final ActionTypeEnum actionType;
		private final String technicalRecordId;
		
		protected JFSAValuationReportImpl(JFSAValuationReport.JFSAValuationReportBuilder builder) {
			this.reportingTimestamp = builder.getReportingTimestamp();
			this.entityResponsibleForReporting = builder.getEntityResponsibleForReporting();
			this.counterparty1 = builder.getCounterparty1();
			this.counterparty2 = builder.getCounterparty2();
			this.counterparty2IdentifierType = builder.getCounterparty2IdentifierType();
			this.submitterIdentifier = builder.getSubmitterIdentifier();
			this.newSDRIdentifier = builder.getNewSDRIdentifier();
			this.uniqueTransactionIdentifier = builder.getUniqueTransactionIdentifier();
			this.utiProprietaryID = builder.getUtiProprietaryID();
			this.valuationAmount = builder.getValuationAmount();
			this.valuationCurrency = builder.getValuationCurrency();
			this.valuationTimestamp = builder.getValuationTimestamp();
			this.valuationMethod = builder.getValuationMethod();
			this.delta = builder.getDelta();
			this.actionType = builder.getActionType();
			this.technicalRecordId = builder.getTechnicalRecordId();
		}
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		public Boolean getCounterparty2IdentifierType() {
			return counterparty2IdentifierType;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		public String getNewSDRIdentifier() {
			return newSDRIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("utiProprietaryID")
		@RuneAttribute("utiProprietaryID")
		public String getUtiProprietaryID() {
			return utiProprietaryID;
		}
		
		@Override
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		public BigDecimal getValuationAmount() {
			return valuationAmount;
		}
		
		@Override
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		public ISOCurrencyCodeEnum getValuationCurrency() {
			return valuationCurrency;
		}
		
		@Override
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		public ZonedDateTime getValuationTimestamp() {
			return valuationTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		public ValuationType1Code getValuationMethod() {
			return valuationMethod;
		}
		
		@Override
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		public BigDecimal getDelta() {
			return delta;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public ActionTypeEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@Override
		public JFSAValuationReport build() {
			return this;
		}
		
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder toBuilder() {
			JFSAValuationReport.JFSAValuationReportBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(JFSAValuationReport.JFSAValuationReportBuilder builder) {
			ofNullable(getReportingTimestamp()).ifPresent(builder::setReportingTimestamp);
			ofNullable(getEntityResponsibleForReporting()).ifPresent(builder::setEntityResponsibleForReporting);
			ofNullable(getCounterparty1()).ifPresent(builder::setCounterparty1);
			ofNullable(getCounterparty2()).ifPresent(builder::setCounterparty2);
			ofNullable(getCounterparty2IdentifierType()).ifPresent(builder::setCounterparty2IdentifierType);
			ofNullable(getSubmitterIdentifier()).ifPresent(builder::setSubmitterIdentifier);
			ofNullable(getNewSDRIdentifier()).ifPresent(builder::setNewSDRIdentifier);
			ofNullable(getUniqueTransactionIdentifier()).ifPresent(builder::setUniqueTransactionIdentifier);
			ofNullable(getUtiProprietaryID()).ifPresent(builder::setUtiProprietaryID);
			ofNullable(getValuationAmount()).ifPresent(builder::setValuationAmount);
			ofNullable(getValuationCurrency()).ifPresent(builder::setValuationCurrency);
			ofNullable(getValuationTimestamp()).ifPresent(builder::setValuationTimestamp);
			ofNullable(getValuationMethod()).ifPresent(builder::setValuationMethod);
			ofNullable(getDelta()).ifPresent(builder::setDelta);
			ofNullable(getActionType()).ifPresent(builder::setActionType);
			ofNullable(getTechnicalRecordId()).ifPresent(builder::setTechnicalRecordId);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			JFSAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierType, _that.getCounterparty2IdentifierType())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(newSDRIdentifier, _that.getNewSDRIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(utiProprietaryID, _that.getUtiProprietaryID())) return false;
			if (!Objects.equals(valuationAmount, _that.getValuationAmount())) return false;
			if (!Objects.equals(valuationCurrency, _that.getValuationCurrency())) return false;
			if (!Objects.equals(valuationTimestamp, _that.getValuationTimestamp())) return false;
			if (!Objects.equals(valuationMethod, _that.getValuationMethod())) return false;
			if (!Objects.equals(delta, _that.getDelta())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierType != null ? counterparty2IdentifierType.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (newSDRIdentifier != null ? newSDRIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (utiProprietaryID != null ? utiProprietaryID.hashCode() : 0);
			_result = 31 * _result + (valuationAmount != null ? valuationAmount.hashCode() : 0);
			_result = 31 * _result + (valuationCurrency != null ? valuationCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationTimestamp != null ? valuationTimestamp.hashCode() : 0);
			_result = 31 * _result + (valuationMethod != null ? valuationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (delta != null ? delta.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "JFSAValuationReport {" +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierType=" + this.counterparty2IdentifierType + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"newSDRIdentifier=" + this.newSDRIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"utiProprietaryID=" + this.utiProprietaryID + ", " +
				"valuationAmount=" + this.valuationAmount + ", " +
				"valuationCurrency=" + this.valuationCurrency + ", " +
				"valuationTimestamp=" + this.valuationTimestamp + ", " +
				"valuationMethod=" + this.valuationMethod + ", " +
				"delta=" + this.delta + ", " +
				"actionType=" + this.actionType + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}';
		}
	}

	/*********************** Builder Implementation of JFSAValuationReport  ***********************/
	class JFSAValuationReportBuilderImpl implements JFSAValuationReport.JFSAValuationReportBuilder {
	
		protected ZonedDateTime reportingTimestamp;
		protected String entityResponsibleForReporting;
		protected String counterparty1;
		protected String counterparty2;
		protected Boolean counterparty2IdentifierType;
		protected String submitterIdentifier;
		protected String newSDRIdentifier;
		protected String uniqueTransactionIdentifier;
		protected String utiProprietaryID;
		protected BigDecimal valuationAmount;
		protected ISOCurrencyCodeEnum valuationCurrency;
		protected ZonedDateTime valuationTimestamp;
		protected ValuationType1Code valuationMethod;
		protected BigDecimal delta;
		protected ActionTypeEnum actionType;
		protected String technicalRecordId;
		
		@Override
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		public ZonedDateTime getReportingTimestamp() {
			return reportingTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		public String getEntityResponsibleForReporting() {
			return entityResponsibleForReporting;
		}
		
		@Override
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		public String getCounterparty1() {
			return counterparty1;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		public String getCounterparty2() {
			return counterparty2;
		}
		
		@Override
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		public Boolean getCounterparty2IdentifierType() {
			return counterparty2IdentifierType;
		}
		
		@Override
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		public String getSubmitterIdentifier() {
			return submitterIdentifier;
		}
		
		@Override
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		public String getNewSDRIdentifier() {
			return newSDRIdentifier;
		}
		
		@Override
		@RosettaAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		public String getUniqueTransactionIdentifier() {
			return uniqueTransactionIdentifier;
		}
		
		@Override
		@RosettaAttribute("utiProprietaryID")
		@RuneAttribute("utiProprietaryID")
		public String getUtiProprietaryID() {
			return utiProprietaryID;
		}
		
		@Override
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		public BigDecimal getValuationAmount() {
			return valuationAmount;
		}
		
		@Override
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		public ISOCurrencyCodeEnum getValuationCurrency() {
			return valuationCurrency;
		}
		
		@Override
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		public ZonedDateTime getValuationTimestamp() {
			return valuationTimestamp;
		}
		
		@Override
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		public ValuationType1Code getValuationMethod() {
			return valuationMethod;
		}
		
		@Override
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		public BigDecimal getDelta() {
			return delta;
		}
		
		@Override
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		public ActionTypeEnum getActionType() {
			return actionType;
		}
		
		@Override
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		public String getTechnicalRecordId() {
			return technicalRecordId;
		}
		
		@RosettaAttribute(value="reportingTimestamp", isRequired=true)
		@RuneAttribute(value="reportingTimestamp", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setReportingTimestamp(ZonedDateTime _reportingTimestamp) {
			this.reportingTimestamp = _reportingTimestamp == null ? null : _reportingTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="entityResponsibleForReporting", isRequired=true)
		@RuneAttribute(value="entityResponsibleForReporting", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setEntityResponsibleForReporting(String _entityResponsibleForReporting) {
			this.entityResponsibleForReporting = _entityResponsibleForReporting == null ? null : _entityResponsibleForReporting;
			return this;
		}
		
		@RosettaAttribute(value="counterparty1", isRequired=true)
		@RuneAttribute(value="counterparty1", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty1(String _counterparty1) {
			this.counterparty1 = _counterparty1 == null ? null : _counterparty1;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2", isRequired=true)
		@RuneAttribute(value="counterparty2", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2(String _counterparty2) {
			this.counterparty2 = _counterparty2 == null ? null : _counterparty2;
			return this;
		}
		
		@RosettaAttribute(value="counterparty2IdentifierType", isRequired=true)
		@RuneAttribute(value="counterparty2IdentifierType", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setCounterparty2IdentifierType(Boolean _counterparty2IdentifierType) {
			this.counterparty2IdentifierType = _counterparty2IdentifierType == null ? null : _counterparty2IdentifierType;
			return this;
		}
		
		@RosettaAttribute(value="submitterIdentifier", isRequired=true)
		@RuneAttribute(value="submitterIdentifier", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setSubmitterIdentifier(String _submitterIdentifier) {
			this.submitterIdentifier = _submitterIdentifier == null ? null : _submitterIdentifier;
			return this;
		}
		
		@RosettaAttribute("newSDRIdentifier")
		@RuneAttribute("newSDRIdentifier")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setNewSDRIdentifier(String _newSDRIdentifier) {
			this.newSDRIdentifier = _newSDRIdentifier == null ? null : _newSDRIdentifier;
			return this;
		}
		
		@RosettaAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@RuneAttribute(value="uniqueTransactionIdentifier", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setUniqueTransactionIdentifier(String _uniqueTransactionIdentifier) {
			this.uniqueTransactionIdentifier = _uniqueTransactionIdentifier == null ? null : _uniqueTransactionIdentifier;
			return this;
		}
		
		@RosettaAttribute("utiProprietaryID")
		@RuneAttribute("utiProprietaryID")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setUtiProprietaryID(String _utiProprietaryID) {
			this.utiProprietaryID = _utiProprietaryID == null ? null : _utiProprietaryID;
			return this;
		}
		
		@RosettaAttribute(value="valuationAmount", isRequired=true)
		@RuneAttribute(value="valuationAmount", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationAmount(BigDecimal _valuationAmount) {
			this.valuationAmount = _valuationAmount == null ? null : _valuationAmount;
			return this;
		}
		
		@RosettaAttribute(value="valuationCurrency", isRequired=true)
		@RuneAttribute(value="valuationCurrency", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationCurrency(ISOCurrencyCodeEnum _valuationCurrency) {
			this.valuationCurrency = _valuationCurrency == null ? null : _valuationCurrency;
			return this;
		}
		
		@RosettaAttribute(value="valuationTimestamp", isRequired=true)
		@RuneAttribute(value="valuationTimestamp", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationTimestamp(ZonedDateTime _valuationTimestamp) {
			this.valuationTimestamp = _valuationTimestamp == null ? null : _valuationTimestamp;
			return this;
		}
		
		@RosettaAttribute(value="valuationMethod", isRequired=true)
		@RuneAttribute(value="valuationMethod", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setValuationMethod(ValuationType1Code _valuationMethod) {
			this.valuationMethod = _valuationMethod == null ? null : _valuationMethod;
			return this;
		}
		
		@RosettaAttribute("delta")
		@RuneAttribute("delta")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setDelta(BigDecimal _delta) {
			this.delta = _delta == null ? null : _delta;
			return this;
		}
		
		@RosettaAttribute(value="actionType", isRequired=true)
		@RuneAttribute(value="actionType", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setActionType(ActionTypeEnum _actionType) {
			this.actionType = _actionType == null ? null : _actionType;
			return this;
		}
		
		@RosettaAttribute(value="technicalRecordId", isRequired=true)
		@RuneAttribute(value="technicalRecordId", isRequired=true)
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder setTechnicalRecordId(String _technicalRecordId) {
			this.technicalRecordId = _technicalRecordId == null ? null : _technicalRecordId;
			return this;
		}
		
		@Override
		public JFSAValuationReport build() {
			return new JFSAValuationReport.JFSAValuationReportImpl(this);
		}
		
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder prune() {
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getReportingTimestamp()!=null) return true;
			if (getEntityResponsibleForReporting()!=null) return true;
			if (getCounterparty1()!=null) return true;
			if (getCounterparty2()!=null) return true;
			if (getCounterparty2IdentifierType()!=null) return true;
			if (getSubmitterIdentifier()!=null) return true;
			if (getNewSDRIdentifier()!=null) return true;
			if (getUniqueTransactionIdentifier()!=null) return true;
			if (getUtiProprietaryID()!=null) return true;
			if (getValuationAmount()!=null) return true;
			if (getValuationCurrency()!=null) return true;
			if (getValuationTimestamp()!=null) return true;
			if (getValuationMethod()!=null) return true;
			if (getDelta()!=null) return true;
			if (getActionType()!=null) return true;
			if (getTechnicalRecordId()!=null) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public JFSAValuationReport.JFSAValuationReportBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			JFSAValuationReport.JFSAValuationReportBuilder o = (JFSAValuationReport.JFSAValuationReportBuilder) other;
			
			
			merger.mergeBasic(getReportingTimestamp(), o.getReportingTimestamp(), this::setReportingTimestamp);
			merger.mergeBasic(getEntityResponsibleForReporting(), o.getEntityResponsibleForReporting(), this::setEntityResponsibleForReporting);
			merger.mergeBasic(getCounterparty1(), o.getCounterparty1(), this::setCounterparty1);
			merger.mergeBasic(getCounterparty2(), o.getCounterparty2(), this::setCounterparty2);
			merger.mergeBasic(getCounterparty2IdentifierType(), o.getCounterparty2IdentifierType(), this::setCounterparty2IdentifierType);
			merger.mergeBasic(getSubmitterIdentifier(), o.getSubmitterIdentifier(), this::setSubmitterIdentifier);
			merger.mergeBasic(getNewSDRIdentifier(), o.getNewSDRIdentifier(), this::setNewSDRIdentifier);
			merger.mergeBasic(getUniqueTransactionIdentifier(), o.getUniqueTransactionIdentifier(), this::setUniqueTransactionIdentifier);
			merger.mergeBasic(getUtiProprietaryID(), o.getUtiProprietaryID(), this::setUtiProprietaryID);
			merger.mergeBasic(getValuationAmount(), o.getValuationAmount(), this::setValuationAmount);
			merger.mergeBasic(getValuationCurrency(), o.getValuationCurrency(), this::setValuationCurrency);
			merger.mergeBasic(getValuationTimestamp(), o.getValuationTimestamp(), this::setValuationTimestamp);
			merger.mergeBasic(getValuationMethod(), o.getValuationMethod(), this::setValuationMethod);
			merger.mergeBasic(getDelta(), o.getDelta(), this::setDelta);
			merger.mergeBasic(getActionType(), o.getActionType(), this::setActionType);
			merger.mergeBasic(getTechnicalRecordId(), o.getTechnicalRecordId(), this::setTechnicalRecordId);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			JFSAValuationReport _that = getType().cast(o);
		
			if (!Objects.equals(reportingTimestamp, _that.getReportingTimestamp())) return false;
			if (!Objects.equals(entityResponsibleForReporting, _that.getEntityResponsibleForReporting())) return false;
			if (!Objects.equals(counterparty1, _that.getCounterparty1())) return false;
			if (!Objects.equals(counterparty2, _that.getCounterparty2())) return false;
			if (!Objects.equals(counterparty2IdentifierType, _that.getCounterparty2IdentifierType())) return false;
			if (!Objects.equals(submitterIdentifier, _that.getSubmitterIdentifier())) return false;
			if (!Objects.equals(newSDRIdentifier, _that.getNewSDRIdentifier())) return false;
			if (!Objects.equals(uniqueTransactionIdentifier, _that.getUniqueTransactionIdentifier())) return false;
			if (!Objects.equals(utiProprietaryID, _that.getUtiProprietaryID())) return false;
			if (!Objects.equals(valuationAmount, _that.getValuationAmount())) return false;
			if (!Objects.equals(valuationCurrency, _that.getValuationCurrency())) return false;
			if (!Objects.equals(valuationTimestamp, _that.getValuationTimestamp())) return false;
			if (!Objects.equals(valuationMethod, _that.getValuationMethod())) return false;
			if (!Objects.equals(delta, _that.getDelta())) return false;
			if (!Objects.equals(actionType, _that.getActionType())) return false;
			if (!Objects.equals(technicalRecordId, _that.getTechnicalRecordId())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (reportingTimestamp != null ? reportingTimestamp.hashCode() : 0);
			_result = 31 * _result + (entityResponsibleForReporting != null ? entityResponsibleForReporting.hashCode() : 0);
			_result = 31 * _result + (counterparty1 != null ? counterparty1.hashCode() : 0);
			_result = 31 * _result + (counterparty2 != null ? counterparty2.hashCode() : 0);
			_result = 31 * _result + (counterparty2IdentifierType != null ? counterparty2IdentifierType.hashCode() : 0);
			_result = 31 * _result + (submitterIdentifier != null ? submitterIdentifier.hashCode() : 0);
			_result = 31 * _result + (newSDRIdentifier != null ? newSDRIdentifier.hashCode() : 0);
			_result = 31 * _result + (uniqueTransactionIdentifier != null ? uniqueTransactionIdentifier.hashCode() : 0);
			_result = 31 * _result + (utiProprietaryID != null ? utiProprietaryID.hashCode() : 0);
			_result = 31 * _result + (valuationAmount != null ? valuationAmount.hashCode() : 0);
			_result = 31 * _result + (valuationCurrency != null ? valuationCurrency.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (valuationTimestamp != null ? valuationTimestamp.hashCode() : 0);
			_result = 31 * _result + (valuationMethod != null ? valuationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (delta != null ? delta.hashCode() : 0);
			_result = 31 * _result + (actionType != null ? actionType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (technicalRecordId != null ? technicalRecordId.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "JFSAValuationReportBuilder {" +
				"reportingTimestamp=" + this.reportingTimestamp + ", " +
				"entityResponsibleForReporting=" + this.entityResponsibleForReporting + ", " +
				"counterparty1=" + this.counterparty1 + ", " +
				"counterparty2=" + this.counterparty2 + ", " +
				"counterparty2IdentifierType=" + this.counterparty2IdentifierType + ", " +
				"submitterIdentifier=" + this.submitterIdentifier + ", " +
				"newSDRIdentifier=" + this.newSDRIdentifier + ", " +
				"uniqueTransactionIdentifier=" + this.uniqueTransactionIdentifier + ", " +
				"utiProprietaryID=" + this.utiProprietaryID + ", " +
				"valuationAmount=" + this.valuationAmount + ", " +
				"valuationCurrency=" + this.valuationCurrency + ", " +
				"valuationTimestamp=" + this.valuationTimestamp + ", " +
				"valuationMethod=" + this.valuationMethod + ", " +
				"delta=" + this.delta + ", " +
				"actionType=" + this.actionType + ", " +
				"technicalRecordId=" + this.technicalRecordId +
			'}';
		}
	}
}
