package drr.standards.iosco.cde.version1.price.reports;

import cdm.base.math.ArithmeticOperationEnum;
import cdm.observable.asset.Price;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.reports.ReportFunction;
import drr.base.trade.link.functions.GetPackageInformation;
import drr.regulation.common.TransactionReportInstruction;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(PackageTransactionPriceRule.PackageTransactionPriceRuleDefault.class)
public abstract class PackageTransactionPriceRule implements ReportFunction<TransactionReportInstruction, Price> {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected GetPackageInformation getPackageInformation;

	/**
	* @param input 
	* @return output 
	*/
	@Override
	public Price evaluate(TransactionReportInstruction input) {
		Price.PriceBuilder outputBuilder = doEvaluate(input);
		
		final Price output;
		if (outputBuilder == null) {
			output = null;
		} else {
			output = outputBuilder.build();
			objectValidator.validate(Price.class, output);
		}
		
		return output;
	}

	protected abstract Price.PriceBuilder doEvaluate(TransactionReportInstruction input);

	public static class PackageTransactionPriceRuleDefault extends PackageTransactionPriceRule {
		@Override
		protected Price.PriceBuilder doEvaluate(TransactionReportInstruction input) {
			Price.PriceBuilder output = Price.builder();
			return assignOutput(output, input);
		}
		
		protected Price.PriceBuilder assignOutput(Price.PriceBuilder output, TransactionReportInstruction input) {
			final MapperS<Price> thenArg = MapperS.of(getPackageInformation.evaluate(input)).<Price>map("getPrice", identifiedList -> identifiedList.getPrice());
			output = toBuilder(thenArg
				.mapSingleToItem(item -> {
					if (notExists(item.<ArithmeticOperationEnum>map("getArithmeticOperator", price -> price.getArithmeticOperator())).getOrDefault(false)) {
						return item;
					}
					return MapperS.<Price>ofNull();
				}).get());
			
			return Optional.ofNullable(output)
				.map(o -> o.prune())
				.orElse(null);
		}
	}
}
