package drr.base.trade;

import cdm.event.common.PositionIdentifier;
import cdm.observable.asset.PriceSchedule;
import com.google.common.collect.ImmutableList;
import com.rosetta.model.lib.RosettaModelObject;
import com.rosetta.model.lib.RosettaModelObjectBuilder;
import com.rosetta.model.lib.annotations.RosettaAttribute;
import com.rosetta.model.lib.annotations.RosettaDataType;
import com.rosetta.model.lib.annotations.RuneAttribute;
import com.rosetta.model.lib.annotations.RuneDataType;
import com.rosetta.model.lib.meta.RosettaMetaData;
import com.rosetta.model.lib.path.RosettaPath;
import com.rosetta.model.lib.process.BuilderMerger;
import com.rosetta.model.lib.process.BuilderProcessor;
import com.rosetta.model.lib.process.Processor;
import com.rosetta.model.lib.records.Date;
import com.rosetta.util.ListEquals;
import drr.base.qualification.event.ReportableActionEnum;
import drr.base.trade.basket.CustomBasket;
import drr.base.trade.meta.ReportableInformationBaseMeta;
import drr.enrichment.common.EnrichmentData;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Optional.ofNullable;

/**
 * Specifies the relevant regulators or other supervisory bodies the transaction is relevant for, and regime specific information required relating to the transaction and the parties.
 * @version 7.0.0-dev.66
 */
@RosettaDataType(value="ReportableInformationBase", builder=ReportableInformationBase.ReportableInformationBaseBuilderImpl.class, version="7.0.0-dev.66")
@RuneDataType(value="ReportableInformationBase", model="drr", builder=ReportableInformationBase.ReportableInformationBaseBuilderImpl.class, version="7.0.0-dev.66")
public interface ReportableInformationBase extends RosettaModelObject {

	ReportableInformationBaseMeta metaData = new ReportableInformationBaseMeta();

	/*********************** Getter Methods  ***********************/
	/**
	 * Specifies the method for confirmation of the transacion.
	 */
	ConfirmationMethodEnum getConfirmationMethod();
	/**
	 * Specifies the type of venue where a trade could be executed.
	 */
	ExecutionVenueTypeEnum getExecutionVenueType();
	/**
	 * Specifies whether the contract was entered into as an intragroup transaction, defined in Article 3 of Regulation (EU) No 648/2012.
	 */
	Boolean getIntragroup();
	/**
	 * Indicates whether the transaction is between two affiliated entities. It is referred to as Inter-affiliate under the Canadian CSA reporting regime.
	 */
	Boolean getAffiliated();
	/**
	 * Specifies whether the sender of this trade considers it to be a large notional trade or block trade for reporting purposes, and thus eligible for delayed public reporting. Normally this will only be applicable for off-facility trades.
	 */
	Boolean getLargeSizeTrade();
	/**
	 * Indicator of whether the swap was executed anonymously on a SEF or DCM. When set to true, counterparty details will be masked on reporting.
	 */
	Boolean getSefOrDcmAnonymousIndicator();
	/**
	 * Date and time a transaction was originally executed, resulting in the generation of a new UTI. This data element remains unchanged throughout the life of the UTI.
	 */
	ZonedDateTime getOriginalExecutionTimestamp();
	/**
	 * Indicator whether the derivative is based on crypto-assets.
	 */
	Boolean getCryptoBased();
	/**
	 * Last Available Spot Price of the underlying asset. This price is only needed for a regulatory purpose, that being the reporting of the notional amount for commodity basis products
	 */
	PriceSchedule getUnderlyingAssetLastAvailableSpotPrice();
	/**
	 * Underlying Asset Price Source is only needed for a regulatory purpose, that being the source of the price used to determine the value or level of the asset or benchmark
	 */
	List<? extends ReportablePriceSource> getUnderlyingAssetPriceSource();
	/**
	 * Specifies the regulatory lifecycle event action for the ReportableEvent
	 */
	ReportableActionEnum getReportableAction();
	/**
	 * The UTI of the position in which a derivative transaction is included.
	 */
	PositionIdentifier getSubsequentPositionUTI();
	/**
	 * The prior position UTI (eg in case of a corporate action of type StockIdentifierChange).
	 */
	PositionIdentifier getCounterpartyPositionPriorUTI();
	/**
	 * Information pertaining to Custom Basket
	 */
	CustomBasket getCustomBasket();
	/**
	 * Date as per the contract, by which all transfer of cash or assets should take place and the counterparties should no longer have any outstanding obligations to each other under that contract.
	 */
	Date getFinalContractualSettlementDate();
	EnrichmentData getEnrichment();

	/*********************** Build Methods  ***********************/
	ReportableInformationBase build();
	
	ReportableInformationBase.ReportableInformationBaseBuilder toBuilder();
	
	static ReportableInformationBase.ReportableInformationBaseBuilder builder() {
		return new ReportableInformationBase.ReportableInformationBaseBuilderImpl();
	}

	/*********************** Utility Methods  ***********************/
	@Override
	default RosettaMetaData<? extends ReportableInformationBase> metaData() {
		return metaData;
	}
	
	@Override
	@RuneAttribute("@type")
	default Class<? extends ReportableInformationBase> getType() {
		return ReportableInformationBase.class;
	}
	
	@Override
	default void process(RosettaPath path, Processor processor) {
		processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
		processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
		processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
		processor.processBasic(path.newSubPath("affiliated"), Boolean.class, getAffiliated(), this);
		processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
		processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
		processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
		processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
		processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.class, getUnderlyingAssetLastAvailableSpotPrice());
		processRosetta(path.newSubPath("underlyingAssetPriceSource"), processor, ReportablePriceSource.class, getUnderlyingAssetPriceSource());
		processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
		processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.class, getSubsequentPositionUTI());
		processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.class, getCounterpartyPositionPriorUTI());
		processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.class, getCustomBasket());
		processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
		processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.class, getEnrichment());
	}
	

	/*********************** Builder Interface  ***********************/
	interface ReportableInformationBaseBuilder extends ReportableInformationBase, RosettaModelObjectBuilder {
		PriceSchedule.PriceScheduleBuilder getOrCreateUnderlyingAssetLastAvailableSpotPrice();
		@Override
		PriceSchedule.PriceScheduleBuilder getUnderlyingAssetLastAvailableSpotPrice();
		ReportablePriceSource.ReportablePriceSourceBuilder getOrCreateUnderlyingAssetPriceSource(int index);
		@Override
		List<? extends ReportablePriceSource.ReportablePriceSourceBuilder> getUnderlyingAssetPriceSource();
		PositionIdentifier.PositionIdentifierBuilder getOrCreateSubsequentPositionUTI();
		@Override
		PositionIdentifier.PositionIdentifierBuilder getSubsequentPositionUTI();
		PositionIdentifier.PositionIdentifierBuilder getOrCreateCounterpartyPositionPriorUTI();
		@Override
		PositionIdentifier.PositionIdentifierBuilder getCounterpartyPositionPriorUTI();
		CustomBasket.CustomBasketBuilder getOrCreateCustomBasket();
		@Override
		CustomBasket.CustomBasketBuilder getCustomBasket();
		EnrichmentData.EnrichmentDataBuilder getOrCreateEnrichment();
		@Override
		EnrichmentData.EnrichmentDataBuilder getEnrichment();
		ReportableInformationBase.ReportableInformationBaseBuilder setConfirmationMethod(ConfirmationMethodEnum confirmationMethod);
		ReportableInformationBase.ReportableInformationBaseBuilder setExecutionVenueType(ExecutionVenueTypeEnum executionVenueType);
		ReportableInformationBase.ReportableInformationBaseBuilder setIntragroup(Boolean intragroup);
		ReportableInformationBase.ReportableInformationBaseBuilder setAffiliated(Boolean affiliated);
		ReportableInformationBase.ReportableInformationBaseBuilder setLargeSizeTrade(Boolean largeSizeTrade);
		ReportableInformationBase.ReportableInformationBaseBuilder setSefOrDcmAnonymousIndicator(Boolean sefOrDcmAnonymousIndicator);
		ReportableInformationBase.ReportableInformationBaseBuilder setOriginalExecutionTimestamp(ZonedDateTime originalExecutionTimestamp);
		ReportableInformationBase.ReportableInformationBaseBuilder setCryptoBased(Boolean cryptoBased);
		ReportableInformationBase.ReportableInformationBaseBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule underlyingAssetLastAvailableSpotPrice);
		ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(ReportablePriceSource underlyingAssetPriceSource);
		ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(ReportablePriceSource underlyingAssetPriceSource, int idx);
		ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSource);
		ReportableInformationBase.ReportableInformationBaseBuilder setUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSource);
		ReportableInformationBase.ReportableInformationBaseBuilder setReportableAction(ReportableActionEnum reportableAction);
		ReportableInformationBase.ReportableInformationBaseBuilder setSubsequentPositionUTI(PositionIdentifier subsequentPositionUTI);
		ReportableInformationBase.ReportableInformationBaseBuilder setCounterpartyPositionPriorUTI(PositionIdentifier counterpartyPositionPriorUTI);
		ReportableInformationBase.ReportableInformationBaseBuilder setCustomBasket(CustomBasket customBasket);
		ReportableInformationBase.ReportableInformationBaseBuilder setFinalContractualSettlementDate(Date finalContractualSettlementDate);
		ReportableInformationBase.ReportableInformationBaseBuilder setEnrichment(EnrichmentData enrichment);

		@Override
		default void process(RosettaPath path, BuilderProcessor processor) {
			processor.processBasic(path.newSubPath("confirmationMethod"), ConfirmationMethodEnum.class, getConfirmationMethod(), this);
			processor.processBasic(path.newSubPath("executionVenueType"), ExecutionVenueTypeEnum.class, getExecutionVenueType(), this);
			processor.processBasic(path.newSubPath("intragroup"), Boolean.class, getIntragroup(), this);
			processor.processBasic(path.newSubPath("affiliated"), Boolean.class, getAffiliated(), this);
			processor.processBasic(path.newSubPath("largeSizeTrade"), Boolean.class, getLargeSizeTrade(), this);
			processor.processBasic(path.newSubPath("sefOrDcmAnonymousIndicator"), Boolean.class, getSefOrDcmAnonymousIndicator(), this);
			processor.processBasic(path.newSubPath("originalExecutionTimestamp"), ZonedDateTime.class, getOriginalExecutionTimestamp(), this);
			processor.processBasic(path.newSubPath("cryptoBased"), Boolean.class, getCryptoBased(), this);
			processRosetta(path.newSubPath("underlyingAssetLastAvailableSpotPrice"), processor, PriceSchedule.PriceScheduleBuilder.class, getUnderlyingAssetLastAvailableSpotPrice());
			processRosetta(path.newSubPath("underlyingAssetPriceSource"), processor, ReportablePriceSource.ReportablePriceSourceBuilder.class, getUnderlyingAssetPriceSource());
			processor.processBasic(path.newSubPath("reportableAction"), ReportableActionEnum.class, getReportableAction(), this);
			processRosetta(path.newSubPath("subsequentPositionUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getSubsequentPositionUTI());
			processRosetta(path.newSubPath("counterpartyPositionPriorUTI"), processor, PositionIdentifier.PositionIdentifierBuilder.class, getCounterpartyPositionPriorUTI());
			processRosetta(path.newSubPath("customBasket"), processor, CustomBasket.CustomBasketBuilder.class, getCustomBasket());
			processor.processBasic(path.newSubPath("finalContractualSettlementDate"), Date.class, getFinalContractualSettlementDate(), this);
			processRosetta(path.newSubPath("enrichment"), processor, EnrichmentData.EnrichmentDataBuilder.class, getEnrichment());
		}
		

		ReportableInformationBase.ReportableInformationBaseBuilder prune();
	}

	/*********************** Immutable Implementation of ReportableInformationBase  ***********************/
	class ReportableInformationBaseImpl implements ReportableInformationBase {
		private final ConfirmationMethodEnum confirmationMethod;
		private final ExecutionVenueTypeEnum executionVenueType;
		private final Boolean intragroup;
		private final Boolean affiliated;
		private final Boolean largeSizeTrade;
		private final Boolean sefOrDcmAnonymousIndicator;
		private final ZonedDateTime originalExecutionTimestamp;
		private final Boolean cryptoBased;
		private final PriceSchedule underlyingAssetLastAvailableSpotPrice;
		private final List<? extends ReportablePriceSource> underlyingAssetPriceSource;
		private final ReportableActionEnum reportableAction;
		private final PositionIdentifier subsequentPositionUTI;
		private final PositionIdentifier counterpartyPositionPriorUTI;
		private final CustomBasket customBasket;
		private final Date finalContractualSettlementDate;
		private final EnrichmentData enrichment;
		
		protected ReportableInformationBaseImpl(ReportableInformationBase.ReportableInformationBaseBuilder builder) {
			this.confirmationMethod = builder.getConfirmationMethod();
			this.executionVenueType = builder.getExecutionVenueType();
			this.intragroup = builder.getIntragroup();
			this.affiliated = builder.getAffiliated();
			this.largeSizeTrade = builder.getLargeSizeTrade();
			this.sefOrDcmAnonymousIndicator = builder.getSefOrDcmAnonymousIndicator();
			this.originalExecutionTimestamp = builder.getOriginalExecutionTimestamp();
			this.cryptoBased = builder.getCryptoBased();
			this.underlyingAssetLastAvailableSpotPrice = ofNullable(builder.getUnderlyingAssetLastAvailableSpotPrice()).map(f->f.build()).orElse(null);
			this.underlyingAssetPriceSource = ofNullable(builder.getUnderlyingAssetPriceSource()).filter(_l->!_l.isEmpty()).map(list -> list.stream().filter(Objects::nonNull).map(f->f.build()).filter(Objects::nonNull).collect(ImmutableList.toImmutableList())).orElse(null);
			this.reportableAction = builder.getReportableAction();
			this.subsequentPositionUTI = ofNullable(builder.getSubsequentPositionUTI()).map(f->f.build()).orElse(null);
			this.counterpartyPositionPriorUTI = ofNullable(builder.getCounterpartyPositionPriorUTI()).map(f->f.build()).orElse(null);
			this.customBasket = ofNullable(builder.getCustomBasket()).map(f->f.build()).orElse(null);
			this.finalContractualSettlementDate = builder.getFinalContractualSettlementDate();
			this.enrichment = ofNullable(builder.getEnrichment()).map(f->f.build()).orElse(null);
		}
		
		@Override
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		public ConfirmationMethodEnum getConfirmationMethod() {
			return confirmationMethod;
		}
		
		@Override
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		public ExecutionVenueTypeEnum getExecutionVenueType() {
			return executionVenueType;
		}
		
		@Override
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		public Boolean getIntragroup() {
			return intragroup;
		}
		
		@Override
		@RosettaAttribute("affiliated")
		@RuneAttribute("affiliated")
		public Boolean getAffiliated() {
			return affiliated;
		}
		
		@Override
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		public Boolean getLargeSizeTrade() {
			return largeSizeTrade;
		}
		
		@Override
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		public Boolean getSefOrDcmAnonymousIndicator() {
			return sefOrDcmAnonymousIndicator;
		}
		
		@Override
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		public ZonedDateTime getOriginalExecutionTimestamp() {
			return originalExecutionTimestamp;
		}
		
		@Override
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		public Boolean getCryptoBased() {
			return cryptoBased;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		public PriceSchedule getUnderlyingAssetLastAvailableSpotPrice() {
			return underlyingAssetLastAvailableSpotPrice;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		public List<? extends ReportablePriceSource> getUnderlyingAssetPriceSource() {
			return underlyingAssetPriceSource;
		}
		
		@Override
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		public ReportableActionEnum getReportableAction() {
			return reportableAction;
		}
		
		@Override
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		public PositionIdentifier getSubsequentPositionUTI() {
			return subsequentPositionUTI;
		}
		
		@Override
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		public PositionIdentifier getCounterpartyPositionPriorUTI() {
			return counterpartyPositionPriorUTI;
		}
		
		@Override
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		public CustomBasket getCustomBasket() {
			return customBasket;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public Date getFinalContractualSettlementDate() {
			return finalContractualSettlementDate;
		}
		
		@Override
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		public EnrichmentData getEnrichment() {
			return enrichment;
		}
		
		@Override
		public ReportableInformationBase build() {
			return this;
		}
		
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder toBuilder() {
			ReportableInformationBase.ReportableInformationBaseBuilder builder = builder();
			setBuilderFields(builder);
			return builder;
		}
		
		protected void setBuilderFields(ReportableInformationBase.ReportableInformationBaseBuilder builder) {
			ofNullable(getConfirmationMethod()).ifPresent(builder::setConfirmationMethod);
			ofNullable(getExecutionVenueType()).ifPresent(builder::setExecutionVenueType);
			ofNullable(getIntragroup()).ifPresent(builder::setIntragroup);
			ofNullable(getAffiliated()).ifPresent(builder::setAffiliated);
			ofNullable(getLargeSizeTrade()).ifPresent(builder::setLargeSizeTrade);
			ofNullable(getSefOrDcmAnonymousIndicator()).ifPresent(builder::setSefOrDcmAnonymousIndicator);
			ofNullable(getOriginalExecutionTimestamp()).ifPresent(builder::setOriginalExecutionTimestamp);
			ofNullable(getCryptoBased()).ifPresent(builder::setCryptoBased);
			ofNullable(getUnderlyingAssetLastAvailableSpotPrice()).ifPresent(builder::setUnderlyingAssetLastAvailableSpotPrice);
			ofNullable(getUnderlyingAssetPriceSource()).ifPresent(builder::setUnderlyingAssetPriceSource);
			ofNullable(getReportableAction()).ifPresent(builder::setReportableAction);
			ofNullable(getSubsequentPositionUTI()).ifPresent(builder::setSubsequentPositionUTI);
			ofNullable(getCounterpartyPositionPriorUTI()).ifPresent(builder::setCounterpartyPositionPriorUTI);
			ofNullable(getCustomBasket()).ifPresent(builder::setCustomBasket);
			ofNullable(getFinalContractualSettlementDate()).ifPresent(builder::setFinalContractualSettlementDate);
			ofNullable(getEnrichment()).ifPresent(builder::setEnrichment);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableInformationBase _that = getType().cast(o);
		
			if (!Objects.equals(confirmationMethod, _that.getConfirmationMethod())) return false;
			if (!Objects.equals(executionVenueType, _that.getExecutionVenueType())) return false;
			if (!Objects.equals(intragroup, _that.getIntragroup())) return false;
			if (!Objects.equals(affiliated, _that.getAffiliated())) return false;
			if (!Objects.equals(largeSizeTrade, _that.getLargeSizeTrade())) return false;
			if (!Objects.equals(sefOrDcmAnonymousIndicator, _that.getSefOrDcmAnonymousIndicator())) return false;
			if (!Objects.equals(originalExecutionTimestamp, _that.getOriginalExecutionTimestamp())) return false;
			if (!Objects.equals(cryptoBased, _that.getCryptoBased())) return false;
			if (!Objects.equals(underlyingAssetLastAvailableSpotPrice, _that.getUnderlyingAssetLastAvailableSpotPrice())) return false;
			if (!ListEquals.listEquals(underlyingAssetPriceSource, _that.getUnderlyingAssetPriceSource())) return false;
			if (!Objects.equals(reportableAction, _that.getReportableAction())) return false;
			if (!Objects.equals(subsequentPositionUTI, _that.getSubsequentPositionUTI())) return false;
			if (!Objects.equals(counterpartyPositionPriorUTI, _that.getCounterpartyPositionPriorUTI())) return false;
			if (!Objects.equals(customBasket, _that.getCustomBasket())) return false;
			if (!Objects.equals(finalContractualSettlementDate, _that.getFinalContractualSettlementDate())) return false;
			if (!Objects.equals(enrichment, _that.getEnrichment())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (confirmationMethod != null ? confirmationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (executionVenueType != null ? executionVenueType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (intragroup != null ? intragroup.hashCode() : 0);
			_result = 31 * _result + (affiliated != null ? affiliated.hashCode() : 0);
			_result = 31 * _result + (largeSizeTrade != null ? largeSizeTrade.hashCode() : 0);
			_result = 31 * _result + (sefOrDcmAnonymousIndicator != null ? sefOrDcmAnonymousIndicator.hashCode() : 0);
			_result = 31 * _result + (originalExecutionTimestamp != null ? originalExecutionTimestamp.hashCode() : 0);
			_result = 31 * _result + (cryptoBased != null ? cryptoBased.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetLastAvailableSpotPrice != null ? underlyingAssetLastAvailableSpotPrice.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetPriceSource != null ? underlyingAssetPriceSource.hashCode() : 0);
			_result = 31 * _result + (reportableAction != null ? reportableAction.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (subsequentPositionUTI != null ? subsequentPositionUTI.hashCode() : 0);
			_result = 31 * _result + (counterpartyPositionPriorUTI != null ? counterpartyPositionPriorUTI.hashCode() : 0);
			_result = 31 * _result + (customBasket != null ? customBasket.hashCode() : 0);
			_result = 31 * _result + (finalContractualSettlementDate != null ? finalContractualSettlementDate.hashCode() : 0);
			_result = 31 * _result + (enrichment != null ? enrichment.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformationBase {" +
				"confirmationMethod=" + this.confirmationMethod + ", " +
				"executionVenueType=" + this.executionVenueType + ", " +
				"intragroup=" + this.intragroup + ", " +
				"affiliated=" + this.affiliated + ", " +
				"largeSizeTrade=" + this.largeSizeTrade + ", " +
				"sefOrDcmAnonymousIndicator=" + this.sefOrDcmAnonymousIndicator + ", " +
				"originalExecutionTimestamp=" + this.originalExecutionTimestamp + ", " +
				"cryptoBased=" + this.cryptoBased + ", " +
				"underlyingAssetLastAvailableSpotPrice=" + this.underlyingAssetLastAvailableSpotPrice + ", " +
				"underlyingAssetPriceSource=" + this.underlyingAssetPriceSource + ", " +
				"reportableAction=" + this.reportableAction + ", " +
				"subsequentPositionUTI=" + this.subsequentPositionUTI + ", " +
				"counterpartyPositionPriorUTI=" + this.counterpartyPositionPriorUTI + ", " +
				"customBasket=" + this.customBasket + ", " +
				"finalContractualSettlementDate=" + this.finalContractualSettlementDate + ", " +
				"enrichment=" + this.enrichment +
			'}';
		}
	}

	/*********************** Builder Implementation of ReportableInformationBase  ***********************/
	class ReportableInformationBaseBuilderImpl implements ReportableInformationBase.ReportableInformationBaseBuilder {
	
		protected ConfirmationMethodEnum confirmationMethod;
		protected ExecutionVenueTypeEnum executionVenueType;
		protected Boolean intragroup;
		protected Boolean affiliated;
		protected Boolean largeSizeTrade;
		protected Boolean sefOrDcmAnonymousIndicator;
		protected ZonedDateTime originalExecutionTimestamp;
		protected Boolean cryptoBased;
		protected PriceSchedule.PriceScheduleBuilder underlyingAssetLastAvailableSpotPrice;
		protected List<ReportablePriceSource.ReportablePriceSourceBuilder> underlyingAssetPriceSource = new ArrayList<>();
		protected ReportableActionEnum reportableAction;
		protected PositionIdentifier.PositionIdentifierBuilder subsequentPositionUTI;
		protected PositionIdentifier.PositionIdentifierBuilder counterpartyPositionPriorUTI;
		protected CustomBasket.CustomBasketBuilder customBasket;
		protected Date finalContractualSettlementDate;
		protected EnrichmentData.EnrichmentDataBuilder enrichment;
		
		@Override
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		public ConfirmationMethodEnum getConfirmationMethod() {
			return confirmationMethod;
		}
		
		@Override
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		public ExecutionVenueTypeEnum getExecutionVenueType() {
			return executionVenueType;
		}
		
		@Override
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		public Boolean getIntragroup() {
			return intragroup;
		}
		
		@Override
		@RosettaAttribute("affiliated")
		@RuneAttribute("affiliated")
		public Boolean getAffiliated() {
			return affiliated;
		}
		
		@Override
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		public Boolean getLargeSizeTrade() {
			return largeSizeTrade;
		}
		
		@Override
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		public Boolean getSefOrDcmAnonymousIndicator() {
			return sefOrDcmAnonymousIndicator;
		}
		
		@Override
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		public ZonedDateTime getOriginalExecutionTimestamp() {
			return originalExecutionTimestamp;
		}
		
		@Override
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		public Boolean getCryptoBased() {
			return cryptoBased;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		public PriceSchedule.PriceScheduleBuilder getUnderlyingAssetLastAvailableSpotPrice() {
			return underlyingAssetLastAvailableSpotPrice;
		}
		
		@Override
		public PriceSchedule.PriceScheduleBuilder getOrCreateUnderlyingAssetLastAvailableSpotPrice() {
			PriceSchedule.PriceScheduleBuilder result;
			if (underlyingAssetLastAvailableSpotPrice!=null) {
				result = underlyingAssetLastAvailableSpotPrice;
			}
			else {
				result = underlyingAssetLastAvailableSpotPrice = PriceSchedule.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		public List<? extends ReportablePriceSource.ReportablePriceSourceBuilder> getUnderlyingAssetPriceSource() {
			return underlyingAssetPriceSource;
		}
		
		@Override
		public ReportablePriceSource.ReportablePriceSourceBuilder getOrCreateUnderlyingAssetPriceSource(int index) {
			if (underlyingAssetPriceSource==null) {
				this.underlyingAssetPriceSource = new ArrayList<>();
			}
			return getIndex(underlyingAssetPriceSource, index, () -> {
						ReportablePriceSource.ReportablePriceSourceBuilder newUnderlyingAssetPriceSource = ReportablePriceSource.builder();
						return newUnderlyingAssetPriceSource;
					});
		}
		
		@Override
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		public ReportableActionEnum getReportableAction() {
			return reportableAction;
		}
		
		@Override
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		public PositionIdentifier.PositionIdentifierBuilder getSubsequentPositionUTI() {
			return subsequentPositionUTI;
		}
		
		@Override
		public PositionIdentifier.PositionIdentifierBuilder getOrCreateSubsequentPositionUTI() {
			PositionIdentifier.PositionIdentifierBuilder result;
			if (subsequentPositionUTI!=null) {
				result = subsequentPositionUTI;
			}
			else {
				result = subsequentPositionUTI = PositionIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		public PositionIdentifier.PositionIdentifierBuilder getCounterpartyPositionPriorUTI() {
			return counterpartyPositionPriorUTI;
		}
		
		@Override
		public PositionIdentifier.PositionIdentifierBuilder getOrCreateCounterpartyPositionPriorUTI() {
			PositionIdentifier.PositionIdentifierBuilder result;
			if (counterpartyPositionPriorUTI!=null) {
				result = counterpartyPositionPriorUTI;
			}
			else {
				result = counterpartyPositionPriorUTI = PositionIdentifier.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		public CustomBasket.CustomBasketBuilder getCustomBasket() {
			return customBasket;
		}
		
		@Override
		public CustomBasket.CustomBasketBuilder getOrCreateCustomBasket() {
			CustomBasket.CustomBasketBuilder result;
			if (customBasket!=null) {
				result = customBasket;
			}
			else {
				result = customBasket = CustomBasket.builder();
			}
			
			return result;
		}
		
		@Override
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		public Date getFinalContractualSettlementDate() {
			return finalContractualSettlementDate;
		}
		
		@Override
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		public EnrichmentData.EnrichmentDataBuilder getEnrichment() {
			return enrichment;
		}
		
		@Override
		public EnrichmentData.EnrichmentDataBuilder getOrCreateEnrichment() {
			EnrichmentData.EnrichmentDataBuilder result;
			if (enrichment!=null) {
				result = enrichment;
			}
			else {
				result = enrichment = EnrichmentData.builder();
			}
			
			return result;
		}
		
		@RosettaAttribute("confirmationMethod")
		@RuneAttribute("confirmationMethod")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setConfirmationMethod(ConfirmationMethodEnum _confirmationMethod) {
			this.confirmationMethod = _confirmationMethod == null ? null : _confirmationMethod;
			return this;
		}
		
		@RosettaAttribute(value="executionVenueType", isRequired=true)
		@RuneAttribute(value="executionVenueType", isRequired=true)
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setExecutionVenueType(ExecutionVenueTypeEnum _executionVenueType) {
			this.executionVenueType = _executionVenueType == null ? null : _executionVenueType;
			return this;
		}
		
		@RosettaAttribute("intragroup")
		@RuneAttribute("intragroup")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setIntragroup(Boolean _intragroup) {
			this.intragroup = _intragroup == null ? null : _intragroup;
			return this;
		}
		
		@RosettaAttribute("affiliated")
		@RuneAttribute("affiliated")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setAffiliated(Boolean _affiliated) {
			this.affiliated = _affiliated == null ? null : _affiliated;
			return this;
		}
		
		@RosettaAttribute("largeSizeTrade")
		@RuneAttribute("largeSizeTrade")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setLargeSizeTrade(Boolean _largeSizeTrade) {
			this.largeSizeTrade = _largeSizeTrade == null ? null : _largeSizeTrade;
			return this;
		}
		
		@RosettaAttribute("sefOrDcmAnonymousIndicator")
		@RuneAttribute("sefOrDcmAnonymousIndicator")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setSefOrDcmAnonymousIndicator(Boolean _sefOrDcmAnonymousIndicator) {
			this.sefOrDcmAnonymousIndicator = _sefOrDcmAnonymousIndicator == null ? null : _sefOrDcmAnonymousIndicator;
			return this;
		}
		
		@RosettaAttribute("originalExecutionTimestamp")
		@RuneAttribute("originalExecutionTimestamp")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setOriginalExecutionTimestamp(ZonedDateTime _originalExecutionTimestamp) {
			this.originalExecutionTimestamp = _originalExecutionTimestamp == null ? null : _originalExecutionTimestamp;
			return this;
		}
		
		@RosettaAttribute("cryptoBased")
		@RuneAttribute("cryptoBased")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setCryptoBased(Boolean _cryptoBased) {
			this.cryptoBased = _cryptoBased == null ? null : _cryptoBased;
			return this;
		}
		
		@RosettaAttribute("underlyingAssetLastAvailableSpotPrice")
		@RuneAttribute("underlyingAssetLastAvailableSpotPrice")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setUnderlyingAssetLastAvailableSpotPrice(PriceSchedule _underlyingAssetLastAvailableSpotPrice) {
			this.underlyingAssetLastAvailableSpotPrice = _underlyingAssetLastAvailableSpotPrice == null ? null : _underlyingAssetLastAvailableSpotPrice.toBuilder();
			return this;
		}
		
		@RosettaAttribute("underlyingAssetPriceSource")
		@RuneAttribute("underlyingAssetPriceSource")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(ReportablePriceSource _underlyingAssetPriceSource) {
			if (_underlyingAssetPriceSource != null) {
				this.underlyingAssetPriceSource.add(_underlyingAssetPriceSource.toBuilder());
			}
			return this;
		}
		
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(ReportablePriceSource _underlyingAssetPriceSource, int idx) {
			getIndex(this.underlyingAssetPriceSource, idx, () -> _underlyingAssetPriceSource.toBuilder());
			return this;
		}
		
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder addUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSources) {
			if (underlyingAssetPriceSources != null) {
				for (final ReportablePriceSource toAdd : underlyingAssetPriceSources) {
					this.underlyingAssetPriceSource.add(toAdd.toBuilder());
				}
			}
			return this;
		}
		
		@RuneAttribute("underlyingAssetPriceSource")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setUnderlyingAssetPriceSource(List<? extends ReportablePriceSource> underlyingAssetPriceSources) {
			if (underlyingAssetPriceSources == null) {
				this.underlyingAssetPriceSource = new ArrayList<>();
			} else {
				this.underlyingAssetPriceSource = underlyingAssetPriceSources.stream()
					.map(_a->_a.toBuilder())
					.collect(Collectors.toCollection(()->new ArrayList<>()));
			}
			return this;
		}
		
		@RosettaAttribute("reportableAction")
		@RuneAttribute("reportableAction")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setReportableAction(ReportableActionEnum _reportableAction) {
			this.reportableAction = _reportableAction == null ? null : _reportableAction;
			return this;
		}
		
		@RosettaAttribute("subsequentPositionUTI")
		@RuneAttribute("subsequentPositionUTI")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setSubsequentPositionUTI(PositionIdentifier _subsequentPositionUTI) {
			this.subsequentPositionUTI = _subsequentPositionUTI == null ? null : _subsequentPositionUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("counterpartyPositionPriorUTI")
		@RuneAttribute("counterpartyPositionPriorUTI")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setCounterpartyPositionPriorUTI(PositionIdentifier _counterpartyPositionPriorUTI) {
			this.counterpartyPositionPriorUTI = _counterpartyPositionPriorUTI == null ? null : _counterpartyPositionPriorUTI.toBuilder();
			return this;
		}
		
		@RosettaAttribute("customBasket")
		@RuneAttribute("customBasket")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setCustomBasket(CustomBasket _customBasket) {
			this.customBasket = _customBasket == null ? null : _customBasket.toBuilder();
			return this;
		}
		
		@RosettaAttribute("finalContractualSettlementDate")
		@RuneAttribute("finalContractualSettlementDate")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setFinalContractualSettlementDate(Date _finalContractualSettlementDate) {
			this.finalContractualSettlementDate = _finalContractualSettlementDate == null ? null : _finalContractualSettlementDate;
			return this;
		}
		
		@RosettaAttribute("enrichment")
		@RuneAttribute("enrichment")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder setEnrichment(EnrichmentData _enrichment) {
			this.enrichment = _enrichment == null ? null : _enrichment.toBuilder();
			return this;
		}
		
		@Override
		public ReportableInformationBase build() {
			return new ReportableInformationBase.ReportableInformationBaseImpl(this);
		}
		
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder toBuilder() {
			return this;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder prune() {
			if (underlyingAssetLastAvailableSpotPrice!=null && !underlyingAssetLastAvailableSpotPrice.prune().hasData()) underlyingAssetLastAvailableSpotPrice = null;
			underlyingAssetPriceSource = underlyingAssetPriceSource.stream().filter(b->b!=null).<ReportablePriceSource.ReportablePriceSourceBuilder>map(b->b.prune()).filter(b->b.hasData()).collect(Collectors.toList());
			if (subsequentPositionUTI!=null && !subsequentPositionUTI.prune().hasData()) subsequentPositionUTI = null;
			if (counterpartyPositionPriorUTI!=null && !counterpartyPositionPriorUTI.prune().hasData()) counterpartyPositionPriorUTI = null;
			if (customBasket!=null && !customBasket.prune().hasData()) customBasket = null;
			if (enrichment!=null && !enrichment.prune().hasData()) enrichment = null;
			return this;
		}
		
		@Override
		public boolean hasData() {
			if (getConfirmationMethod()!=null) return true;
			if (getExecutionVenueType()!=null) return true;
			if (getIntragroup()!=null) return true;
			if (getAffiliated()!=null) return true;
			if (getLargeSizeTrade()!=null) return true;
			if (getSefOrDcmAnonymousIndicator()!=null) return true;
			if (getOriginalExecutionTimestamp()!=null) return true;
			if (getCryptoBased()!=null) return true;
			if (getUnderlyingAssetLastAvailableSpotPrice()!=null && getUnderlyingAssetLastAvailableSpotPrice().hasData()) return true;
			if (getUnderlyingAssetPriceSource()!=null && getUnderlyingAssetPriceSource().stream().filter(Objects::nonNull).anyMatch(a->a.hasData())) return true;
			if (getReportableAction()!=null) return true;
			if (getSubsequentPositionUTI()!=null && getSubsequentPositionUTI().hasData()) return true;
			if (getCounterpartyPositionPriorUTI()!=null && getCounterpartyPositionPriorUTI().hasData()) return true;
			if (getCustomBasket()!=null && getCustomBasket().hasData()) return true;
			if (getFinalContractualSettlementDate()!=null) return true;
			if (getEnrichment()!=null && getEnrichment().hasData()) return true;
			return false;
		}
	
		@SuppressWarnings("unchecked")
		@Override
		public ReportableInformationBase.ReportableInformationBaseBuilder merge(RosettaModelObjectBuilder other, BuilderMerger merger) {
			ReportableInformationBase.ReportableInformationBaseBuilder o = (ReportableInformationBase.ReportableInformationBaseBuilder) other;
			
			merger.mergeRosetta(getUnderlyingAssetLastAvailableSpotPrice(), o.getUnderlyingAssetLastAvailableSpotPrice(), this::setUnderlyingAssetLastAvailableSpotPrice);
			merger.mergeRosetta(getUnderlyingAssetPriceSource(), o.getUnderlyingAssetPriceSource(), this::getOrCreateUnderlyingAssetPriceSource);
			merger.mergeRosetta(getSubsequentPositionUTI(), o.getSubsequentPositionUTI(), this::setSubsequentPositionUTI);
			merger.mergeRosetta(getCounterpartyPositionPriorUTI(), o.getCounterpartyPositionPriorUTI(), this::setCounterpartyPositionPriorUTI);
			merger.mergeRosetta(getCustomBasket(), o.getCustomBasket(), this::setCustomBasket);
			merger.mergeRosetta(getEnrichment(), o.getEnrichment(), this::setEnrichment);
			
			merger.mergeBasic(getConfirmationMethod(), o.getConfirmationMethod(), this::setConfirmationMethod);
			merger.mergeBasic(getExecutionVenueType(), o.getExecutionVenueType(), this::setExecutionVenueType);
			merger.mergeBasic(getIntragroup(), o.getIntragroup(), this::setIntragroup);
			merger.mergeBasic(getAffiliated(), o.getAffiliated(), this::setAffiliated);
			merger.mergeBasic(getLargeSizeTrade(), o.getLargeSizeTrade(), this::setLargeSizeTrade);
			merger.mergeBasic(getSefOrDcmAnonymousIndicator(), o.getSefOrDcmAnonymousIndicator(), this::setSefOrDcmAnonymousIndicator);
			merger.mergeBasic(getOriginalExecutionTimestamp(), o.getOriginalExecutionTimestamp(), this::setOriginalExecutionTimestamp);
			merger.mergeBasic(getCryptoBased(), o.getCryptoBased(), this::setCryptoBased);
			merger.mergeBasic(getReportableAction(), o.getReportableAction(), this::setReportableAction);
			merger.mergeBasic(getFinalContractualSettlementDate(), o.getFinalContractualSettlementDate(), this::setFinalContractualSettlementDate);
			return this;
		}
	
		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || !(o instanceof RosettaModelObject) || !getType().equals(((RosettaModelObject)o).getType())) return false;
		
			ReportableInformationBase _that = getType().cast(o);
		
			if (!Objects.equals(confirmationMethod, _that.getConfirmationMethod())) return false;
			if (!Objects.equals(executionVenueType, _that.getExecutionVenueType())) return false;
			if (!Objects.equals(intragroup, _that.getIntragroup())) return false;
			if (!Objects.equals(affiliated, _that.getAffiliated())) return false;
			if (!Objects.equals(largeSizeTrade, _that.getLargeSizeTrade())) return false;
			if (!Objects.equals(sefOrDcmAnonymousIndicator, _that.getSefOrDcmAnonymousIndicator())) return false;
			if (!Objects.equals(originalExecutionTimestamp, _that.getOriginalExecutionTimestamp())) return false;
			if (!Objects.equals(cryptoBased, _that.getCryptoBased())) return false;
			if (!Objects.equals(underlyingAssetLastAvailableSpotPrice, _that.getUnderlyingAssetLastAvailableSpotPrice())) return false;
			if (!ListEquals.listEquals(underlyingAssetPriceSource, _that.getUnderlyingAssetPriceSource())) return false;
			if (!Objects.equals(reportableAction, _that.getReportableAction())) return false;
			if (!Objects.equals(subsequentPositionUTI, _that.getSubsequentPositionUTI())) return false;
			if (!Objects.equals(counterpartyPositionPriorUTI, _that.getCounterpartyPositionPriorUTI())) return false;
			if (!Objects.equals(customBasket, _that.getCustomBasket())) return false;
			if (!Objects.equals(finalContractualSettlementDate, _that.getFinalContractualSettlementDate())) return false;
			if (!Objects.equals(enrichment, _that.getEnrichment())) return false;
			return true;
		}
		
		@Override
		public int hashCode() {
			int _result = 0;
			_result = 31 * _result + (confirmationMethod != null ? confirmationMethod.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (executionVenueType != null ? executionVenueType.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (intragroup != null ? intragroup.hashCode() : 0);
			_result = 31 * _result + (affiliated != null ? affiliated.hashCode() : 0);
			_result = 31 * _result + (largeSizeTrade != null ? largeSizeTrade.hashCode() : 0);
			_result = 31 * _result + (sefOrDcmAnonymousIndicator != null ? sefOrDcmAnonymousIndicator.hashCode() : 0);
			_result = 31 * _result + (originalExecutionTimestamp != null ? originalExecutionTimestamp.hashCode() : 0);
			_result = 31 * _result + (cryptoBased != null ? cryptoBased.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetLastAvailableSpotPrice != null ? underlyingAssetLastAvailableSpotPrice.hashCode() : 0);
			_result = 31 * _result + (underlyingAssetPriceSource != null ? underlyingAssetPriceSource.hashCode() : 0);
			_result = 31 * _result + (reportableAction != null ? reportableAction.getClass().getName().hashCode() : 0);
			_result = 31 * _result + (subsequentPositionUTI != null ? subsequentPositionUTI.hashCode() : 0);
			_result = 31 * _result + (counterpartyPositionPriorUTI != null ? counterpartyPositionPriorUTI.hashCode() : 0);
			_result = 31 * _result + (customBasket != null ? customBasket.hashCode() : 0);
			_result = 31 * _result + (finalContractualSettlementDate != null ? finalContractualSettlementDate.hashCode() : 0);
			_result = 31 * _result + (enrichment != null ? enrichment.hashCode() : 0);
			return _result;
		}
		
		@Override
		public String toString() {
			return "ReportableInformationBaseBuilder {" +
				"confirmationMethod=" + this.confirmationMethod + ", " +
				"executionVenueType=" + this.executionVenueType + ", " +
				"intragroup=" + this.intragroup + ", " +
				"affiliated=" + this.affiliated + ", " +
				"largeSizeTrade=" + this.largeSizeTrade + ", " +
				"sefOrDcmAnonymousIndicator=" + this.sefOrDcmAnonymousIndicator + ", " +
				"originalExecutionTimestamp=" + this.originalExecutionTimestamp + ", " +
				"cryptoBased=" + this.cryptoBased + ", " +
				"underlyingAssetLastAvailableSpotPrice=" + this.underlyingAssetLastAvailableSpotPrice + ", " +
				"underlyingAssetPriceSource=" + this.underlyingAssetPriceSource + ", " +
				"reportableAction=" + this.reportableAction + ", " +
				"subsequentPositionUTI=" + this.subsequentPositionUTI + ", " +
				"counterpartyPositionPriorUTI=" + this.counterpartyPositionPriorUTI + ", " +
				"customBasket=" + this.customBasket + ", " +
				"finalContractualSettlementDate=" + this.finalContractualSettlementDate + ", " +
				"enrichment=" + this.enrichment +
			'}';
		}
	}
}
